/*
 Copyright (C) 1998       Gerald L. Gay
 
 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Library General Public License 
 version 2 as published by the Free Software Foundation.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Library General Public License for more details.

 You should have received a copy of the GNU Library General Public
 License along with this library; see the file COPYING.  If not,
 write to the Free Software Foundation, Inc., 675 Mass Ave, 
 Cambridge, MA 02139, USA.
*/

/* The ToolTip functions are actually independent of the Xarm
 * library.  They will work with minor modification on their own.
 * They are implemented here in the WObjectClass because this is
 * where all Widget objects start.  Any widget can have a tooltip,
 * not just buttons, although that is the most common application.
 *
 * The functions that don't need to be visible outside of this file
 * are declared static and aren't even mentioned in WObjectClass. This
 * keeps the implementation hidden as much as possible.
 */

#include <Xarm/Base.h>
#include <Xm/Label.h>

/* The ToolTip data structure */

typedef struct _TOOL_TIP_STRUCT_
{
    Widget widget;
    XtPointer data;
    XtIntervalId timer_id;
    unsigned long timeout;
    unsigned int margin;
    Bool show;
    Widget window;
    Boolean timer_fired;
    Boolean unmap_event_handlers_registered;
}ToolTip, *ToolTipPtr;

/* Only one is needed. */

static ToolTip *tool_tip = NULL;

/* Internal function prototypes. */

static void CreateTipStuff(Widget widget);
static Widget ToolTipPopup(Widget widget, char *tip_text);
static void motion_timeout_cb(XtPointer client_data, XtIntervalId *timer_id);
static void add_unmap_event_handlers(XtPointer tip_data);
static void remove_unmap_event_handlers(XtPointer tip_data);

static void buttonPress(Widget widget, XtPointer client_data, XEvent *xevent,
                        Boolean *continue_to_dispatch);
static void pointerEnter(Widget widget, XtPointer client_data, XEvent *xevent,
                         Boolean *continue_to_dispatch);
static void pointerLeave(Widget widget, XtPointer client_data, XEvent *xevent,
                         Boolean *continue_to_dispatch);
static void widgetUnmap(Widget widget, XtPointer client_data, XEvent *xevent,
                        Boolean *continue_to_dispatch);

static void widgetDestroy(Widget widget, XtPointer client_data, XtPointer cb_data);
static int widgetViewable(Widget widget);

static void free_tip_data(Widget widget, XtPointer client_data, XtPointer cb_data);
static bool ToolTipIsInitialized();



/* This function is used to determine if the user has
 * specified ToolTip resources in an AppDefaults file.
 */

static void GetAppResource(Widget widget,
                    String resource_name,
                    String resource_class,
                    String resource_type,
                    Cardinal resource_size,
                    String default_type,
                    XtPointer default_value,
                    XtPointer app_value,
                    XtPointer base)
{
    XtResource resource;
    Arg arg;
    int nargs = 0;

    resource.resource_name = resource_name;
    resource.resource_class = resource_class;
    resource.resource_type = resource_type;
    resource.resource_size = resource_size;
    resource.resource_offset = 0;
    resource.default_type = default_type;
    resource.default_addr = default_value;

    if (app_value)
    {
        arg.name = resource_name;
        arg.value = (int)app_value;
	nargs = 1;
    }

    if ( base != NULL )
    {
        XtGetApplicationResources(widget,
            base,
            &resource,
            1,
	    &arg,
	    nargs);
    }
}

static void remove_old_tooltip(void)
{
    if (!tool_tip->timer_fired)
    {
        XtRemoveTimeOut(tool_tip->timer_id);
        tool_tip->timer_fired = True;
    }

    if (tool_tip->window)
    {
        XtDestroyWidget(tool_tip->window);
        tool_tip->window = NULL;
    }

    remove_unmap_event_handlers(tool_tip->data);
}


static void CreateTipStuff(Widget widget)
{

    GetAppResource(widget, "tipTimeout", "TipTimeout",
                   XtRInt, sizeof(unsigned long),
                   XtRImmediate, (XtPointer)1000,
                   (XtPointer)NULL, (XtPointer)&tool_tip->timeout);

    GetAppResource(widget, "tipShow", "TipShow",
                   XtRBool, sizeof(Bool),
                   XtRImmediate, (XtPointer)True,
                   (XtPointer)NULL, (XtPointer)&tool_tip->show);

    GetAppResource(widget, "tipMargin", "TipMargin",
                   XtRInt, sizeof(unsigned int),
                   XtRImmediate, (XtPointer)2,
                   (XtPointer)NULL, (XtPointer)&tool_tip->margin);
}


static Widget ToolTipPopup(Widget widget, char *tip_text)
{
    Widget tipShell = NULL;
    Widget tipLabel;
    XmString xmStr;
    Position x, y;
    Position abs_x, abs_y;
    Dimension width, height, border_width;
    Dimension lab_width, lab_height, lab_border;
    Screen *screenPtr;


    screenPtr = XtScreen(widget);
    if (tool_tip->show && tip_text && !widget->core.being_destroyed)
    {
        XtVaGetValues(widget,
            XmNx, &x,
            XmNy, &y,
            XmNwidth, &width,
            XmNheight, &height,
            XmNborderWidth, &border_width,
            NULL);

        XtTranslateCoords(widget, 0, 0, &abs_x, &abs_y);

        tipShell = XtVaAppCreateShell("toolTipShell", "ToolTipShell",
                                      wmShellWidgetClass, XtDisplay(widget),
          XmNoverrideRedirect, True,
          XmNsaveUnder,        True,
          XmNallowShellResize, True,
          XmNx, abs_x,
          XmNy, abs_y + height + border_width,
          NULL);

	xmStr = XmStringCreateLocalized(tip_text);

	tipLabel = XtVaCreateManagedWidget("toolTipLabel", xmLabelWidgetClass, tipShell,
					   XmNlabelString, xmStr,
					   NULL);

        XtVaGetValues(tipLabel,
                      XmNwidth, &lab_width,
                      XmNheight, &lab_height,
                      XmNborderWidth, &lab_border,
                      NULL);

        lab_width = lab_width + (lab_border * 2);
        lab_height = lab_height + (lab_border * 2);

        if ((lab_width + abs_x + border_width) > screenPtr->width) {
            XtVaSetValues(tipShell, XmNx, screenPtr->width - lab_width, NULL);
        }

        if ((lab_height + abs_y + border_width + height) > screenPtr->height) {
            XtVaSetValues(tipShell, XmNy, abs_y - lab_height, NULL);
        }

	XmStringFree(xmStr);

        XtPopup(tipShell, XtGrabNone);

    }

    return (tipShell);
}


static void motion_timeout_cb(XtPointer client_data, XtIntervalId *timer_id)
{
    tool_tip->timer_fired = True;

    if (tool_tip->window)
    {
        XtDestroyWidget(tool_tip->window);
        tool_tip->window = NULL;
        remove_unmap_event_handlers(tool_tip->data);
    }

    tool_tip->window = ToolTipPopup(tool_tip->widget, (char *)tool_tip->data);
}


bool WObjectClass::toolTipEnable()
{
    bool show = false;

    if (ToolTipIsInitialized())
    {
        show = tool_tip->show;
    }

    return(show);
}


bool WObjectClass::toolTipEnable(const bool show)
{
    bool old_show = false;

    if (ToolTipIsInitialized())
    {
        old_show = tool_tip->show;
        tool_tip->show = show;
        if (!tool_tip->show && tool_tip->window)
        {
            XtDestroyWidget(tool_tip->window);
            tool_tip->window = NULL;
        }
    }

    return(old_show);
}


unsigned int WObjectClass::toolTipMargin(const unsigned int margin)
{
    unsigned int old_margin = 0;

    if (ToolTipIsInitialized())
    {
        old_margin = tool_tip->margin;
        tool_tip->margin = margin;
    }

    return(old_margin);
}

unsigned int WObjectClass::toolTipMargin()
{
    unsigned int margin = 0;

    if (ToolTipIsInitialized())
    {
        margin = tool_tip->margin;
    }

    return(margin);
}

unsigned long WObjectClass::toolTipTimeout(const unsigned long timeout)
{
    unsigned long old_timeout = 0;

    if (ToolTipIsInitialized())
    {
        old_timeout = tool_tip->timeout;
        tool_tip->timeout = timeout;
    }

    return(old_timeout);
}

unsigned long WObjectClass::toolTipTimeout()
{
    unsigned long timeout = 0;

    if (ToolTipIsInitialized())
    {
        timeout = tool_tip->timeout;
    }

    return(timeout);
}

static bool ToolTipIsInitialized(void)
{
    bool b;

    if (tool_tip)
        b = true;
    else
        b = false;

    return(b);
}


ToolTipPtr ToolTipInitialize(Widget widget)
{
    Widget tipShell;

    if (!ToolTipIsInitialized())
    {
        tool_tip = (ToolTipPtr)calloc(1, sizeof(ToolTip));
        tool_tip->timeout = 1000;
        tool_tip->show = True;
        tool_tip->margin = 5;
        tool_tip->timer_fired = True;
        tool_tip->unmap_event_handlers_registered = False;

        tipShell = XtVaAppCreateShell("toolTipShell", "ToolTipShell",
                                      wmShellWidgetClass, XtDisplay(widget),
          XmNoverrideRedirect, True,
          XmNsaveUnder,        True,
          XmNallowShellResize, True,
          NULL);

        CreateTipStuff(tipShell);

        XtDestroyWidget(tipShell);
    }

    return(tool_tip);
}


void WObjectClass::addToolTip(const char *tip_text)
{

   if (tipData != NULL) removeToolTip();

   tipData = (XtPointer)strdup(tip_text);

   if (tipData == NULL) return;

   if (!ToolTipIsInitialized())
       ToolTipInitialize(wid);

   XtAddEventHandler(wid, ButtonPressMask, False,
     buttonPress, tipData);
   XtAddEventHandler(wid, EnterWindowMask, False,
     pointerEnter, tipData);
   XtAddEventHandler(wid, LeaveWindowMask, False,
     pointerLeave, tipData);

   XtAddCallback(wid, XtNdestroyCallback,
                 free_tip_data, tipData);
}

void WObjectClass::removeToolTip()
{
    if (ToolTipIsInitialized() && (tipData != NULL))
    {
        XtRemoveEventHandler(wid, ButtonPressMask, False,
          buttonPress, tipData);
        XtRemoveEventHandler(wid, EnterWindowMask, False,
          pointerEnter, tipData);
        XtRemoveEventHandler(wid, LeaveWindowMask, False,
          pointerLeave, tipData);
        if (wid == tool_tip->widget && tool_tip->window)
        {
            XtDestroyWidget(tool_tip->window);
            tool_tip->window = NULL;
            remove_unmap_event_handlers(tool_tip->data);
        }
    }
}

static void buttonPress (Widget widget, XtPointer client_data, XEvent *xevent,
                         Boolean *continue_to_dispatch)
{
    remove_old_tooltip();
}

static void pointerEnter (Widget widget, XtPointer client_data, XEvent *xevent,
                          Boolean *continue_to_dispatch)
{
    XCrossingEvent *event = (XCrossingEvent *)xevent;

/* Note:  We had better not get two Pointer Enter events consecutively! */

    if (event->mode == NotifyNormal)
    {
        if (!tool_tip->timer_fired)
        {
            XtRemoveTimeOut(tool_tip->timer_id);
            tool_tip->timer_fired = True;
        }

        if (widgetViewable(widget))
        {
            tool_tip->timer_fired = False;
            tool_tip->widget = widget;
            tool_tip->data = client_data;
            tool_tip->timer_id = XtAppAddTimeOut(
                                   XtWidgetToApplicationContext(widget),
                                   tool_tip->timeout,
                                   motion_timeout_cb, client_data);
            add_unmap_event_handlers(tool_tip->data);
        }
    }
    else
    {
        remove_old_tooltip();
    }
}

static void pointerLeave (Widget widget, XtPointer client_data, XEvent *xevent,
                          Boolean *continue_to_dispatch)
{
    XCrossingEvent *event = (XCrossingEvent *)xevent;

    if (event->mode == NotifyNormal)
    {
        remove_old_tooltip();
    }
}

static void widgetUnmap (Widget widget, XtPointer client_data, XEvent *xevent,
                         Boolean *continue_to_dispatch)
{
    if (xevent->type == UnmapNotify)
    {
        remove_old_tooltip();
    }
}

static int widgetViewable(Widget widget)
{
    int viewable_status = FALSE;
    Status attr_status;
    XWindowAttributes window_attributes;

    attr_status = XGetWindowAttributes(XtDisplay(widget),
                                       XtWindow(widget),
                                       &window_attributes);

    if (attr_status)
        viewable_status = window_attributes.map_state == IsViewable ? TRUE :
                                                                      FALSE;

    return(viewable_status);
}

static void free_tip_data(Widget w, XtPointer client_data, XtPointer cb_data)
{
    free(client_data);
}

static void widgetDestroy(Widget w, XtPointer client_data, XtPointer cb_data)
{
    remove_old_tooltip();
}

static void add_unmap_event_handlers(XtPointer tip_data)
{
    Widget parent;

    if (!tool_tip->unmap_event_handlers_registered)
    {

        tool_tip->unmap_event_handlers_registered = True;
        if (!tool_tip->widget->core.being_destroyed)
        {
            XtAddCallback(tool_tip->widget, XtNdestroyCallback, widgetDestroy,
              tip_data);
            XtAddEventHandler(tool_tip->widget, StructureNotifyMask, False,
              widgetUnmap, tip_data);
        }
        parent = XtParent(tool_tip->widget);
        while (parent)
        {
            if (!parent->core.being_destroyed)
            {
                XtAddEventHandler(parent, StructureNotifyMask, False,
                  widgetUnmap, tip_data);
            }
            parent = XtParent(parent);
        }
    }
}


static void remove_unmap_event_handlers(XtPointer tip_data)
{
    Widget parent;

    if (tool_tip->unmap_event_handlers_registered)
    {
        tool_tip->unmap_event_handlers_registered = False;
        if (!tool_tip->widget->core.being_destroyed)
        {
            XtRemoveCallback(tool_tip->widget, XtNdestroyCallback,
              widgetDestroy, tip_data);
            XtRemoveEventHandler(tool_tip->widget, StructureNotifyMask, False,
                widgetUnmap, tip_data);
        }
        parent = XtParent(tool_tip->widget);
        while (parent)
        {
            if (!parent->core.being_destroyed)
            {
                XtRemoveEventHandler(parent, StructureNotifyMask, False,
                  widgetUnmap, tip_data);
            }
            parent = XtParent(parent);
        }
    }
}
