/* ************************************************************************ 
 *         The Amulet User Interface Development Environment              *
 * ************************************************************************
 * This code was written as part of the Amulet project at                 *
 * Carnegie Mellon University, and has been placed in the public          *
 * domain.  If you are using this code or any part of Amulet,             *
 * please contact amulet@cs.cmu.edu to be put on the mailing list.        *
 * ************************************************************************/

// This file contains dialog box widgets
// Alan Ferrency 12-13-95

#include <amulet.h>

#include WIDGETS_ADVANCED__H

Am_Object Am_Text_Input_Dialog, Am_Alert_Dialog, Am_Choice_Dialog;

Am_Define_Formula (int, height_of_layout)
{
  int h = (int)Am_Height_Of_Parts(cc, self) + 2 * ((int)self.GV(Am_Y_OFFSET));
  return h < 75 ? 75 : h;
}

Am_Define_Formula (int, width_of_layout)
{
  int w = (int)Am_Width_Of_Parts(cc, self) + 2 * ((int)self.GV(Am_X_OFFSET));
  return w < 120 ? 120 : w;
}

// The items in a dialog's map of text items comes from the dialog's 
// Am_ITEMS slot.
Am_Define_Formula (Am_Wrapper*, dialog_items)
{
  return self.GV_Owner().GV_Owner().GV(Am_ITEMS);
}

// returns dialog's am_valid_input slot
Am_Define_Formula (bool, valid_input)
{
  return self.GV_Owner().GV_Object(Am_WINDOW).GV(Am_VALID_INPUT);
}

// Returns the value of the dialog box
Am_Define_Value_Formula (value_from_dialog)
{
  value = self.GV_Owner().GV_Object(Am_WINDOW).GV(Am_VALUE);
}

Am_Define_Value_Formula (value_from_text_part)
{
  value = self.GV_Owner().GV_Object(Am_WINDOW).GV_Part(Am_DIALOG_GROUP)
    .GV_Part(Am_TEXT_WIDGET).GV(Am_VALUE);
}

// this goes in a dialog's Am_VALUE slot, and returns its command's value
Am_Define_Value_Formula (value_from_command)
{
  value = self.GV_Part(Am_COMMAND).GV(Am_VALUE);
}

// Put this in a dialog box widget somewhere; it finds the widget's top
// level Am_COMMAND (for implementation parents
Am_Define_Object_Formula (cmd_from_dialog)
{ 
  // the window is the widget
  return self.GV_Owner().GV_Object(Am_WINDOW).GV_Part(Am_COMMAND);
}

// set the choice button return value into the choice dialog value slot
Am_Define_Method(Am_Object_Method, void, choice_dialog_stop, (Am_Object cmd))
{
  Am_Object widget = cmd.Get_Owner().Get(Am_WINDOW);
  Am_Value return_value = widget.Get_Part(Am_DIALOG_GROUP)
    .Get_Part(Am_DIALOG_BUTTONS).Get(Am_VALUE);
  widget.Get_Part(Am_COMMAND).Set(Am_VALUE, return_value);
}

// set the text widget return value into the input dialog's Am_VALUE slot
Am_Define_Method(Am_Object_Method, void, input_dialog_stop, (Am_Object cmd))
{
  Am_Object widget = cmd.Get_Owner().Get(Am_WINDOW);
  Am_Value return_value = widget.Get_Part(Am_DIALOG_GROUP)
    .Get_Part(Am_TEXT_WIDGET).Get(Am_VALUE);
  widget.Get_Part(Am_COMMAND).Set(Am_VALUE, return_value);
}

// abort the input dialog box from cancel button: 
// abort text widget and call dialog's abort method.
Am_Define_Method(Am_Object_Method, void, input_dialog_abort, (Am_Object cmd))
{
  Am_Object widget = cmd.Get_Owner().Get_Object(Am_WINDOW);
  Am_Object text_widget = 
    widget.Get_Part(Am_DIALOG_GROUP).Get_Part(Am_TEXT_WIDGET);
  Am_Abort_Widget(text_widget);
  // explicitly call dialog box's abort method.  We don't have any 
  // implementation parent here, because then this stop method would call
  // the dialog's stop method as well as the abort method.
  // Is there a better way?
  Am_Object widget_cmd = widget.Get_Part(Am_COMMAND);
  Am_Object_Method aborter = widget_cmd.Get(Am_ABORT_DO_METHOD);
  aborter.Call(widget_cmd);
}

// explicitly finish a dialog box using finish_pop_up
Am_Define_Method(Am_Explicit_Widget_Run_Method, void, 
		 Am_Finish_Dialog_Method,
		 (Am_Object widget, Am_Value initial_value))
{
  if (initial_value.Valid()) widget.Set(Am_VALUE, initial_value);
  if (widget.Valid()) {
    Am_Object window = widget.Get(Am_WINDOW);
    if (window.Valid()) 
      Am_Finish_Pop_Up_Waiting (window, Am_Value(false));
  }
  // else do nothing, something's not valid.
}

// explicitly run a dialog box non-modally using pop_up_window
Am_Define_Method(Am_Explicit_Widget_Run_Method, void, 
		 Am_Show_Dialog_Method, 
		 (Am_Object widget, Am_Value iv))
{
  Am_Pop_Up_Window_And_Wait (widget, iv, false);
}


void Am_Dialog_Widgets_Initialize ()
{
  Am_Object am_empty_dialog;
  
  am_empty_dialog = Am_Window.Create("empty_dialog_window")
    .Set (Am_X_OFFSET, 5) // used in width_of_layout
    .Set (Am_Y_OFFSET, 5) // used in height_of_layout
    .Set (Am_WIDTH, width_of_layout)
    .Set (Am_HEIGHT, height_of_layout)
    .Set (Am_FILL_STYLE, Am_Motif_Gray)
    .Set (Am_ITEMS, 0)
    .Set (Am_V_SPACING, 5)
    .Set (Am_H_SPACING, 10)
    .Set (Am_H_ALIGN, Am_CENTER_ALIGN)
    .Set (Am_WIDGET_LOOK, (int)Am_MOTIF_LOOK)
    .Set (Am_WIDGET_START_METHOD, Am_Show_Dialog_Method)
    .Set (Am_WIDGET_ABORT_METHOD, Am_Finish_Dialog_Method)
    .Set (Am_WIDGET_STOP_METHOD, Am_Finish_Dialog_Method)
    .Set (Am_VALUE, "")
    .Set (Am_VALUE, value_from_command)
    .Set_Single_Constraint_Mode (Am_VALUE, false)
    .Add_Part (Am_COMMAND, Am_Command.Create("Command_In_Dialog")
	       .Set(Am_VALUE, ""))
    .Add_Part (Am_DIALOG_GROUP, Am_Group.Create ("Group_In_Dialog")
	       .Set (Am_LEFT, Am_Center_X_Is_Center_Of_Owner)
	       .Set (Am_TOP, Am_Center_Y_Is_Center_Of_Owner)
	       .Set (Am_WIDTH, Am_Width_Of_Parts)
	       .Set (Am_HEIGHT, Am_Height_Of_Parts)
	       .Set (Am_LAYOUT, Am_Vertical_Layout)
	       .Set (Am_V_SPACING, Am_From_Owner (Am_V_SPACING))
	       .Set (Am_H_SPACING, Am_From_Owner (Am_H_SPACING))
	       .Set (Am_H_ALIGN, Am_From_Owner (Am_H_ALIGN))
	       .Set (Am_FILL_STYLE, Am_From_Owner (Am_FILL_STYLE))
	       .Set (Am_WIDGET_LOOK, Am_From_Owner (Am_WIDGET_LOOK))
	       .Add_Part (Am_Map.Create ("Dialog_Prompt_Text")
			  .Set (Am_ITEM_PROTOTYPE, Am_Text.Create ("Prompt_Text_Proto")
				.Set (Am_ITEM, "") //initialize
				.Set (Am_TEXT, Am_Same_As(Am_ITEM))
				.Set (Am_X_OFFSET, 0)
				)
			  .Set (Am_LAYOUT, Am_Vertical_Layout)
			  .Set (Am_V_SPACING, Am_From_Owner(Am_V_SPACING))
			  .Set (Am_H_ALIGN, Am_From_Owner (Am_H_ALIGN))
			  .Set (Am_ITEMS, dialog_items)
			  )
	       )
    .Get_Part (Am_COMMAND)
      .Set (Am_VALUE, "")
      .Set_Single_Constraint_Mode (Am_VALUE, false)
    .Get_Owner()
    ;
  
  Am_Alert_Dialog = am_empty_dialog.Create ("alert_dialog")
    .Set (Am_TITLE, "Alert")
    .Set (Am_ICON_TITLE, "Alert")
    .Get_Part(Am_DIALOG_GROUP)
      .Add_Part (Am_Button.Create ("alert_dialog_ok")
		 .Set (Am_WIDTH, 50)
		 .Set (Am_FILL_STYLE, Am_From_Owner(Am_FILL_STYLE))
		 .Set (Am_WIDGET_LOOK, Am_From_Owner(Am_WIDGET_LOOK))
		 .Get_Part (Am_COMMAND)
		 .Set (Am_LABEL, "OK")
		 .Set (Am_IMPLEMENTATION_PARENT, cmd_from_dialog)
		 //		 .Set (Am_DO_METHOD, do_hide_dialog)
		 .Get_Owner()
		 )
    .Get_Owner();
    ;
  
  Am_Text_Input_Dialog = am_empty_dialog.Create ("text_input_dialog")
    .Set (Am_TITLE, "Text Input Dialog")
    .Set (Am_ICON_TITLE, "Text Input Dialog")
    .Set (Am_VALID_INPUT, true)
    // A high priority one-shot interactor to catch return presses.
    // It needs high priority to take precedence over the running text
    // widget.
    .Add_Part (Am_One_Shot_Interactor.Create("text_catch_return")
	       .Set (Am_START_WHEN, "RETURN")
	       .Set (Am_PRIORITY, 105)
	       .Set (Am_ACTIVE, valid_input)
	       .Get_Part(Am_COMMAND)
	         .Set (Am_IMPLEMENTATION_PARENT, cmd_from_dialog)
	         .Set (Am_DO_METHOD, input_dialog_stop)
	       .Get_Owner()
	       )
    .Get_Part (Am_COMMAND)
      .Set (Am_VALUE, "")
      .Set (Am_VALUE, value_from_text_part)
      .Get_Owner()
    .Get_Part (Am_DIALOG_GROUP)
      .Add_Part (Am_TEXT_WIDGET, 
		 Am_Text_Input_Widget.Create("text_input_in_dialog")
		 .Set (Am_LEFT, Am_Center_X_Is_Center_Of_Owner)
		 .Set (Am_WIDTH, 200)
		 .Set (Am_X_OFFSET, 0)
		 .Set (Am_FILL_STYLE, Am_From_Owner(Am_FILL_STYLE))
		 .Set (Am_WIDGET_LOOK, Am_From_Owner(Am_WIDGET_LOOK))
		 .Get_Part (Am_TEXT_WIDGET_TEXT_OBJ)
		   .Set (Am_TEXT, value_from_dialog)
		 .Get_Owner()
		 // we don't want the text widget to do anything on 
		 // stop, because when you click outside a text widget
		 // it stops.  It interferes with the buttons.
		 .Get_Part(Am_COMMAND)
		   .Set (Am_LABEL, "")
		 .Get_Owner()
		 )
      .Add_Part (Am_DIALOG_BUTTONS, Am_Button_Panel
		 .Create ("buttons_in_text_dialog")
		 .Set (Am_FIXED_WIDTH, true)
		 .Set (Am_LAYOUT, Am_Horizontal_Layout)
		 .Set (Am_H_SPACING, Am_From_Owner (Am_H_SPACING))
		 .Set (Am_FILL_STYLE, Am_From_Owner(Am_FILL_STYLE))
		 .Set (Am_WIDGET_LOOK, Am_From_Owner(Am_WIDGET_LOOK))
		 .Set (Am_ITEMS, Am_Value_List()
		       .Add (Am_Command
			     .Create ("text_input_okay")
			     .Set (Am_ACTIVE, valid_input)
			     .Set (Am_LABEL, "OK")
			     .Set (Am_IMPLEMENTATION_PARENT, cmd_from_dialog)
			     .Set (Am_DO_METHOD, input_dialog_stop)
			     )
		       .Add (Am_Command
			     .Create ("text_input_cancel")
			     .Set (Am_LABEL, "Cancel")
			     // don't want an implementation parent;
			     // call the abort method of the dialog's command
			     // explicitly.
			     //.Set (Am_IMPLEMENTATION_PARENT, cmd_from_dialog)
			     .Set (Am_DO_METHOD, input_dialog_abort)
			     )
		       )
		 )
    .Get_Owner();
    ;

  Am_Choice_Dialog = am_empty_dialog.Create("Choice_Dialog")
    .Set (Am_TITLE, "Choice Dialog")
    .Set (Am_ICON_TITLE, "Choice Dialog")
    //    .Set (Am_VALUE, value_from_button_part)
    .Get_Part (Am_DIALOG_GROUP)
      .Add_Part (Am_DIALOG_BUTTONS,
		      Am_Button_Panel.Create("buttons_in_choice_dialog")
		 .Set (Am_FIXED_WIDTH, true)
		 .Set (Am_LAYOUT, Am_Horizontal_Layout)
		 .Set (Am_H_SPACING, Am_From_Owner (Am_H_SPACING))
		 .Set (Am_FILL_STYLE, Am_From_Owner(Am_FILL_STYLE))
		 .Set (Am_WIDGET_LOOK, Am_From_Owner(Am_WIDGET_LOOK))
		 .Set (Am_ITEMS, Am_Value_List()
		       .Add (Am_Command
			     .Create ("choice_okay")
			     .Set (Am_LABEL, "OK")
			     .Set (Am_IMPLEMENTATION_PARENT, cmd_from_dialog)
			     .Set (Am_DO_METHOD, choice_dialog_stop)
			     )
		       .Add (Am_Command
			     .Create ("choice_cancel")
			     .Set (Am_LABEL, "Cancel")
			     .Set (Am_IMPLEMENTATION_PARENT, cmd_from_dialog)
			     .Set (Am_DO_METHOD, choice_dialog_stop)
			     )
		       )
		 )
    .Get_Owner();
}

// This goes in a dialog widget's Am_COMMAND object's stop method
Am_Define_Method(Am_Object_Method, void, do_stop_waiting_for_dialog,
		 (Am_Object cmd))
{
  Am_Object widget = cmd.Get_Owner();
  if (widget.Valid()) {
    // we want to stop the text input widget if it exists
    Am_Value v;
    widget.Get(Am_VALUE, v);
    Am_Finish_Pop_Up_Waiting (widget, v);
  }
  // else do nothing, something's not valid.
}

// same as stop waiting, but return a NULL value.
Am_Define_Method(Am_Object_Method, void, do_abort_waiting_for_dialog,
		 (Am_Object cmd))
{
  Am_Object widget = cmd.Get_Owner();
  if (widget.Valid())
      Am_Finish_Pop_Up_Waiting (widget, NULL);
}

// show a message in a dialog box and wait for the user to click okay.
void Am_Show_Alert_Dialog (Am_Value_List alert_texts, int x /* = 100*/, 
			   int y /*= 100*/, bool modal /*= false*/)
{
  Am_Object the_dialog = Am_Alert_Dialog.Create();
  the_dialog.Set (Am_ITEMS, alert_texts)
    .Set (Am_LEFT, x)
    .Set (Am_TOP, y);
  the_dialog.Get_Part (Am_COMMAND)
    .Set (Am_DO_METHOD, do_stop_waiting_for_dialog)
    .Set (Am_ABORT_DO_METHOD, do_abort_waiting_for_dialog);
  Am_Screen.Add_Part (the_dialog);
  Am_Value dummy;
  Am_Pop_Up_Window_And_Wait (the_dialog, dummy, modal);
  Am_Screen.Remove_Part (the_dialog);
}

// get a line of text input from a dialog box
Am_Value Am_Get_Input_From_Dialog (Am_Value_List prompt_texts, 
				   Am_String initial_value /*= ""*/, 
				   int x /*= 100*/, int y /* = 100*/, 
				   bool modal /*=false*/)
{
  Am_Object the_dialog = Am_Text_Input_Dialog.Create();
  the_dialog.Set (Am_ITEMS, prompt_texts)
    .Set (Am_LEFT, x)
    .Set (Am_TOP, y);
  the_dialog.Get_Part (Am_COMMAND)
    .Set (Am_DO_METHOD, do_stop_waiting_for_dialog)
    .Set (Am_ABORT_DO_METHOD, do_abort_waiting_for_dialog);
  the_dialog.Set (Am_VALUE, initial_value);
  Am_Screen.Add_Part (the_dialog);
  Am_Object text_widget = the_dialog.Get_Part(Am_DIALOG_GROUP)
    .Get_Part (Am_TEXT_WIDGET);
  Am_Start_Widget(text_widget, initial_value);
  Am_Value v;
  Am_Pop_Up_Window_And_Wait (the_dialog, v, modal);
  Am_Screen.Remove_Part (the_dialog);
  return v;
}

// get a choice from a dialog box: "OK" or "Cancel"
Am_Value Am_Get_Choice_From_Dialog (Am_Value_List prompt_texts, 
				    int x /*= 100*/, int y /*= 100*/,
				    bool modal /*=false*/)
{
  Am_Object the_dialog = Am_Choice_Dialog.Create();
  the_dialog.Set (Am_ITEMS, prompt_texts)
    .Set (Am_LEFT, x)
    .Set (Am_TOP, y);
  the_dialog.Get_Part (Am_COMMAND)
    .Set (Am_DO_METHOD, do_stop_waiting_for_dialog)
    .Set (Am_ABORT_DO_METHOD, do_abort_waiting_for_dialog);
  Am_Screen.Add_Part (the_dialog);
  Am_Value v;
  Am_Pop_Up_Window_And_Wait (the_dialog, v, modal);
  Am_Screen.Remove_Part (the_dialog);
  return v;
}

// this pops up some (custom?) dialog box that a user creates, 
// waits for it to complete, and returns its value.
Am_Value Am_Show_Dialog_And_Wait (Am_Object the_dialog, 
				  bool modal /*= false*/)
{
  the_dialog.Get_Part (Am_COMMAND)
    .Set (Am_DO_METHOD, do_stop_waiting_for_dialog)
    .Set (Am_ABORT_DO_METHOD, do_abort_waiting_for_dialog);
  Am_Value v;
  Am_Pop_Up_Window_And_Wait (the_dialog, v, modal);
  return v;
}
