// $Id: WidgetLayout.cpp,v 1.9 1997/11/01 23:46:28 jharris Exp $

#include <qapp.h>
#include <qpainter.h>
#include <qmsgbox.h>
#include <qfile.h>
#include <qstrlist.h>

#include "DialogWnd.h"
#include "GuiTools.h"

#include "widget/DlgButton.h"
#include "widget/DlgButtonGroup.h"
#include "widget/DlgCheckBox.h"
#include "widget/DlgComboBox.h"
#include "widget/DlgFrame.h"
#include "widget/DlgGroupBox.h"
#include "widget/DlgLCDNumber.h"
#include "widget/DlgLabel.h"
#include "widget/DlgLineEdit.h"
#include "widget/DlgListBox.h"
#include "widget/DlgMenuBar.h"
#include "widget/DlgMultiLineEdit.h"
#include "widget/DlgPushButton.h"
#include "widget/DlgRadioButton.h"
#include "widget/DlgScrollBar.h"
#include "widget/DlgUser.h"

#include "layout/BoxLayout.h"

#include "GridDialog.h"
#include "DlgParser.h"

#define Inherited QObject

int WidgetLayout::layoutNumber = 1;

WidgetLayout::WidgetLayout
(
    DialogWnd*	parent,
    const char*	name
)
    :
    Inherited( (QObject *)parent, name ),
    itsParent( parent ),
    itsDlgEditWnd( NULL ),
    isDragging( FALSE),
    itsDragSquare( none ),
    itsParser( NULL ),
    widgetInConstruction( NULL ),
    itsLayoutItemInConstruction( NULL ),
    autoSelect( FALSE ),
    itsLayoutType( None ),
    itsBoxLayout( NULL ),
    itsSelectedLayoutWidget( NULL ),
    itsScreenFlags( SF_None ),	
    itsWindowMinSize( 0, 0 ),
    itsWindowMaxSize( QCOORD_MAX, QCOORD_MAX ),
    itsWindowPos( -1, -1 )
{
    itsCursors[ topLeft		] = &sizeFDiagCursor;
    itsCursors[ left		] = &sizeHorCursor;
    itsCursors[ bottomLeft	] = &sizeBDiagCursor;
    itsCursors[ bottom		] = &sizeVerCursor;
    itsCursors[ bottomRight 	] = &sizeFDiagCursor;
    itsCursors[ right		] = &sizeHorCursor;
    itsCursors[ topRight	] = &sizeBDiagCursor;
    itsCursors[ top		] = &sizeVerCursor;	
    itsCursors[ none		] = &sizeAllCursor;

    itsSelectedWidget.setAutoDelete( TRUE );

    itsNoMoveWidgets.setAutoDelete( FALSE );
    itsDlgWidgets.setAutoDelete( TRUE );
    itsZOrdering.setAutoDelete( FALSE );

    // Generate a name
    QString tmp;
    tmp.setNum( layoutNumber );

    ++layoutNumber;

    itsWidgetPopup = new QPopupMenu;
    itsWidgetPopup->insertItem( "Raise", this, SLOT(RaiseSelectedWidget()) );
    itsWidgetPopup->insertItem( "Lower", this, SLOT(LowerSelectedWidget()) );
    itsWidgetPopup->insertSeparator();
    itsWidgetPopup->insertItem( "Delete", this, SLOT(DeleteSelectedWidget()) );
    itsWidgetPopup->insertSeparator();
    itsWidgetPopup->insertItem( "Properties", this,	
				SLOT(SelectedWidgetProperties()) );

    itsWidgetFramePopup = new QPopupMenu;
    itsWidgetFramePopup->insertItem( "Properties", this,
				     SLOT(SelectedWidgetProperties()) );
}


WidgetLayout::~WidgetLayout()
{
}


DlgWidget* WidgetLayout::AddWidget
(
    DlgWidget::Type	type
)
{
    DlgWidget* newWidget = DlgWidget::NewDlgWidget( itsDlgEditWnd, type );

    bool widgetOnTop = TRUE;

    if( type == DlgWidget::DW_ButtonGroup ||
	type == DlgWidget::DW_GroupBox )
    {
	widgetOnTop = FALSE;
    }

    if( widgetOnTop )
    {
	itsZOrdering.append( newWidget );
    }
    else
    {
	itsZOrdering.insert( 0, newWidget );
    }

    QWidget* widget = newWidget->GetWidget();
    QString name = widget->name();

    ConnectToWidget( widget );

    itsDlgWidgets.insert( name, newWidget );
    
    widget->show();
    SetDragging( widget );

    if( autoSelect )
	AddToSelection( widget );

    emit Changed();

    return newWidget;
}
	


void WidgetLayout::SaveContents
(
    QTextStream&	stream
)
{
    //QRect rect( itsWindowPos, itsDlgEditWnd->size() );

    stream << "WidgetLayout {\n";

    // Don't write the Rect field, use InitialPos and Size instead
    //stream << "Rect {" << rect << "}\n";
    stream << "InitialPos {" << itsWindowPos << "}\n";
    stream << "Size {" << itsDlgEditWnd->size() << "}\n";
    stream << "MinSize {" << itsWindowMinSize << "}\n";
    stream << "MaxSize {" << itsWindowMaxSize << "}\n";
    stream << "Grid {" << itsDlgEditWnd->GetGrid() << "}\n";

    QListIterator< DlgWidget > widgetIter( itsZOrdering );
    DlgWidget* dlgWidget;
    
    while( (dlgWidget = widgetIter()) )
    {
	dlgWidget->SaveContents( stream );
    }

    stream << "\nLayout {";
    switch( itsLayoutType )
    {
	case None:
	{
	    stream << "None";
	    break;
	}
	case Box:
	{
	    stream << "Box";
	    break;
	}
    }
    stream << "}\n";
    
    switch( itsLayoutType )
    {
	case None:
	{
	    break;
	}
	case Box:
	{
	    itsBoxLayout->SaveContents( stream, "1" );
	    break;
	}
    }

    stream << "}" << endl;
}


void WidgetLayout::ResetContents( void )
{
    //itsDlgWidgets.setAutoDelete( TRUE );
    itsDlgWidgets.clear();
    //itsDlgWidgets.setAutoDelete( FALSE );
    
    itsZOrdering.clear();
    itsNoMoveWidgets.clear();
}



void WidgetLayout::RestoreContents
(
    DlgParser *theParser
)
{
    // Remember the Parser 
    itsParser = theParser;
    
  // Connecting the Parser
    connect( itsParser, SIGNAL(NewWidget(QString &)), 
	     SLOT(RestoreNewWidget(QString &)));
    connect( itsParser, SIGNAL(WidgetKeyValue(QString &,QString &)), 
	     SLOT(RestoreWidgetKeyValue(QString &,QString &)) );
    connect( itsParser, SIGNAL(EndWidget()), 
	     SLOT(RestoreEndWidget()) );

    connect( itsParser, SIGNAL(LayoutKeyValue(QString &,QString &)), 
	     SLOT(RestoreLayoutKeyValue(QString &,QString &)) );
    connect( itsParser, SIGNAL(EndLayout()), 
	     SLOT(ContentsRestored()) );

    connect( itsParser, SIGNAL(NewLayoutItem(QString&)),
	     SLOT(RestoreNewLayoutItem(QString&)) );
    connect( itsParser, SIGNAL(LayoutItemKeyValue(QString&,QString&)),
	     SLOT(RestoreLayoutItemKeyValue(QString&,QString&)) );
    connect( itsParser, SIGNAL(EndLayoutItem()),
	     SLOT(RestoreEndLayoutItem()) );
}

/* Slot, called when all Widgets for this layout 
 * have been read
 *
 * Undo all connections done by RestoreContents
 */
void WidgetLayout::ContentsRestored()
{
    if( ! itsParser )
	return;

    itsParser->disconnect( this );

    itsParser = NULL;

    switch( itsLayoutType )
    {
	case None:
	{
	    break;
	}
	case Box:
	{
	    itsBoxLayout->Recreate();
	    LowerLayout();
	    break;
	}
    }
}

void WidgetLayout::RestoreNewWidget( QString &widgetName )
{
    // Create a new Widget
    widgetInConstruction = DlgWidget::NewDlgWidget( itsDlgEditWnd, 
						    widgetName );
	    
    // No Widget => Error in File, stop action
    if( ! widgetInConstruction )
    {
	itsParser->StopParsing();
	return;
    }	
    else
    {
	ConnectToWidget( widgetInConstruction->GetWidget() );
    }
}

void WidgetLayout::RestoreLayoutKeyValue( QString &key, QString &value )
{

    // Still read in the depreciated Rect field
    if( key == "Rect" )
    {
	QTextStream read( value, IO_ReadOnly );
	QRect aRect;
	read >> aRect;
	itsDlgEditWnd->move( aRect.left(), aRect.top() );
	SetWindowSize( aRect.size() );
	itsWindowPos = QPoint( -1,-1 );
    } 
    else if( key == "InitialPos" )
    {
	QTextStream read( value, IO_ReadOnly );
	QPoint point;
	read >> point;
	SetWindowPos( point );
    }
    else if( key == "Size" )
    {
	QTextStream read( value, IO_ReadOnly );
	QSize size;
	read >> size;
	itsDlgEditWnd->move( 0,0 );
	SetWindowSize( size );
    }
    else if( key == "MinSize" )
    {
	QTextStream read( value, IO_ReadOnly );
	QSize size;
	read >> size;
	SetWindowMinSize( size );
    }
    else if( key == "MaxSize" )
    {
	QTextStream read( value, IO_ReadOnly );
	QSize size;
	read >> size;
	SetWindowMaxSize( size );
    }
    else if( key == "Grid" )
    {
	itsDlgEditWnd->SetGrid( value.toInt() );
    }
    else if( key == "Layout" )
    {
	if( value == "None" )
	{
	    SetLayoutType( None );
	}
	else if( value == "Box" )
	{
	    SetLayoutType( Box );
	}
    }
    else
    {
	fprintf( stderr, "Unknown key value Pair: < %s : %s >\n",
		 (const char *)key, (const char *) value );
    }
}

void WidgetLayout::RestoreWidgetKeyValue( QString &key, QString &value )
{
    // The Name token is of special interest for us
    if( key == "Name" && widgetInConstruction )
    {
	// Check for double names, and rename if necessary
	if( itsDlgWidgets[value] )
	{
	    itsDlgWidgets.insert( widgetInConstruction->GetWidget()->name(), 
				  widgetInConstruction );
	    return;
	} else
	{
	    itsDlgWidgets.insert( value, widgetInConstruction );
	}
    }

    // Just send the values to the widget
    if( ! widgetInConstruction->RestoreKeyValue( key, value ) )
    {
	fprintf( stderr, "Unknown key value Pair: < %s : %s >\n",
		 (const char *)key, (const char *) value );
    }
}

void WidgetLayout::RestoreEndWidget( )
{
    if( widgetInConstruction )
    {
	widgetInConstruction->GetWidget()->show();
	itsZOrdering.append( widgetInConstruction );
	UpdateWidgetLayoutState( widgetInConstruction );

	if( autoSelect )
	    AddToSelection( widgetInConstruction->GetWidget() );
    }
}


void WidgetLayout::RestoreNewLayoutItem( QString&	itemName )
{
    switch( itsLayoutType )
    {
	case None:
	{
	    break;
	}
	case Box:
	{
	    // break item name into type and id
	    int pos = itemName.find( '-' );
	    QString type = itemName.left( pos );
	    QString id = itemName.mid( pos + 1, itemName.length() - pos );

	    // root level layout
	    if( id == "1" ) 
	    {
		itsLayoutItemInConstruction = itsBoxLayout;
	    }
	    else
	    {
		// strip off the leading index, "1" in this case since we
		// know that its a child of itsBoxLayout
		pos = id.find( '.' );
		id = id.mid( pos + 1, id.length() - pos - 1 );

		itsLayoutItemInConstruction = 
		    itsBoxLayout->NewLayoutItem( type, id );
	    }

	    break;
	}
    }	
}


void WidgetLayout::RestoreLayoutItemKeyValue( QString&	key,
					      QString&	value )
{
    if( itsLayoutItemInConstruction && 
	!itsLayoutItemInConstruction->RestoreKeyValue( key, value ) )
    {
	fprintf( stderr, "Unknown key value Pair: < %s : %s >\n",
		 (const char *)key, (const char *) value );
    }
    
}


void WidgetLayout::RestoreEndLayoutItem()
{
}


void WidgetLayout::GenerateDataHeader
(
 QTextStream&	stream,
 const QString&	dataName,
 const QString&	dataBaseName,
 const QString&	dataBaseHeader
 )
{
  stream << "#ifndef " << dataName << "_included\n";
  stream << "#define " << dataName << "_included\n";
  stream << endl;

  // find the list of all the header's we will need
  QListIterator< DlgWidget > iter( itsZOrdering );
  DlgWidget* widget;
  QDict<int> headerDict;
  QString header;
  int dummyInt = 0;

  // create a dictionary of header files that are needed for the mapped
  // variables, without duplicates
  while( (widget = iter()) )
  {
    header = widget->GetHeaderFilename();

    if( !headerDict[ header ] && widget->IsVariableMapped() )
    {
	    headerDict.insert( header, &dummyInt );
    }
  }


  QDictIterator<int> headerIter( headerDict );

  stream << "#include " << dataBaseHeader << '\n';

  while( headerIter.current() )
  {
    stream << "#include " << headerIter.currentKey() << '\n';
    ++headerIter;
  }
    
  stream << '\n';
  stream << "class " << dataName << " : public " << dataBaseName << '\n';
  stream << "{\n";
  stream << "    Q_OBJECT\n";
  stream << '\n';
  stream << "public:\n";
  stream << '\n';
  stream << "    " << dataName << '\n';
  stream << "    (\n";
  stream << "        QWidget* parent = NULL,\n";
  stream << "        const char* name = NULL\n";
  stream << "    );\n";
  stream << '\n';
  stream << "    virtual ~" << dataName << "();\n";
  stream << "\n";

  // define slots
    
  QDict<QString> publicSlots, protectedSlots;
  SignalConnection* sig;
  QString slot;
  const QStrList& qdialogSlots = GuiTools::GetQDialogSlots();
    
  publicSlots.setAutoDelete( TRUE );
  protectedSlots.setAutoDelete( TRUE );

  // scan through all the widgets and their slots, placing the slots into
  // the correct dictionary for the scope making sure there are no 
  // duplicates
  iter.toFirst();
  while( (widget = iter()) )
  {
    QListIterator<SignalConnection> 
      signalIter( widget->GetConnectedSignalList() );
	
    while( (sig = signalIter()) )
    {
      // Signals which connect to base class slots must not be generated
      if( sig->GetScope() == SignalConnection::BaseClass )
        continue;

      slot = sig->GetSlot();

      // if the base is a dialog and the slot is defined by the QDialog
      // class, skip it
      if( dataBaseName != "QDialog" || !qdialogSlots.contains( slot ) )
      {
        switch( sig->GetScope() )
        {
        case SignalConnection::Protected:
          {
            if( !protectedSlots[ slot ] )
            {
              protectedSlots.insert( slot, new QString( slot ) );
            }
            break;
          }
        case SignalConnection::Public:
          {
            if( !publicSlots[ slot ] )
            {
              publicSlots.insert( slot, new QString( slot ) );
            }
            break;
          }
        case SignalConnection::BaseClass:
          break;
        }
      }
    }
  }
    
  QDictIterator<QString> publicIter( publicSlots );
  QString* slotStr;

  stream << "public slots:\n\n";
  while( (slotStr = publicIter()) )
  {
    stream << "    virtual void " << *slotStr << ";\n";
  }
  stream << '\n';

  QDictIterator<QString> protectedIter( protectedSlots );
    
  stream << "protected slots:\n\n";
  while( (slotStr = protectedIter()) )
  {
    // if the user declared a slot to be both protected and private,
    // make it public
    if( !publicSlots[ *slotStr ] )
    {
      stream << "    virtual void " << *slotStr << ";\n";
    }
  }
  stream << '\n';    

  // declare mapped variables
    
  stream << "protected:\n";

  iter.toFirst();
  while( (widget = iter()) )
  {
    if( widget->IsVariableMapped() )
    {
      stream << "    " << widget->GetWidgetClass() << "* " <<
        widget->GetVariableName() << ";\n";
    }
  }
    
  stream << '\n';
  stream << "};\n";

  stream << '\n';
  stream << "#endif // " << dataName << "_included\n";
}



void WidgetLayout::GenerateDataSource
(
    QTextStream&	stream,
    const QString&	dataHeader,
    const QString&	dataName,
    const QString&	dataBaseName,
    bool		modalWindow,
    WFlags		windowFlags
)
{
    stream << "#include \"" << dataHeader << "\"\n";
    stream << '\n';
    stream << "#define Inherited " << dataBaseName << "\n";
    stream << "\n";

    QDict<int> nonMappedClasses;
    QListIterator<DlgWidget> widgetIter( itsZOrdering );
    DlgWidget* widget;
    QString header;
    int dummyInt = 0;

    // create a dictionary of the header files needed for the widgets that
    // have not been mapped
    while( (widget = widgetIter()) )
    {
	header = widget->GetHeaderFilename();

	if( !nonMappedClasses[ header ] && !widget->IsVariableMapped() )
	{
	    nonMappedClasses.insert( header, &dummyInt );
	}
    }
    
    QDictIterator<int> headerIter( nonMappedClasses );
    
    while( headerIter.current() )
    {
	stream << "#include " << headerIter.currentKey() << '\n';
	++headerIter;
    }

    if( itsLayoutType != None )
    {
	stream << "#include <qlayout.h>\n";
    }
    
    stream << '\n';
    stream << dataName << "::" << dataName << '\n';
    stream << "(\n";
    stream << "\tQWidget* parent,\n";
    stream << "\tconst char* name\n";
    stream << ")\n";
    stream << "\t:\n";
    
    if( modalWindow )
    {
	stream << "\tInherited( parent, name, TRUE, " << windowFlags << " )\n";
    }
    else
    {
	stream << "\tInherited( parent, name, " << windowFlags << " )\n";
    }

    stream << "{\n";
    
    // initialize widgets, do the button groups last to make sure the 
    // widgets inserted are constructed
    QString varName;

    widgetIter.toFirst();
    while( (widget = widgetIter()) )
    {	
	GenerateWidgetSource( widget, stream );
    }

    widgetIter.toFirst();
    while( (widget = widgetIter()) )
    {	
	GenerateWidgetExtraSource( widget, stream );
    }
    
    GenerateLayoutSource( stream );

    // Set initial position and size
    if( (itsWindowPos.x() >= 0) && (itsWindowPos.y() >= 0) )
    {
	stream << "\tsetGeometry( " << 
	    itsWindowPos.x() << "," << itsWindowPos.y() << ", " << 
	    itsDlgEditWnd->width() << "," << itsDlgEditWnd->height() << 
	    " );\n";
    }
    else
    {
	stream << "\tresize( " <<
	    itsDlgEditWnd->width() << "," << itsDlgEditWnd->height() << 
	    " );\n";
    }
    
    // Set min and max sizes
    stream << "\tsetMinimumSize( " << 
	itsWindowMinSize.width() << ", " << itsWindowMinSize.height() << 
	" );\n";
    stream << "\tsetMaximumSize( " <<
	itsWindowMaxSize.width() << ", " << itsWindowMaxSize.height() <<
	" );\n";
    
    stream << "}\n";
    stream << "\n\n";

    // define the descructor

    stream << dataName << "::~" << dataName << "()\n";
    stream << "{\n";
    stream << "}\n";
    
    // define all the slots

    QDict<SignalConnection> slotDict;
    SignalConnection* sig;
    QString slot;
    const QStrList& qdialogSlots = GuiTools::GetQDialogSlots();

    slotDict.setAutoDelete( FALSE );
    widgetIter.toFirst();
    while( (widget = widgetIter()) )
    {
	QListIterator<SignalConnection> 
	    signalIter( widget->GetConnectedSignalList() );

	while( (sig = signalIter()) )
	{
      // Signals which connect to base class slots must not be generated
      if( sig->GetScope() == SignalConnection::BaseClass )
        continue;

	    slot = sig->GetSlot();

	    // If the base class is QDialog, and the slot is defined by the 
	    // QDialog class, skip it
	    if( dataBaseName != "QDialog" || !qdialogSlots.contains(slot) )
	    {
		if( !slotDict[ slot ] )
		{
		    slotDict.insert( slot, sig );
		
		    stream << "void " << dataName << "::" << slot << '\n';
		    stream << "{\n";
		    stream << "}\n";
		}
	    }
	}
    }
}


void WidgetLayout::GenerateWidgetSource
(
    DlgWidget*		widget,
    QTextStream&	stream
)
{
    QString varName = widget->GetGeneratedVarName();
    
    // construct a new widget
    if( !widget->IsVariableMapped() )
    {
	// declare new variable
	stream << '\t' << widget->GetWidgetClass() << "* " << 
	    varName << ";\n";
    }
	
    // 'new' variable
    widget->GenerateCtor( stream, varName );

    varName = "\t" + varName + widget->Dot();
    widget->GenerateSource( stream, varName, itsZOrdering );

    stream << '\n';
}


void WidgetLayout::GenerateWidgetExtraSource( DlgWidget*	widget,
					      QTextStream&	stream )
{
    QString varName = "\t" + widget->GetGeneratedVarName() + widget->Dot();
    
    widget->GenerateExtraSource( stream, varName, itsZOrdering );
}


void WidgetLayout::GenerateLayoutSource( QTextStream&	stream )
{
    // generate layout code
    switch( itsLayoutType )
    {
	case None:
	{
	    break;
	}
	case Box:
	{
	    QString emptyLayoutName;
	    QString id;
	    itsBoxLayout->GenerateSource( stream, emptyLayoutName, 
					  id.setNum( 1 ) );
	}
    }
}


void WidgetLayout::SetDragging
(
    QWidget*	widget
)
{
    ClearSelection();
    AddToSelection( widget );
    ProcessResize( widget, NULL );
    
    itsDragSquare = none;
    
    UpdateCursor( widget );
    
    itsWidgetClick.setX( widget->width()/2 );
    itsWidgetClick.setY( widget->height()/2 );
    
    widget->move( 0,0 );
}


void WidgetLayout::ConnectToWidget( QWidget* widget )
{
    connect( widget, SIGNAL(MousePress(QWidget*,QMouseEvent*)),
	     SLOT(ProcessMousePress(QWidget*,QMouseEvent*)) );
    connect( widget, SIGNAL(MouseDoubleClick(QWidget*,QMouseEvent*)),
	     SLOT(ProcessMouseDoubleClick(QWidget*,QMouseEvent*)) );
    connect( widget, SIGNAL( MouseMove( QWidget*, QMouseEvent* ) ),
	     SLOT( ProcessMouseMove( QWidget*, QMouseEvent* ) ) );
    connect( widget, SIGNAL( MouseRelease() ),
	     SLOT( ProcessMouseRelease() ) );
    connect( widget, SIGNAL( Paint( QWidget* ) ),
	     SLOT( ProcessPaint( QWidget* ) ) );
    connect( widget, SIGNAL( Resize( QWidget*, QResizeEvent* ) ),
	     SLOT( ProcessResize( QWidget*, QResizeEvent* ) ) );
}


void WidgetLayout::AlignTop()
{
    if( !itsSelectedWidget.isEmpty() )
    {
	QListIterator<Selected> iter( itsSelectedWidget );

	Selected *sel = iter();
	int top = sel->widget->geometry().top();
	QWidget* widget;
    
	// skip first item
	while( (sel = iter()) )
	{
	    widget = sel->widget;
	    
	    // make sure you don't mave one that isn't supposed to be moved
	    if( itsNoMoveWidgets[widget->name()] == NULL )
	    {
		widget->move( widget->x(), top );	
		emit Changed();
	    }
	}
    }
}


void WidgetLayout::AlignBottom()
{
    if( !itsSelectedWidget.isEmpty() )
    {
	QListIterator<Selected> iter( itsSelectedWidget );

	Selected *sel = iter();
	int bottom = sel->widget->geometry().bottom();
	QRect rect;
	QWidget* widget;
    
	// skip first item
	while( (sel = iter()) )
	{
	    widget = sel->widget;
	    
	    // make sure you don't mave one that isn't supposed to be moved
	    if( itsNoMoveWidgets[widget->name()] == NULL )
	    {
		rect = widget->geometry();
		rect.moveBottomRight( QPoint( rect.right(), bottom ) );
		widget->setGeometry( rect );
		emit Changed();
	    }
	}
    }
}


void WidgetLayout::AlignLeft()
{
    if( !itsSelectedWidget.isEmpty() )
    {
	QListIterator<Selected> iter( itsSelectedWidget );

	Selected *sel = iter();
	int left = sel->widget->geometry().left();
	QWidget* widget;
    
	// skip first item
	while( (sel = iter()) )
	{
	    widget = sel->widget;
	    
	    // make sure you don't mave one that isn't supposed to be moved
	    if( itsNoMoveWidgets[widget->name()] == NULL )
	    {
		widget->move( left, widget->y() );
		emit Changed();
	    }
	}
    }
}


void WidgetLayout::AlignRight()
{
    if( !itsSelectedWidget.isEmpty() )
    {
	QListIterator<Selected> iter( itsSelectedWidget );

	Selected* sel = iter();
	int right = sel->widget->geometry().right();
	QRect rect;
	QWidget* widget;
    
	// skip first item
	while( (sel = iter()) )
	{
	    widget = sel->widget;
	    
	    // make sure you don't mave one that isn't supposed to be moved
	    if( itsNoMoveWidgets[widget->name()] == NULL )
	    {
		rect = widget->geometry();
		rect.moveBottomRight( QPoint( right, rect.bottom() ) );
		widget->setGeometry( rect );
		emit Changed();
	    }
	}
    }
}


void WidgetLayout::SizeHorizontal()
{
    if( !itsSelectedWidget.isEmpty() )
    {
	QListIterator<Selected> iter( itsSelectedWidget );

	Selected* sel = iter();
	int width = sel->widget->width();
	QWidget* widget;
	
	// skip first item
	while( (sel = iter()) )
	{
	    widget = sel->widget;

	    // make sure you don't size one that isn't supposed to be sized
	    // no move == no size
	    if( itsNoMoveWidgets[widget->name()] == NULL )
	    {
		widget->resize( width, widget->height() );
		emit Changed();
	    }
	}
    }
}


void WidgetLayout::SizeVertical()
{
    if( !itsSelectedWidget.isEmpty() )
    {
	QListIterator<Selected> iter( itsSelectedWidget );

	Selected* sel = iter();
	int height = sel->widget->height();
	QWidget* widget;
	
	// skip first item
	while( (sel = iter()) )
	{
	    widget = sel->widget;

	    // make sure you don't size one that isn't supposed to be sized
	    // no move == no size
	    if( itsNoMoveWidgets[widget->name()] == NULL )
	    {
		widget->resize( widget->width(), height );
		emit Changed();
	    }
	}
    }
}


void WidgetLayout::SizeBothHV()
{
    SizeHorizontal();
    SizeVertical();
}


void WidgetLayout::SizeToFit()
{
    if( !itsSelectedWidget.isEmpty() )
    {
	QListIterator<Selected> iter( itsSelectedWidget );
	Selected* sel;
	
	while( (sel = iter()) )
	{
	    QSize hint = sel->widget->sizeHint();
	    if( !hint.isValid() )
	    {
		// if no size hint given, don't change size
		hint = sel->widget->size();
	    }

	    ResizeWidget( sel->widget, QRect(sel->widget->pos(), hint) );
	}
    }
}


void WidgetLayout::MoveUp()
{
    MoveWidget( QPoint(0, -itsDlgEditWnd->GetGrid()) );
}


void WidgetLayout::MoveDown()
{
    MoveWidget( QPoint(0, itsDlgEditWnd->GetGrid()) );
}


void WidgetLayout::MoveLeft()
{
    MoveWidget( QPoint(-itsDlgEditWnd->GetGrid(), 0) );
}


void WidgetLayout::MoveRight()
{
    MoveWidget( QPoint(itsDlgEditWnd->GetGrid(), 0) );
}


void WidgetLayout::MoveWidget( const QPoint&	newPos )
{
    if( !itsSelectedWidget.isEmpty() )
    {
	// send a fake mouse move event

	QWidget* widget = itsSelectedWidget.first()->widget;
	QMouseEvent event( Event_MouseMove, newPos, LeftButton, LeftButton );

	// remember old state of dragging and point of widget click and cursor
	bool oldDragging = isDragging;
	isDragging = TRUE;

	QPoint oldClick = itsWidgetClick;
	itsWidgetClick = QPoint(0,0);

	ResizeSquare oldSquare = itsDragSquare;
	itsDragSquare = none;

	ProcessMouseMove( widget, &event );

	itsDragSquare = oldSquare;
	itsWidgetClick = oldClick;
	isDragging = oldDragging;
    }
}


void WidgetLayout::ResizeShorter()
{
    if( !itsSelectedWidget.isEmpty() )
    {
	QListIterator<Selected> iter( itsSelectedWidget );
	Selected* sel;
	
	while( (sel = iter()) )
	{
	    QRect widgetRect = sel->widget->geometry();
	    widgetRect.setBottom( widgetRect.bottom() - 
				  itsDlgEditWnd->GetGrid() );
	    
	    ResizeWidget( sel->widget, widgetRect );
	}
    }
}


void WidgetLayout::ResizeWider()
{
    if( !itsSelectedWidget.isEmpty() )
    {
	QListIterator<Selected> iter( itsSelectedWidget );
	Selected* sel;
	
	while( (sel = iter()) )
	{
	    QWidget* parent = sel->widget->parentWidget();
	    QRect widgetRect = sel->widget->geometry();
	    widgetRect.setRight( widgetRect.right() + 
				 itsDlgEditWnd->GetGrid() );

	    // make sure the widgetRect doesn't go off the widget frame
	    widgetRect.setRight( QMIN( widgetRect.right(), 
				       parent->width() - 1 ) );
	    
	    ResizeWidget( sel->widget, widgetRect );
	}
    }
}


void WidgetLayout::ResizeNarrower()
{
    if( !itsSelectedWidget.isEmpty() )
    {
	QListIterator<Selected> iter( itsSelectedWidget );
	Selected* sel;
	
	while( (sel = iter()) )
	{
	    QRect widgetRect = sel->widget->geometry();
	    widgetRect.setRight( widgetRect.right() - 
				 itsDlgEditWnd->GetGrid() );
	    
	    ResizeWidget( sel->widget, widgetRect );
	}
    }
}


void WidgetLayout::ResizeTaller()
{
    if( !itsSelectedWidget.isEmpty() )
    {
	QListIterator<Selected> iter( itsSelectedWidget );
	Selected* sel;
	
	while( (sel = iter()) )
	{
	    QWidget* parent = sel->widget->parentWidget();
	    QRect widgetRect = sel->widget->geometry();
	    widgetRect.setBottom( widgetRect.bottom() + 
				  itsDlgEditWnd->GetGrid() );

	    // make sure the widgetRect doesn't go off the widget frame
	    widgetRect.setBottom( QMIN( widgetRect.bottom(), 
					parent->height() - 1 ) );
	    
	    ResizeWidget( sel->widget, widgetRect );
	}
    }
}


void WidgetLayout::ResizeWidget( QWidget*	widget,
				 const QRect&	newRect )
{
    widget->move( newRect.left(), newRect.top() );

    // currently only the widget frame is not moveable
    if( itsNoMoveWidgets[ widget->name() ] != NULL )
    {
	QRect origRect = widget->geometry();
	QSize min = widget->minimumSize();
	QSize max = widget->maximumSize();
		
	// Only set the min/max sizes for the dimensions that changed
	if( origRect.width() != newRect.width() )
	{
	    widget->setMinimumSize( newRect.width(), min.height() );
	    widget->setMaximumSize( newRect.width(), max.height() );
	}
		
	if( origRect.height() != newRect.height() )
	{
	    widget->setMinimumSize( min.width(), newRect.height() );
	    widget->setMaximumSize( max.width(), newRect.height() );
	}

	emit WidgetWndChanged();
    }
    else
    {
	widget->resize( newRect.width(), newRect.height() );
    }
	    
	    
    // WidgetGeometryChange now called in ProcessResize since the 
    // widget was just resized in the previous code
    //WidgetGeometryChange( widget );
    emit Changed();
}


void WidgetLayout::ProcessMousePress
(
    QWidget* 		widget,
    QMouseEvent*	event
)
{
    switch( event->button() )
    {
	case LeftButton:
	{
	    if( (event->state() & ControlButton) && 
		(widget != itsDlgEditWnd ) )
	    {
		if( IsSelected( widget ) )
		{
		    RemoveFromSelection( widget );
		}
		else
		{
		    AddToSelection( widget );
		}
	    }
	    else if( !IsSelected( widget ) || (itsSelectedWidget.count() > 1) )
	    {
		ClearSelection();
		AddToSelection( widget );
	    }

	    int i;
	    itsDragSquare = none;
	    QPoint pos = event->pos();
	    Selected *sel = itsSelectedWidget.first();
	    
	    if( itsSelectedWidget.count() == 1 )
	    {
		for( i = 0; (i < 8) && (itsDragSquare == none); i++ )
		{
		    if( sel->itsResizeSquares[i].contains( pos ) )
		    {
			itsDragSquare = (enum ResizeSquare)i;
		    }
		}
	    }
	    UpdateCursor( widget );
	
	    itsWidgetClick = event->pos();
	    isDragging = TRUE;	
	
	    break;
	}
	
	case MidButton:
	{
	    ClearSelection();
	    emit WidgetDeselected();
	    break;
	}
		
	case RightButton:
	{
	    ClearSelection();
	    AddToSelection( widget );

	    DlgWidget* dlgWidget = GetDlgWidget( widget );
	    if( dlgWidget )
	    { 
		itsWidgetPopup->popup( QCursor::pos() );
	    }
	    else if( widget == itsDlgEditWnd )
	    {
		itsWidgetFramePopup->popup( QCursor::pos() );
	    }
	    
	    break;
	}
    }
}


void WidgetLayout::ProcessMouseDoubleClick( QWidget*		/*widget*/,
					    QMouseEvent*	event )
{
    if( event->button() == LeftButton )
    {
	SelectedWidgetProperties();
    }
}


void WidgetLayout::ProcessMouseMove
(
    QWidget*	 	widget,
    QMouseEvent*	event
)
{
    if( isDragging && 
	itsSelectedWidget.count() > 1 && 
	itsDragSquare == none )              // Moving only
    {
	QPoint oldWidgetPos = widget->pos();
	QRect  newWidgetRect = CalculateGeometryRect( widget, event->pos() );
	QPoint newWidgetPos = newWidgetRect.topLeft();
	bool   moveOKForAll = TRUE;
	int    maxWidth  = itsDlgEditWnd->width();
	int    maxHeight = itsDlgEditWnd->height();

	if( oldWidgetPos != newWidgetPos )
	{
	    QPoint delta = newWidgetPos - oldWidgetPos;
	    QListIterator<Selected> iter( itsSelectedWidget );
	    Selected *sel;

	    while( (sel = iter()) && moveOKForAll )
	    {
		// don't move one that's not allowed
		if( itsNoMoveWidgets[ sel->widget->name() ] )
		{
		    moveOKForAll = FALSE;
		}
		else
		{
		    if( sel->widget == widget )
		    {
			sel->newPos = newWidgetPos;
		    }
		    else
		    {
			sel->newPos = sel->widget->pos() + delta;
			int x = sel->newPos.x();
			int y = sel->newPos.y();

			if( x < 0 || x+sel->widget->width() > maxWidth ||
			    y < 0 || y+sel->widget->height() > maxHeight )
			{
			    moveOKForAll = FALSE;
			}
		    }
		}
	    }

	    if( moveOKForAll )
	    {
		iter.toFirst();

		while( (sel = iter()) )
		{
		    sel->widget->move( sel->newPos );
		}
	  
		WidgetWndChanged();
		WidgetGeometryChange( widget );
		emit Changed();
	    }      
	}
    } 
    else if( isDragging && itsSelectedWidget.first() &&
	     itsSelectedWidget.first()->widget == widget) 
    {
	// don't move the top left corner of a non movable widget
	if( ( itsNoMoveWidgets[ widget->name() ] == NULL ) ||
	    ( (itsDragSquare == right) || (itsDragSquare == bottomRight) ||
	      (itsDragSquare == bottom) ) )
	{
	    QRect rect = CalculateGeometryRect( widget, event->pos() );

	    ResizeWidget( widget, rect );
	}
    }

    UpdateCursor( widget );
}

void WidgetLayout::ProcessMouseRelease()
{
    isDragging = FALSE;
}


void WidgetLayout::ProcessPaint
(
    QWidget*		widget
)
{
    Selected findSel( widget );
    
    if( itsSelectedWidget.find( &findSel ) != -1 )
    {
	Selected* sel = itsSelectedWidget.current();

	QPainter painter( sel->widget );
	QColor color( sel->widget->foregroundColor() );
	
	DlgWidget* dlgWidget = GetDlgWidget( sel->widget );
	if( dlgWidget && dlgWidget->IsInLayout() )
	{
	    color = darkBlue;
	}

	painter.setPen( QPen( color, 3, DotLine ) );

	// draw border around widget
	painter.drawRect( painter.window() );

	QBrush brush( color, Dense4Pattern );
	// draw rects to resize
	int i;
	for( i=0; i < 8; i++ )
	{
	    painter.fillRect( sel->itsResizeSquares[i], brush );
	}
    }
}
	

void WidgetLayout::ProcessResize
(
    QWidget* 		widget,
    QResizeEvent*	// event
)
{
    Selected findSel( widget );
    if( itsSelectedWidget.find( &findSel ) != -1 )
    {	
	Selected *sel = itsSelectedWidget.current();
	QRect	rect = widget->geometry();
	QRect	sq( 0,0, 7,7 );
	QRect	null( 0,0, 0,0 );
	bool	moveable = (itsNoMoveWidgets[ widget->name() ]  == NULL );
    
	rect.moveTopLeft( QPoint(0,0) );

	sq.moveTopLeft( rect.topLeft());
	sel->itsResizeSquares[ topLeft	] = moveable ? sq : null;

	sq.moveBy( 0, rect.height()/2 - sq.height()/2 );
	sel->itsResizeSquares[ left		] = moveable ? sq : null;

	sq.moveBottomLeft( rect.bottomLeft() );
	sel->itsResizeSquares[ bottomLeft	] = moveable ? sq : null;

	sq.moveBy( rect.width()/2 - sq.width()/2, 0 );
	sel->itsResizeSquares[ bottom	] = sq;

	sq.moveBottomRight( rect.bottomRight() );
	sel->itsResizeSquares[ bottomRight	] = sq;

	sq.moveBy( 0, -(rect.height()/2 - sq.height()/2 ) );
	sel->itsResizeSquares[ right		] = sq;

	sq.moveTopRight( rect.topRight() );
	sel->itsResizeSquares[ topRight	] = moveable ? sq : null;

	sq.moveBy( -(rect.width()/2 - sq.width()/2), 0 );
	sel->itsResizeSquares[ top		] = moveable ? sq : null;

	WidgetGeometryChange( widget );
    }
}
    

void WidgetLayout::RaiseSelectedWidget()
{
    QListIterator<Selected> iter( itsSelectedWidget );
    Selected* sel;
    
    while( (sel = iter()) )
    {
	DlgWidget* dlgWidget = GetDlgWidget( sel->widget );
	if( itsZOrdering.removeRef( dlgWidget ) )
	{
	    itsZOrdering.append( dlgWidget );
	}
    
	emit Changed();
    }
}


void WidgetLayout::LowerSelectedWidget()
{
    QListIterator<Selected> iter( itsSelectedWidget );
    Selected* sel;
    
    while( (sel = iter()) )
    {
	sel->widget->lower();

	DlgWidget* dlgWidget = GetDlgWidget( sel->widget );
	if( itsZOrdering.removeRef( dlgWidget ) )
	{
	    itsZOrdering.insert( 0, dlgWidget );
	}
	emit Changed();
    }

    // The layout widgets must be lowered beneath the dialog widgets
    LowerLayout();
}


void WidgetLayout::DeleteSelectedWidget()
{
    // deleting a widget
    if( itsSelectedWidget.count() > 0 )
    {
	QListIterator<Selected> iter( itsSelectedWidget );
	Selected* sel;

	while( (sel = iter()) )
	{
	    DlgWidget* dlgWidget = GetDlgWidget( sel->widget );
	    if( dlgWidget )
	    {
		itsZOrdering.removeRef( dlgWidget );
	
		QString name = sel->widget->name();

		if( dlgWidget->IsInLayout() )
		{
		    if( itsLayoutType == Box )
		    {
			itsBoxLayout->RemoveWidget( dlgWidget );
			itsBoxLayout->Recreate();
		    }
		}

		// remove will delete dlgWidget but not the actual widget
		itsDlgWidgets.remove( name ); 
		delete sel->widget;
	
		itsNoMoveWidgets.remove( name );

		emit Changed();
	    }
	}
	itsSelectedWidget.clear();
    }
    
    if( itsSelectedLayoutWidget )
    {
	if( itsSelectedLayoutWidget == itsDlgEditWnd )
	{
	    SetLayoutType( None );
	}
	else
	{
	    switch( itsLayoutType )
	    {
		case None:
		{
		    break;
		}
		case Box:
		{
		    BoxLayout* layout = FindLayoutForSelectedItem();
		    if( layout )
		    {
			layout->RemoveItem( FindSelectedLayoutItem() );
		    }
		    
		    itsBoxLayout->Recreate();
		    
		    break;
		}
	    }

	    itsSelectedLayoutWidget = NULL;
	}
    }

    RemoveScreenFlags( SF_WidgetSelected | SF_Align | 
		       SF_Layout | SF_BoxLayout );
}

void WidgetLayout::SelectedWidgetProperties()
{
    if( itsSelectedWidget.count() > 1 )
	return;

    Selected *sel = itsSelectedWidget.first();
    DlgWidget* dlgWidget;

    if( sel && (dlgWidget = GetDlgWidget(sel->widget)) )
    {
	// Remember the name first
	QString origName( sel->widget->name() );

	// Find the layout the widget might be on
	BoxLayout* layout = NULL;
	if( itsLayoutType == Box )
	{
	    layout = 
		itsBoxLayout->FindLayout( 
		    GuiTools::GetGlobalGeometry( dlgWidget->GetWidget() ) );
	}

	// Remember widget's layout state
	bool wasInLayout = dlgWidget->IsInLayout();

	if( dlgWidget->DoPropertyDialog( layout ) == QDialog::Accepted )
	{
	    emit Changed();

	    if( itsLayoutType == Box )
	    {
		itsBoxLayout->Recreate();
	    }

	    // If different layout state, update
	    if( wasInLayout != dlgWidget->IsInLayout() )
	    {
		UpdateWidgetLayoutState( dlgWidget );
	    }

	    // Check the name and register if neccesary
	    QString newName( sel->widget->name() );

	    // user might have changed the widget's name or variable mapping
	    emit WidgetSelected( newName, dlgWidget->GetVariableName() );

	    if( newName != origName )
	    {
		if( itsDlgWidgets[newName] )
		{
		    QMessageBox::message( "Name conflict...",
					  "The new widget name is already in use\n"
					  "The old name is restored !",
					  "OK" );
		    sel->widget->setName( origName );
		} else
		{
		    dlgWidget = itsDlgWidgets.take( origName );
		    itsDlgWidgets.insert( newName, dlgWidget );
		}
	    }
	}
    }
    // Do properties on the whole window
    else if( (itsSelectedLayoutWidget == itsDlgEditWnd) ||
	     (itsLayoutType == None) )
    {
	if( itsParent->DoPropertyDialog() == QDialog::Accepted )
	{
	    emit Changed();

	    switch( itsLayoutType )
	    {
		case None:
		{
		    break;
		}
		case Box:
		{
		    itsBoxLayout->Recreate();

		    // update the layout status on the menu bar in case
		    // it was added
		    QString menuName = itsBoxLayout->GetMenuBarName();
		    if( !menuName.isEmpty() )
		    {
			UpdateWidgetLayoutState( GetWidget( menuName ) );
		    }
		    
		    break;
		}
	    }
	}
    }
    // Do properties on any layout items
    else
    {
	switch( itsLayoutType )
	{
	    case None:
	    {
		break;
	    }
	    case Box:
	    {
		LayoutItem* item = FindSelectedLayoutItem();
		
		if( item )
		{
		    if( item->EditProperties() )
		    {
			itsBoxLayout->Recreate();
		    }
		}
	    }
	}
    }
}

void WidgetLayout::OptionsGrid( )
{
    GridDialog gridDialog( NULL, "GridDialog", itsDlgEditWnd->GetGrid() );
    if( gridDialog.exec() == QDialog::Accepted )
    {
	itsDlgEditWnd->SetGrid( gridDialog.GetNewGrid() );
	emit Changed();
    }
}

void WidgetLayout::EnableLayout()
{

    if( ! itsDlgEditWnd )
	return;

    connect( itsDlgEditWnd, 
	     SIGNAL(MousePress(QWidget*,QMouseEvent*)),
	     SLOT(ProcessMousePress(QWidget*,QMouseEvent*)) );
    connect( itsDlgEditWnd,
	     SIGNAL(MouseDoubleClick(QWidget*,QMouseEvent*)),
	     SLOT(ProcessMouseDoubleClick(QWidget*,QMouseEvent*)) );
    connect( itsDlgEditWnd, 
	     SIGNAL(MouseMove(QWidget*,QMouseEvent*)),
	     SLOT(ProcessMouseMove(QWidget*,QMouseEvent*)) );
    connect( itsDlgEditWnd, 
	     SIGNAL(MouseRelease()), 
	     SLOT(ProcessMouseRelease()) );
    connect( itsDlgEditWnd, 
	     SIGNAL(Paint(QWidget*)),
	     SLOT(ProcessPaint(QWidget*)) );
    connect( itsDlgEditWnd, 
	     SIGNAL(Resize(QWidget*,QResizeEvent*)),
	     SLOT(ProcessResize(QWidget*,QResizeEvent*)) );
}
 
void WidgetLayout::DisableLayout()
{
    if( ! itsDlgEditWnd )
	return;

    itsDlgEditWnd->disconnect( this );

    ClearSelection();
}

void WidgetLayout::SetSelected
(
    QWidget*	widget,
    bool	selected
)
{
    if( widget )
    {
	widget->setMouseTracking( selected );
    
	widget->update();

	if( selected )
	{
	    WidgetGeometryChange( widget );

	    QString varName;
	    DlgWidget* dlgWidget = itsDlgWidgets[ widget->name() ];
	    if( dlgWidget )
	    {
		varName = dlgWidget->GetVariableName();
	    }
	    
	    emit WidgetSelected( widget->name(), varName );
	    
	    if( widget == itsDlgEditWnd )
	    {
		AddScreenFlags( SF_Layout );
	    }
	}
	// Unselect any selected layout widgets when the widget frame
	// is unselected
 	else if( (widget == itsDlgEditWnd) )
	{
	    if( itsSelectedLayoutWidget )
	    {
		QWidget* oldWidget = itsSelectedLayoutWidget;
		itsSelectedLayoutWidget = NULL;
		oldWidget->update();
	    }
	    
	    RemoveScreenFlags( SF_Layout | SF_BoxLayout );
 	}
    }
}

void WidgetLayout::ClearSelection
(
)
{
    QListIterator<Selected> iter( itsSelectedWidget );
    Selected* sel;
    
    while( (sel = iter()) )
    {
	SetSelected( sel->widget, FALSE );
	sel->widget->setCursor( arrowCursor );
    }

    itsSelectedWidget.clear();

    RemoveScreenFlags( SF_WidgetSelected | SF_Align );
}

void WidgetLayout::AddToSelection
(
    QWidget *widget
)
{
    if( ! IsSelected( widget ) )
    {
	itsSelectedWidget.append( new Selected( widget ) );
    
	SetSelected( widget, TRUE );
	ProcessResize( widget, NULL );
    }

    if( itsSelectedWidget.count() == 1 )
    {
	AddScreenFlags( SF_WidgetSelected );
	RemoveScreenFlags( SF_Align );
    }
    else
    {
	AddScreenFlags( SF_WidgetSelected | SF_Align );
    }
}

void WidgetLayout::RemoveFromSelection
(
    QWidget *widget
)
{
    QListIterator<Selected> iter( itsSelectedWidget );
    Selected *sel;

    while( (sel = iter()) && (sel->widget != widget) );

    if( sel )
    {
	// SetSelected( sel->widget, FALSE );
	//sel->widget->setCursor( arrowCursor );
	itsSelectedWidget.remove( sel );
	SetSelected( widget, FALSE );
	ProcessResize( widget, NULL );
    }

    if( itsSelectedWidget.count() == 1 )
    {
	AddScreenFlags( SF_WidgetSelected );
	RemoveScreenFlags( SF_Align );
    }
    else if( itsSelectedWidget.count() > 0 )
    {
	AddScreenFlags( SF_WidgetSelected | SF_Align );
    }
}


bool WidgetLayout::IsSelected( QWidget*	widget )
{
    Selected sel( widget );
    return (itsSelectedWidget.find( &sel ) >= 0);
}


void WidgetLayout::UpdateCursor
(
    QWidget*	widget
)
{
    if( !isDragging )
    {
	if( IsSelected( widget ) )
	{
	    Selected *sel = itsSelectedWidget.first();
	    QPoint cursorPos = widget->mapFromGlobal( QCursor::pos() );

	    ResizeSquare rs = none;
	    int i;
	
	    for( i = 0; (i < 8) && (rs == none); i++ )
	    {
		if( sel->itsResizeSquares[i].contains( cursorPos ) )
		{
		    rs = (ResizeSquare) i;
		}
	    }
	
	    widget->setCursor( *itsCursors[rs] );
	}
	else
	{
	    widget->setCursor( arrowCursor );
	}
    }
}


void WidgetLayout::WidgetGeometryChange( QWidget* widget )
{
    if( widget == itsDlgEditWnd )
    {
	emit WidgetGeometryChanged( 0, 0, 
				    widget->width(), widget->height() );
    }
    else
    {
	QPoint pos = GetDlgWidget( widget )->GetPos();
	
	emit WidgetGeometryChanged( pos.x(), pos.y(),
				    widget->width(), widget->height() );
    }
}


QRect WidgetLayout::CalculateGeometryRect
(
    QWidget*	widget,
    QPoint	pos
)
{
    QPoint eventPos = widget->mapToParent( pos );
    QRect rect;
    QRect widgetRect = widget->geometry();
    int minWidth = 10; //widget->minimumSize().width();
    int minHeight = 10; //widget->minimumSize().height();
    int grid = itsDlgEditWnd->GetGrid();
    int width, height;

    switch( itsDragSquare )
    {
	case none: // move widget	
	{
	    rect = widgetRect;
	    QPoint gridPoint = eventPos - itsWidgetClick;
	    if( grid > 1 )
	    {
		gridPoint /= grid;
		gridPoint *= grid;
	    }
	    rect.moveTopLeft( gridPoint );

	    QRect parentRect = widget->parentWidget()->geometry();

	    if( rect.left() < 0 )
	    {
		rect.moveTopLeft( QPoint(0,rect.top()) );
	    }
	    
	    if( rect.top() < 0 )
	    {
		rect.moveTopLeft( QPoint(rect.left(),0) );
	    }
	    
	    if( rect.right() > parentRect.width() )
	    {
		rect.moveBottomRight(QPoint(parentRect.width(),rect.bottom()));
	    }
	    
	    if( rect.bottom() > parentRect.height() )
	    {
		rect.moveBottomRight(QPoint(rect.right(),parentRect.height()));
	    }
	    
	    break;
	}

	case topLeft:
	{
	    width = widgetRect.right() - eventPos.x();
	    height = widgetRect.bottom() - eventPos.y();

	    if( grid > 1 )
	    {
		width /= grid; width *= grid;
		height /= grid; height *= grid;
	    }
	    rect.setWidth( QMAX( width, minWidth) );
	    rect.setHeight( QMAX( height, minHeight) );
	    rect.moveBottomRight( widgetRect.bottomRight() );
	    
	    break;
	}
	    
	case left:
	{
	    width = widgetRect.right() - eventPos.x();

	    if( grid > 1 )
	    {
		width /= grid; width *= grid;
	    }

	    rect = widgetRect;
	    rect.setWidth( QMAX(width, minWidth) );
	    rect.moveTopRight( widgetRect.topRight() );

	    break;
	}
	    
	case bottomLeft:
	{
	    width = widgetRect.right() - eventPos.x();
	    height = eventPos.y() - widgetRect.y();

	    if( grid > 1 )
	    {
		width /= grid; width *= grid;
		height /= grid; height *= grid;
	    }

	    rect.setWidth( QMAX(width, minWidth) );
	    rect.setHeight( QMAX(height,	minHeight) );
	    rect.moveTopRight( widgetRect.topRight() );
	    
	    break;
	}

	case bottom:
	{
	    height = eventPos.y() - widgetRect.y();

	    if( grid > 1 )
	    {
		height /= grid; height *= grid;
	    }

	    rect = widgetRect;
	    rect.setHeight( QMAX( height, minHeight ) );

	    break;
	}
	    
	case bottomRight:
	{
	    width = eventPos.x() - widgetRect.x();
	    height = eventPos.y() - widgetRect.y(); 

	    if( grid > 1 )
	    {
		width /= grid; width *= grid;
		height /= grid; height *= grid;
	    }

	    rect.setWidth( QMAX(width, minWidth) );
	    rect.setHeight( QMAX(height,	minHeight) );
	    rect.moveTopLeft( widgetRect.topLeft() );
	    
	    break;
	}
	    
	case right:
	{
	    width =  eventPos.x() - widgetRect.x();

	    if( grid > 1 )
	    {
		width /= grid; width *= grid;
	    }

	    rect = widgetRect;
	    rect.setWidth( QMAX( width, minWidth ) );

	    break;
	}
	    
	case topRight:
	{
	    width = eventPos.x() - widgetRect.x();
	    height = widgetRect.bottom() - eventPos.y();

	    if( grid > 1 )
	    {
		width /= grid; width *= grid;
		height /= grid; height *= grid;
	    }

	    rect.setWidth( QMAX(width, minWidth) );
	    rect.setHeight( QMAX(height, minHeight) );
	    rect.moveBottomLeft( widgetRect.bottomLeft() );

	    break;
	}
	    
	    
	case top:
	{
	    height = widgetRect.bottom() - eventPos.y();
			
	    if( grid > 1 )
	    {
		height /= grid; height *= grid;
	    }

	    rect = widgetRect;
	    rect.setHeight( QMAX(height, minHeight) );
	    rect.moveBottomLeft( widgetRect.bottomLeft() );

	    break;
	}
    }

    QRect parentRect = widget->parentWidget()->geometry();

    if( itsDragSquare == none )
    {
	if( rect.left() < 0 )
	{
	    rect.moveTopLeft( QPoint(0,rect.top()) );
	}
	    
	if( rect.top() < 0 )
	{
	    rect.moveTopLeft( QPoint(rect.left(),0) );
	}
	    
	if( rect.right() > parentRect.width() - 1 )
	{
	    rect.moveBottomRight(QPoint(parentRect.width() - 1,rect.bottom()));
	}
	    
	if( rect.bottom() > parentRect.height() - 1 )
	{
	    rect.moveBottomRight(QPoint(rect.right(),parentRect.height() - 1));
	}
    }
    else
    {	    
	parentRect.moveTopLeft( QPoint(0,0) );

	if( rect.left() < parentRect.left() )
	{
	    rect.setLeft( parentRect.left() );
	}

	if( rect.top() < parentRect.top() )
	{
	    rect.setTop( parentRect.top() );
	}

	if( rect.right() > parentRect.right() )
	{
	    rect.setRight( parentRect.right() );
	}
    
	if( rect.bottom() > parentRect.bottom() )
	{
	    rect.setBottom( parentRect.bottom() );
	}
    }
    

    return rect;
}


bool WidgetLayout::CopySelected( QTextStream &stream )
{
    if( itsSelectedWidget.isEmpty() )
	return FALSE;

    stream << "WidgetLayout {\n";

    QListIterator<Selected> iter( itsSelectedWidget );
    Selected* sel;

    while( (sel = iter()) )
    {
	DlgWidget *dlgWidget = GetDlgWidget( sel->widget );
	if( ! dlgWidget )
	    continue;

	dlgWidget->SaveContents( stream );
    }
    stream << "}" << endl;

    return TRUE;
}


bool WidgetLayout::CutSelected( QTextStream &stream )
{
    if( CopySelected( stream ) )
    {
	DeleteSelectedWidget();
	return TRUE;
    }

    return FALSE;
}


void WidgetLayout::RaiseDlgWidget
(
    const DlgWidget*	dlgWidget
)
{
    if( itsZOrdering.removeRef( dlgWidget ) )
    {
	itsZOrdering.append( dlgWidget );
    }
}


void WidgetLayout::LowerDlgWidget
(
    const DlgWidget*	dlgWidget
)
{
    if( itsZOrdering.removeRef( dlgWidget ) )
    {
	itsZOrdering.insert( 0, dlgWidget );
    }
}

    
QWidget* WidgetLayout::GetSelectedWidget( void )
{
    QWidget* widget = NULL;

    if( itsSelectedWidget.count() == 1 )
    {
	widget = itsSelectedWidget.first()->widget;
    }

    return widget;
}


DlgWidget* WidgetLayout::GetDlgWidget
(
    const QWidget*	widget 
)
{
    DlgWidget* dlgWidget = NULL;

    if( widget )
    {
	QString name( widget->name() );
	dlgWidget = itsDlgWidgets[ name ];	
    }

    return dlgWidget;
}


DlgWidget* WidgetLayout::GetSelectedDlgWidget( void )
{
    DlgWidget* dlgWidget = NULL;

    if( itsSelectedWidget.count() == 1 )
    {
	dlgWidget =  GetDlgWidget( itsSelectedWidget.first()->widget );
    }
    
    return dlgWidget;
}


void WidgetLayout::GetWidgetNames
( 
    QStrList&	nameList
) const
{
    QDictIterator<DlgWidget> iter( itsDlgWidgets );

    nameList.clear();
    while( iter.current() )
    {
	nameList.inSort( iter.currentKey() );
	++iter;
    }
}


void WidgetLayout::GetWidgetTypeNames( DlgWidget::Type	type,
				       QStrList&	nameList ) const
{
    QDictIterator<DlgWidget> iter( itsDlgWidgets );

    nameList.clear();
    while( iter.current() )
    {
	if( iter.current()->GetType() == type )
	{
	    nameList.inSort( iter.currentKey() );
	}
	
	++iter;
    }
}

    
void WidgetLayout::SetLayoutType
(
    WidgetLayout::LayoutType	type
)
{
    if( type != itsLayoutType )
    {
	// Clean up old setting
	switch( itsLayoutType )
	{
	    case None:
	    {
		break;
	    }
	    case Box:
	    {
		delete itsBoxLayout;
		itsBoxLayout = NULL;

		// update all the unmoveable widgets that may have been
		// freed up by the deletion
		QDictIterator<QWidget> iter( itsNoMoveWidgets );
		QWidget* widget;
		DlgWidget* dlgWidget;
		
		while( (widget = iter()) )
		{
		    dlgWidget = GetDlgWidget( widget );
		    
		    if( dlgWidget )
		    {
			UpdateWidgetLayoutState( dlgWidget );
		    }
		}

		RemoveScreenFlags( SF_BoxLayout );
		
		break;
	    }
	}
	
	// Set new setting	
	itsLayoutType = type;

	switch( itsLayoutType )
	{
	    case None:
	    {
		break;
	    }
	    case Box:
	    {
		itsBoxLayout = new BoxLayout( itsDlgEditWnd, 
					      this,
					      itsDlgEditWnd );
		itsBoxLayout->Recreate();

		itsSelectedLayoutWidget = itsDlgEditWnd;
		
		AddScreenFlags( SF_BoxLayout );

		break;
	    }
	}
    }
}


void WidgetLayout::InsertBoxLayout()
{
    if( itsLayoutType == None )
    {
	SetLayoutType( Box );
	AddScreenFlags( SF_BoxLayout );
    }
    else if( itsSelectedLayoutWidget )
    {
	BoxLayout* layout = 
	    itsBoxLayout->FindLayout( itsSelectedLayoutWidget );
	    
	if( layout )
	{
	    layout->AddBoxLayout();
	    itsBoxLayout->Recreate();
	    LowerLayout();
	}
    }
}


void WidgetLayout::InsertGridLayout()
{
}


void WidgetLayout::InsertBoxSpacing()
{
    if( itsSelectedLayoutWidget && (itsLayoutType == Box) )
    {
	BoxLayout* layout = itsBoxLayout->FindLayout(itsSelectedLayoutWidget);
	
	if( layout )
	{
	    layout->AddBoxSpacing();
	    itsBoxLayout->Recreate();
	    LowerLayout();
	}
    }
}


void WidgetLayout::InsertBoxStretch()
{
    if( itsSelectedLayoutWidget && (itsLayoutType == Box) )
    {
	BoxLayout* layout = itsBoxLayout->FindLayout(itsSelectedLayoutWidget);
	
	if( layout )
	{
	    layout->AddBoxStretch();
	    itsBoxLayout->Recreate();
	    LowerLayout();
	}
    }
}


void WidgetLayout::LayoutMousePress
(
    QWidget*		widget,
    QMouseEvent*	event
)
{
    if( widget != itsDlgEditWnd )
    {
	// Send a event to the widget frame as if the user had clicked on the
	// widget frame itself, and not one of the layout items.  The state of
	// the fake event does not have any sort of keyboard modifiers in it.
	QMouseEvent fakeEvent( event->type(), event->pos(), 
			       event->button(), event->button() );
    
	ProcessMousePress( itsDlgEditWnd, &fakeEvent );
    }

    switch( event->button() )
    {
	case LeftButton:	
	case RightButton:
	{
	    QWidget* oldSelected = itsSelectedLayoutWidget;
	    itsSelectedLayoutWidget = widget;

	    if( oldSelected )
	    {
		oldSelected->update();
	    }
	    
	    widget->update();

	    QString selName = itsDlgEditWnd->name();
	    LayoutItem* item = FindSelectedLayoutItem();

	    if( item )
	    {
		selName += " (" + item->GetItemName() + ")";
	    
		emit WidgetSelected( selName, "" );

		if( item->IsLayout() )
		{
		    AddScreenFlags( SF_Layout | SF_BoxLayout );
		}
		else if( itsSelectedLayoutWidget == itsDlgEditWnd )
		{
		    AddScreenFlags( SF_Layout );
		    RemoveScreenFlags( SF_BoxLayout );
		}
		else
		{
		    RemoveScreenFlags( SF_Layout | SF_BoxLayout );
		}
	    }
	    
	    break;
	}
	case MidButton:
	{
	    itsSelectedLayoutWidget = NULL;
	    
	    widget->update();

	    RemoveScreenFlags( SF_Layout | SF_BoxLayout );
	    
	    break;
	}
    }
}


void WidgetLayout::LayoutPaint
(
    QWidget*	widget
)
{
    if( (widget == itsSelectedLayoutWidget) &&
	(widget != itsDlgEditWnd) )
    {
	QPainter painter( widget );

	painter.setRasterOp( NotXorROP );
	painter.setPen( QPen( widget->foregroundColor(), 2, DotLine ) );
	painter.drawRect( painter.window() );
    }
}


void WidgetLayout::LowerLayout( void )
{
    switch( itsLayoutType )
    {
	case None:
	{
	    break;
	}
	case Box:
	{
	    itsBoxLayout->LowerWidget();
	    break;
	}
    }
}


LayoutItem* WidgetLayout::FindSelectedLayoutItem( void ) const
{
    LayoutItem* item = NULL;
    
    switch( itsLayoutType )
    {
	case None:
	{
	    break;
	}
	case Box:
	{
	    item = itsBoxLayout->FindLayoutItem( itsSelectedLayoutWidget );
	    break;
	}
    }
    
    return item;
}


BoxLayout* WidgetLayout::FindLayoutForSelectedItem( void ) const
{
    BoxLayout* layout = NULL;

    if( itsLayoutType == Box )
    {
	LayoutItem* item = FindSelectedLayoutItem();
	
	if( item )
	{
	    layout = itsBoxLayout->FindLayout( item );
	}
    }
    
    return layout;
}


void WidgetLayout::UpdateWidgetLayoutState
(
    const DlgWidget*	dlgWidget
)
{
    QWidget* widget = dlgWidget->GetWidget();

    if( dlgWidget->IsInLayout() )
    {
	itsNoMoveWidgets.insert( widget->name(), widget );
    }
    else
    {
	itsNoMoveWidgets.remove( widget->name() );
    }
    
    widget->repaint();
    ProcessResize( widget, NULL );
}


	
