/*
	HackTool -- A simple texteditor written with TOAD
	Copyright (C) 1999 by Mark-Andr Hopf

	- Some of the stuff in TMainWindow should be moved into a special class.
	- Use file locking to avoid some of the troubles I've had with joe
	- multiple files
	- multiple views
	- undo/redo
	- font selection
	- session management
	- the filedialog should stay in the last recently used directory
	  and offer a list of recently accessed directories
	- the message for `save file?' on exit states `before loading'
*/

#include <toad/toad.hh>
#include <toad/form.hh>
#include <toad/menubar.hh>
#include <toad/filedialog.hh>
#include <toad/dialog.hh>
#include <toad/pushbutton.hh>
#include <toad/textfield.hh>

#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>

#include <strstream>

#include <toad/textarea.hh>

static const string programname("HackTool");

class TMainWindow;

class TStatusBar:
	public TWindow
{
	public:
		TStatusBar(TWindow *, const string&, TMainWindow*, TTextArea*);
		
		void paint();
		void mouseLDown(int,int,unsigned);
		
		void update();
		
	protected:
		TMainWindow *main;
		TTextArea *ta;							// textarea in TMainWindow

		TTextField  *txt_row;
		int ypos;										// ypos for text
		int xpos_row;								// xpos for `Row:' text
};

class TMainWindow:
	public TForm
{
		typedef TForm super;
	public:
		TMainWindow(TWindow *, const string&);
		void menuNew();
		void menuOpen();
		bool menuSave();
		void menuSaveAs();
		void menuSearch();
		void menuAbout();
		void menuCopyright();

	protected:
		void closeRequest();
	
		TTextArea *ta;
		string filename;

		bool _Check();
		bool _Save(const string& title);
		
		void search(TStringManipulator*);
};

TBitmap *logo;

void CreateMemoryFiles();

int ToadMain()
{
	CreateMemoryFiles();
	logo = new TBitmap;
	try {
		logo->Load("memory://hacktool.png");
	} catch (exception &e) {
		cout << e.what() << endl;
		if (logo) {
			delete logo;
			logo = NULL;
		}
	}
	int result = TMainWindow(NULL, programname).Run();
	delete logo;
	return result;
}


TMainWindow::TMainWindow(TWindow *p, const string &t):
	super(p,t)
{
	SetSize(498, 374);

	TMenuBar *mb = new TMenuBar(this, "menubar");
		
		mb->BgnPulldown("File");
			CONNECT(mb->AddItem("New")->sigActivate, 
				this, menuNew);
			CONNECT(mb->AddItem("Open..")->sigActivate, 
				this, menuOpen);
			CONNECT(mb->AddItem("Save")->sigActivate, 
				this, menuSave);
			CONNECT(mb->AddItem("Save As..")->sigActivate,
				this, menuSaveAs);
			CONNECT(mb->AddItem("Exit")->sigActivate,
				this, closeRequest);
		mb->EndPulldown();
		
		mb->BgnPulldown("Edit");
			CONNECT(mb->AddItem("Search")->sigActivate,
				this, menuSearch);
		mb->EndPulldown();
		
		mb->BgnPulldown("Info");
			CONNECT(mb->AddItem("About")->sigActivate,
				this, menuAbout);
			CONNECT(mb->AddItem("Copyright")->sigActivate,
				this, menuCopyright);
		mb->EndPulldown();

	ta = new TTextArea(this, "textarea");
	TTextArea::TConfiguration config;
	config.tabwidth = 2;
	ta->SetConfiguration(&config);

	TStatusBar *sb = new TStatusBar(this, "statusbar", this, ta);
	CONNECT(ta->sigStatus, sb, update);
	
	Attach(mb, TOP | LEFT | RIGHT);
	Attach(ta, TOP, mb);
	Attach(ta, LEFT | RIGHT);
	Attach(ta, BOTTOM, sb);
	Attach(sb, LEFT | RIGHT | BOTTOM);
}

void TMainWindow::menuNew()
{
	if (!_Check())
		return;
		
	filename.erase();
	ta->SetValue("");
}

bool TMainWindow::_Check()
{
	if (ta->Modified()) {
		unsigned r = MessageBox(NULL,
									 "Open..",
									 "Do you want to save the current file before loading "
									 "a new one?",
									 TMessageBox::ICON_QUESTION |
									 TMessageBox::YES | TMessageBox::NO );
		if (r==TMessageBox::YES) {
			if (!menuSave())
				return false;
		} else if (r!=TMessageBox::NO) {
			return false;
		}
	}
	return true;
}

void TMainWindow::menuOpen()
{
	if (!_Check())
		return;

	TFileDialog dlg(this, "Open..");
	dlg.DoModal();
	if (dlg.GetResult()==IDOK) {
		string data;
		FILE *in = fopen(dlg.GetFilename().c_str(), "r");
		if (!in)
			return;
		int n;
		char buffer[8193];
		while(true) {
			n = fread(buffer, 1, 8192, in);
			if (n<=0)
				break;
			data.append(buffer,n);
		};
		fclose(in);
		ta->SetValue(data);
		filename = dlg.GetFilename();
		SetTitle(programname+ ": " + filename);
	}
}

bool TMainWindow::menuSave()
{
  string title = "Save";
  if (filename.size()==0) {
		TFileDialog dlg(this, title);
		dlg.DoModal();
    if (dlg.GetResult()==IDOK) {
			filename = dlg.GetFilename();
		} else {
			return false;
		}
  }
	return _Save(title);
}

void TMainWindow::menuSaveAs()
{
	string title = "Save As..";

	TFileDialog dlg(this, title);
	dlg.SetFilename(filename);
	dlg.DoModal();
	if (dlg.GetResult()==IDOK) {
		filename = dlg.GetFilename();
		if (_Save(title))
			SetTitle(programname+ ": " + filename);
  }
}

//. Save file and use `title' as the title for the message boxes
//---------------------------------------------------------------
bool TMainWindow::_Save(const string &title)
{
	struct stat st;

	bool makebackup = false;
		
	// check original filename
	//-------------------------
	if (stat(filename.c_str(), &st)==0) {
		if (S_ISDIR(st.st_mode)) {
			MessageBox(NULL, 
								 title, 
								 "You've specified a directory but not a filename.",
								 TMessageBox::ICON_STOP |
								 TMessageBox::OK);
			return false;
		}
		if (!S_ISREG(st.st_mode)) {
			MessageBox(NULL,
								 title,
								 "Sanity check. You haven't specified a regular file.",
								 TMessageBox::ICON_STOP |
								 TMessageBox::OK);
			return false;
		}
		makebackup = true;
	}
	
	// check backup filename
	//-----------------------
	string backupfile = filename+"~";
	if (stat(backupfile.c_str(), &st)==0) {
		if (!S_ISREG(st.st_mode)) {
			if (MessageBox(NULL,
								 		 title,
								 		 "I can't create the backup file.\n\n"
								 		 "Do you want to continue?",
										 TMessageBox::ICON_QUESTION |
								 		 TMessageBox::YES | TMessageBox::NO
								 		)	!= TMessageBox::YES)
			{
				return false;
			}
			makebackup = false;
		}
	}
	if (makebackup) {
		if (rename(filename.c_str(), backupfile.c_str())!=0) {
			if (MessageBox(NULL,
										 title,
										 "Failed to create the backup file.\n\n"
										 "Do you want to continue?",
										 TMessageBox::ICON_QUESTION |
										 TMessageBox::YES | TMessageBox::NO |
										 TMessageBox::BUTTON2) != TMessageBox::YES)
			{
				return false;
			}
		}
	}

	int fd = open(filename.c_str(), O_WRONLY|O_CREAT|O_TRUNC, 0644);
	if (fd==-1) {
		if (makebackup) {
			MessageBox(NULL,
								 title,
								 "Damn! I've failed to create the file.",
								 TMessageBox::ICON_EXCLAMATION | TMessageBox::OK);
			return false;
		}
	}
	
	string data = ta->Value();

	if (write(fd, data.c_str(), data.size())!=data.size()) {
		MessageBox(NULL,
							 title,
							 "Couldn't write everything...",
							 TMessageBox::ICON_EXCLAMATION | TMessageBox::OK);
		close(fd);
		return false;
	}
	close(fd);
	ta->SetModified(false);
	return true;
}

void TMainWindow::closeRequest()
{
	if (!_Check())
		return;
	super::closeRequest();
}

TTextField *tf;

void TMainWindow::menuSearch()
{
	static PDialog dlg;
//	TTextField *tf;
	TPushButton *btn;
	
	static string phrase;

	if (!dlg) {
		dlg = new TDialog(NULL, "Search");

		dlg->SetSize(175,34);
		tf  = new TTextField(dlg, "phrase", &phrase);
		tf->SetDirect(true);
		tf->SetShape(5,5,100,22);
		btn = new TPushButton(dlg, "do_search");
		btn->SetLabel("Search");
		btn->SetShape(110,5,60,22);
	
		CONNECT(tf->sigActivate, this, search, tf);
		CONNECT(btn->sigActivate, this, search, tf);
	}
	if (!dlg->IsRealized()) {
		dlg->DoModeless();
		dlg->SetVisible(true);
	} else {
		cout << "REALIZED\n";
	}
	dlg->SetVisible(true);
	dlg->RaiseWindow();
}

void TMainWindow::search(TStringManipulator *str)
{
	cout << "search phrase:" << str->Value() << endl;
	ta->Find(str->Value());
}

void TMainWindow::menuAbout()
{
 	MessageBox(this, "About",
		"HackTool v0.1\n"
 		"Copyright  1999 by Mark-Andr Hopf\n\n"
 		"Currently just another simple 8bit texteditor."
 		, MB_OK, logo);
}

void TMainWindow::menuCopyright()
{
 	MessageBox(this, "Copyright",
 		"HackTool v0.1\n"
 		"Copyright  1999 by Mark-Andr Hopf\n"
 		"eMail: hopf@informatik.uni-rostock.de\n\n"
 		"This program is free software; you can redistribute it and/or modify "
 		"it under the terms of the GNU General Public License as published by "
 		"the Free Software Foundation; either version 2 of the License, or "
 		"(at your option) any later version.\n\n"
 		"This program is distributed in the hope that it will be useful, "
 		"but WITHOUT ANY WARRANTY; without even the implied warranty of "
 		"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the "
 		"GNU General Public License for more details.\n\n"
 		"You should have received a copy of the GNU General Public License "
 		"along with this program; if not, write to the Free Software "
 		"Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA."
 		, MB_OK, logo);
}

// TStatusBar
//---------------------------------------------------------------------------
class TEditorControl
{
	public:
		TEditorControl(TTextArea *ta) {
			this->ta = ta;
			row = ta->CursorY();
		}

		unsigned row;
		void rowChanged() {
			row = ta->GotoLine(row);
			TDataManipulator::Assimilate(&row);
		}

	protected:
		TTextArea *ta;
};

TEditorControl *editor = NULL;

TStatusBar::TStatusBar(TWindow *p, 
											 const string &t,
											 TMainWindow *m,
											 TTextArea *tx):
	TWindow(p, t)
{
	main = m;
	ta = tx;
	editor = new TEditorControl(ta);
	SetBackground(TColor::DIALOG);
	SetSize(1,24);
	ypos = (Height()-DefaultFont().Height())>>1;
	xpos_row = Height()+5;

	txt_row = new TTextField(this, "row", &editor->row);
	txt_row->SetDirect(true);	// this should default to `true' in the future!!!
	CONNECT(txt_row->sigActivate, editor, rowChanged);
	CONNECT(txt_row->sigActivate, ta, SetFocus);
		
	txt_row->SetShape(xpos_row+DefaultFont().TextWidth("Row:")+3,2,
										40, Height()-4);
}

void TStatusBar::paint()
{
	TPen pen(this);

	pen.DrawString(xpos_row, ypos, "Row:");
	
	strstream str;
	
	str << "Col " << ta->CursorX()
			<< "   "
			<< "Lines " << ta->Lines();
	
	pen.DrawString(txt_row->XPos()+40+5,ypos, str.str(), str.pcount());
	
	if (ta->Modified()) {
		pen.SetColor(128,0,0);
		pen.FillCircle(4,4,Height()-8, Height()-8);
		pen.SetColor(TColor::DIALOG);
		pen.FillCircle(6,6,Height()-12, Height()-12);
		pen.SetColor(128,0,0);
		pen.FillCircle(8,8,Height()-16, Height()-16);
	} else {
		pen.DrawCircle(4,4,Height()-8, Height()-8);
	}
}

void TStatusBar::update()
{
	char buffer[255];
	sprintf(buffer, "%d", ta->CursorY());
	txt_row->SetValue(buffer);
	Invalidate();
}

void TStatusBar::mouseLDown(int x, int y, unsigned)
{
	TRect rect(4,4,Height()-8, Height()-8);
	if (ta->Modified() && rect.IsInside(x,y)) {
		main->menuSave();
	}
}
