/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

#include <toad/toadbase.hh>
#include <toad/pen.hh>
#include <toad/window.hh>

#include <toad/buttonbase.hh>

//!TButtonBase
//. The base class for <A HREF="TPushButton.html">TPushButton</A> and
//. <A HREF="TRadioButtonBase.html">TRadioButtonBase</A>.

/*---------------------------------------------------------------------------*
 | Constructor                                                               |
 *---------------------------------------------------------------------------*/
TButtonBase::TButtonBase(TWindow *parent, const string &title)
	:super(parent,title)
{
	bitmap = NULL;
	bDown=false;
	SetBackground(TColor::BTNFACE);
	SetSize(100, DefaultFont().Height() + 8);
}

TButtonBase::~TButtonBase()
{
	if (bitmap)
		delete bitmap;
}

/*---------------------------------------------------------------------------*
 | SetDown		                                                               |
 *---------------------------------------------------------------------------*/
//. `true' will move the button down and trigger `sigArm', `false' will release
//. the button and trigger `sigDisarm'.
void TButtonBase::SetDown(bool down)
{
	if (down==bDown) return;
	bDown=down;
	Invalidate();
	if (down)
		sigArm();
	else
		sigDisarm();
}

//. TButtonBase objects can display either a text label or a bitmap. When
//. this method succeeds the bitmap is used and `true' is returned.
bool TButtonBase::LoadBitmap(const string& name)
{
	if (IsRealized())
		Invalidate();
	if (!bitmap)
		bitmap = new TBitmap;
	try {
		bitmap->Load(name.c_str());
	}
	catch(...) {
		delete bitmap;
		bitmap = NULL;
		return false;
	}
	return true;
}

/*---------------------------------------------------------------------------*
 | DrawText	                                                                 |
 *---------------------------------------------------------------------------*/
//. Draws the labels text when no bitmap was loaded or the bitmap.
//. <H3>Attention</H3>
//. `DrawLabel' would be a better name now!
void TButtonBase::DrawText(TPen &pen,const string &text, bool bDown, bool bEnabled)
{
	int n=bDown?1:0;
	if (!bitmap) {
		int x = (Width()-pen.TextWidth(text)) >> 1;
		int y = (Height()-pen.Ascent()) >> 1;
		if(Enabled() && bEnabled)	{
			pen.SetColor(TColor::BTNTEXT);
			pen.DrawString(x+n, y+n, text);
			if (IsFocus()) {
				pen.SetLineStyle(TPen::DOT);
				pen.DrawRectangle(x+n-3,y+n-1,pen.TextWidth(text)+6,pen.Height()+1);
			}
		}	else {
			pen.SetColor(TColor::BTNLIGHT);
			pen.DrawString(x+1,y+1,text);
			pen.SetColor(TColor::BTNSHADOW);
			pen.DrawString(x,y,text);		
		}
	}	else {
		pen.DrawBitmap(
			(Width()-bitmap->width) / 2 + n,
			(Height()-bitmap->height) / 2 + n,
			bitmap);
	}
}
	
/*---------------------------------------------------------------------------*
 | DrawShadow                                                                |
 *---------------------------------------------------------------------------*/
//. Draws the buttons shadow.
void TButtonBase::DrawShadow(TPen &pen, bool bDown)
{
	int a=IsFocus()?1:0;

	TPoint p[6];
	if (bDown) {
		pen.SetColor(TColor::BTNSHADOW);
		p[0].Set(a,Height()-a);
		p[1].Set(a,a);
		p[2].Set(Width()-a,a);
		pen.DrawLines(p, 3);
	}	else {
		pen.SetColor(TColor::BTNSHADOW);
		p[0].Set(a,Height()-a-1);
		p[1].Set(Width()-a-1, Height()-a-1	);
		p[2].Set(Width()-a-1,	a 			 		);
		p[3].Set(Width()-a-2, a+1 				);
		p[4].Set(Width()-a-2,Height()-a-2	);
		p[5].Set(a+2				,	Height()-a-2	);
		pen.DrawLines(p, 6);

		pen.SetColor(TColor::BTNLIGHT);
		p[0].Set(a					, Height()-a-1 );
		p[1].Set(a					, a						);
		p[2].Set(Width()-a-1,	a						);
		p[3].Set(Width()-a-2, a+1					);
		p[4].Set(a+1				, a+1					);
		p[5].Set(a+1				, Height()-a-2	);
		pen.DrawLines(p, 6);
	}

	if (IsFocus()) {
		pen.SetColor(0,0,0);
		pen.DrawRectangle(0,0, Width(), Height());
	}
}
