/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

#ifndef TDataInterface
#define TDataInterface TDataInterface

// Note:
//  The type casts to and from `void*' may look dangerous but they will
// only be used when the `_data' attribute of all participating data
// interfaces do point at the same adress. So there're safe at last.

#include <string>

class TDataManipulator;

class TDataInterface
{
		friend TDataManipulator;
	public:
		// allocate memory for the global exchange point (GDataStorage)
		virtual void* Allocate() = 0;

		// return a pointer to the data to be modified by the interface
		virtual void* Pointer() const = 0;
	protected:
		static void* _global;		// global data exchange pointer for SetX & GetX
		virtual void _SetX(const void*) = 0;
};

template <class T>
class GDataInterface:
	public virtual TDataInterface
{
	public:
		// store & get data via the pointer this interface is associated with
		void SetValue(const T& v) {	Write(v, Pointer()); }
		T Value() const {	return Read(Pointer());	}

		// store & get data via the global exchange point
		void SetX(const T& v) {
			if (_global) free(_global);	// no destructor called...!!!
			_global = Allocate();
			Write(v, _global);
		}
		T GetX() const {
			return Read(_global);
		}

		void _SetX(const void *ptr) {	SetX( *((T*)ptr) );	}	// for `TDataManipulator::Assimilate'

	protected:
		// a subclass must implemented these to convert the data, the `ptr'
		// must be feed to `safe_cast' to get a valid pointer to be modified
		// by this interface
		virtual void Write(const T &data, void* ptr) = 0;
		virtual T Read(const void* ptr) const = 0;
};

// predefined interfaces
//-----------------------

class TStringInterface:
	public GDataInterface<string>
{
};

class TIntInterface:
	public GDataInterface<int>
{
};

class TBoolInterface:
	public GDataInterface<bool>
{
};

TStringInterface* NewStringInterface(string *d);
TStringInterface* NewStringInterface(int *d);
TStringInterface* NewStringInterface(unsigned *d);
TStringInterface* NewStringInterface(unsigned char *d);
TStringInterface* NewStringInterface(double *d);

TIntInterface* NewIntInterface(int *d);
TIntInterface* NewIntInterface(unsigned char*);

TBoolInterface* NewBoolInterface(bool *d);

#endif
