/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

#include <toad/toadbase.hh>
#include <toad/pen.hh>
#include <toad/window.hh>
#include <toad/dialog.hh>
#include <toad/dialogeditor.hh>
#include <toad/control.hh>
#include <toad/gadget.hh>
#include <toad/labelowner.hh>
#include <toad/io/urlstream.hh>

#include <map>

//!TDialog
//. TDialog can be used as a parent for dialog windows. It has some special
//. methods to control the children.

// Constructor
//---------------------------------------------------------------------------
TDialog::TDialog(TWindow* parent,const string &title):super(parent,title)
{
	SetBackground(TColor::DIALOG);
	bShell = bStaticFrame = true;
	bFocusManager = true;
	bDrawFocus = false;
	_placement = PLACE_PARENT_CENTER;
}

TDialog::~TDialog()
{
	// force dialog editor to store its data
  if (TDialogEditor::EditWindow()==this)
		TDialogEditor::SetEditWindow(NULL);
		
	if (TDialogEditor::running && bDialogEditRequest)	{
		try {
			string filename = ResourcePrefix() + "dialog#" + _resource_name;
			ourlstream url(filename);
			TOutObjectStream out(&url);
			Store(out);
		}
		catch(exception &e) {
			cerr << "exception while loading resource:" << e.what() << endl;
		}
	}
}

// paint
//---------------------------------------------------------------------------
void TDialog::paint()
{
	super::paint();
	if (bDrawFocus) {
		TWindow *wnd = FocusWindow();
		if (wnd && wnd->Parent()==this) {
			TPen pen(this);
			TRectangle r;
			wnd->GetShape(&r);
			pen.DrawRectangle(r.x-3,r.y-2,r.w+6,r.h+4);
			pen.DrawRectangle(r.x-2,r.y-3,r.w+4,r.h+6);
		}
	}
}

void TDialog::childNotify(TWindow *who, EChildNotify type)
{
	if (type==TCHILD_FOCUS && bDrawFocus) {
		TRectangle r;
		who->GetShape(&r);
		r.x-=3; r.w+=6;
		r.y-=3; r.h+=6;
		Invalidate(r);
	}
}

//. When you want to use TOADs dialog editor you'll have to call this
//. method after all children are created. The `resource_name' will
//. be used to identify the layout information e.g. the class name of 
//. the dialog.
//---------------------------------------------------------------------------
void TDialog::DoLayout(const string &resource_name)
{
	bDialogEditRequest = true;
	_resource_name=resource_name;

	try {
		string filename = ResourcePrefix() + "dialog#" + _resource_name;
		iurlstream url(filename);
		TInObjectStream in(&url);
		Restore(in);
	}
	catch(exception &e) {
		cerr << "exception while loading resource:" << e.what() << endl;
	}

	Arrange();
}

void TDialog::Store(TOutObjectStream &out)
{
	out.WriteDWord(6);											// header size
	out.WriteWord(Width());									// 2 byte
	out.WriteWord(Height());								// 2 byte
	out.WriteByte(bDrawFocus ? 255 : 0);		// 1 byte
	out.WriteByte(0);												// 1 byte (dummy)
	TGadgetWindow::Store(out);
}

void TDialog::Restore(TInObjectStream &in)
{
	ulong header_size  = in.ReadDWord();
	ulong header_start = in.TellRead();
	unsigned w = in.ReadWord();
	unsigned h = in.ReadWord();
	SetSize(w,h);
	bDrawFocus = in.ReadByte();
	in.SeekRead(header_start + header_size);
	TGadgetWindow::Restore(in);
}

typedef map<string, TWindow*> TTitleWindowMap;

// see `TDialog::Arrange' for more information
static void ArrangeHelper(TGadgetWindow::TGadgetStorage::iterator p, 
													TGadgetWindow::TGadgetStorage::iterator e,
													TTitleWindowMap &wmap)
{
	while(p!=e) {
		TGWindow *gw = dynamic_cast<TGWindow*>(*p);
		if (gw) {
			TTitleWindowMap::iterator pm = wmap.find(gw->title);
			if (pm!=wmap.end()) {
				gw->window = (*pm).second;
				TRectangle r;
				gw->getShape(r);
				gw->window->SetShape(r.x,r.y,r.w,r.h);
				TLabelOwner *lo = dynamic_cast<TLabelOwner*>(gw->window);
				if (lo) {
					lo->SetLabel(gw->label);
					gw->label.erase();
				}
				gw->window->taborder = gw->taborder;
				wmap.erase(pm);
			}
		}
		TGGroup *gg = dynamic_cast<TGGroup*>(*p);
		if (gg) {
			ArrangeHelper(gg->gadgets.begin(), gg->gadgets.end(), wmap);
		}
		p++;
	}
}

// Called from `DoLayout' after `Restore'.<P>
// 
void TDialog::Arrange()
{
	// create a map of all child windows
	//-----------------------------------
	TTitleWindowMap wmap;
	TWindow *wnd = FirstChild();
	while(wnd) {
		if (wmap.find(wnd->Title())!=wmap.end()) {
			cerr << "TDialog: child title \"" << wnd->Title() << "\" isn't unique" << endl;
		} else {
			wmap[wnd->Title()] = wnd;
		}
		wnd=TWindow::NextSibling(wnd);
	}

	// arrange all child windows with the information store 
	// in their related TGWindows
	//-----------------------------------------------------
	ArrangeHelper(gadgets.begin(), gadgets.end(), wmap);

	// add TGWindow's for the remaining windows
	//-----------------------------------------------------
	TRectangle noshape(4,4,33,33);
	TTitleWindowMap::iterator p,e;
	p = wmap.begin();
	e = wmap.end();
	while(p!=e) {
		TGWindow *gw = new TGWindow();
		gw->title  = (*p).second->Title();
		gw->window = (*p).second;
		gw->SetShape(noshape.x, noshape.y, noshape.w, noshape.h);
		gw->window->SetShape(noshape.x, noshape.y, noshape.w, noshape.h);
		gadgets.push_back(gw);
		noshape.x+=36;
		if (noshape.x>Width()) {
			noshape.x=4;
			noshape.y+=36;
		}
		p++;
	}
}

//. Create window as a modal dialog and wait until the dialog is closed
//. again.
//---------------------------------------------------------------------------
void TDialog::DoModal()
{
	TOADBase::DoModal(this);
	Flush();
}

//. Create window as a modeless dialog.
//---------------------------------------------------------------------------
void TDialog::DoModeless()
{
	TOADBase::DoModeless(this);
	Flush();
}

// adjust
//---------------------------------------------------------------------------
void TDialog::adjust()
{
	if (!bShell)	// uhh, someone is playing tricks with us, so we stop playing
		return;			// tricks on our own

	TOADBase::PlaceWindow(this, _placement, Parent());
}

// closeRequest
//---------------------------------------------------------------------------
class TActionEndDialog:
	public TAction
{
		TDialog *_dialog;
	public:
		TActionEndDialog(TDialog *d) { _dialog = d; }
		void execute() { TOADBase::EndDialog(_dialog); }
};

void TDialog::closeRequest()
{
	// modeless dialogs must be destroyed from outside
	SendMessage( new TActionEndDialog(this) );
}

//. Calls 'Apply()' for all TControl children.<BR>
//. !!!get the better code from dialogeditor/simpledlgeditor.cc!!!
//---------------------------------------------------------------------------
void TDialog::Apply()
{
	TWindow *p = FirstChild();
	while(p) {
		TDataManipulator *ctrl = dynamic_cast<TDataManipulator*>(p);
		if (ctrl)
			ctrl->Apply();
		p = NextSibling(p);
	}
}

//. Calls 'Reset()' for all TControl children.
//---------------------------------------------------------------------------
void TDialog::Reset()
{
	TWindow *p = FirstChild();
	while(p) {
		TDataManipulator *ctrl = dynamic_cast<TDataManipulator*>(p);
		if (ctrl)
			ctrl->Reset();
		p = NextSibling(p);
	}
}

//. Calls 'Apply()' and close the dialog.
//---------------------------------------------------------------------------
void TDialog::Ok()
{
	Apply();
	closeRequest();
}
