/*
 * TOAD -- A Simple and Powerful C++ GUI Toolkit for the X Window System
 * Copyright (C) 1996-99 by Mark-Andr Hopf
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
 * MA  02111-1307,  USA
 */

/*
    Focus Manager version 3.1
    -------------------------

		For every top level window (aka shell window, aka child of the root
		window) TOAD creates a focus domain (TDomain) object with informations
		on how to distribute keyboard events to the top level window and most of
		its' children.
		
		The focus traversal is limited to the domains and sub domains,
		which can be part of the top level windows' domain.
		
		All sub domains are linked in a tree structure whose root node is
		the top domain.
		
		ATTENTION:
		Errors are bound to happen when the TWindow attributes `bFocusManager'
		and `bShell' are modified between calls to `Create()' and `Destroy()'.
		
		STUFF TO ADD:
		A sub domain should store it's last focus and reactivate it when it
		gets the focus again.
		
		Since the X11 support for internationalized text input is bound to the
		focus management, the i18n related code is placed here also.
*/

/*
	A VERY IMPORTANT THING TO DO IS:
	- to modify `XNFocusWindow' when the focus changes
*/

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xlocale.h>

#define _TOAD_PRIVATE

#include <toad/toadbase.hh>
#include <toad/pen.hh>
#include <toad/window.hh>
#include <toad/dialogeditor.hh>

#include <map>
#include <assert.h>

static XIM xim = NULL;
static XIMStyle xim_style;

XIC xic_current = NULL;

struct TDomain
{
	TDomain() {
		owner=focus_window=NULL; focus=next=first_child=parent=NULL;
		xic = NULL;
	}
	~TDomain() {
		if (xic)
			XDestroyIC(xic);
	}
	TWindow *owner, *focus_window;
	TDomain *focus, *next, *first_child, *parent;
	XIC xic;
};

typedef map<TWindow*,TDomain*> TDomainMap;
static TDomainMap top_domain_map;

static TDomain *current_domain = NULL;

static void AddSubDomain(TWindow*);
static void DelSubDomain(TWindow*);
static TDomain* GetDomain(TWindow*);
static TDomain* GetTopDomain(TWindow *wnd);
static TWindow* Walk(TWindow *top, TWindow *focus, bool bNext);
static TDomain* SetPathTo(TWindow *wnd, TWindow *who);

// called everytime a new window is created on the screen and it creates
// a new domain if necessary
//---------------------------------------------------------------------------
void TOADBase::FocusNewWindow(TWindow* wnd)
{
	assert(wnd!=NULL);
	
	if (wnd->bShell) {
//if(wnd->bPopup) cout << "new popup: " << wnd->Title() << endl;
		// create a new domain for window `wnd'
		//----------------------------------------------------------------
		TDomainMap::iterator dp = top_domain_map.find(wnd);
		if (dp!=top_domain_map.end()) {
			cerr<< __FILE__ << ":" << __LINE__
					<< ": toad internal warning: domain is already there\n";
			return;
		}

		TDomain *domain = new TDomain();
		domain->owner = wnd;
		if (xim) {
			domain->xic = XCreateIC(xim,
															XNInputStyle, xim_style,
															XNClientWindow, wnd->x11window,
															XNFocusWindow, wnd->x11window,
															NULL);
			if (domain->xic==NULL)
				cerr << "toad: Couldn't create X Input Context for window \"" 
						 << wnd->Title() << "\"" << endl;
		}

		top_domain_map[wnd]=domain;
	} else if (wnd->bFocusManager) {
		// add a new sub domain for window `wnd' in the windows' top level 
		// window parents' domain
		//----------------------------------------------------------------
		AddSubDomain(wnd);
	}
}

void TOADBase::FocusDelWindow(TWindow* wnd)
{
	assert(wnd!=NULL);

	TDomain *domain = GetTopDomain(wnd);

	// when the window is a `normal' window, one without a focus domain,
	// then clear the focus window pointers in it's superior domains if
	// necessary
	//------------------------------------------------------------------
	if (!wnd->bFocusManager && !wnd->bShell) {
		// check top level domain
		if (domain->focus_window==wnd) {
			domain->focus_window=NULL;
			wnd->_SetFocus(false);
		}
		
		// check sub domain
		TDomain *subdomain = GetDomain(wnd);
		if (subdomain->focus_window==wnd)
			subdomain->focus_window=NULL;
		
		return;
	}
	
	// remove domain
	//------------------------------------------------------------------
	if (wnd->bShell) {
//if(wnd->bPopup) cout << "del popup: " << wnd->Title() << endl;
		assert(domain->first_child==NULL);
		TDomainMap::iterator dp = top_domain_map.find(wnd);
		top_domain_map.erase(dp);
		if (current_domain == domain)
			current_domain = NULL;
		delete domain;
		return;
	}
	
	// remove sub domain
	//------------------------------------------------------------------
	if (wnd->bFocusManager)
		DelSubDomain(wnd);
}

void TOADBase::DomainToWindow(TWindow *wnd)
{
	TDomain *new_domain = wnd ? GetTopDomain(wnd) : NULL;

	if (new_domain!=current_domain) {
	
		// deactivate old domain
		//-----------------------
		if (current_domain) {
			if (current_domain->xic) {
				xic_current = NULL;
				XUnsetICFocus(current_domain->xic);
			}
			// deactivate subtree
			TDomain *p = current_domain;
			while(p) {
				p->owner->_SetFocus(false);
				p=p->focus;
			}
			
			if (current_domain->focus_window)
				current_domain->focus_window->_SetFocus(false);
		}

		current_domain = new_domain;

		// activate new domain
		//---------------------
		if (current_domain) {
			if (current_domain->xic) {
				xic_current = current_domain->xic;
				XSetICFocus(current_domain->xic);
			}

			// activate subtree
			TDomain *p = current_domain;
			while(p) {
				p->owner->_SetFocus(true);
				p=p->focus;
			}

			// just in case there is no focus window: find one
			//-------------------------------------------------
			// OUCH!!! The same code is duplicated below in `KeyDown'
			if (!current_domain->focus_window) {
				wnd = Walk(current_domain->owner,	NULL,	true);
				if (!wnd)
					wnd=current_domain->owner;
				SetFocusWindow(wnd);
			}
		
			if (current_domain->focus_window)
				current_domain->focus_window->_SetFocus(true);
		}
	}
}

void TOADBase::SetFocusWindow(TWindow* wnd)
{
	assert(wnd!=NULL);

//	cout << "SetFocusWindow" << endl;
	if (wnd->bNoFocus && !wnd->bFocusManager && !wnd->bShell) {
#if 0
		cout << "window \"" << wnd->Title() << "\" doesn't need focus" << endl;
		cout << "  bNoFocus     : " << (wnd->bNoFocus?"true":"false") << endl;
		cout << "  bFocusManager: " << (wnd->bFocusManager?"true":"false") << endl;
		cout << "  bShell       : " << (wnd->bShell?"true":"false") << endl;
#endif
		return;
	}
	
	if (!wnd->IsRealized()) {
#if 0
		cerr << "window \"" << wnd->Title() << "\": can't set focus because parent isn't realized yet\n"
						"  (don't worry, it's TOADs' fault)\n";
#endif
		return;
	}
	
	TDomain *top_domain = GetTopDomain(wnd);
	assert(top_domain);
	if (top_domain->focus_window==wnd)
		return;

	// when `wnd' is a window with a focus domain try to take one of its
	// children
	//------------------------------------------------------------------
	if (wnd->bFocusManager || wnd->bShell) {
		// set focus to an inferior window that wants to receive keyboard
		// events
		TWindow *inferior = Walk(wnd, NULL, true);
		if (inferior) {
			wnd = inferior;
		} else {
			// there is no inferior window, instead walk down the list of
			// domains
			TDomain *bottom = GetDomain(wnd);
			while(bottom->first_child) {
				if (!bottom->focus)
					bottom->focus = bottom->first_child;
				bottom = bottom->focus;
			}
			wnd = Walk(bottom->owner, NULL, true);
		}
	}

	// set a new path to `wnd' in the domain tree
	if (wnd)
		SetPathTo(wnd, wnd);
}

TDomain* SetPathTo(TWindow *wnd, TWindow *new_focus)
{
	// top of recursion: return the top level domain 
	// and set new focus window
	//-----------------------------------------------
	if (wnd->bShell) {
		TDomainMap::iterator dp = top_domain_map.find(wnd);
		assert(dp!=top_domain_map.end());
		TWindow *focus = (*dp).second->focus_window;
		if (focus && focus!=new_focus)
			focus->_SetFocus(false);
		(*dp).second->focus_window = new_focus;
		return (*dp).second;
	}
	
	// recurse upwards
	//-----------------------------------------------
	TDomain *superior_domain = SetPathTo(wnd->Parent(), new_focus);
	
	if (wnd->bFocusManager) {
		TDomain *domain = superior_domain->first_child;
		while(domain) {
			if (domain->owner == wnd)
				break;
			domain = domain->next;
		}
		assert(domain!=NULL);
		
//		cout << "superior domain points to " << (superior_domain->focus ? superior_domain->focus->owner->Title() : "(NULL)") << endl;
//		cout << "  new focus will be " << wnd->Title() << endl;
		if (superior_domain->focus && superior_domain->focus->owner != wnd) {
			// deactivate old focus subtree
//			cout << "deactivating old subtree" << endl;
			TDomain *p = superior_domain->focus;
			while(p) {
				p->owner->_SetFocus(false);
				p = p->focus;
			}
			superior_domain->focus = domain;
		}

//		cout << "calling _SetFocus(true) in " << domain->owner->Title() << endl;
		domain->owner->_SetFocus(true);
	
		return domain;
	}	
	
	if (wnd==new_focus)
		wnd->_SetFocus(true);
	
	return superior_domain;
}

TWindow* TOADBase::FocusWindow()
{
	return current_domain ? current_domain->focus_window : NULL;
}

// called from the message loop to distribute the `keyDown' event
void TOADBase::KeyDown(TKey key, char* t, unsigned m)
{
	if (!current_domain)		// paranoia check
		return;

	// focus traversal
	//-------------------------------------------------
	TWindow *wnd = NULL;
	
	if ((key==TK_TAB && !(m & MK_SHIFT)) || 
			key==TK_F8) {
		wnd = Walk( current_domain->focus_window
									? GetDomain(current_domain->focus_window)->owner
									: current_domain->owner,
								current_domain->focus_window,
								true );
	} else if (key==TK_LEFT_TAB || 
						(key==TK_TAB && (m & MK_SHIFT)) ||
						key==TK_F7 ) 
	{
		wnd = Walk( current_domain->focus_window
									? GetDomain(current_domain->focus_window)->owner
									: current_domain->owner,
								current_domain->focus_window,
								false );
	}

	if ( (key==TK_TAB || key==TK_LEFT_TAB) && 
				current_domain->focus_window &&
				current_domain->focus_window->bTabKey )
		wnd = NULL;

	if (wnd && wnd!=current_domain->focus_window) {
		SetFocusWindow(wnd);
		return;
	}

	// just in case there is no focus window: find one
	//-------------------------------------------------
	if (!current_domain->focus_window) {
		wnd = Walk(current_domain->owner,	NULL,	true);
		// BAD STYLE!!!
		if (!wnd) {	// // make exception for single toplevel window apps :(
			if (!current_domain->owner->bNoFocus)
				wnd=current_domain->owner;
		}
		if (wnd) {
			current_domain->focus_window = wnd;
			current_domain->focus_window->_SetFocus(true);
		}
	}
		
	// dispatch the key event
	//------------------------
	if (current_domain->focus_window)
		current_domain->focus_window->keyDown(key,t,m);
}

void TOADBase::KeyUp(TKey key, char* t, unsigned m)
{
	if (current_domain && current_domain->focus_window)
		current_domain->focus_window->keyUp(key,t,m);
}


//---------------------------------------------------------------------------
void AddSubDomain(TWindow *wnd)
{
	assert(wnd!=NULL);
	
	// get the windows domain
	//------------------------
	TDomain *domain = GetDomain(wnd);
	assert(domain!=NULL);
	
	// create a new domain and add it to the windows' domain
	//------------------------------------------------------
	TDomain *new_domain = new TDomain();
	new_domain->owner = wnd;
	new_domain->parent = domain;
	new_domain->next = domain->first_child;
	domain->first_child = new_domain;
	
	// the first domain added to another domain gets the focus
	//--------------------------------------------------------
	if (!domain->focus)
		domain->focus = new_domain;
}

void DelSubDomain(TWindow *wnd)
{
	assert(wnd!=NULL);
	
	TDomain *domain = GetDomain(wnd);
	assert(domain!=NULL);
#if 0
cout << "domain->owner:" << domain->owner->Title() << endl;
cout << "popup?       :" << domain->owner->bPopup << endl;
cout << "domain for   :" << wnd->Title() << endl;
#endif
	assert(domain->owner==wnd);		// window isn't a focus manager
#if 0
if (domain->first_child)
	cout << "uh, remaining domain for window " << domain->first_child->owner->Title() << endl;
#endif
	assert(domain->first_child==NULL);	// should be removed already
	
	// remove `domain' from the parent domain
	//----------------------------------------
	TDomain *p1 = domain->parent->first_child;
	TDomain *p2;
	
	if (p1==domain) {
		domain->parent->first_child = p1->next;
	} else {
		while(true) {
			p2 = p1->next;
			assert(p2);								// couldn't find domain in parent
			if (p2==domain) {
				p1->next = p2->next;
				break;
			}
			p1 = p2;
		}
	}
	
	// when necessary modify the parent domains focus
	//-----------------------------------------------
	if (domain->parent->focus == domain)
		domain->parent->focus = domain->parent->first_child;

	// okay, remove the domains data
	//------------------------------
	delete domain;
}

TDomain* GetDomain(TWindow *wnd)
{
	assert(wnd!=NULL);

	// this method is called recursive and the recursion stops when we've
	// reached the top level window	
	//-------------------------------------------------------------------
	if (wnd->bShell) {
		TDomainMap::iterator p = top_domain_map.find(wnd);
		assert(p!=top_domain_map.end());	// shell without top domain
		return (*p).second;
	}

	TDomain *domain = GetDomain(wnd->Parent());

	// in most situations the domain we wanted to find is now available in
	// `domain'; but when we encounter a sub domain on our way down from
	// top level window, we do a loop to find it and return it instead of
	// the domain found earlier
	//--------------------------------------------------------------------
	if (wnd->bFocusManager) {
		TDomain *p = domain->first_child;
		while(p) {
			if (p->owner == wnd)
				return p;
			p = p->next;
		}
	}
	return domain;
}

TDomain* GetTopDomain(TWindow *wnd)
{
	assert(wnd!=NULL);
	
	while(wnd->Parent() && !wnd->bShell)
		wnd = wnd->Parent();
		
	TDomainMap::iterator dp = top_domain_map.find(wnd);
	assert(dp!=top_domain_map.end());
	return (*dp).second;
}

TWindow* Walk(TWindow *top, TWindow *focus, bool bNext)
{
	assert(top!=NULL);
	
	TWindow *old_focus = focus;
	
	if (!focus)
		focus=top;

	TWindow *ptr = focus;
	
	do {
		if ( ptr->FirstChild() &&					// go down when `ptr' has a child
				 !(ptr->bShell&&ptr!=top) &&	// but skip alien focus domains
				 ptr->bFocusTraversal &&			// and windows that don't want to be part of the focus traversal
				 ptr->IsRealized() &&					// and when the window is mapped
				 ( ptr->bNoFocus							// and only go down when window rejects focus
					 || ptr->bFocusManager      //   or when it's a focusmanager
					 || ptr->bShell							//   or a shell window
				))
		{
			// go one step down
			bNext ? ptr = ptr->FirstChild() : ptr = ptr->LastChild();
		} else {
			// couldn't go downwards and can't go sideways or up either because
			// `ptr' is the `top' window and we have to stay in the tops' subtree
			// so leave =>
			if (ptr==top)
				return NULL;
				
			// try to get the next/previous sibling of the window; when there is
			// no such sibling walk upwards until there's one
			TWindow *next;
			while( bNext ? (next=TWindow::NextSibling(ptr))==NULL
									 : (next=TWindow::PrevSibling(ptr))==NULL )
			{
				// no next/prev sibling so go one step upwards
				ptr = ptr->Parent();
				
				// when we've reached the top of the tree, leave the loop to
				// walk down again
				if (ptr==top) {
					next = top;
					break;
				}
			}
			ptr = next;
			// home run, we haven't found another focus window => leave
			if (ptr==focus)
				return old_focus;
		}
	} while(
			ptr->bNoFocus ||					// continue when window doesn't want the focus
			!ptr->IsRealized() ||			// or isn't mapped
			(ptr!=top && ptr->bShell) // continue when window belongs to another domain
	);
		
	return ptr;		
}

/*
 *
 * The X Input Context is bound to the focus management so it's here:
 *
 */

// Find and create an X11 Input Context
//---------------------------------------------------------------------------
void TOADBase::InitXInput()
{
	int idx, quality;

	if (!XSupportsLocale()) {
		cerr << "X doesn't support locale " << setlocale(LC_ALL,NULL) << endl;
		return;
	}

	if (XSetLocaleModifiers("")==NULL) {
		cerr << "Can't set locale modifier" << endl;
		return;
	}
	
	if ((xim = XOpenIM(TOADBase::x11display, NULL, NULL, NULL))==NULL) {
		cerr << "Failed to open X Input Method for locale " << 
						setlocale(LC_ALL,NULL) << endl;
		return;
	}

	XIMStyles *xim_styles=NULL;

	if (XGetIMValues(xim, XNQueryInputStyle, &xim_styles, NULL)) {
		cerr << "XGetIMValues failed" << endl;
		goto error1;
	}
	if (!xim_styles) {
		cerr << "Input method doesn't support any style" << endl;
		goto error1;
	}

	idx = -1;
	quality = 0;
	for (unsigned i=0; i<xim_styles->count_styles; i++) {
		if (quality<5 && xim_styles->supported_styles[i] & 
					(XIMPreeditNone|XIMStatusNone|XIMPreeditNothing|XIMStatusNothing) )
		{
			idx=i; quality = 5;
		}
		if (quality<10 &&	xim_styles->supported_styles[i] &
					(XIMPreeditNone|XIMStatusNone) )
		{
			idx=i; quality = 10;
		}
		if (quality<15 &&	xim_styles->supported_styles[i] &
					(XIMPreeditNothing|XIMStatusNothing) )
		{
			idx=i; quality = 15;
		}
	}

	if (idx==-1) {
		cerr << "Didn't found a X Input Method TOAD can handle" << endl;
		goto error1;
	}

	xim_style = xim_styles->supported_styles[idx];

	if (xim_styles)
		XFree(xim_styles);

	return;

error1:
	if (xim_styles)
		XFree(xim_styles);
	if (xim)
		XCloseIM(xim);	
	xim = NULL;
}

void TOADBase::CloseXInput()
{
	if (xim)
		XCloseIM(xim);
	xim = NULL;
}
