#include <qpainter.h>

#include "pixmap/Pixmap.h"
#include "pixmap/PixmapWnd.h"
#include "pixmap/PixmapOption.h"
#include "pixmap/PixmapRotate.h"
#include "pixmap/ColorFrame.h"
#include "prop/ColorSelectDialog.h"
#include "bmp/bitmaps.h"

Previewer::Previewer( QWidget *p )
  : QFrame( p ),
    contents( NULL )
{
  horScr = new QScrollBar( this );
  horScr->setOrientation( QScrollBar::Horizontal );
  horScr->hide();
  connect( horScr, SIGNAL(valueChanged(int)), this, SLOT(ScrBar(int)) );
  verScr = new QScrollBar( this );
  verScr->setOrientation( QScrollBar::Vertical );
  verScr->hide();
  connect( verScr, SIGNAL(valueChanged(int)), this, SLOT(ScrBar(int)) );
}

void Previewer::SetPixmap( QPixmap *p )
{
  contents = p;
  Resize();
}

void Previewer::ScrBar( int )
{
  repaint();
}

void Previewer::Resize()
{
  QSize cS = contents->size();

  // Positioning the scrollbars
  verScr->setGeometry( size().width() - SCRBAR_DIMENSION, 0,
                       SCRBAR_DIMENSION, size().height() - SCRBAR_DIMENSION );

  horScr->setGeometry( 0, size().height() - SCRBAR_DIMENSION, 
                       size().width() - SCRBAR_DIMENSION, SCRBAR_DIMENSION );

  int w = width() - SCRBAR_DIMENSION;
  if( cS.width() > width() )
  {
    horScr->show();
    horScr->setRange( 0, cS.width() - w );
  } else
  {
    horScr->hide();
    horScr->setRange( 0, 0 );
  }

  int h = height() - SCRBAR_DIMENSION;
  if( cS.height() > height() )
  {
    verScr->show();
    verScr->setRange( 0, cS.height() - h );
  } else
  {
    verScr->hide();
    verScr->setRange( 0, 0 );
  }

  QPainter p;
  p.begin( this );
  p.fillRect( rect(), backgroundColor() );
  p.end();

}

void Previewer::drawContents( QPainter * )
{
  if( ! contents )
    return;

  bitBlt( this, 0, 0, contents, 
          horScr->value(), verScr->value(), 
          (horScr->isVisible() ? width() - SCRBAR_DIMENSION : contents->width() ), 
          (verScr->isVisible() ? height() - SCRBAR_DIMENSION : contents->height() ) );
}

void Previewer::resizeEvent( QResizeEvent * )
{
  Resize();
}
 
PixmapWnd::PixmapWnd( Pixmap *p, ProjectWnd *prj, const char *name )
  : QFrame( NULL, name ),
    itsPixmapDraw( new PixmapDraw( this ) ),
    itsLayout( this, QBoxLayout::Down ),
    itsMenu( this ),
    itsToolbar( this ),
    pixmap( p ),
    itsStatusBar( 20, this ),
    theProjectWnd( prj )
{
    setCaption( "Qt Architect" );

    QPixmap iconPixmap;
    iconPixmap.loadFromData( QtArch_bmp_data, QtArch_bmp_len );

    setIcon( iconPixmap );
    setIconText( caption() );

  // Dialog knows what to do when we want to close
  connect( this, SIGNAL(Closed()), pixmap, SLOT(Hide()) );
  
  // Best for us if layout reports changes directly to pixmap
  connect( itsPixmapDraw, SIGNAL(Changed()), pixmap, SLOT(PixmapChanged()) );

  // setup menu items
  QPopupMenu* file = new QPopupMenu;
  file->insertItem( "Save", pixmap, SLOT(Save()), CTRL+Key_S );
  file->insertItem( "Generate Files", pixmap, SLOT(Generate()), CTRL+Key_G );
  file->insertSeparator();
  file->insertItem( "Import...", this, SLOT(Import()) );
  file->insertItem( "Export...", this, SLOT(Export()) );
  file->insertSeparator();
  file->insertItem( "Options...", this, SLOT(SetOptions()) );
  file->insertSeparator();
  file->insertItem( "Close", this, SLOT(Close()) );

  edit = new QPopupMenu;
  edit->insertItem( "Select", this, SLOT(Select()) );
  deselectEntry = edit->insertItem( "Deselect", this, SLOT(Deselect()) );
  edit->insertSeparator();
  clipEntry = edit->insertItem( "Clip Selection", this, SLOT(Clip()) );
  edit->insertSeparator();
  cutEntry   = edit->insertItem( "Cut", this, SLOT(Cut()) );
  copyEntry  = edit->insertItem( "Copy", this, SLOT(Copy()) );
  pasteEntry = edit->insertItem( "Paste", itsPixmapDraw, SLOT(Paste()) );
  edit->setItemEnabled( deselectEntry, FALSE );
  edit->setItemEnabled( clipEntry, FALSE );
  edit->setItemEnabled( cutEntry, FALSE );
  edit->setItemEnabled( copyEntry, FALSE );
  edit->setItemEnabled( pasteEntry, FALSE );
  edit->setCheckable( TRUE );

  modes = new QPopupMenu;
  modes->insertItem( "Point", PixmapDraw::DrawPoint );
  modes->insertItem( "Line", PixmapDraw::DrawLine );
  modes->insertItem( "Rectangle", PixmapDraw::DrawRect );
  modes->insertItem( "Circle", PixmapDraw::DrawCircle );
  modes->insertItem( "Fill", PixmapDraw::DrawFill );
  modes->setCheckable( TRUE );

  tools = new QPopupMenu;
  tools->insertItem( "Rotate...", this, SLOT(Rotate()) );
  useGridMenu = tools->insertItem( "Grid", this, SLOT(UseGrid()) );
  tools->setCheckable( TRUE );
  tools->insertSeparator();

  sizes = new QPopupMenu;
  sizes->insertItem( "1", 1 );
  sizes->insertItem( "2", 2 );
  sizes->insertItem( "4", 4 );
  sizes->insertItem( "8", 8 );
  sizes->insertItem( "Other...", 0 );
  sizes->setCheckable( TRUE );
  connect( sizes, SIGNAL(activated(int)), this, SLOT(PixelSize(int)) );
  tools->insertItem( "Pixel size", sizes );

  styles = new QPopupMenu;
  styles->insertItem( "Solid", SolidLine );
  styles->insertItem( "Dashed", DashLine );
  styles->insertItem( "Dot", DotLine );
  styles->insertItem( "Dash Dot", DashDotLine );
  styles->insertItem( "Dash Dot Dot", DashDotDotLine );
  styles->setCheckable( TRUE );
  connect( styles, SIGNAL(activated(int)), this, SLOT(LineStyles(int)) );
  tools->insertItem( "Line style", styles );

  itsMenu.insertItem( "Pixmap", file );
  itsMenu.insertItem( "Edit", edit );
  itsMenu.insertItem( "Draw", modes );
  itsMenu.insertItem( "Tool", tools );

  // Connect the drawing modes
  connect( modes, SIGNAL(activated(int)), this, SLOT(ModeChange(int)) );

  // connect some links from PixmapDraw to us
  connect( itsPixmapDraw, SIGNAL(MouseTracking(int,int,int,int)), 
           this, SLOT(MouseTracking(int,int,int,int)) );
  connect( itsPixmapDraw, SIGNAL(CPAvailable()), 
           this, SLOT(CPAvailable()) );

  // insert menu into layout
  itsLayout.setMenuBar( &itsMenu );

  // Setup Toolbar
  QPixmap pixmap;
  pixmap.loadFromData( DrawPixel_bmp_data, DrawPixel_bmp_len );
  itsToolbar.Insert( PixmapDraw::DrawPoint, pixmap, 0, 0,
		     "Draw Pixel", "Draw single pixels", "pixel", true );
  pixmap.loadFromData( DrawLine_bmp_data, DrawLine_bmp_len );
  itsToolbar.Insert( PixmapDraw::DrawLine, pixmap, 0, 0,
		     "Draw Line", "Draw a line", "line", true );
  pixmap.loadFromData( DrawRect_bmp_data, DrawRect_bmp_len );
  itsToolbar.Insert( PixmapDraw::DrawRect, pixmap, 0, 0,
		     "Draw Rectangle", "Draw a rectangle", "rect", true );
  pixmap.loadFromData( DrawCircle_bmp_data, DrawCircle_bmp_len );
  itsToolbar.Insert( PixmapDraw::DrawCircle, pixmap, 0, 0,
		     "Draw Circle", "Draw a circle", "circle", true );
  pixmap.loadFromData( DrawFill_bmp_data, DrawFill_bmp_len );
  itsToolbar.Insert( PixmapDraw::DrawFill, pixmap, 0, 0,
		     "Fill", "Fill area", "fill", true );

  itsToolbar.InsertSeparator( 10 );
  pixmap.loadFromData( PixmapZoomIn_bmp_data, PixmapZoomIn_bmp_len );
  itsToolbar.Insert( pixmap, itsPixmapDraw, SLOT(ZoomIn()),
		     "Zoom in", "Zoom into Pixmap" );
  pixmap.loadFromData( PixmapZoomOut_bmp_data, PixmapZoomOut_bmp_len );
  itsToolbar.Insert( pixmap, itsPixmapDraw, SLOT(ZoomOut()),
		     "Zoom out", "Zoom out of Pixmap" );
  connect( itsPixmapDraw, SIGNAL(ZoomChanged(int)), 
           this, SLOT(ZoomChanged(int)));

  connect( &itsToolbar, SIGNAL(clicked(int)), this, SLOT(ModeChange(int)) );

   connect( &itsToolbar.GetToolTipGroup(), SIGNAL(showTip(const char*)),
 	   SLOT(ShowToolTip(const char*)) );
   connect( &itsToolbar.GetToolTipGroup(), SIGNAL(removeTip()),
 	   SLOT(RemoveToolTip()) );

  itsLayout.addSpacing( 3 );

  // add toolbar to layout
  itsLayout.addWidget( &itsToolbar );
  itsToolbar.setMinimumSize( itsToolbar.size() );
 
  // create a new layout to handle horizontal spacing of WidgetFrame
  QBoxLayout* pixLayout = new QBoxLayout( QBoxLayout::LeftToRight, 0 );
  itsLayout.addLayout( pixLayout, 1 );

  // set layout of horizontal layout
  //pixLayout->addSpacing( 0 );
  pixLayout->addWidget( itsPixmapDraw, 1, AlignTop );	
  itsPixmapDraw->setMinimumSize( 400, 300 );
  // pixLayout->addSpacing( 1 );
  //  pixLayout->addStretch( 0 );
  
  QFrame *sepLine = new QFrame( this );
  sepLine->setMinimumSize( 5, 300 );
  sepLine->setFrameStyle( QFrame::Panel | QFrame::Raised );
  pixLayout->addWidget( sepLine, 1, AlignTop );
  
  QBoxLayout* colLayout = new QBoxLayout( QBoxLayout::TopToBottom, 0 );
  pixLayout->addLayout( colLayout, 1 );

  // Set up color choosing
  itsColorArea = new ColorFrame( this );
  itsColorArea->setMinimumSize( 200, 200 );
  itsColorArea->setFrameStyle( QFrame::Panel | QFrame::Raised );
  connect( itsColorArea, SIGNAL(ColorChanged(int,const QColor &)),
           this, SLOT(ColorChanged(int,const QColor &)) );
  colLayout->addWidget( itsColorArea, 1, AlignTop );

  // Set up previewer
  itsPreviewArea = new Previewer( this );
  itsPreviewArea->setMinimumSize( 200, 100 );
  itsPreviewArea->setFrameStyle( QFrame::Panel | QFrame::Raised );
  colLayout->addWidget( itsPreviewArea, 1, AlignTop );
  itsPixmapDraw->SetupPreview( itsPreviewArea );

  // itsLayout.addSpacing( 5 );
  //  itsLayout.addStretch( 0 );

  // setup status bar
  itsStatusMsg = itsStatusBar.InsertTextRegion( 1 );
  itsStatusBar.InsertSpacing( 5 );
  itsStatusZoom = itsStatusBar.InsertTextRegion( 0, "Zoom:  x" );
  itsStatusBar.SetText( itsStatusZoom, "Zoom:  8" );
  itsStatusPSize = itsStatusBar.InsertTextRegion( 0, "Width:000 Height:000" );
  itsStatusBar.SetText( itsStatusPSize, "Width: 32 Height: 32" );
  itsStatusPos = itsStatusBar.InsertTextRegion( 0, "X:000 Y:000" );
  itsStatusBar.SetText( itsStatusPos, "X:    Y:   " );
  itsStatusSize = itsStatusBar.InsertTextRegion( 0, "W:000 H:000");
  itsStatusBar.SetText( itsStatusSize, "W:    H:   " );
  
  // add status bar to layout
  itsLayout.addWidget( &itsStatusBar );
  itsStatusBar.setMinimumSize( 0, itsStatusBar.geometry().height() );

  itsLayout.activate();

  // Set initial drawing area
  ModeChange( PixmapDraw::DrawPoint );

  sizes->setItemChecked( 1, TRUE );
  itsPixmapDraw->SetPixelSize( 1 );
  styles->setItemChecked( SolidLine, TRUE );
  itsPixmapDraw->SetLineStyle( SolidLine );

  itsPixmapDraw->SetSize( QSize( 32, 32 ) );
  itsPixmapDraw->SetLeftColor( black );
  itsColorArea->SetLeftColor( black );
  itsPixmapDraw->SetRightColor( white );
  itsColorArea->SetRightColor( white );

  // resize to minimum size
  resize( 0,0 );
}

PixmapWnd::~PixmapWnd()
{
  delete itsPixmapDraw;
}

void PixmapWnd::closeEvent( QCloseEvent*	event )
{
  // shut up
  event = event;

  emit Closed();
}

void PixmapWnd::resizeEvent( QResizeEvent *event )
{

  //  itsPixmapDraw->resize( 
  QFrame::resizeEvent( event );
}

void PixmapWnd::Close()
{
  emit Closed();
}

void PixmapWnd::Import()
{
  QString file = QFileDialog::getOpenFileName();

  if( ! file.isEmpty() )
    itsPixmapDraw->LoadFile( file );
}

void PixmapWnd::Export()
{
  QString file = QFileDialog::getSaveFileName();

  if( ! file.isEmpty() )
    itsPixmapDraw->WriteFile( file, "BMP" );
}

void PixmapWnd::SetOptions()
{
  // Collect some data
  const QColor &actBackground = itsPixmapDraw->GetBackgroundColor();
  QSize actSize = itsPixmapDraw->GetSize();

  // Display data
  PixmapOption *dlg = new PixmapOption( actBackground, actSize, this );

  // Write data
  if( dlg->exec() )
  {
    if( actBackground != dlg->GetBackgroundColor() )
    {
      itsPixmapDraw->SetBackgroundColor( dlg->GetBackgroundColor() );
    }
    QSize size = dlg->GetSize();
    itsPixmapDraw->SetSize( size );

    QString str;
    str.sprintf( "Width:%3d Height:%3d", size.width(), size.height() );
    itsStatusBar.SetText( itsStatusPSize, str );
  }

  delete dlg;
}

void PixmapWnd::ModeChange( int id )
{
  if( id >= PixmapDraw::LastDrawMode ||
      id < PixmapDraw::DrawPoint )
    return;

  // Uncheck all items
  for( unsigned int i = 1; i < modes->count() + 1; i++ )
    modes->setItemChecked( i, FALSE );
  // Check the new item
  modes->setItemChecked( id, TRUE );

  if( &itsToolbar != sender() )
    itsToolbar.ToggleItem( id );

  itsPixmapDraw->ModeChange( (PixmapDraw::DrawMode) id );
}

void PixmapWnd::PixelSize( int id )
{
  // Uncheck all items
  int item = 1;
  for( unsigned int i = 1; i < sizes->count() + 1; i++, item *= 2 )
    sizes->setItemChecked( item, FALSE );
  // Check the new item
  sizes->setItemChecked( id, TRUE );

  if( id == 0 )
  {
    id = 1;
  }

  itsPixmapDraw->SetPixelSize( id );
}

void PixmapWnd::LineStyles( int id )
{
  // Uncheck all items
  for( unsigned int i = 1; i < styles->count() + 1; i++ )
    styles->setItemChecked( i, FALSE );
  // Check the new item
  styles->setItemChecked( id, TRUE );

  itsPixmapDraw->SetLineStyle( id );
}

void PixmapWnd::Rotate()
{
  PixmapRotate *dlg = new PixmapRotate( this );

  if( dlg->exec() )
  {
    // First rotation
    itsPixmapDraw->Rotate( dlg->GetRotation() );

    // Then eventually flip
    if( dlg->FlipVertical() )
      itsPixmapDraw->Flip( PixmapDraw::Vertical );
    if( dlg->FlipHorizontal() )
      itsPixmapDraw->Flip( PixmapDraw::Horizontal );
  }
}

void PixmapWnd::UseGrid()
{
  tools->setItemChecked( useGridMenu, ! tools->isItemChecked( useGridMenu ) );
  itsPixmapDraw->UseGrid( tools->isItemChecked( useGridMenu ));
}

void PixmapWnd::MouseTracking( int x, int y,
                               int width, int height )
{
  QString str;

  str.sprintf( "X:%3d Y:%3d", x, y );
  itsStatusBar.SetText( itsStatusPos, str );

  if( width || height )
  {
    str.sprintf( "W:%3d H:%3d", width, height );
    itsStatusBar.SetText( itsStatusSize, str );
  }
}

void PixmapWnd::ColorChanged( int button, const QColor &c )
{
  if( button == LeftButton )
    itsPixmapDraw->SetLeftColor( c );
  else if( button == RightButton )
    itsPixmapDraw->SetRightColor( c );
}

void PixmapWnd::ZoomChanged( int zoom )
{
  QString str;

  str.sprintf( "Zoom: %2d", zoom );
  itsStatusBar.SetText( itsStatusZoom, str );

}

void PixmapWnd::ShowToolTip( const char* tip )
{
    itsStatusBar.SetText( itsStatusMsg, tip );
}

void PixmapWnd::RemoveToolTip()
{
    itsStatusBar.SetText( itsStatusMsg, itsLastStatusMsg );
}

void PixmapWnd::Select()
{
  ModeChange( PixmapDraw::DrawRegion );
}

void PixmapWnd::Deselect()
{
  itsPixmapDraw->Deselect();
  edit->setItemEnabled( deselectEntry, FALSE );
  edit->setItemEnabled( clipEntry, FALSE );
  edit->setItemEnabled( cutEntry, FALSE );
  edit->setItemEnabled( copyEntry, FALSE );
  edit->setItemEnabled( pasteEntry, FALSE );

  edit->setItemChecked( clipEntry, FALSE );
  ModeChange( PixmapDraw::DrawPoint );
}

void PixmapWnd::Clip()
{
  edit->setItemChecked( clipEntry, ! edit->isItemChecked( clipEntry ) );
  itsPixmapDraw->ClipRegion();
}

void PixmapWnd::CPAvailable()
{
  edit->setItemEnabled( deselectEntry, TRUE );
  edit->setItemEnabled( clipEntry, TRUE );
  edit->setItemEnabled( cutEntry, TRUE );
  edit->setItemEnabled( copyEntry, TRUE );
}

void PixmapWnd::Cut()
{
  if( itsPixmapDraw->Cut() )
    edit->setItemEnabled( pasteEntry, TRUE );
}

void PixmapWnd::Copy()
{
  if( itsPixmapDraw->Copy() )
    edit->setItemEnabled( pasteEntry, TRUE );
}




