/*
solcal.c Version 0.4.0. Solar Calendar
Copyright (C) 2001-2010  dondalah721@yahoo.com (Dondalah)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to:

	Free Software Foundation, Inc.
	59 Temple Place - Suite 330
	Boston, MA  02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <asm/errno.h>
#include <zlib.h>
#include <math.h>
#include "font.h"

#define PI M_PI
#define XSZ 624
#define YSZ 480

/* all times are standard time */

/* julian day at epoch = 1990.0 */
#define EPOCH 2447891.5
/* ecliptic longitude (deg) at epoch 1990.0 */
#define EPSILON_G 279.403303
/* ecliptic longitude (deg) of perigee */
#define OMEGA_G   282.768422 
/* eccentricity of the Sun-Earth orbit */
#define ECCENTRICITY 0.016713
/* semi-major axis (kilometers) */
#define R_ZERO 1.495985e8
/* angular diameter (deg) at r = r_zero */

int mlen[16] = {
    0, 31, 28, 31, 30, 31, 30, 31,
   31, 30, 31, 30, 31, 31, 28,  0 };

char mtbl[16][4] = {
   "???", "Jan", "Feb", "Mar",
   "Apr", "May", "Jun", "Jul",
   "Aug", "Sep", "Oct", "Nov",
   "Dec", "Jan", "Feb", "???" };

char mtx[YSZ][XSZ];

void putstx(char *pgm)
   {
   fprintf(stderr,"Usage: %s year "
      "longitude latitude timezone\n",
      pgm);
   fprintf(stderr,"Where year      is 1900   to 2999\n");
   fprintf(stderr,"      longitude is -180.0 to 180.0\n");
   fprintf(stderr,"      west longitude is negative\n");
   fprintf(stderr,"      latitude  is -66.5  to 66.5\n");
   fprintf(stderr,"      south latitude is negative\n");
   fprintf(stderr,"      timezone  is -12.0  to 12.0\n");
   fprintf(stderr,"      west timezones are negative\n");
   exit(1);
   } /* putstx */

void putimg(hndl,str,len)
gzFile hndl;
unsigned char *str;
int len;
   {
   int wrtlen,tot;
   tot = 0;
   while (tot < len)
      {
      wrtlen = gzwrite(hndl,str,len);
      if (wrtlen < 1)
         {
         fprintf(stderr,"Write error %d\n", errno);
         perror("Write error");
         exit(1);
         } /* write err */
      tot += wrtlen;
      } /* write loop */
   } /* putimg */

void wrtppm(hndl)
gzFile hndl;
   {
   int len,x,y;
   unsigned char str[32];
   putimg(hndl,"P6\n",3);
   sprintf(str,"%d %d\n%d\n",
      XSZ, YSZ, 255);
   len = strlen(str);
   putimg(hndl,str,len);
   for (y=0;y<YSZ;y++)
      {
      for (x=0;x<XSZ;x++)
         {
         if (mtx[y][x])
            {
            str[0] = str[1] = str[2] = 0;
            } /* if black */
         else
            {
            str[0] = str[1] = str[2] = 255;
            } /* else white */
         putimg(hndl,str,3);
         } /* for each col in mtx */
      } /* for each row in mtx */
   } /* wrtppm */

void drawpxl(x,y)
int x,y;
   {
   if (x < 0 || x >= XSZ)
      {
      fprintf(stderr,"drawpxl: x out of range %d\n", x);
      exit(0);
      return;
      } /* x out of range */
   else if (y < 0 || y >= YSZ)
      {
      fprintf(stderr,"drawpxl: y out of range %d\n", y);
      exit(0);
      return;
      } /* y out of range */
   else mtx[y][x] = 1;
   } /* drawpxl */

void drawtxt(str,x,y)
char *str;
int x,y;
   {
   int len,ch,i,j,k;
   int ofst,row,col;
   char *p,*q;
   len = strlen(str);
   p = str;
   q = p + len;
   ofst = 0;
   while (p < q)
      {
      i = *p - 32;
      row = y;
      for (j=0;j<16;j++)
         {
         col = x + ofst;
         for (k=0;k<8;k++)
            {
            if (col >= (XSZ-8)
               || row >= (YSZ-16))
               {
               fprintf(stderr,"Text overflow "
                  "%c in %d %d\n",
                  *p, col, row);
               return;
               } /* if too far to right */
            if (font[i][j][k])
               drawpxl(col,row);
            col++;
            } /* for each pxl col */
         row++;
         } /* for each pxl row */
      p++;
      ofst += 8;
      } /* for each char in string */
   } /* drawtxt */

void drawline(x1,y1,x2,y2)
int x1,y1,x2,y2;
   {
   int x,y,dx,dy,xdlta,ydlta,absxdlta,absydlta;
   int xchg,ychg,xlim,ylim;
   if (x1 < 0 || x1 > XSZ)
      {
      fprintf(stderr,"x1 %d out of range\n", x1);
      return;
      } /* x1 err */
   if (y1 < 0 || y1 > YSZ)
      {
      fprintf(stderr,"y1 %d out of range\n", y1);
      return;
      } /* y1 err */
   if (x2 < 0 || x2 > XSZ)
      {
      fprintf(stderr,"x2 %d out of range\n", x2);
      return;
      } /* x2 err */
   if (y2 < 0 || y2 > YSZ)
      {
      fprintf(stderr,"y2 %d out of range\n", y2);
      return;
      } /* y2 err */
   if (x1 < x2)
      {
      dx = x2 - x1;
      dy = y2 - y1;
      if (dy) xdlta = (dx << 8) / dy;
      else xdlta = 256;
      ydlta = (dy << 8) / dx;
      x = x1;
      y = y1;
      xlim = x2;
      ylim = y2;
      } /* if slope to right */
   else if (x1 > x2)
      {
      dx = x1 - x2;
      dy = y1 - y2;
      if (dy) xdlta = (dx << 8) / dy;
      else xdlta = 256;
      ydlta = (dy << 8) / dx;
      x = x2;
      y = y2;
      xlim = x1;
      ylim = y1;
      } /* if slope to left */
   else
      {
      x = x1;
      y = y1;
      if (y2 > y1)
         {
         y = y1;
         ylim = y2;
         } /* if y2 > y1 */
      else
         {
         y = y2;
         ylim = y1;
         } /* else y1 > y2 */
      while (y < ylim)
         drawpxl(x,y++);
      return;
      } /* vertical line */
   if (xdlta < 0) absxdlta = -xdlta;
   else absxdlta = xdlta;
   if (ydlta < 0) absydlta = -ydlta;
   else absydlta = ydlta;
   xchg = 0;
   ychg = 0;
   while (x < xlim)
      {
      xchg += absxdlta;
      if (xchg > 256)
         {
         if (x < xlim) x++;
         xchg -= 256;
         } /* if whole number */
      ychg += absydlta;
      if (ychg > 128)
         {
         if (ydlta < 0)
            {
            if (y >= ylim) y--;
            } /* if negative slope */
         else
            {
            if (y <= ylim) y++;
            } /* else positive slope and not too far */
         ychg -= 256;
         } /* if whole number */
      drawpxl(x,y);
      } /* draw line */
   } /* drawline */

void drawcircle(xc,yc,r)
int xc,yc,r;
   {
   int x,y,p;
   x = 0;
   y = r;
   p = 3 - (r + r);
   while (x <= y)
      {
      drawpxl(xc+x,yc+y);
      drawpxl(xc-x,yc+y);
      drawpxl(xc+x,yc-y);
      drawpxl(xc-x,yc-y);
      drawpxl(xc+y,yc+x);
      drawpxl(xc-y,yc+x);
      drawpxl(xc+y,yc-x);
      drawpxl(xc-y,yc-x);
      if (p < 0)
         p += 4 * x++ + 6;
      else
         p += 4 * (x++ - y--) + 10;
      } /* bresenham's algorithm */
   } /* drawcircle */

int initmlen(yr)
int yr;
   {
   int lpyr;
   if (!(yr % 400)) lpyr = 1;
   else if (!(yr % 100)) lpyr = 0;
   else if (!(yr % 4)) lpyr = 1;
   else lpyr = 0;
   if (lpyr)
      {
      mlen[2] = 29;
      mlen[14] = 29;
      } /* if leap year */
   else
      {
      mlen[2] = 28;
      mlen[14] = 28;
      } /* else not leap year */
   return(lpyr);
   } /* initmlen */

/* Input rtn = name of calling routine */
/* Output */
/* return code = 1 (valid date) */
/* return code = 0 (invalid date) */
int valdate(rtn,yr,mon,day,lpyr)
char *rtn;
int yr,mon,day,*lpyr;
   {
   *lpyr = initmlen(yr);
   if (day < 1 || day > mlen[mon]
      || mon < 1 || mon > 12
      || yr < -4713 || yr > 2999)
      {
      fprintf(stderr,"%s: "
         "Invalid date %d/%d/%d\n",
         rtn, yr, mon, day);
      return(0);
      } /* invalid date */
   return(1);
   } /* valdate */

/* Input rtn = name of calling routine */
/* Output */
/* return code = 1 (valid time) */
/* return code = 0 (invalid time) */
int valtime(rtn,hr,min,sec)
char *rtn;
int hr,min;
double sec;
   {
   if (hr < 0 || hr > 24
      || min < 0 || min > 60
      || sec < 0.0 || sec > 60.0)
      {
      fprintf(stderr,"%s: "
         "Invalid time %d:%d:%f\n",
         rtn, hr, min, sec);
      return(0);
      } /* invalid time */
   return(1);
   } /* valtime */

void getjul(yr,mon,day,jul,lpyr)
int yr,mon,day;
int *jul,*lpyr;
   {
   int yrday,wrkday;
   /* lpyr is a pointer */
   if (!(valdate("getjul",yr,mon,day,lpyr)))
      {
      *jul = 0;
      *lpyr = 0;
      return;
      }
   if (mon < 3)
      {
      if (*lpyr) yrday = ((mon - 1) * 62) >> 1;
      else yrday = ((mon - 1) * 63) >> 1;
      } /* if jan or feb */
   else
      {
      wrkday = (int) floor((mon + 1.0) * 30.6);
      if (*lpyr) yrday = wrkday - 62;
      else yrday = wrkday - 63;
      } /* else mar - dec */
   *jul = yrday + day;
   } /* getjul */

/* Duffet-Smith Section 7 */
double hms2hhh(hr,min,sec)
int hr,min;
double sec;
   {
   double hhh;
   if (!(valtime("hms2hhh",hr,min,sec)))
      return(0);
   hhh = hr
      + (min / 60.0)
      + (sec / 3600.0);
   return(hhh);
   } /* hms2hhh */

/* Duffet-Smith Section 8 */
void hhh2hms(hhh,hr,min,sec)
double hhh;
int *hr,*min;
double *sec;
   {
   double wholemin;
   *hr  = (int) floor(hhh);
   *min = (int) floor(hhh * 60.0) % 60;
   *sec = modf((hhh * 60.0),&wholemin) * 60.0;
   if (!(valtime("hhh2hms",*hr,*min,*sec)))
      {
      *hr = *min = 0;
      *sec = 0.0;
      return;
      } /* invalid time */
   } /* hhh2hms */

/* Duffet-Smith Section 9 Inner Routine */
double lthhh2uthhh(lthhh,timezone)
double lthhh,timezone;
   {
   double zonetime,uthhh;
   zonetime = lthhh - timezone;
   if (zonetime > 24.0) uthhh = zonetime - 24.0;
   else if (zonetime < 0.0) uthhh = zonetime + 24.0;
   else uthhh = zonetime;
   return(uthhh);
   } /* lthhh2uthhh */

/* Duffet-Smith Section 9 Outer Routine */
void lt2ut(lthr,ltmin,ltsec,timezone,uthr,utmin,utsec)
int lthr,ltmin;
double ltsec,timezone;
int *uthr,*utmin;
double *utsec;
   {
   double lthhh,uthhh;
   lthhh = hms2hhh(lthr,ltmin,ltsec);
   uthhh = lthhh2uthhh(lthhh,timezone);
   /* uthr, utmin, utsec are pointers */
   hhh2hms(uthhh,uthr,utmin,utsec);
   } /* lt2ut */

/* Duffet-Smith Section 10 Inner Routine */
double uthhh2lthhh(uthhh,timezone)
double uthhh,timezone;
   {
   double zonetime,lthhh;
   zonetime = uthhh + timezone;
   if (zonetime > 24.0) lthhh = zonetime - 24.0;
   else if (zonetime < 0.0) lthhh = zonetime + 24.0;
   else lthhh = zonetime;
   return(lthhh);
   } /* uthhh2lthhh */

/* Duffet-Smith Section 10 Outer Routine */
void ut2lt(uthr,utmin,utsec,timezone,lthr,ltmin,ltsec)
int uthr,utmin;
double utsec,timezone;
int *lthr,*ltmin;
double *ltsec;
   {
   double uthhh,lthhh;
   uthhh = hms2hhh(uthr,utmin,utsec);
   lthhh = uthhh2lthhh(uthhh,timezone);
   /* lthr, ltmin, ltsec are pointers */
   hhh2hms(lthhh,lthr,ltmin,ltsec);
   } /* ut2lt */

/* Duffet-Smith Section 4 */
/* calculate julian day using 0/0/1900 as the epoch */
double ymd2jd(yr,mon,day,hr,min,sec)
int yr,mon,day,hr,min;
double sec;
   {
   int tmpmon,tmpyr;
   int greg,century,leapcentury;
   int totdays,yrdays;
   int lpyr;
   double jul,dblday;
   if (!(valdate("ymd2jd",yr,mon,day,&lpyr)))
      return(0.0);
   /* number of days in current month */
   /* including time as a fraction */
   /* convert day/hr/min/sec.frac to dd.sss */
   /* where sss = day_seconds / (24*60*60) */
   dblday = day
      + (hms2hhh(hr, min, sec) / 24.0);
   /* year begins 3/1 and ends 2/28 or 2/29 */
   /* yyyy/1 ==> (yyyy-1)/13 */
   /* yyyy/2 ==> (yyyy-1)/14 */
   if (mon < 3)
      {
      tmpyr = yr - 1;
      tmpmon = mon + 12;
      } /* if jan or feb */
   else
      {
      tmpyr = yr;
      tmpmon = mon;
      } /* mar - dec */
   /* is date gregorian? */
   if (yr > 1582) greg = 1;
   else if (yr == 1582
      && mon > 10) greg = 1;
   else if (yr == 1582
      && mon == 10
      && day >= 15) greg = 1;
   else greg = 0;
   if (greg)
      {
      century = tmpyr / 100;
      leapcentury = 2 - century + (century/4);
      } /* if gregorian date */
   else
      {
      leapcentury = 0;
      } /* else not gregorian date */
   /* convert years to days */
   if (tmpyr < 0)
      {
      totdays = (int) floor((365.25 * tmpyr) - 0.75);
      } /* if negative year */
   else
      {
      totdays = (int) floor(365.25 * tmpyr);
      } /* else positive year */
   /* number of days in current year */
   /* not including current month */
   yrdays = (int) floor(30.6001 * (tmpmon + 1.0));
   jul = leapcentury + totdays + yrdays
      + dblday + 1720994.5;
   return(jul);
   } /* ymd2jd */

/* Duffet-Smith Section 5 */
/* convert julian day to gregorian date (yyyy/mm/dd.frac) */
/* using 0/0/1900 as the epoch */
void jd2ymd(jd,yr,mon,ddd)
double jd;
int *yr,*mon;
double *ddd;
   {
   double tmpjd,jdint,jdfrac;
   double jul,dblday;
   double leapcent,tmpdays,totdays;
   double mthday,tmpmon,absyr;
   double mmm;
   tmpjd = jd + 0.5;
   jdint = floor(tmpjd);
   jdfrac = tmpjd - jdint;
   if (jdint > 2299160.0)
      {
      leapcent = floor((jdint - 1867216.25)
         / 36524.25);
      tmpdays = jdint + 1.0 + leapcent
         - floor(leapcent / 4.0);
      } /* if gregorian era */
   else
      {
      tmpdays = jdint;
      } /* else before gregorian era */
   totdays = tmpdays + 1524;
   absyr = floor((totdays - 122.1) / 365.25);
   mthday = floor(365.25 * absyr);
   tmpmon = floor((totdays - mthday) / 30.6001);
   *ddd = totdays - mthday + jdfrac
      - floor(tmpmon * 30.6001);
   if (tmpmon > 13.5)
      {
      mmm  = tmpmon - 13.0;
      *mon = (int) floor(tmpmon - 13.0);
      } /* if not after October */
   else
      {
      mmm  = tmpmon - 1.0;
      *mon = (int) floor(tmpmon - 1.0);
      } /* else before October */
   if (mmm > 2.5)
      {
      *yr = (int) floor(absyr - 4716.0);
      } /* if not jan or feb */
   else
      {
      *yr = (int) floor(absyr - 4715.0);
      } /* else jan or feb */
   } /* jd2ymd */

/* Duffet-Smith Section 12 Inner Routine */
/* called directly by routine 49 */
double utzero2t0(yr,mon,day)
int yr,mon,day;
   {
   double jd,epochjd,tmpyr,lpyr;
   double tmpt0,tmphr,wholehr;
   double t0;
   if (!(valdate("utzero2t0",yr,mon,day,&lpyr)))
      return(0.0);
   jd = ymd2jd(yr,mon,day,0,0,0.0);
   epochjd = jd - 2451545.0;
   tmpyr = epochjd / 36525.0;
   tmpt0 = 6.697374558
      + (2400.051336 * tmpyr)
      + (0.000025862 * tmpyr * tmpyr);
   tmphr = modf((tmpt0 / 24.0), &wholehr) * 24.0;
   if (tmphr < 0)
      {
      t0 = tmphr + 24.0;
      wholehr += 1.0;
      } /* if just below range of 0.0 - 24.0 */
   else t0 = tmphr;
   return(t0);
   } /* utzero2t0 */

/* Duffet-Smith Section 12 Outer Routine */
void ut2gst(yr,mon,day,hr,min,sec,gsthr,gstmin,gstsec)
int yr,mon,day,hr,min;
double sec;
int *gsthr,*gstmin;
double *gstsec;
   {
   double t0;
   double uthhh,sidhhh,tmpgst,gsthhh;
   t0 = utzero2t0(yr,mon,day);
   uthhh  = hms2hhh(hr,min,sec);
   sidhhh = uthhh * 1.002737909;
   tmpgst = t0 + sidhhh;
   if (tmpgst > 24.0) gsthhh = tmpgst - 24.0;
   else if (tmpgst < 0.0) gsthhh = tmpgst + 24.0;
   else gsthhh = tmpgst;
   /* gsthr, gstmin, gstsec are pointers */
   hhh2hms(gsthhh,gsthr,gstmin,gstsec);
   } /* ut2gst */

/* Duffet-Smith Section 13 Inner Routine */
double gsthhh2uthhh(gstyr,gstmon,gstday,gsthhh)
int gstyr,gstmon,gstday;
double gsthhh;
   {
   double jd,epochjd,tmpyr,lpyr;
   double t0,adjt0,tmphr,wholehr;
   double adjhhh,tmput;
   double uthhh;
   jd = ymd2jd(gstyr,gstmon,gstday,0,0,0.0);
   epochjd = jd - 2451545.0;
   tmpyr = epochjd / 36525.0;
   t0 = 6.697374558
      + (2400.051336 * tmpyr)
      + (0.000025862 * tmpyr * tmpyr);
   tmphr = modf((t0 / 24.0), &wholehr) * 24.0;
   if (tmphr < 0)
      {
      adjt0 = tmphr + 24.0;
      wholehr += 1.0;
      } /* if just below range of 0.0 - 24.0 */
   else adjt0 = tmphr;
   adjhhh = gsthhh - adjt0;
   if (adjhhh > 24.0) tmput = adjhhh - 24.0;
   else if (adjhhh < 0.0) tmput = adjhhh + 24.0;
   else tmput = adjhhh;
   uthhh = tmput * 0.9972695663;
   return(uthhh);
   } /* gsthhh2uthhh */

/* Duffet-Smith Section 13 Outer Routine */
void gst2ut(gstyr,gstmon,gstday,gsthr,gstmin,gstsec,
   uthr,utmin,utsec)
int gstyr,gstmon,gstday,gsthr,gstmin;
double gstsec;
int *uthr,*utmin;
double *utsec;
   {
   double jd,epochjd,tmpyr,lpyr;
   double t0,adjt0,tmphr,wholehr;
   double gsthhh,adjhhh,tmput,uthhh;
   gsthhh = hms2hhh(gsthr,gstmin,gstsec);
   uthhh = gsthhh2uthhh(gstyr,gstmon,gstday,gsthhh);
   /* uthr, utmin, utsec are pointers */
   hhh2hms(uthhh,uthr,utmin,utsec);
   } /* gst2ut */

/* Duffet-Smith Section 14 */
void gst2lst(gsthr,gstmin,gstsec,lon,
   lsthr,lstmin,lstsec)
int gsthr,gstmin;
double gstsec,lon;
int *lsthr,*lstmin;
double *lstsec;
   {
   double gsthhh,tmphr,lsthhh;
   gsthhh = hms2hhh(gsthr,gstmin,gstsec);
   tmphr  = gsthhh + (lon / 15.0);
   if (tmphr < 0.0) lsthhh = tmphr + 24.0;
   else if (tmphr > 24.0) lsthhh = tmphr - 24.0;
   else lsthhh = tmphr;
   /* lsthr, lstmin, lstsec are pointers */
   hhh2hms(lsthhh,lsthr,lstmin,lstsec);
   } /* gst2lst */

/* Duffet-Smith Section 15 Inner Routine */
double lsthhh2gsthhh(lsthhh,lon)
double lsthhh,lon;
   {
   double tmphr,gsthhh;
   tmphr  = lsthhh - (lon / 15.0);
   if (tmphr < 0.0) gsthhh = tmphr + 24.0;
   else if (tmphr > 24.0) gsthhh = tmphr - 24.0;
   else gsthhh = tmphr;
   return(gsthhh);
   } /* lsthhh2gsthhh */

/* Duffet-Smith Section 15 Outer Routine */
void lst2gst(lsthr,lstmin,lstsec,lon,
   gsthr,gstmin,gstsec)
int lsthr,lstmin;
double lstsec,lon;
int *gsthr,*gstmin;
double *gstsec;
   {
   double lsthhh,tmphr,gsthhh;
   lsthhh = hms2hhh(lsthr,lstmin,lstsec);
   gsthhh = lsthhh2gsthhh(lsthhh,lon);
   /* gsthr, gstmin, gstsec are pointers */
   hhh2hms(gsthhh,gsthr,gstmin,gstsec);
   } /* lst2gst */

/* Duffet-Smith Section 27 */
/* input  lambda and beta  are in decimal degrees */
/* output alpha  and delta are in decimal degrees */
void elp2equ(lambda,beta,alpha,delta)
double lambda,beta,*alpha,*delta;
   {
   double lambdarad,betarad,d2r,r2d;
   /* obliquity of the eliptic */
   /* the angle between the planes of the */
   /* equator and the eliptic */
   double epsilonrad;
   double sindeltarad;
   double x,y;
   double tmpalpha,aprime;
   d2r = M_PI / 180.0;
   r2d = 180.0 / M_PI;
   lambdarad = lambda * d2r;
   betarad = beta * d2r;
   epsilonrad = 23.441884 * d2r;
   sindeltarad = (sin(betarad) * cos(epsilonrad))
      + (cos(betarad) * sin(epsilonrad) * sin(lambdarad));
   *delta = asin(sindeltarad) * r2d;
   y = (sin(lambdarad) * cos(epsilonrad))
      - (tan(betarad) * sin(epsilonrad));
   x = cos(lambdarad);
   tmpalpha = atan(y / x) * r2d;
   if (x < 0.0 && y >= 0.0) aprime = tmpalpha + 180.0;
   else if (x >= 0.0 && y < 0.0) aprime = tmpalpha + 360.0;
   else if (x < 0.0 && y < 0.0) aprime = tmpalpha + 180.0;
   else aprime = tmpalpha;
   *alpha = aprime / 15.0;
   } /* elp2equ */

/* Duffet-Smith Section 47 */
/* Iterative Routine R2 */
/* input msun is in radians */
double calcecc(msun)
double msun;
   {
   double ecc,eccadj;
   double delta,epsilon,kecc,lim;
   lim = 1e-6;
   kecc = ECCENTRICITY;
   ecc = msun;

   delta = -(kecc * sin(ecc));
   while (fabs(delta) > lim)
      {
      eccadj = delta / (1.0 - (kecc * cos(ecc)));
      ecc = ecc - eccadj;
      delta = ecc - (kecc * sin(ecc)) - msun;
      } /* do while limit exceeded */

   return(ecc);
   } /* calcecc */

/* Duffet-Smith Section 47 */
/* lambda = sun's geocentric eliptic longitude (deg) */
/* alpha = right ascension */
/* delta = declination */
void solpos(yr,mon,day,hr,min,sec,lambda,alpha,delta)
int yr,mon,day,hr,min;
double sec,*lambda,*alpha,*delta;
   {
   double currjd,epkday;
   double wrkpos,tmppos,tmp;
   double wholedeg;
   /* motion of a mean sun moving in a circle */
   double tmpmsun,msun;
   /* eccentricity */
   double ecc;
   double tanhalfv,halfv,v;
   /* lambda = sun's geocentric eliptic longitude (deg) */
   double tmplambda;
   double d2r,r2d;
   d2r = M_PI / 180.0;
   r2d = 180.0 / M_PI;
   currjd = ymd2jd(yr,mon,day,hr,min,sec);
   epkday = currjd - EPOCH;
   wrkpos = modf((epkday / 365.242191),
      &wholedeg) * 360.0;
   if (wrkpos < 0.0) tmppos = wrkpos + 360.0;
   else tmppos = wrkpos;
   tmpmsun = tmppos + EPSILON_G - OMEGA_G;
   if (tmpmsun < 0.0) msun = tmpmsun + 360.0;
   else msun = tmpmsun;
   ecc = calcecc(msun * d2r);
   tanhalfv = sqrt((1+ECCENTRICITY)/(1-ECCENTRICITY))
      * tan(ecc * 0.5);
   halfv = atan(tanhalfv);
   v = halfv * 2.0 * r2d;
   tmplambda = v + OMEGA_G;
   if (tmplambda > 360.0)
      *lambda = tmplambda - 360.0;
   else if (tmplambda < 0.0)
      *lambda = tmplambda + 360.0;
   else *lambda = tmplambda;
   /* lambda is input to elp2equ */
   /* output: alpha and delta are pointers */
   elp2equ(*lambda,0.0,alpha,delta);
   } /* solpos */

/* Duffet-Smith Section 33 */
void riselst(alpha,delta,lat,lst_r,lst_s,azr,azs)
double alpha,delta,lat;
double *lst_r,*lst_s;
double *azr,*azs;
   {
   double cosazrad,azimuthrad;
   double azimuth;
   double alpharad,deltarad,latrad;
   double tanprod;
   double hanglerad,hangle;
   double lstr,lsts;
   double d2r,r2d;
   d2r = M_PI / 180.0;
   r2d = 180.0 / M_PI;
   alpharad = alpha * d2r;
   deltarad = delta * d2r;
   latrad = lat * d2r;
   cosazrad = sin(deltarad) / cos(latrad);
   if (cosazrad > 1 || cosazrad < -1)
      {
      fprintf(stderr,"riselst: delta %f lat %f\n",
         delta, lat);
      fprintf(stderr,"riselst: sin(delta) %f cos(lat) %f\n",
         sin(deltarad), cos(latrad));
      fprintf(stderr,"riselst: sin(delta)/cos(lat) %f\n",
         cosazrad);
      fprintf(stderr,"riselst: Invalid cosine\n");
      *lst_r = *lst_s = *azr = *azs = 0.0;
      return;
      } /* if invalid cosine */
   azimuth = acos(cosazrad) * r2d;
   tanprod = -tan(latrad) * tan(deltarad);
   if (tanprod > 1 || tanprod < -1)
      {
      fprintf(stderr,"riselst: delta %f lat %f\n",
         delta, lat);
      fprintf(stderr,"riselst: tan(delta) %f -tan(lat) %f\n",
         tan(deltarad), -tan(latrad));
      fprintf(stderr,"riselst: tan(delta) * (-tan(lat)) %f\n",
         tanprod);
      fprintf(stderr,"riselst: Invalid cosine\n");
      *lst_r = *lst_s = *azr = *azs = 0.0;
      return;
      } /* if invalid cosine */
   hanglerad = acos(tanprod) / 15.0;
   hangle = hanglerad * r2d;
   lstr = 24.0 + alpha - hangle;
   if (lstr > 24.0) *lst_r = lstr - 24.0;
   else *lst_r = lstr;
   lsts = alpha + hangle;
   if (lsts > 24.0) *lst_s = lsts - 24.0;
   else *lst_s = lsts;
   *azr = azimuth;
   *azs = 360.0 - azimuth;
   } /* riselst */

/* Duffet-Smith Section 49 */
void riseset(yr,mon,day,hr,min,sec,lon,lat,timezone,
   ltr,lts,azr,azs)
int yr,mon,day,hr,min;
double sec,lon,lat,timezone;
double *ltr,*lts,*azr,*azs;
   {
   int gsthr,gstmin,lsthr,lstmin;
   int risehr,risemin;
   double risesec;
   double gstsec,lstsec;
   double lambda,alpha,delta;
   double lambda2,alpha2,delta2;
   double lst1r,lst1s,lst2r,lst2s;
   double gst1r,gst1s,gst2r,gst2s;
   double t00,tmpt00p,t00prime;
   double gstr,gsts;
   double avgdelta,avgdeltarad,d2r,r2d;
   double latrad,psi,x,y;
   double deltat,deltatsec;
   double utr,uts;
   d2r = M_PI / 180.0;
   r2d = 180.0 / M_PI;
   solpos(yr,mon,day,hr,min,sec,
      &lambda,&alpha,&delta);
   lambda2 = lambda + 0.985647;
   elp2equ(lambda2,0.0,&alpha2,&delta2);
   /* azr, azs are pointers */
   riselst(alpha,delta,lat,
      &lst1r,&lst1s,azr,azs);
   /* azr, azs are pointers */
   riselst(alpha2,delta2,lat,
      &lst2r,&lst2s,azr,azs);
   gst1r = lsthhh2gsthhh(lst1r,lon);
   gst1s = lsthhh2gsthhh(lst1s,lon);
   gst2r = lsthhh2gsthhh(lst2r,lon);
   gst2s = lsthhh2gsthhh(lst2s,lon);
   if (gst1r > gst2r)
      gst2r = gst2r + 24.0;
   if (gst1s > gst2s)
      gst2s = gst2s + 24.0;
   t00 = utzero2t0(yr,mon,day);
   tmpt00p = t00 - (lon * 1.002738 / 15.0);
   if (tmpt00p < 0.0) t00prime = tmpt00p + 24.0;
   else t00prime = tmpt00p;
   if (gst1r < t00prime)
      {
      gst1r = gst1r + 24.0;
      gst2r = gst2r + 24.0;
      }
   if (gst1s < t00prime)
      {
      gst1s = gst1s + 24.0;
      gst2s = gst2s + 24.0;
      }
   gstr = (((24.07 * gst1r)
      - (t00 * (gst2r - gst1r)))
      / (24.07 + gst1r - gst2r));
   gsts = (((24.07 * gst1s)
      - (t00 * (gst2s - gst1s)))
      / (24.07 + gst1s - gst2s));
   avgdelta = (delta + delta2) / 2.0;
   latrad = lat * d2r;
   avgdeltarad = avgdelta * d2r;
   psi = acos(sin(latrad)
      / cos(avgdeltarad)) * r2d;
   x = (0.533 / 2.0) - (8.79 / 3600.0)
      + (34.0 / 60.0);
   y = asin(sin(x * d2r) / sin(latrad)) * r2d;
   deltatsec = (240.0 * y) / cos(avgdeltarad);
   deltat = deltatsec / 3600.0;
   gstr = gstr - deltat;
   gsts = gsts + deltat;
   utr = gsthhh2uthhh(yr,mon,day,gstr);
   uts = gsthhh2uthhh(yr,mon,day,gsts);
   *ltr = uthhh2lthhh(utr,timezone);
   *lts = uthhh2lthhh(uts,timezone);
   } /* riseset */

void draw(sumaz,winaz,lat,yr)
double sumaz,winaz,lat;
int yr;
   {
   int x1,y1,x2,y2;
   int rslt;
   double dblx,dbly,theta;
   double sin();
   double cos();
   gzFile hndl;
   int black,white;
   char ttlstr[64];

   drawcircle(300,240,150);
   /* draw center point */
   drawpxl(300,240);
   drawpxl(301,240);
   drawpxl(299,240);
   drawpxl(300,239);
   drawpxl(300,241);
   drawcircle(300,240,20);

   /* Summer Solstice */

   theta = (180.0 - sumaz) * M_PI / 180.0;
   dblx = sin(theta) * 150.0;
   dbly = cos(theta) * 150.0;
   x1 = (int) floor(dblx);
   y1 = (int) floor(dbly);
   drawpxl(300+x1,240+y1);
   drawpxl(301+x1,240+y1);
   drawpxl(299+x1,240+y1);
   drawpxl(300+x1,239+y1);
   drawpxl(300+x1,241+y1);
   sprintf(ttlstr,"Sunrise 6/21/%d", yr);
   drawtxt(ttlstr, 320+x1, 230+y1);
   sprintf(ttlstr,"Az = %3.1f deg.", sumaz);
   drawtxt(ttlstr, 320+x1+10, 230+y1+20);

   theta = (-sumaz) * M_PI / 180.0;
   dblx = sin(theta) * 150.0;
   dbly = cos(theta) * 150.0;
   x2 = (int) floor(dblx);
   y2 = (int) floor(dbly);
   drawpxl(300+x2,240+y2);
   drawpxl(301+x2,240+y2);
   drawpxl(299+x2,240+y2);
   drawpxl(300+x2,239+y2);
   drawpxl(300+x2,241+y2);
   drawline(300+x1,240+y1,300+x2,240+y2);
   drawtxt("Shadow", 235+x2, 230+y2);

   /* Winter Solstice */

   theta = (180.0 - winaz) * M_PI / 180.0;
   dblx = sin(theta) * 150.0;
   dbly = cos(theta) * 150.0;
   x1 = (int) floor(dblx);
   y1 = (int) floor(dbly);
   drawpxl(300+x1,240+y1);
   drawpxl(301+x1,240+y1);
   drawpxl(299+x1,240+y1);
   drawpxl(300+x1,239+y1);
   drawpxl(300+x1,241+y1);
   sprintf(ttlstr,"Sunrise 12/21/%d", yr);
   drawtxt(ttlstr, 320+x1, 230+y1);
   sprintf(ttlstr,"Az = %3.1f deg.", winaz);
   drawtxt(ttlstr, 320+x1, 230+y1+20);

   theta = (-winaz) * M_PI / 180.0;
   dblx = sin(theta) * 150.0;
   dbly = cos(theta) * 150.0;
   x2 = (int) floor(dblx);
   y2 = (int) floor(dbly);
   drawpxl(300+x2,240+y2);
   drawpxl(301+x2,240+y2);
   drawpxl(299+x2,240+y2);
   drawpxl(300+x2,239+y2);
   drawpxl(300+x2,241+y2);
   drawline(300+x1,240+y1,300+x2,240+y2);
   drawtxt("Shadow", 235+x2, 230+y2);

   drawtxt("Solar Calendar", 50, 40);
   sprintf(ttlstr,"Latitude = %3.1f deg.", lat);
   drawtxt(ttlstr, 40, 60);
   drawtxt("Pole", 300-15, 240+20);
   drawtxt("North", 300-18, 50);

   hndl = gzopen("solcal.ppm.gz", "wb9");
   if (hndl == NULL)
      {
      fprintf(stderr,"Error opening ppm file\n");
      perror("Open error");
      exit(1);
      } /* if open error */
   wrtppm(hndl);
   rslt = gzclose(hndl);
   if (rslt < 0)
      {
      fprintf(stderr,"Error closing ppm file\n");
      perror("Close error");
      exit(1);
      } /* if closing error */
   } /* draw */

int main(argc,argv)
int argc;
char **argv;
   {
   int yr,mon,day;
   int lsthr,lstmin;
   int lpyr;
   int hr_r,min_r,hr_s,min_s;
   int x,y;
   double lon,lat,timezone;
   double ltr,lts;
   double azimuthr,azimuths;
   double sec_r,sec_s;
   double sumaz,winaz;
   if (argc != 5) putstx(*argv);
   yr  = atoi(*(argv+1));
   if (yr < 1900    || yr > 2999)   putstx(*argv);
   lon = atof(*(argv+2));
   if (lon < -180.0 || lon > 180.0) putstx(*argv);
   lat = atof(*(argv+3));
   if (lat < -66.5  || lat > 66.5)  putstx(*argv);
   timezone = atof(*(argv+4));
   if (timezone < -12.0 || timezone > 12.0)
      putstx(*argv);
   lpyr = initmlen(yr);
   riseset(yr,6,21,0,0,0.0,
      lon,lat,timezone,
      &ltr,&lts,&azimuthr,&azimuths);
   sumaz = azimuthr;
   hhh2hms(ltr,&hr_r,&min_r,&sec_r);
   hhh2hms(lts,&hr_s,&min_s,&sec_s);
   printf("06/21  %02d:%02d:%02.f  %15f   "
      "%02d:%02d:%02.f  %15f\n",
      hr_r, min_r, sec_r, azimuthr,
      hr_s, min_s, sec_s, azimuths);
   riseset(yr,12,21,0,0,0.0,
      lon,lat,timezone,
      &ltr,&lts,&azimuthr,&azimuths);
   winaz = azimuthr;
   hhh2hms(ltr,&hr_r,&min_r,&sec_r);
   hhh2hms(lts,&hr_s,&min_s,&sec_s);
   printf("12/21  %02d:%02d:%02.f  %15f   "
      "%02d:%02d:%02.f  %15f\n",
      hr_r, min_r, sec_r, azimuthr,
      hr_s, min_s, sec_s, azimuths);
   for (y=0;y<YSZ;y++)
      {
      for (x=0;x<XSZ;x++) mtx[y][x] = 0;
      } /* for each row */
   draw(sumaz,winaz,lat,yr);
   return(0);
   } /* main */
