/*******************************************************************************

 		BIM-XXX-RS232 Radio Transceiver driver.
	Copyright (c) 1997 Walter Fetter Lages <w.fetter@ieee.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Please note that the GPL allows you to use the driver, NOT the radio.
    In order to use the radio, you need a license from the communications
    authority of your country.

    Command line options (insmod command line)
 
    major    major number the driver should use; default 51 
    iobase#  base address of the port; common values are 0x3f8, 0x2f8, 0x3e8,
	     and 0x2e8
    irq#     interrupt line of the port; common values are 3 and 4
    type#    modem type; 0=none, 1=BIM RS232 modem, 2=NULL MODEM cable
    baud#    default baud rate; default 19200
    tx_delay# transmitter keyup delay in 10 ms; default 20
    tx_tail#  transmitter tail time in 10 ms; default 2
    slottime# transmitter slot time in 10 ms; default 10
    ppersist# tranmitter p persistence scaled to 0..255 range; default 63
    fulldup# full duplex; default 0
		
             # = 0, 1, 2 or 3.	

    History:
    0.0  05.31.97  Start development based on Baycom ser12 and 
		   par96 radio modem driver version 0.3, Copyright (C) 1996
		   Thomas Sailer (sailer@ife.ee.ethz.ch), distributed with
		   Linux kernel version 2.0.28.

    0.1  06.20.97  First working version.
    1.0  07.06.97  First released version.
	
*******************************************************************************/

#define MODULE
#include <linux/module.h>

extern "C"
{
#include <linux/fs.h>
#include <linux/ioport.h>
#include <linux/malloc.h>
#include <linux/mm.h>
#include <linux/sched.h>
};

#include <asm/io.h>

#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/interrupt.h>
#include <linux/tqueue.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>

#include <bim.h>

#define BIM_TYPE_NORMAL 0
#define TTY_DRIVER_TYPE_BIM 6

#define BUFLEN_RX 8192
#define BUFLEN_TX 8192

#define NR_PORTS 4

#define KISS_VERBOSE

#define BIM_MAGIC (0x42494d31)		

#define BIM_BH IMMEDIATE_BH	// defined in linux/interrupt.h

// UART registers
#define RBR(iobase) (iobase+0)
#define THR(iobase) (iobase+0)
#define IER(iobase) (iobase+1)
#define IIR(iobase) (iobase+2)
#define FCR(iobase) (iobase+2)
#define LCR(iobase) (iobase+3)
#define MCR(iobase) (iobase+4)
#define LSR(iobase) (iobase+5)
#define MSR(iobase) (iobase+6)
#define SCR(iobase) (iobase+7)
#define DLL(iobase) (iobase+0)
#define DLM(iobase) (iobase+1)

#define BIM_EXTENT 8

// Modem state flags
#define ARB_TIMER_ON	0x01

// default baud rate
#define DFLT_BAUD	19200

static int major=BIM_MAJOR;

static int base0=0x3f8;
static int irq0=4;
static int type0=BIM_INVALID;
static int baud0=DFLT_BAUD;
static int tx_delay0=20;
static int tx_tail0=2;
static int slottime0=10;
static int ppersist0=63;
static int fulldup0=0;

static int base1=0x2f8;
static int irq1=3;
static int type1=BIM_INVALID;
static int baud1=DFLT_BAUD;
static int tx_delay1=20;
static int tx_tail1=2;
static int slottime1=10;
static int ppersist1=63;
static int fulldup1=0;

static int base2=0x3e8;
static int irq2=4;
static int type2=BIM_INVALID;
static int baud2=DFLT_BAUD;
static int tx_delay2=20;
static int tx_tail2=2;
static int slottime2=10;
static int ppersist2=63;
static int fulldup2=0;


static int base3=0x2e8;
static int irq3=3;
static int type3=BIM_INVALID;
static int baud3=DFLT_BAUD;
static int tx_delay3=20;
static int tx_tail3=2;
static int slottime3=10;
static int ppersist3=63;
static int fulldup3=0;

enum UART
{
	UART_UNKNOWN,
	UART_8250,
	UART_16450,
	UART_16550,
	UART_16550A
};

enum IER_BITS
{
	ERB=	0x01,
	ETBEI=	0x02,
	ELSI=	0x04,
	EDSSI=	0x08
};

enum IIR_BITS
{
	IP0=	0x01,
	IID=	0x0e,
	IRLS=	0x06,
	IRDA=	0x04,
	ICTO=	0x0c,
	ITHRE=	0x02,
	IMS=	0x00,
	IFIFO=	0xc0
};

enum FCR_BITS
{
	FIFOE=	0x01,
	RFIFOR=	0x02,
	TFIFOR=	0x04,
	DMAMS=	0x08,
	RTL=	0x40,
	RTM=	0x80
};

enum LCR_BITS
{
	WLS0=	0x01,
	WLS1=	0x02,
	WL8=	0x03,
	STB=	0x04,
	PEN=	0x08,
	EPS=	0x10,
	SP=	0x20,
	BREAK=	0x40,
	DLAB=	0x80
};
	

enum MCR_BITS
{
	DTR=	0x01,
	RTS=	0x02,
	OUT1=	0x04,
	OUT2=	0x08,
	LOOP=	0x10,
};

enum MSR_BITS
{
	DCTS=	0x01,
	DDSR=	0x02,
	TERI=	0x04,
	DDCD=	0x08,
	CTS=	0x10,
	DSR=	0x20,
	RI=	0x40,
	DCD=	0x80
};

enum LSR_BITS
{
	DR=	0x01,
	OE=	0x02,
	PE=	0x04,
	FE=	0x08,
	BI=	0x10,
	THRE=	0x20,
	TSRE=	0x40
};

enum BH_SCHED_BITS
{
	BH_SCHED_RX=	0x01,
	BH_SCHED_TX=	0x02
};
	

struct access_params
{
	int tx_delay;  // the transmitter keyup delay in 10ms units
	int tx_tail;   // the transmitter keyoff delay in 10ms units
	int slottime;  // the slottime in 10ms; usually 10 = 100ms
	int ppersist;  // the p-persistence 0..255
	int fulldup;   // the driver does not support full duplex, setting
	               // this just makes the driver send even if DCD is on
};

#define BH_RX_BUFLEN	256	// bottom half RX buffer
#define BH_TX_BUFLEN	16	// bottom half TX buffer

struct hdlc_state_rx
{
	int rx_state;			// 0 = sync hunt, != 0 receiving
	unsigned int bitstream;		// keep track of bitstream to catch stuffed bit
	unsigned int bitbuf;		// hdlc input bit buffer
	int numbits;			// number of bits at bitbuf

	unsigned char bh_buffer[BH_RX_BUFLEN];	// bottom-half buffer
	int bh_wr;		// bottom-half buffer write point
	int bh_rd;		// bottom-half buffer read point
	int bh_buflen;		// bottom-half buffer lenght; redundant but simpler

	int len;			// buffer lenght in use
	unsigned char *bp;		// buffer pointer
	unsigned char buffer[BIM_MAXFLEN+2];	   // make room for CRC
};

struct hdlc_state_tx
{

	int tx_state;		// 0=send flags,1=send txtail(flags),2=send packet
	int numflags;		// number of flags to send
	unsigned int bitstream;	// keep track of bit stream to insert stuff bit
	unsigned char ptt;	// push to talk state
	unsigned char holdptt;	// flag to release ptt after transmit

	unsigned int bitbuf;	// buffer for hdlc output bits
	int numbits;		// number of bits available at bitbuf

	unsigned char bh_buffer[BH_TX_BUFLEN];	// bottom-half buffer
	int bh_wr;		// bottom-half buffer write point
	int bh_rd;		// bottom-half buffer read point
	int bh_buflen;		// bottom-half buffer lenght; redundant but simpler

	int len;		// frame buffer lenght in use
	unsigned char *bp;	// frame buffer pointer
	unsigned char buffer[BIM_MAXFLEN+2];		// make room for CRC
};

struct modem_state
{
	unsigned char dcd;
	unsigned int flags;
};

struct packet_buffer
{
	unsigned int rd;
	unsigned int wr;
	
	unsigned int buflen;
	unsigned char *buffer;
};

struct packet_hdr
{
	unsigned int next;
	unsigned int len;
	// packet following
};

#ifdef BIM_DEBUG
struct bit_buffer
{
	unsigned int rd;
	unsigned int wr;
	unsigned char buffer[64];
};

struct debug_vals
{
	unsigned long last_jiffies;
	unsigned cur_intcnt;
	unsigned last_intcnt;
};
#endif // BIM_DEBUG

struct kiss_decode
{
	unsigned char dec_state; // 0 = hunt FEND
	unsigned char escaped;
	unsigned char pkt_buf[BIM_MAXFLEN+1];
	unsigned int wr;
};





struct bim_state
{
	int magic;

	unsigned int iobase;
	unsigned int irq;
	int type;
	int dflt_baud;
	int uart;

	int opened;
	struct tty_struct *tty;

#ifdef BIM_USE_BH
	struct tq_struct tq_receiver;
	struct tq_struct tq_transmitter;
#endif // BIM_USE_BH

	struct packet_buffer rx_buf;
	struct packet_buffer tx_buf;

	struct access_params ch_params;

	struct hdlc_state_rx hdlc_rx;
	struct hdlc_state_tx hdlc_tx;

	struct modem_state modem;

	struct timer_list arb_timer;

#ifdef BIM_DEBUG
	struct bit_buffer bitbuf_hdlc;
	
	struct debug_vals debug_vals;
#endif // BIM_DEBUG

	struct kiss_decode kiss_decode;

	struct bim_statistics stat;

	int preamble;
};

#ifdef BIM_USE_BH
DECLARE_TASK_QUEUE(tq_bim);
#endif // BIM_USE_BH

static struct
{
	int iobase;
	int irq;
	int type;
	int baud;
	struct access_params dflt_ch_params;
} bim_ports[NR_PORTS];

static struct tty_struct *bim_table[NR_PORTS];
static struct termios *bim_termios[NR_PORTS];
static struct termios *bim_termios_locked[NR_PORTS];

static int bim_refcount;

static struct tty_driver bim_driver;

static struct bim_state bim_state[NR_PORTS];



/*******************************************************************************

	 The CRC routines are stolen from WAMPES by Dieter Deyke

*******************************************************************************/

static const unsigned short crc_ccitt_table[]=
{
	0x0000, 0x1189, 0x2312, 0x329b, 0x4624, 0x57ad, 0x6536, 0x74bf,
	0x8c48, 0x9dc1, 0xaf5a, 0xbed3, 0xca6c, 0xdbe5, 0xe97e, 0xf8f7,
	0x1081, 0x0108, 0x3393, 0x221a, 0x56a5, 0x472c, 0x75b7, 0x643e,
	0x9cc9, 0x8d40, 0xbfdb, 0xae52, 0xdaed, 0xcb64, 0xf9ff, 0xe876,
	0x2102, 0x308b, 0x0210, 0x1399, 0x6726, 0x76af, 0x4434, 0x55bd,
	0xad4a, 0xbcc3, 0x8e58, 0x9fd1, 0xeb6e, 0xfae7, 0xc87c, 0xd9f5,
	0x3183, 0x200a, 0x1291, 0x0318, 0x77a7, 0x662e, 0x54b5, 0x453c,
	0xbdcb, 0xac42, 0x9ed9, 0x8f50, 0xfbef, 0xea66, 0xd8fd, 0xc974,
	0x4204, 0x538d, 0x6116, 0x709f, 0x0420, 0x15a9, 0x2732, 0x36bb,
	0xce4c, 0xdfc5, 0xed5e, 0xfcd7, 0x8868, 0x99e1, 0xab7a, 0xbaf3,
	0x5285, 0x430c, 0x7197, 0x601e, 0x14a1, 0x0528, 0x37b3, 0x263a,
	0xdecd, 0xcf44, 0xfddf, 0xec56, 0x98e9, 0x8960, 0xbbfb, 0xaa72,
	0x6306, 0x728f, 0x4014, 0x519d, 0x2522, 0x34ab, 0x0630, 0x17b9,
	0xef4e, 0xfec7, 0xcc5c, 0xddd5, 0xa96a, 0xb8e3, 0x8a78, 0x9bf1,
	0x7387, 0x620e, 0x5095, 0x411c, 0x35a3, 0x242a, 0x16b1, 0x0738,
	0xffcf, 0xee46, 0xdcdd, 0xcd54, 0xb9eb, 0xa862, 0x9af9, 0x8b70,
	0x8408, 0x9581, 0xa71a, 0xb693, 0xc22c, 0xd3a5, 0xe13e, 0xf0b7,
	0x0840, 0x19c9, 0x2b52, 0x3adb, 0x4e64, 0x5fed, 0x6d76, 0x7cff,
	0x9489, 0x8500, 0xb79b, 0xa612, 0xd2ad, 0xc324, 0xf1bf, 0xe036,
	0x18c1, 0x0948, 0x3bd3, 0x2a5a, 0x5ee5, 0x4f6c, 0x7df7, 0x6c7e,
	0xa50a, 0xb483, 0x8618, 0x9791, 0xe32e, 0xf2a7, 0xc03c, 0xd1b5,
	0x2942, 0x38cb, 0x0a50, 0x1bd9, 0x6f66, 0x7eef, 0x4c74, 0x5dfd,
	0xb58b, 0xa402, 0x9699, 0x8710, 0xf3af, 0xe226, 0xd0bd, 0xc134,
	0x39c3, 0x284a, 0x1ad1, 0x0b58, 0x7fe7, 0x6e6e, 0x5cf5, 0x4d7c,
	0xc60c, 0xd785, 0xe51e, 0xf497, 0x8028, 0x91a1, 0xa33a, 0xb2b3,
	0x4a44, 0x5bcd, 0x6956, 0x78df, 0x0c60, 0x1de9, 0x2f72, 0x3efb,
	0xd68d, 0xc704, 0xf59f, 0xe416, 0x90a9, 0x8120, 0xb3bb, 0xa232,
	0x5ac5, 0x4b4c, 0x79d7, 0x685e, 0x1ce1, 0x0d68, 0x3ff3, 0x2e7a,
	0xe70e, 0xf687, 0xc41c, 0xd595, 0xa12a, 0xb0a3, 0x8238, 0x93b1,
	0x6b46, 0x7acf, 0x4854, 0x59dd, 0x2d62, 0x3ceb, 0x0e70, 0x1ff9,
	0xf78f, 0xe606, 0xd49d, 0xc514, 0xb1ab, 0xa022, 0x92b9, 0x8330,
	0x7bc7, 0x6a4e, 0x58d5, 0x495c, 0x3de3, 0x2c6a, 0x1ef1, 0x0f78
};


static void append_crc_ccitt(unsigned char *buffer,int len)
{
 	unsigned short crc=0xffff;

	for(;len > 0;len--) crc=(crc >> 8) ^ crc_ccitt_table[(crc ^ *buffer++) & 0xff];
	crc^= 0xffff;
	*buffer++=crc;
	*buffer++=crc >> 8;
}


static int check_crc_ccitt(const unsigned char *buf,int cnt)
{
	unsigned short crc=0xffff;

	for(; cnt > 0; cnt--) crc=(crc >> 8) ^ crc_ccitt_table[(crc ^ *buf++) & 0xff];
	return (crc & 0xffff) == 0xf0b8;
}



static int baud_table[]=
{
	0,50,75,110,134,150,200,300,600,1200,1800,2400,4800,
	9600,19200,38400,57600,115200,0
};

static int baud(struct bim_state *bim)
{
	switch(bim->type)
	{
		case BIM_MODEM:
		case BIM_CABLE:
		{
			unsigned int i=bim->tty->termios->c_cflag & CBAUD;
			if(i & CBAUDEX)
			{	
				i&=~CBAUDEX;
				i+=15;
			}
			return baud_table[i];
		}
	}
	return 0;
}

static inline unsigned int tenms_to_flags(struct bim_state *bim,unsigned int tenms)
{
	return (tenms*baud(bim)+999)/1000;
}




/*******************************************************************************

			 KISS decoder functions

*******************************************************************************/

static int store_packet(struct packet_buffer *buf,unsigned char *data,char from_user,unsigned int len)
{
	struct packet_hdr *hdr;
	unsigned int needed=sizeof(struct packet_hdr)+len;
	
	unsigned int free=buf->rd-buf->wr;
	if(buf->rd <= buf->wr)
	{
		free=buf->buflen-buf->wr;
		if((free < needed) && (buf->rd >= needed))
		{
			hdr=(struct packet_hdr *)(buf->buffer+buf->wr);
			hdr->next=0;
			hdr->len=0;
			buf->wr=0;
			free=buf->rd;
		}
	}
	if(free < needed) return 0;		// buffer overrun
	hdr=(struct packet_hdr *)(buf->buffer+buf->wr);
	if(from_user) memcpy_fromfs(hdr+1,data,len);
	else memcpy(hdr+1,data,len);
	hdr->len=len;
	hdr->next=buf->wr+needed;
	if(hdr->next+sizeof(struct packet_hdr) >= buf->buflen) hdr->next=0;
	buf->wr=hdr->next;
	return 1;
}
	

static void get_packet(struct packet_buffer *buf,unsigned char **data,unsigned int *len)
{
	*data=NULL;
	*len=0;
	if(buf->rd == buf->wr) return;
	struct packet_hdr *hdr=(struct packet_hdr *)(buf->buffer+buf->rd);
	while(!(hdr->len))
	{
		buf->rd=hdr->next;
		if (buf->rd == buf->wr)	return;
		hdr=(struct packet_hdr *)(buf->buffer+buf->rd);
	}
	*data=(unsigned char *)(hdr+1);
	*len=hdr->len;
}


static void ack_packet(struct packet_buffer *buf)
{
	if (buf->rd == buf->wr)	return;
	struct packet_hdr *hdr=(struct packet_hdr *)(buf->buffer+buf->rd);
	buf->rd=hdr->next;
}


static int store_kiss_packet(struct packet_buffer *buf,unsigned char *data,unsigned int len)
{
	unsigned char *bp=data;
	int ln=len;

	// variables of buf
	
	unsigned int rd;
	unsigned int wr;
	unsigned int buflen;
	unsigned char *buffer;

	if(!len || !data || !buf) return 0;
	buflen=buf->buflen;
	rd=buf->rd;
	wr=buf->wr;
	buffer=buf->buffer;
	
#define ADD_CHAR(c) {\
		buffer[wr++] = c;\
		if (wr >= buflen) wr = 0;\
		if (wr == rd) return 0;\
	}
#define ADD_KISSCHAR(c) {\
		if (((c) & 0xff) == KISS_FEND) {\
			ADD_CHAR(KISS_FESC);\
			ADD_CHAR(KISS_TFEND);\
		} else if (((c) & 0xff) == KISS_FESC) {\
			ADD_CHAR(KISS_FESC);\
			ADD_CHAR(KISS_TFESC);\
		} else {\
			ADD_CHAR(c);\
		}\
	}

	ADD_CHAR(KISS_FEND);
	ADD_KISSCHAR(KISS_CMD_DATA);
	for(;ln > 0;ln--,bp++) ADD_KISSCHAR(*bp);

	ADD_CHAR(KISS_FEND);
	buf->wr=wr;
#undef ADD_CHAR
#undef ADD_KISSCHAR

	return 1;
}


static void bim_put_fend(struct bim_state *bim)
{
	if(bim->kiss_decode.wr <= 0 || (bim->kiss_decode.pkt_buf[0] & 0xf0) != 0) return;
	switch (bim->kiss_decode.pkt_buf[0] & 0xf)
	{
		case KISS_CMD_DATA:
		{
			if(bim->kiss_decode.wr <= 8) break;
			if(!store_packet(&bim->tx_buf,bim->kiss_decode.pkt_buf+1,0,bim->kiss_decode.wr-1)) bim->stat.tx_bufferoverrun++;
			break;
		}

		case KISS_CMD_TXDELAY:
		{
			if(bim->kiss_decode.wr < 2) break;
			bim->ch_params.tx_delay=bim->kiss_decode.pkt_buf[1];

#ifdef KISS_VERBOSE
			printk(KERN_INFO "BIM: TX delay = %ums\n",bim->ch_params.tx_delay * 10);
#endif // KISS_VERBOSE

			break;
		}

		case KISS_CMD_PPERSIST:
		{
			if(bim->kiss_decode.wr < 2) break;
			bim->ch_params.ppersist=bim->kiss_decode.pkt_buf[1];

#ifdef KISS_VERBOSE
			printk(KERN_INFO "BIM: p-persistence = %u\n",bim->ch_params.ppersist);
#endif // KISS_VERBOSE

			break;
		}

		case KISS_CMD_SLOTTIME:
		{
			if(bim->kiss_decode.wr < 2) break;
			bim->ch_params.slottime=bim->kiss_decode.pkt_buf[1];

#ifdef KISS_VERBOSE
			printk(KERN_INFO "BIM: slottime = %ums\n",bim->ch_params.slottime*10);
#endif // KISS_VERBOSE

			break;
		}

		case KISS_CMD_TXTAIL:
		{
			if(bim->kiss_decode.wr < 2) break;
			bim->ch_params.tx_tail=bim->kiss_decode.pkt_buf[1];

#ifdef KISS_VERBOSE
			printk(KERN_INFO "BIM: TX tail = %ums\n",bim->ch_params.tx_tail*10);
#endif // KISS_VERBOSE

			break;
		}

		case KISS_CMD_FULLDUP:
		{
			if(bim->kiss_decode.wr < 2) break;
			bim->ch_params.fulldup=bim->kiss_decode.pkt_buf[1];

#ifdef KISS_VERBOSE
			printk(KERN_INFO "BIM: %s duplex\n",bim->ch_params.fulldup ? "full" : "half");
#endif // KISS_VERBOSE
	
			break;
		}

		default:
		{

#ifdef KISS_VERBOSE
			printk(KERN_INFO "BIM: unhandled KISS packet code %u\n",bim->kiss_decode.pkt_buf[0] & 0xf);
#endif // KISS_VERBOSE

			break;
		}
	}
}

/*******************************************************************************

			 Tx Arbitration functions

*******************************************************************************/

/*
 * ANSI sample implementation borrowed form Metaware High C Reference Manual
 * ANSI uses 1 as seed. Here seed is get from jiffies to be more, well, random.
 */
static unsigned long int rand_seed=(unsigned short) jiffies;	//random seed;

static inline int rand(void)
{
	rand_seed=1103515245UL*rand_seed+12345;
	return (unsigned int) (rand_seed/65536) % 32768;
}

inline unsigned char preamble_tx(struct bim_state *bim)
{
	if(bim->preamble < 0) return 0;
	return (bim->preamble--)? 'U':0xff;
}

static inline void ptt(struct bim_state *bim,int value)
{
	unsigned int mcr=inb(MCR(bim->iobase));
	if(value)
	{
		outb(mcr | RTS,MCR(bim->iobase));	// set RTS
		bim->hdlc_tx.holdptt=1;
		bim->hdlc_tx.ptt=1;
		bim->stat.ptt_keyed++;
		bim->preamble=(3*baud(bim)+9999)/10000;

		// enable TX interrupt
		unsigned char ier=inb(IER(bim->iobase));
		outb(ier | ETBEI,IER(bim->iobase));
	}
	else
	{
		bim->hdlc_tx.ptt=0;
		// disable tx interrupt
		unsigned char ier=inb(IER(bim->iobase));
		outb(ier & ~ETBEI,IER(bim->iobase));
		outb(mcr & ~RTS,MCR(bim->iobase));	// reset RTS
		bim->hdlc_tx.holdptt=0;
	}
}

static inline void tx_arbitrate(struct bim_state *bim)
{
	unsigned char *bp;
	unsigned int len;
	
	if(!bim || bim->hdlc_tx.ptt || bim->modem.dcd) return;

	get_packet(&bim->tx_buf,&bp,&len);
	if(!bp || !len) return;
	if(!bim->ch_params.fulldup)
	{
		if((rand() % 256) > bim->ch_params.ppersist) return;
	}
	bim->hdlc_tx.tx_state=0;
	if(!(bim->hdlc_tx.numflags=tenms_to_flags(bim,bim->ch_params.tx_delay))) bim->hdlc_tx.numflags=1;
	bim->hdlc_tx.numbits=bim->hdlc_tx.bitbuf=bim->hdlc_tx.bitstream=0;
	ptt(bim,1);
}

static inline void tx_release(struct bim_state *bim)
{
	bim->hdlc_tx.tx_state=1;
	if(!(bim->hdlc_tx.numflags=tenms_to_flags(bim,bim->ch_params.tx_tail))) bim->hdlc_tx.numflags=1;
	if(bim->hdlc_tx.numbits % 8) bim->hdlc_tx.numflags++;
}

static inline void rx_arbitrate(struct bim_state *bim)
{
	if(!bim || !bim->hdlc_tx.ptt) return;

	if(!bim->ch_params.fulldup)
	{
		if((rand() % 256) > bim->ch_params.ppersist) return;
		tx_release(bim);
	}
}

static void arb_sched(unsigned long data)
{
	struct bim_state *bim=(struct bim_state *)data;

	if(!bim || bim->magic != BIM_MAGIC) return;

	unsigned long ms=bim->ch_params.slottime*10;
	del_timer(&bim->arb_timer);
	bim->arb_timer.expires=jiffies+(ms*HZ+999)/1000+1;
	add_timer(&bim->arb_timer);

	 // if transmitting don't even test DCD 
	if(bim->hdlc_tx.ptt) return;

	switch(bim->type)
	{
		case BIM_MODEM:
		{
			// BIM CD logis IS inverted !!!
			bim->modem.dcd=(~inb(MSR(bim->iobase)) & DCD) >> 7;
			break;
		}
		case BIM_CABLE:
		{
			// use CTS as CD
			bim->modem.dcd=(inb(MSR(bim->iobase)) & CTS) >> 4;
			break;
		}
		default: bim->modem.dcd=(inb(MSR(bim->iobase)) & DCD) >> 7;
	}

	tx_arbitrate(bim);
}


/*******************************************************************************

				HDLC functions

********************************************************************************/

#ifdef BIM_DEBUG
inline static void add_bitbuffer_byte(struct bit_buffer *buf,unsigned char byte)
{
	buf->buffer[buf->wr]=byte;
	buf->wr=(buf->wr+1)%sizeof(buf->buffer);
}
#endif // BIM_DEBUG

/*
 * This routine moves received bytes from rx buffer to flip buffer.
 * The flip buffer is located in the tty structure, and is used as a high 
 * speed interface between the tty driver and the tty line discipline.
 */
static inline void rx_chars_to_flip(struct bim_state *bim) 
{
	if((!bim) || (!bim->tty) || (bim->tty->flip.count >= TTY_FLIPBUF_SIZE) ||
	    (bim->rx_buf.rd == bim->rx_buf.wr) || 
	    (!bim->tty->flip.char_buf_ptr) ||
	    (!bim->tty->flip.flag_buf_ptr)) return;
	for(;;)
	{
		// computes the number of bytes to transfer
		int flip_free=TTY_FLIPBUF_SIZE-bim->tty->flip.count;
		unsigned int cnt;
		if(bim->rx_buf.rd <= bim->rx_buf.wr) cnt=bim->rx_buf.wr-bim->rx_buf.rd;
		else cnt=bim->rx_buf.buflen-bim->rx_buf.rd;
		if((flip_free <= 0) || (!cnt))
		{
			tty_schedule_flip(bim->tty);
			return;
		}
		if(cnt > (unsigned int)flip_free) cnt=flip_free;

		// transfer and update flip buffer pointers
		unsigned long flags;
		save_flags(flags);
		cli();
		memcpy(bim->tty->flip.char_buf_ptr,bim->rx_buf.buffer+bim->rx_buf.rd,cnt);
		memset(bim->tty->flip.flag_buf_ptr,TTY_NORMAL,cnt);
		bim->tty->flip.count+=cnt;
		bim->tty->flip.char_buf_ptr+= cnt;
		bim->tty->flip.flag_buf_ptr+= cnt;
		restore_flags(flags);

		// updates rx buffer pointer
		unsigned int new_rd=bim->rx_buf.rd+cnt;
		if(new_rd >= bim->rx_buf.buflen) new_rd -= bim->rx_buf.buflen;
		bim->rx_buf.rd=new_rd;
	}
}

static inline int hdlc_rx_add_bytes(struct bim_state *bim,unsigned int bits,int num)
{
	int added=0;
	while(bim->hdlc_rx.rx_state && num >= 8)
	{
		if((unsigned int)bim->hdlc_rx.len >= sizeof(bim->hdlc_rx.buffer))
		{
			// buffer overflow
			bim->hdlc_rx.rx_state=0;
			return 0;
		}
		*bim->hdlc_rx.bp++=bits >> (16-num);
		bim->hdlc_rx.len++;
		num-=8;
		added+=8;
	}
	return added;
}

static inline void hdlc_rx_flag(struct bim_state *bim)
{
	if(bim->hdlc_rx.len < 4) return;
	if(!check_crc_ccitt(bim->hdlc_rx.buffer,bim->hdlc_rx.len)) return;
       	bim->stat.rx_packets++;
	if(!store_kiss_packet(&bim->rx_buf,bim->hdlc_rx.buffer,bim->hdlc_rx.len-2)) bim->stat.rx_bufferoverrun++;
	rx_chars_to_flip(bim);	// send packet to flip buffer
}

static void hdlc_rx_byte(struct bim_state *bim,unsigned int byte)
{
	if(!bim) return;


	byte&=0xff;		// mask unused bits

#ifdef BIM_DEBUG
	add_bitbuffer_byte(&bim->bitbuf_hdlc,byte);
#endif // BIM_DEBUG
       	bim->hdlc_rx.bitstream >>= 8;
	bim->hdlc_rx.bitstream |= byte << 8;
	bim->hdlc_rx.bitbuf >>= 8;
	bim->hdlc_rx.bitbuf |= byte << 8;
	bim->hdlc_rx.numbits+=8;
	// check stuffed bit
	int i;
	unsigned int mask1;
	unsigned int mask2;
	unsigned int mask3;
	unsigned int mask4;
	unsigned int mask5;
	unsigned int mask6;

	for(i=7,mask1=0x1fc,mask2=0x1fe,mask3=0x0fc,
	    mask4=0x1f8,mask5=0xf8,mask6=0xff; 
	    i >= 0; 
	    i--,mask1 <<= 1,mask2 <<= 1,mask3 <<= 1,mask4 <<= 1, 
	    mask5 <<= 1,mask6 = (mask6 << 1) | 1)
	{
		if((bim->hdlc_rx.bitstream & mask1) == mask1) bim->hdlc_rx.rx_state=0; // abort received
		else if((bim->hdlc_rx.bitstream & mask2) == mask3)
		{
			// flag received
			if(bim->hdlc_rx.rx_state)
			{
				// terminate frame
				hdlc_rx_add_bytes(bim,bim->hdlc_rx.bitbuf << (8 + i),bim->hdlc_rx.numbits-8-i);
				hdlc_rx_flag(bim);
			}
			// start new frame
			bim->hdlc_rx.len=0;
			bim->hdlc_rx.bp=bim->hdlc_rx.buffer;
			bim->hdlc_rx.rx_state=1;
			bim->hdlc_rx.numbits=i;
		}
		else if((bim->hdlc_rx.bitstream & mask4) == mask5)
		{
			// stuffed bit
			bim->hdlc_rx.numbits--;
			bim->hdlc_rx.bitbuf=(bim->hdlc_rx.bitbuf & (~mask6)) |	((bim->hdlc_rx.bitbuf & mask6) << 1);
		}
	}
	bim->hdlc_rx.numbits-=hdlc_rx_add_bytes(bim,bim->hdlc_rx.bitbuf,bim->hdlc_rx.numbits);
}

static unsigned char hdlc_tx_byte(struct bim_state *bim)
{
	if(!bim || !bim->hdlc_tx.ptt) return 0;
	for(;;)
	{
		// if there are bits availabe at bitbuf return them
		if(bim->hdlc_tx.numbits >= 8)
		{
			unsigned char ret=bim->hdlc_tx.bitbuf & 0xff;
			bim->hdlc_tx.bitbuf>>=8;
			bim->hdlc_tx.numbits-=8;
			return ret;
		}

		switch(bim->hdlc_tx.tx_state)
		{
			default:	// reset
			{
				ptt(bim,0);
				bim->hdlc_tx.tx_state=0;
				return 0;
			}
			case 0:		// send flags
			case 1:		// send txtail(flags)
			{
				if(bim->hdlc_tx.numflags)
				{
					// insert flags into bitbuf
					bim->hdlc_tx.numflags--;
					bim->hdlc_tx.bitbuf|=0x7e << bim->hdlc_tx.numbits;
					bim->hdlc_tx.numbits+=8;
					break;
				}
				if(bim->hdlc_tx.tx_state == 1)
				{
					// no more flags, reset ptt
					bim->hdlc_tx.holdptt=0;
					return 0;
				}
				// get packet pointer and lenght
				get_packet(&bim->tx_buf,&bim->hdlc_tx.bp,(unsigned int *)&bim->hdlc_tx.len);
				if(!bim->hdlc_tx.bp || !bim->hdlc_tx.len)
				{
					// no more packets
					tx_release(bim);
					break;
				}
				if(bim->hdlc_tx.len >= BIM_MAXFLEN)
				{
					// packet too big, discard
					bim->hdlc_tx.tx_state=0;
					bim->hdlc_tx.numflags=1;
					ack_packet(&bim->tx_buf);

					// chance to revert channel
					rx_arbitrate(bim);
					break;
				}
				// copy packet to buffer
				memcpy(bim->hdlc_tx.buffer,bim->hdlc_tx.bp,bim->hdlc_tx.len);
				ack_packet(&bim->tx_buf);
				bim->hdlc_tx.bp=bim->hdlc_tx.buffer;
				append_crc_ccitt(bim->hdlc_tx.buffer,bim->hdlc_tx.len);
				// the appended CRC
				bim->hdlc_tx.len+=2; 
				bim->hdlc_tx.tx_state=2;	// send packet
				bim->hdlc_tx.bitstream=0;
				bim->stat.tx_packets++;
				break;
			}
			case 2:		// send packet
			{
				if(!bim->hdlc_tx.len)
				{
					// packet end
					bim->hdlc_tx.tx_state=0;
					bim->hdlc_tx.numflags=1;

					// chance to revert channel
					rx_arbitrate(bim); 
					break;
				}
				bim->hdlc_tx.len--;
				bim->hdlc_tx.bitbuf |= *bim->hdlc_tx.bp << bim->hdlc_tx.numbits;
				bim->hdlc_tx.bitstream >>= 8;
				bim->hdlc_tx.bitstream |= (*bim->hdlc_tx.bp++) << 16;
				// test if stuff bit is needed
				unsigned int mask1=0x1f000;
				unsigned int mask2=0x10000;
				unsigned int mask3=0xffffffff >> (31-bim->hdlc_tx.numbits);
				bim->hdlc_tx.numbits+=8;
				for(int i=0;i < 8;i++,mask1 <<= 1,mask2 <<= 1,mask3 = (mask3 << 1) | 1)
				{
					if((bim->hdlc_tx.bitstream & mask1) != mask1) continue;
					// insert stuff bit
					bim->hdlc_tx.bitstream&=~mask2; 
					bim->hdlc_tx.bitbuf=(bim->hdlc_tx.bitbuf & mask3) | ((bim->hdlc_tx.bitbuf & (~mask3)) << 1);
					bim->hdlc_tx.numbits++;
					mask3=(mask3 << 1) | 1;
				}
				break;
			}			
		}
	}
}


/*******************************************************************************

			 Interrupt & Bottom-half functions

*******************************************************************************/

#ifdef BIM_DEBUG
/*
 * This routine measures how many interrupts are received per second.
 */
inline static void bim_int_freq(struct bim_state *bim)
{
	unsigned long cur_jiffies=jiffies;

	// measure the interrupt frequency

	bim->debug_vals.cur_intcnt++;
	if((cur_jiffies-bim->debug_vals.last_jiffies) >= HZ)
	{
		bim->debug_vals.last_jiffies=cur_jiffies;
		bim->debug_vals.last_intcnt=bim->debug_vals.cur_intcnt;
		bim->debug_vals.cur_intcnt=0;
	}
}
#endif // BIM_DEBUG



static void bim_interrupt(int irq,void *dev_id,struct pt_regs *regs)
{
	register struct bim_state *bim=(struct bim_state *)dev_id;
	unsigned int data;

	if(!bim || bim->magic != BIM_MAGIC) return;

#ifdef BIM_DEBUG
	bim_int_freq(bim);
#endif // BIM_DEBUG

#ifdef BIM_USE_BH
	int bh_sched=0;
#endif // BIM_USE_BH
	
	unsigned char iir;
	while(!((iir=inb(IIR(bim->iobase))) & IP0))
	{
		switch(iir & IID)
		{
			case IRLS:	
			{
				// receiver line status

#ifdef BIM_INTR_DEBUG
				printk("IRLS ");
#endif // BIM_INTR_DEBUG

				inb(LSR(bim->iobase));
				break;
			}
			case ICTO:
			{
				// character timeout
#ifdef BIM_INTR_DEBUG
				printk("ICTO ");
#endif // BIM_INTR_DEBUG
				//fall through
			}
			case IRDA:
			{	
				// received data available

#ifdef BIM_INTR_DEBUG
				printk("IRDA ");
#endif // BIM_INTR_DEBUG

				while(inb(LSR(bim->iobase)) & DR)
				{
					data=inb(RBR(bim->iobase));

#ifdef BIM_USE_BH
					if(bim->hdlc_rx.bh_buflen >= BH_RX_BUFLEN)
					{
						// bh buffer overflow
#ifdef BIM_DEBUG	
						printk("BIM: RX bottom-half buffer overflow\n");
#endif	// BIM_DEBUG
						break;
					}

					bim->hdlc_rx.bh_buffer[bim->hdlc_rx.bh_wr]=data;
					bim->hdlc_rx.bh_wr=(bim->hdlc_rx.bh_wr+1)%BH_RX_BUFLEN;
					bim->hdlc_rx.bh_buflen++;
					bh_sched|=BH_SCHED_RX;

#else // BIM_USE_BH
					hdlc_rx_byte(bim,data);
//					rx_chars_to_flip(bim);	// send bytes from rx buffer to flip buffer
#endif // BIM_USE_BH

				}
				break;
			}
			case ITHRE:
			{

#ifdef BIM_INTR_DEBUG
				printk("ITHRE ");
#endif // BIM_INTR_DEBUG

				// check if transmitter active
				if(!bim->hdlc_tx.ptt) break;
				if(!(inb(LSR(bim->iobase)) & THRE)) break; // not THRE

				int fifosize=(bim->uart==UART_16550A)? 16:1;
				for(int i=0;i < fifosize;i++)
				{
					data=preamble_tx(bim);

					if(!data)
					{
						// no more preamble
#ifdef BIM_USE_BH
						// check end of transmition
						if(!bim->hdlc_tx.bh_buflen)
						{
							if(!bim->hdlc_tx.holdptt) ptt(bim,0);
							else
							{
								// disable TX interrupt while waiting bottom half
								unsigned char ier=inb(IER(bim->iobase));
								outb(ier & ~ETBEI,IER(bim->iobase));
							}
							break;
						}

						data=bim->hdlc_tx.bh_buffer[bim->hdlc_tx.bh_rd];
						bim->hdlc_tx.bh_rd=(bim->hdlc_tx.bh_rd+1)%BH_TX_BUFLEN;
						bim->hdlc_tx.bh_buflen--;

#else // BIM_USE_BH
						if(!bim->hdlc_tx.holdptt)
						{
							ptt(bim,0);
							break;
						}
						data=hdlc_tx_byte(bim);
						if(!bim->hdlc_tx.holdptt)
						{
							if(!i) ptt(bim,0);
							break;
						}
#endif // BIM_USE_BH			

					}
					outb(data,THR(bim->iobase));

#ifdef BIM_USE_BH
					bh_sched|=BH_SCHED_TX;
#endif

				}
				break;
			}
			case IMS:
			{

#ifdef BIM_INTR_DEBUG
				printk("IMS ");
#endif // BIM_INTR_DEBUG

				inb(MSR(bim->iobase));
				break;
			}
		}
	}

#ifdef BIM_USE_BH
	if(bh_sched & BH_SCHED_RX)
	{
		queue_task_irq_off(&bim->tq_receiver,&tq_bim);
		mark_bh(BIM_BH);
	}
	if(bh_sched & BH_SCHED_TX)
	{
		queue_task_irq_off(&bim->tq_transmitter,&tq_bim);
		mark_bh(BIM_BH);
	}
#endif // BIM_USE_BH
}

// Bottom half (soft interrupt)
#ifdef BIM_USE_BH
static void bh_receiver(void *priv)
{
	struct bim_state *bim=(struct bim_state *)priv;

	if(!bim || bim->magic != BIM_MAGIC) return;
	while(bim->hdlc_rx.bh_buflen)
	{
		unsigned long flags;
		save_flags(flags);
		cli();

		unsigned char data=bim->hdlc_rx.bh_buffer[bim->hdlc_rx.bh_rd];
		bim->hdlc_rx.bh_rd=(bim->hdlc_rx.bh_rd+1)%BH_RX_BUFLEN;
		bim->hdlc_rx.bh_buflen--;

		restore_flags(flags);

		hdlc_rx_byte(bim,data);
	}
//	rx_chars_to_flip(bim);	// send bytes from rx buffer to flip buffer
}


static void bh_transmitter(void *priv)
{
	struct bim_state *bim=(struct bim_state *)priv;

	if(!bim || bim->magic != BIM_MAGIC) return;

	while(bim->hdlc_tx.bh_buflen < BH_TX_BUFLEN)
	{
		unsigned char data=hdlc_tx_byte(bim);
		if(!bim->hdlc_tx.holdptt) break;

		unsigned long flags;
		save_flags(flags);
		cli();

		bim->hdlc_tx.bh_buffer[bim->hdlc_tx.bh_wr]=data;
		bim->hdlc_tx.bh_wr=(bim->hdlc_tx.bh_wr+1)%BH_TX_BUFLEN;
		bim->hdlc_tx.bh_buflen++;

		restore_flags(flags);
	}
	// enable TX interrupt
	unsigned char ier=inb(IER(bim->iobase));
	outb(ier | ETBEI,IER(bim->iobase));
}


static void bim_bottom_half(void)
{
	run_task_queue(&tq_bim);
}
#endif // BIM_USE_BH




/*******************************************************************************

		 Hardware initialization & cleanup functions

*******************************************************************************/



static const char *uart_str[]=
{
	"unknown",
	"8250",
	"16450",
	"16550",
	"16550A"
};



inline static void bim_set_divisor(struct bim_state *bim,unsigned short divisor)
{
	outb(DLAB | WLS0,LCR(bim->iobase));	// DLAB = 1
	outb(divisor & 0xff,DLL(bim->iobase));
	outb(divisor >> 8,DLM(bim->iobase));
	outb(WL8,LCR(bim->iobase));	// word length=8, 1 stop bit, no parity
}



static enum UART bim_check_uart(unsigned int iobase)
{
	unsigned char b1;
	unsigned char b2;
	unsigned char b3;
	enum UART u;
	enum UART uart_tab[]=
	{
		UART_16450,
		UART_UNKNOWN,
		UART_16550,
		UART_16550A
	};

	b1=inb(MCR(iobase));
	outb(b1 | LOOP,MCR(iobase));	// loopback mode
	b2=inb(MSR(iobase));
	outb(LOOP | OUT2 | RTS,MCR(iobase));
	b3=inb(MSR(iobase)) & (DCD | RI | DSR | CTS);
	outb(b1,MCR(iobase));		// restore old values
	outb(b2,MSR(iobase));
	if(b3 != (DCD | CTS)) return UART_UNKNOWN;
	inb(RBR(iobase));
	inb(RBR(iobase));
	outb(FIFOE,FCR(iobase));		// enable FIFOs
	u=uart_tab[(inb(IIR(iobase)) & IFIFO) >> 6];
	if(u==UART_16450)
	{
		outb(0x5a,SCR(iobase));
		b1=inb(SCR(iobase));
		outb(0xa5,SCR(iobase));
		b2=inb(SCR(iobase));
		if((b1 != 0x5a) || (b2 != 0xa5)) u=UART_8250;
	}
	return u;
}

static void bim_set_baud(struct bim_state *bim)
{
	if(!bim->iobase) return;

	unsigned int baud_base=115200;
	unsigned int baud=0;

	if(bim->opened)
	{
		unsigned int cflag=bim->tty->termios->c_cflag;

		// can change only baud rate
		bim->tty->termios->c_cflag=CS8 | CREAD | CLOCAL | (cflag & CBAUD);

		unsigned int i=cflag & CBAUD;

		if(i & CBAUDEX)
		{
			i&=~CBAUDEX;
			// can't support rates above 115200 bps
			if(i < 1 || i > 2) bim->tty->termios->c_cflag &=~CBAUDEX;
			else i+=15;
		}

		baud=baud_table[i];
	}
	else baud=bim->dflt_baud;

	unsigned short quot=0;
	if(baud == 134) quot=2*baud_base/269;
	else if(baud) quot=baud_base/baud;
	unsigned int mcr=inb(MCR(bim->iobase));
	if(quot) outb(mcr | DTR | OUT1 | OUT2,MCR(bim->iobase));
	else
	{
		outb(0,MCR(bim->iobase));
		return;
	}

	bim_set_divisor(bim,quot);

#ifdef BIM_DEBUG
	printk(KERN_INFO "BIM: baud rate: %d\n",baud);
#endif
}


static int bim_allocate_resources(unsigned int iobase,unsigned int irq)
{
	enum UART u;

	if(!iobase || iobase > 0xfff || irq < 2 || irq > 15) return -ENXIO;
	if(check_region(iobase,BIM_EXTENT)) return -EACCES;
	if((u=bim_check_uart(iobase)) == UART_UNKNOWN) return -EIO;
	request_region(iobase,BIM_EXTENT,"bim");
	outb(0,FCR(iobase));		// disable FIFOs
	outb(DTR | OUT1 | OUT2,MCR(iobase));
	printk(KERN_INFO "BIM: iobase 0x%x irq %u uart %s\n",iobase,irq,uart_str[u]);
	return u;
}
	

static void bim_deallocate_resources(struct bim_state *bim) 
{
	if(!bim || bim->type == BIM_INVALID) return;

	// remove arbitration timer
	if(bim->modem.flags & ARB_TIMER_ON) del_timer(&bim->arb_timer);
	bim->modem.flags&=~ARB_TIMER_ON;

	outb(0,FCR(bim->iobase));		// disable FIFOs
	// disable interrupts
	outb(0,IER(bim->iobase));

	// reset DTR
	outb(0,MCR(bim->iobase));

	// this should prevent kernel: Trying to free IRQx messages

	if(bim->opened > 0) free_irq(bim->irq,bim);
	release_region(bim->iobase,BIM_EXTENT);
	bim->type=BIM_INVALID;
	printk(KERN_INFO "BIM: release iobase 0x%x irq %u\n",bim->iobase,bim->irq);
	bim->iobase=bim->irq;
	bim->uart=UART_UNKNOWN;
}




static int bim_on_open(struct bim_state *bim) 
{

	if(!bim || bim->type == BIM_INVALID) return -ENXIO;

	// set baud rate
	bim_set_baud(bim);

	// set DTR
	outb(DTR | OUT1 | OUT2,MCR(bim->iobase));
	outb(0,FCR(bim->iobase));		// disable FIFOs
	outb(0,IER(bim->iobase));
	if(request_irq(bim->irq,bim_interrupt,SA_INTERRUPT,"bim",bim)) return -EBUSY;

	// enable and reset FIFOs
	if(bim->uart==UART_16550A) outb(FIFOE |	RFIFOR | TFIFOR | RTL | RTM,FCR(bim->iobase));

	// clear any pending interrupt
	inb(LSR(bim->iobase));
	inb(RBR(bim->iobase));
	inb(IIR(bim->iobase));
	inb(MSR(bim->iobase));

	// enable received data available interrupt
	outb(ERB,IER(bim->iobase));  

	// start arbitration
	if(!(bim->modem.flags & ARB_TIMER_ON))
	{
		unsigned long ms=bim->ch_params.slottime*10;
		bim->modem.flags|=ARB_TIMER_ON;
		bim->arb_timer.expires=jiffies+(ms*HZ+999)/1000+1;
		add_timer(&bim->arb_timer);
	}

	return 0;
}

static void bim_on_close(struct bim_state *bim) 
{
	if(!bim || bim->type == BIM_INVALID) return;

	// remove arbitration timer
	if(bim->modem.flags & ARB_TIMER_ON) del_timer(&bim->arb_timer);
	bim->modem.flags&=~ARB_TIMER_ON;

	outb(0,FCR(bim->iobase));		// disable FIFOs

	// disable interrupts
	outb(0,IER(bim->iobase));

	// reset DTR
	outb(0,MCR(bim->iobase));
	free_irq(bim->irq,bim);	
}

static inline void bim_dealloc_hw(struct bim_state *bim) 
{
	if(!bim || bim->magic != BIM_MAGIC || bim->type == BIM_INVALID) return;
	bim_deallocate_resources(bim);
}


static int bim_set_hardware(struct bim_state *bim,int modem_type,unsigned int iobase,unsigned int irq,int baud)
{
	if(!bim) return -EINVAL;

	int i;
	switch(modem_type)
	{
		case BIM_MODEM:
		case BIM_CABLE:
		{
			i=bim_allocate_resources(iobase,irq);
			if(i < 0) return i;
			bim->uart=i;
			break;
		}
		case BIM_INVALID:
		{
			iobase=irq=0;
			break;
		}
		default: return -ENXIO;
	}

	bim_dealloc_hw(bim);
	bim->iobase=iobase;
	bim->irq=irq;
	bim->type=modem_type;
	bim->dflt_baud=baud;
	i=0;
	if(bim->opened > 0 && bim->type != BIM_INVALID) i=bim_on_open(bim);
	return i;
}


/*******************************************************************************

			 TTY Driver functions

*******************************************************************************/


static inline int bim_paranoia_check(struct bim_state *bim,const char *routine)
{
	if(!bim || bim->magic != BIM_MAGIC)
	{
		printk(KERN_ERR "BIM: bad magic number for bim struct in routine %s\n",routine);
		return 1;
	}
	return 0;
}



/*
 * 	This routine is called when a particular tty device is opened.
 * 	This routine is mandatory; if this routine is not filled in,
 * 	the attempted open will fail with ENODEV.
 */
int bim_open(struct tty_struct *tty,struct file *filp)
{
	if(!tty) return -ENODEV;

	int line=MINOR(tty->device)-tty->driver.minor_start;
	if(line < 0 || line >= NR_PORTS) return -ENODEV;
	struct bim_state *bim=bim_state+line;

	if(bim->opened > 0)
	{
		bim->opened++;
		MOD_INC_USE_COUNT;
		return 0;
	}


	// allocate the buffer space

	if(bim->rx_buf.buffer) kfree_s(bim->rx_buf.buffer,bim->rx_buf.buflen);
	if(bim->tx_buf.buffer) kfree_s(bim->tx_buf.buffer,bim->tx_buf.buflen);
	bim->rx_buf.buflen=BUFLEN_RX;
	bim->tx_buf.buflen=BUFLEN_TX;
	bim->rx_buf.rd=bim->rx_buf.wr=0;
	bim->tx_buf.rd=bim->tx_buf.wr=0;
	bim->rx_buf.buffer=(unsigned char *)kmalloc(bim->rx_buf.buflen,GFP_KERNEL);
	bim->tx_buf.buffer=(unsigned char *)kmalloc(bim->tx_buf.buflen,GFP_KERNEL);
	if(!bim->rx_buf.buffer || !bim->tx_buf.buffer)
	{
		if(bim->rx_buf.buffer) kfree_s(bim->rx_buf.buffer,bim->rx_buf.buflen);
		if(bim->tx_buf.buffer) kfree_s(bim->tx_buf.buffer,bim->tx_buf.buflen);
		bim->rx_buf.buffer=bim->tx_buf.buffer=NULL;
		bim->rx_buf.buflen=bim->tx_buf.buflen=0;
		return -ENOMEM;
	}

	int i=0;				// enable to open a unitialized port



	switch(bim->type)
	{
		case BIM_MODEM:
		case BIM_CABLE:
		{
			i=bim_on_open(bim);	// initialize serial port
			break;
		}
		case BIM_INVALID:
		{
			i=0;
			break;
		}
		default: return -ENODEV;
	}

	if(i) return i;

	bim->opened++;
	MOD_INC_USE_COUNT;

	tty->driver_data=bim;
	bim->tty=tty;
	return 0;   
}


/*
 * 	This routine is called when a particular tty device is closed.
 */
static void bim_close(struct tty_struct *tty,struct file * filp)
{
	if(!tty) return;

	struct bim_state *bim;
	if(bim_paranoia_check(bim=(struct bim_state *)tty->driver_data,"close")) return;

	MOD_DEC_USE_COUNT;
	bim->opened--;
	if(bim->opened <= 0)
	{
		bim_on_close(bim);	// clear serial port
		tty->driver_data=NULL;
		bim->tty=NULL;
		bim->opened=0;
		
		// free the buffers 

		bim->rx_buf.rd=bim->rx_buf.wr=0;
		bim->tx_buf.rd=bim->tx_buf.wr=0;
		if(bim->rx_buf.buffer) kfree_s(bim->rx_buf.buffer,bim->rx_buf.buflen);
		if(bim->tx_buf.buffer) kfree_s(bim->tx_buf.buffer,bim->tx_buf.buflen);
		bim->rx_buf.buffer=bim->tx_buf.buffer=NULL;
		bim->rx_buf.buflen=bim->tx_buf.buflen=0;
	}
}


/*
 * 	This routine is called by the kernel to write a single
 * 	character to the tty device.  If the kernel uses this routine,
 * 	it must call the flush_chars() routine (if defined) when it is
 * 	done stuffing characters into the driver.  If there is no room
 * 	in the queue, the character is ignored.
 */
static void bim_put_char(struct tty_struct *tty,unsigned char ch)
{
	if(!tty) return;
	struct bim_state *bim;
	if(bim_paranoia_check(bim=(struct bim_state *)tty->driver_data,"put_char")) return;
		
	if(ch==KISS_FEND)
	{
		bim_put_fend(bim);
		bim->kiss_decode.wr=0;
		bim->kiss_decode.escaped=0;
		bim->kiss_decode.dec_state=1;
		return;
	}
	if(!bim->kiss_decode.dec_state) return;
	if(ch==KISS_FESC)
	{
		bim->kiss_decode.escaped=1;
		return;
	}
	if(bim->kiss_decode.wr >= sizeof(bim->kiss_decode.pkt_buf))
	{
		bim->kiss_decode.wr=0;
		bim->kiss_decode.dec_state=0;
		return;
	}
	if(bim->kiss_decode.escaped)
	{
		if(ch==KISS_TFEND) bim->kiss_decode.pkt_buf[bim->kiss_decode.wr++]=KISS_FEND;
		else if(ch==KISS_TFESC) bim->kiss_decode.pkt_buf[bim->kiss_decode.wr++]=KISS_FESC;
		else
		{
			bim->kiss_decode.wr=0;
			bim->kiss_decode.dec_state=0;
		}
		bim->kiss_decode.escaped=0;
		return;
	}
	bim->kiss_decode.pkt_buf[bim->kiss_decode.wr++]=ch;
}


/*
 * 	This routine is called by the kernel to write a series of
 * 	characters to the tty device.  The characters may come from
 * 	user space or kernel space.  This routine will return the
 *	number of characters actually accepted for writing.  This
 *	routine is mandatory.
 */
static int bim_write(struct tty_struct *tty,int from_user,const unsigned char *buf,int count)
{
	if(!tty || !buf || count <= 0) return count;

	struct bim_state *bim;
	if(bim_paranoia_check(bim=(struct bim_state *)tty->driver_data,"write")) return count; 

	const unsigned char *bp;		
	int c;
	if(from_user) for(c=count,bp=buf;c > 0;c--,bp++) bim_put_char(tty,get_user(bp));
	else for(c=count,bp=buf;c > 0;c--,bp++) bim_put_char(tty,*bp);

	if((tty->flags & (1 << TTY_DO_WRITE_WAKEUP)) &&	tty->ldisc.write_wakeup) (tty->ldisc.write_wakeup)(tty);

	wake_up_interruptible(&tty->write_wait);
	return count;
}



/*
 * 	This routine returns the numbers of characters the tty driver
 * 	will accept for queuing to be written.  This number is subject
 * 	to change as output buffers get emptied, or if the output flow
 *	control is acted.
 */
static int bim_write_room(struct tty_struct *tty)
{
	if(!tty) return 0;

	struct bim_state *bim;
	if(bim_paranoia_check(bim=(struct bim_state *)tty->driver_data,"write_room")) return 0;
		
	int free=bim->tx_buf.rd-bim->tx_buf.wr;
	if(free <= 0)
	{
		free=bim->tx_buf.buflen-bim->tx_buf.wr;
		if((unsigned int)free < bim->tx_buf.rd) free=bim->tx_buf.rd;	/* we may fold */
	}

	return free / 2; /* a rather pessimistic estimate */
}


/*
 * 	This routine returns the numbers of characters the tty driver
 * 	has in queue to be written to tty device.  This number is subject
 * 	to change as output buffers get emptied.
 */
static int bim_chars_in_buffer(struct tty_struct *tty)
{
	if(!tty) return 0;

	struct bim_state *bim;
	if(bim_paranoia_check(bim=(struct bim_state *)tty->driver_data,"chars_in_buffer")) return 0;

	int cnt=bim->tx_buf.wr-bim->tx_buf.rd;
	if(cnt < 0) cnt+=bim->tx_buf.buflen;
		
	return cnt;
}


/*
 * 	This routine is called by the kernel after it has written a
 * 	series of characters to the tty device using write().  
 */
static void bim_flush_buffer(struct tty_struct *tty)
{
	if(!tty) return;

	struct bim_state *bim;
	if(bim_paranoia_check(bim=(struct bim_state *)tty->driver_data,"flush_buffer")) return;

	wake_up_interruptible(&tty->write_wait);
	if((tty->flags & (1 << TTY_DO_WRITE_WAKEUP)) &&	tty->ldisc.write_wakeup) (tty->ldisc.write_wakeup)(tty);
}


/* 
 * 	This routine allows the tty driver to implement
 *	device-specific ioctl's.  If the ioctl number passed in cmd
 * 	is not recognized by the driver, it should return ENOIOCTLCMD.
 */
static int bim_ioctl(struct tty_struct *tty,struct file * file,unsigned int cmd,unsigned long arg)
{
	if (!tty) return -EINVAL;

	struct bim_state *bim;
	if(bim_paranoia_check(bim=(struct bim_state *)tty->driver_data,"ioctl")) return -EINVAL;
		
	switch(cmd)
	{
		default: return -ENOIOCTLCMD;

		case TIOCMGET:
		{
			int i=verify_area(VERIFY_WRITE,(void *)arg,sizeof(int));
			if(i) return i;
			i=(bim->modem.dcd ? TIOCM_CAR : 0) | (bim->hdlc_tx.ptt ? TIOCM_RTS : 0);
			put_user(i,(int *)arg);
			return 0;
		}
		
		case BIMCTL_GETDCD:
		{
			int i=verify_area(VERIFY_WRITE,(void *)arg,sizeof(unsigned char));
			if(!i) put_user(bim->modem.dcd,(unsigned char *) arg);
			return i;
		}
		
		case BIMCTL_GETPTT:
		{
			int i=verify_area(VERIFY_WRITE,(void *)arg,sizeof(unsigned char));
			if(!i) put_user(bim->hdlc_tx.ptt,(unsigned char *)arg);
			return i;
		}
		
		case BIMCTL_PARAM_TXDELAY:
		{
			if(arg > 255) return -EINVAL;
			bim->ch_params.tx_delay=arg;
			return 0;
		}
		
		case BIMCTL_PARAM_PPERSIST:
		{
			if(arg > 255) return -EINVAL;
			bim->ch_params.ppersist=arg;
			return 0;
		}
		
		case BIMCTL_PARAM_SLOTTIME:
		{
			if (arg > 255) return -EINVAL;
			bim->ch_params.slottime=arg;
			return 0;
		}
		
		case BIMCTL_PARAM_TXTAIL:
		{
			if(arg > 255) return -EINVAL;
			bim->ch_params.tx_tail=arg;
			return 0;
		}
		
		case BIMCTL_PARAM_FULLDUP:
		{
			bim->ch_params.fulldup=arg ? 1 : 0;
			return 0;
		}

		case BIMCTL_GETPARAMS:
		{
			struct bim_params par;

			int i = verify_area(VERIFY_WRITE,(void *) arg,sizeof(par));
			if(i) return i;
			par.iobase=bim->iobase;
			par.irq=bim->irq;
			par.type=bim->type;
			par.baud=bim->dflt_baud;
			par.tx_delay=bim->ch_params.tx_delay;
			par.tx_tail=bim->ch_params.tx_tail;
			par.slottime=bim->ch_params.slottime;
			par.ppersist=bim->ch_params.ppersist;
			par.fulldup=bim->ch_params.fulldup;
			memcpy_tofs((void *)arg,&par,sizeof(par));
			return 0;
		}

		case BIMCTL_SETPARAMS:
		{
			if(!suser()) return -EPERM;

			struct bim_params par;
			int i=verify_area(VERIFY_READ,(void *) arg,sizeof(par));
			if(i) return i;
			memcpy_fromfs(&par,(void *)arg,sizeof(par));
			switch(par.type)
			{
				case BIM_MODEM:
				{
					printk(KERN_INFO "BIM: changing hardware type: modem iobase 0x%x irq %u baudrate %d\n",par.iobase,par.irq,par.baud);
					break;
				}
				case BIM_CABLE:
				{
					printk(KERN_INFO "BIM: changing hardware type: cable iobase 0x%x irq %u baudrate %d\n",par.iobase,par.irq,par.baud);
					break;
				}
				case BIM_INVALID:
				{
					printk(KERN_INFO "BIM: changing hardware type: off\n");
					break;
				}
				default: printk(KERN_INFO "BIM: changing hardware type: unknown\n");
			}

			i=bim_set_hardware(bim,par.type,par.iobase,par.irq,par.baud); 
			if(i) return i;

			// set termios baud flag to default baud
			int baud=0;
			while((baud_table[baud] > 0 || baud == 0) && (baud_table[baud] != bim->dflt_baud)) baud++;
			if(baud > 15) baud |= CBAUDEX;


			tty->termios->c_cflag&=~CBAUD;
			tty->termios->c_cflag|=baud;
			bim->ch_params.tx_delay=par.tx_delay;
			bim->ch_params.tx_tail=par.tx_tail;
			bim->ch_params.slottime=par.slottime;
			bim->ch_params.ppersist=par.ppersist;
			bim->ch_params.fulldup=par.fulldup;


			return 0;
		}

		case BIMCTL_GETSTAT:
		{
			int i=verify_area(VERIFY_WRITE,(void *) arg,sizeof(struct bim_statistics));
			if(i) return i;
			memcpy_tofs((void *)arg,&bim->stat,sizeof(struct bim_statistics));
			return 0;
		}
		

#ifdef BIM_DEBUG
		case BIMCTL_GETBITS:
		{
			if(bim->bitbuf_hdlc.rd == bim->bitbuf_hdlc.wr) return -EAGAIN;
			int i=verify_area(VERIFY_WRITE,(void *) arg,sizeof(unsigned char));
			if(!i)
			{
				put_user(bim->bitbuf_hdlc.buffer[bim->bitbuf_hdlc.rd],(unsigned char *) arg);
				bim->bitbuf_hdlc.rd=(bim->bitbuf_hdlc.rd+1) % sizeof(bim->bitbuf_hdlc.buffer);
			}
			return i;
		}
		
		case BIMCTL_DEBUG1:
		{
			int i=verify_area(VERIFY_WRITE,(void *) arg,sizeof(unsigned long));
			if(!i) put_user((bim->rx_buf.wr-bim->rx_buf.rd) % bim->rx_buf.buflen,(unsigned long *)arg);
			return i;
		}
		
		case BIMCTL_DEBUG2:
		{
			int i=verify_area(VERIFY_WRITE,(void *) arg,sizeof(unsigned long));
			if(!i) put_user(bim->debug_vals.last_intcnt,(unsigned long *)arg);
			return i;
		}
		
#endif // BIM_DEBUG
	}
}



/*
 * 	This routine allows the tty driver to be notified when
 * 	device's termios settings have changed.  Note that a
 * 	well-designed tty driver should be prepared to accept the case
 * 	where old == NULL, and try to do something rational.
 */
static void bim_set_termios(struct tty_struct *tty,struct termios *old)
{
	if(!tty) return;

	struct bim_state *bim;
	if(bim_paranoia_check(bim=(struct bim_state *)tty->driver_data,"set_termios")) return;

	if(tty->termios->c_cflag == old->c_cflag) return;

	bim_set_baud(bim);
}



/*******************************************************************************

		 Driver initialization & cleanup functions

*******************************************************************************/


static void init_channel(struct bim_state *bim,int port)
{

	if(!bim) return;

	// HDLC initialization

	bim->hdlc_rx.rx_state=0;

	bim->hdlc_tx.tx_state=bim->hdlc_tx.numflags=0;
	bim->hdlc_tx.bitstream=0;
	bim->hdlc_tx.ptt=0;
	bim->hdlc_tx.holdptt=0;

	bim->hdlc_rx.bh_rd=bim->hdlc_rx.bh_wr=bim->hdlc_rx.bh_buflen=0;
	bim->hdlc_tx.bh_rd=bim->hdlc_tx.bh_wr=bim->hdlc_tx.bh_buflen=0;

#ifdef BIM_DEBUG
	bim->bitbuf_hdlc.rd=bim->bitbuf_hdlc.wr=0;
#endif // BIM_DEBUG

	// KISS initialization

	bim->kiss_decode.dec_state=bim->kiss_decode.escaped=bim->kiss_decode.wr=0;

	bim->ch_params=bim_ports[port].dflt_ch_params;

	// Arbitration initialization
	bim->modem.dcd=0;
	init_timer(&bim->arb_timer);
	bim->arb_timer.function=arb_sched;
	bim->arb_timer.data=(unsigned long) bim;
	bim->modem.flags=0;

	// Bottom-half initialization

#ifdef BIM_USE_BH
	bim->tq_receiver.next=bim->tq_transmitter.next=NULL;
	bim->tq_receiver.sync=bim->tq_transmitter.sync=0;
	bim->tq_receiver.data=bim->tq_transmitter.data=bim;
	bim->tq_receiver.routine=bh_receiver;
	bim->tq_transmitter.routine=bh_transmitter;
#endif // BIM_USE_BH

}

static void init_datastructs(void)
{
	for(int i=0;i < NR_PORTS;i++)
	{
		struct bim_state *bim=bim_state+i;

		bim->magic=BIM_MAGIC;
		bim->iobase=bim->irq=bim->opened=0;
		bim->type=BIM_INVALID;
		bim->tty=NULL;

		bim->rx_buf.rd=bim->rx_buf.wr = 0;
		bim->rx_buf.buflen=0;
		bim->rx_buf.buffer=NULL;

		bim->tx_buf.rd=bim->tx_buf.wr = 0;
		bim->tx_buf.buflen=0;
		bim->tx_buf.buffer=NULL;

		memset(&bim->stat,0,sizeof(bim->stat));

		init_channel(bim,i);
	}
}

static int bim_init(void)
{

	// initialize the data structures

	init_datastructs();

	// initialize bottom half handler

#ifdef BIM_USE_BH
	init_bh(BIM_BH,bim_bottom_half);
#endif // BIM_USE_BH

	// register the driver as tty driver

	memset(&bim_driver,0, sizeof(struct tty_driver));
	bim_driver.magic=TTY_DRIVER_MAGIC;
	bim_driver.name="bim";
	bim_driver.major=major;
	bim_driver.minor_start=0;
	bim_driver.num=NR_PORTS;
	bim_driver.type=TTY_DRIVER_TYPE_BIM;
	bim_driver.subtype=BIM_TYPE_NORMAL;
	bim_driver.init_termios.c_iflag=0;
	bim_driver.init_termios.c_oflag=0;

	int baud=0;
	while((baud_table[baud] > 0 || baud == 0) && (baud_table[baud] != DFLT_BAUD)) baud++;
	if(baud > 15) baud |= CBAUDEX;
	
	bim_driver.init_termios.c_cflag=CS8 | CREAD | CLOCAL | baud;
	bim_driver.init_termios.c_lflag=0;
	bim_driver.flags=TTY_DRIVER_REAL_RAW;
	bim_driver.refcount=&bim_refcount;
	bim_driver.table=bim_table;
	bim_driver.termios=bim_termios;
	bim_driver.termios_locked=bim_termios_locked;

	// the functions

	bim_driver.open=bim_open;
	bim_driver.close=bim_close;
	bim_driver.write= bim_write;
	bim_driver.put_char=bim_put_char;
	bim_driver.flush_chars=NULL;
	bim_driver.write_room=bim_write_room;
	bim_driver.chars_in_buffer=bim_chars_in_buffer;
	bim_driver.flush_buffer=bim_flush_buffer;
	bim_driver.ioctl=bim_ioctl;

	// cannot throttle the transmitter on this layer

	bim_driver.throttle=NULL;
	bim_driver.unthrottle=NULL;

	// no special actions on termio changes
	bim_driver.set_termios=bim_set_termios;

	// no XON/XOFF and no hangup on the radio port
	bim_driver.stop=NULL;
	bim_driver.start=NULL;
	bim_driver.hangup=NULL;
	bim_driver.set_ldisc=NULL;

	if(tty_register_driver(&bim_driver))
	{
		printk(KERN_WARNING "BIM: tty_register_driver failed\n");
		return -EIO;
	}

	for(int i=0;i < NR_PORTS && bim_ports[i].iobase;i++)
	{
		int j=bim_set_hardware(bim_state+i,bim_ports[i].type,bim_ports[i].iobase,bim_ports[i].irq,bim_ports[i].baud); 
		if(j < 0)
		{
			const char *s;
			switch(-j)
			{
				case ENXIO:
				{
					s="invalid iobase and/or irq";
					break;
				}
				case EACCES:
				{
					s="io region already used";
					break;
				}
				case EIO:
				{
					s="no uart port at iobase";
					break;
				}
				case EBUSY:
				{
					s="interface already in use";
					break;
				}
				case EINVAL:
				{
					s="internal error";
					break;
				}
				default:
				{
					s="unknown error";
					break;
				}
			}
			printk(KERN_WARNING "BIM: iobase 0x%x irq %u: (%i) %s\n",bim_ports[i].iobase, bim_ports[i].irq,j,s);
		}
	}

	return 0;
}


extern "C" int init_module(void)
{
	printk("BIM: 1.0 Copyright (c) 1997 Walter Fetter Lages <w.fetter@ieee.org>.\n");

	printk(KERN_INFO "BIM: init_module called\n");

	bim_ports[0].iobase=base0;
	bim_ports[0].irq=irq0;
	bim_ports[0].type=type0;
	bim_ports[0].baud=baud0;
	bim_ports[0].dflt_ch_params.tx_delay=tx_delay0;
	bim_ports[0].dflt_ch_params.tx_tail=tx_tail0;
	bim_ports[0].dflt_ch_params.slottime=slottime0;
	bim_ports[0].dflt_ch_params.ppersist=ppersist0;
	bim_ports[0].dflt_ch_params.fulldup=fulldup0;


	bim_ports[1].iobase=base1;
	bim_ports[1].irq=irq1;
	bim_ports[1].type=type1;
	bim_ports[1].baud=baud1;
	bim_ports[1].dflt_ch_params.tx_delay=tx_delay1;
	bim_ports[1].dflt_ch_params.tx_tail=tx_tail1;
	bim_ports[1].dflt_ch_params.slottime=slottime1;
	bim_ports[1].dflt_ch_params.ppersist=ppersist1;
	bim_ports[1].dflt_ch_params.fulldup=fulldup1;

	bim_ports[2].iobase=base2;
	bim_ports[2].irq=irq2;
	bim_ports[2].type=type2;
	bim_ports[2].baud=baud2;
	bim_ports[2].dflt_ch_params.tx_delay=tx_delay2;
	bim_ports[2].dflt_ch_params.tx_tail=tx_tail2;
	bim_ports[2].dflt_ch_params.slottime=slottime2;
	bim_ports[2].dflt_ch_params.ppersist=ppersist2;
	bim_ports[2].dflt_ch_params.fulldup=fulldup2;

	bim_ports[3].iobase=base3;
	bim_ports[3].irq=irq3;
	bim_ports[3].type=type3;
	bim_ports[3].baud=baud3;
	bim_ports[3].dflt_ch_params.tx_delay=tx_delay3;
	bim_ports[3].dflt_ch_params.tx_tail=tx_tail3;
	bim_ports[3].dflt_ch_params.slottime=slottime3;
	bim_ports[3].dflt_ch_params.ppersist=ppersist3;
	bim_ports[3].dflt_ch_params.fulldup=fulldup3;

	return bim_init();
}


extern "C" void cleanup_module(void)
{
	printk(KERN_INFO "BIM: cleanup_module called\n");

	disable_bh(BIM_BH);
	if(tty_unregister_driver(&bim_driver)) printk(KERN_WARNING "BIM: failed to unregister tty driver\n");
	for(int i=0;i < NR_PORTS;i++)
	{
		struct bim_state *bim=bim_state+i;

		if(bim->magic != BIM_MAGIC) printk(KERN_ERR "BIM: invalid magic in cleanup_module\n");
		else
		{
			bim_dealloc_hw(bim);

			// free the buffers 

			bim->rx_buf.rd=bim->rx_buf.wr=0;
			bim->tx_buf.rd=bim->tx_buf.wr=0;
			if(bim->rx_buf.buffer) kfree_s(bim->rx_buf.buffer,bim->rx_buf.buflen);
			if(bim->tx_buf.buffer) kfree_s(bim->tx_buf.buffer,bim->tx_buf.buflen);
			bim->rx_buf.buffer=bim->tx_buf.buffer=NULL;
			bim->rx_buf.buflen=bim->tx_buf.buflen=0;
		}
	}
}

