/*******************************************************************************

 		BIM-XXX-RS232 Radio Transceiver driver.
	Copyright (c) 1997 Walter Fetter Lages <w.fetter@ieee.org>
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Please note that the GPL allows you to use the driver, NOT the radio.
    In order to use the radio, you need a license from the communications
    authority of your country.

    History:
    0.0  05.31.97  Start development based on Baycom ser12 and 
		   par96 radio modem driver version 0.3, Copyright (C) 1996
		   Thomas Sailer (sailer@ife.ee.ethz.ch), distributed with
		   Linux kernel version 2.0.28.

    0.1  06.20.97  First working version.
    1.0  07.06.97  First released version.
	
*******************************************************************************/

#include <stdlib.h>
#include <sys/time.h>
#include <fcntl.h>
#include <iomanip.h>
#include <iostream.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>

#include <bim.h>

enum SET_KISS
{
	SET_KEYUP=	0x01,
	SET_TAIL=	0x02,
	SET_SLOT=	0x04,
	SET_P=		0x08,
	SET_MODE=	0x10,
	SET_DISPLAY=	0x20
};

static int fd_bim;

static int baud_table[]=
{
	0,50,75,110,134,150,200,300,600,1200,1800,2400,4800,
	9600,19200,38400,57600,115200,0
};

void display_packet(unsigned char *bp,unsigned int len)
{
	unsigned char v1=1;
	unsigned char cmd=0;
	unsigned char i;
	unsigned char j;

	if(!bp || !len) return;
	if(len < 8) return;
	if(bp[1] & 1)
	{
		// FlexNet Header Compression
		v1=0;
		cmd=(bp[1] & 2) != 0;
		cout << "fm ? to ";
		i=(bp[2] >> 2) & 0x3f;
		if(i) cout << (char) (i+0x20);
		i=((bp[2] << 4) | ((bp[3] >> 4) & 0xf)) & 0x3f;
		if(i) cout << (char) (i+0x20);
		i=((bp[3] << 2) | ((bp[4] >> 6) & 3)) & 0x3f;
		if(i) cout << (char) (i+0x20);
		i=bp[4] & 0x3f;
		if(i) cout << (char) (i+0x20);
		i=(bp[5] >> 2) & 0x3f;
		if(i) cout << (char) (i+0x20);
		i=((bp[5] << 4) | ((bp[6] >> 4) & 0xf)) & 0x3f;
		if(i) cout << (char) (i+0x20);
		cout << "-" << (bp[6] & 0xf) << " QSO Nr " << ((bp[0] << 6) | (bp[1] >> 2));
		bp+=7;
		len-=7;
	} 
	else
	{
		 // normal header
		if(len < 15) return;
		if((bp[6] & 0x80) != (bp[13] & 0x80))
		{
			v1=0;
			cmd=(bp[6] & 0x80);
		}
		cout << "fm ";
		for(i=7;i < 13;i++) if((bp[i] & 0xfe) != 0x40) cout << (char) (bp[i] >> 1);
		cout << "-" << ((bp[13] >> 1) & 0xf) << " to ";
		for(i=0;i < 6;i++) if ((bp[i] & 0xfe) != 0x40) cout << (char) (bp[i] >> 1);
		cout << "-" << ((bp[6] >> 1) & 0xf);
		bp+=14;
		len-=14;
		if((!(bp[-1] & 1)) && (len >= 7)) cout << " via ";
		while((!(bp[-1] & 1)) && (len >= 7))
		{
			for(i=0;i < 6;i++) if ((bp[i] & 0xfe) != 0x40) cout << (char) (bp[i] >> 1);
			cout << "-" << ((bp[6] >> 1) & 0xf);
			bp+=7;
			len-=7;
			if((!(bp[-1] & 1)) && (len >= 7)) cout << ",";
		}
	}
	if(!len) return;
	i=*bp++;
	len--;
	j=v1 ? ((i & 0x10) ? '!' : ' ') : ((i & 0x10) ? (cmd ? '+' : '-') : (cmd ? '^' : 'v'));
	if(!(i & 1))
	{
		// Info frame
		cout << " I" << ((i >> 5) & 7) << ((i >> 1) & 7) << j;
	}
	else if(i & 2)
	{
		// U frame

		switch(i & (~0x10))
		{
			case 0x03:
			{
				cout << " UI" << j;
				break;
			}
			case 0x2f:
			{
				cout << " SABM" << j;
				break;
			}
			case 0x43:
			{
				cout << " DISC" << j;
				break;
			}
			case 0x0f:
			{
				cout << " DM" << j;
				break;
			}
			case 0x63:
			{
				cout << " UA" << j;
				break;
			}
			case 0x87:
			{
				cout << " FRMR" << j;
				break;	
			}
			default:
			{
				cout << " unknown U (0x" << setfill('0') << setw(2) << hex << (int) (i & (~0x10)) << ")" << j;
				break;
			}
		}
	}
	else
	{
		// supervisory
		switch(i & 0xf)
		{
			case 0x1:
			{
				cout << " RR" << ((i >> 5) & 7) << j;
				break;
			}
			case 0x5:
			{
				cout << " RNR" << ((i >> 5) & 7) << j;
				break;
			}
			case 0x9:
			{
				cout << " REJ" << ((i >> 5) & 7) << j;
				break;
			}
			default:
			{
				cout << " unknown S (0x" << setfill('0') << setw(2) << hex << (int) (i & 0xf) << ")" << ((i >> 5) & 7) << j;
				break;
			}
		}
	}
	if(!len)
	{
		cout << "\n";
		return;
	}
	cout << " pid=0x" << setfill('0') << setw(2) << hex << (int) (*bp++) << "\n";
	len--;
	j=0;
	while(len)
	{
		i=*bp++;
		if((i >= 32) && (i < 128)) cout << i;
		else if(i == 13)
		{
			if(j) cout << "\n";
			j=0;
		} 
		else cout << ".";
		if(i >= 32) j = 1;
		len--;
	}
	if(j) cout << "\n";
}


static void kiss_input(void)
{
	unsigned char buffer[1024];	
	unsigned char *bp=buffer;
	int len=read(fd_bim,buffer,sizeof(buffer));
	if(len < 0)
	{
		if(errno != EAGAIN)
		{
			cerr << "Error: " << strerror(errno) << " (" << errno << ") reading from KISS port \n";
			exit(-3);
		}
		return;
	}
	static unsigned char inbuf[BIM_MAXFLEN+1];
	static int inptr=0;
	static unsigned char escaped=0;
	static unsigned char valid=0;
	for(;len > 0; len--,bp++)
	{
		unsigned char cur=*bp;

#ifdef DISPLAY_RXBYTES
		cout << "0x" << setfill('0') << setw(2)  << hex << (int) cur << " ";
#endif // DISPLAY_RXBYTES

		if(cur == KISS_FEND)
		{
			if((inptr > 0) && valid)
			{
				if((inbuf[0] & 0xf0) == 0)
				{	
					// only one channel supported
					switch(inbuf[0] & 0xf)
					{
						case KISS_CMD_DATA:
						{

#ifdef DISPLAY_RXBYTES	
							cout << "\n";
#endif

							if(inptr > 7) display_packet(inbuf+1,inptr-1);
							break;
						}
					}
				}
			}
			valid=1;
			inptr=0;
			escaped=0;
		}
		else if(cur == KISS_FESC)
		{
			escaped=1;
		} 
		else
		{
			if(inptr >= (int) sizeof(inbuf)) valid=0;
			if(valid)
			{
				if(escaped)
				{
					if(cur == KISS_TFEND) inbuf[inptr++]=KISS_FEND;
					else if(cur == KISS_TFESC) inbuf[inptr++]=KISS_FESC;
					else valid=0;
					escaped=0;
				}
				else inbuf[inptr++]=cur;
			}
		}
	}
}


void print_bits(unsigned int ctl)
{
	for(;;)
	{
		unsigned char bits;
		int ret=ioctl(fd_bim,ctl,&bits);
		if(ret < 0) return;
		for(int i=0;i < 8;i++)
		{
			cout << (bits & 0x80 ? '1':'0');
			bits<<=1;
		}
	}
}

void print_bytes(unsigned int ctl)
{
	for(;;)
	{
		unsigned char byte;
		int ret=ioctl(fd_bim,ctl,&byte);
		if(ret < 0) return;

		cout << "0x" <<  setfill('0') << setw(2) << hex << (int) byte << " ";
	}
}


static void display_params(const struct bim_params *par) 
{
	if(!par || par->type == BIM_INVALID)
	{
		cout << "off\n";
		return;
	}
	switch(par->type)
	{
		case BIM_MODEM:
		{
			cout << "modem: ";
			break;
		}
		case BIM_CABLE:
		{
			cout << "cable: ";
			break;
		}
		default:
		{
			cout << "invalid parameter structure!\n";
			return;
		}
	}

	cout << "iobase=0x" << hex << par->iobase << dec;
	cout << " irq=" << par->irq;
	cout << " baudrate=" << par->baud << "\n";
}


static void do_set_params(int argc,char **argv) 
{
	struct bim_params par1;
	int ret;

	ret=ioctl(fd_bim,BIMCTL_GETPARAMS,&par1);
	if(ret < 0)
	{
		if(ret < 0)
		{
			cerr << "setbim: Error: " << strerror(errno) << " (" << errno << ")\n";
			cerr << "\tioctl(BIMCTL_GETPARAMS) failed\n";
		}
		exit(-1);
	}
	struct bim_params par2=par1;
	par2.irq=par2.iobase=0;
	par2.type=BIM_INVALID;
	char set=1;
	if(argc < 1) set=0;
	else
	{
		if(!strcasecmp(argv[0], "off") || !strcasecmp(argv[0], "none"))	par2.type=BIM_INVALID;
		else if(!strcasecmp(argv[0],"modem")) par2.type=BIM_MODEM;
		else if(!strcasecmp(argv[0],"cable")) par2.type=BIM_CABLE;
		else
		{
			cerr << "Invalid modem type\n";
			set=0;
		}
	}
	if(set && par2.type != BIM_INVALID)
	{
		if(argc < 3) set=0;
		else
		{
			par2.iobase=strtoul(argv[1],NULL,0);
			par2.irq=strtoul(argv[2],NULL,0);
			par2.baud=(argc > 3)? strtoul(argv[3],NULL,0):par1.baud;
			int i=0;
			while((baud_table[i] > 0 || i == 0) && (baud_table[i] != par2.baud)) i++;
			if(par2.baud > 0 && !i)
			{
				set=0;
				cerr << "Invalid baud rate\n";
			}
		}
	}
	if(!set)
	{
		cout << "setbim: current UART parameters: ";
		display_params(&par1);
	}
	else
	{
		ret=ioctl(fd_bim,BIMCTL_SETPARAMS,&par2);
		if(ret < 0)
		{
			cerr << "setbim: Error: " << strerror(errno) << " (" << errno << ")\n";
			cerr << "\tcannot set parameters, trying to restore old parameters\n";

			ret=ioctl(fd_bim,BIMCTL_SETPARAMS,&par1);
			if(ret < 0)
			{
				cerr << "setbim: Error: " << strerror(errno) << " (" << errno << ")\n";
				cerr << "\tcannot restore old parameters\n";
			}
		}
		else
		{
			cout << "setbim: new UART parameters:     ";
			display_params(&par2);
		}
		exit(-1);
	}
}

static void display_kiss(const struct bim_params *par) 
{
	cout << "\n\ttransmitter keyup delay in 10 ms units=" << par->tx_delay;
	cout << "\n\ttransmition tail time in 10 ms units=" << par->tx_tail;
	cout << "\n\tslot interval in 10 ms units=" << par->slottime;
	cout << "\n\tp persistence scaled to the range 0..255=" << par->ppersist;
	cout << "\n\tmode=" << ((par->fulldup)? "full":"half") << " duplex\n";
}

static void do_set_kiss(int setkiss,struct bim_params *par) 
{
	struct bim_params par1;
	int ret;

	ret=ioctl(fd_bim,BIMCTL_GETPARAMS,&par1);
	if(ret < 0)
	{
		if(ret < 0)
		{
			cerr << "setbim: Error: " << strerror(errno) << " (" << errno << ")\n";
			cerr << "\tioctl(BIMCTL_GETPARAMS) failed\n";
		}
		exit(-1);
	}
	struct bim_params par2=par1;

	char set=0;	
	if(setkiss & SET_KEYUP)
	{
		par2.tx_delay=par->tx_delay;
		set=1;
	}
	if(setkiss & SET_TAIL)
	{
		par2.tx_tail=par->tx_tail;
		set=1;
	}
	if(setkiss & SET_SLOT)
	{
		par2.slottime=par->slottime;
		set=1;
	}
	if(setkiss & SET_P)
	{
		par2.ppersist=par->ppersist;
		set=1;
	}
	if(setkiss & SET_MODE)
	{
		par2.fulldup=par->fulldup;
		set=1;
	}
		
	if(!set)
	{
		cout << "setbim: current KISS parameters: ";
		display_kiss(&par1);
	}
	else
	{
		ret=ioctl(fd_bim,BIMCTL_SETPARAMS,&par2);
		if(ret < 0)
		{
			cerr << "setbim: Error: " << strerror(errno) << " (" << errno << ")\n";
			cerr << "\tcannot set parameters, trying to restore old parameters\n";

			ret=ioctl(fd_bim,BIMCTL_SETPARAMS,&par1);
			if(ret < 0)
			{
				cerr << "setbim: Error: " << strerror(errno) << " (" << errno << ")\n";
				cerr << "\tcannot restore old parameters\n";
			}
			exit(-1);
		}
		else
		{
			cout << "setbim: new KISS parameters:     ";
			display_kiss(&par2);
		}
	}
}


static const char *usage_str= 
"device [-b] [-B] [-d] [-k delay] [-t tail] [-s slot] [-p persist] [-m mode] [-K] [-h] [-u [type iobase irq [default_baud]]]\n"
"  -b: trace demodulated bits\n"
"  -B: trace demodulated bytes\n"
"  -d: trace dcd and ptt status on stdout\n"
"  -k: set transmitter keyup delay in 10 ms units\n"
"  -t: set transmition tail time in 10 ms units\n"
"  -s: set slot interval in 10 ms units\n"
"  -p: set p persistence scaled to the range 0..255\n"
"  -m: set mode\n"
"      mode: half | full\n"
"  -K: display KISS parameters\n"
"  -u: set or display UART parameters\n"
"      type: none | off | modem | cable\n"
"  -h: this help\n\n";

int main(int argc,char *argv[])
{
	int ret;

	char getsetparams=0;

	if((argc < 2) || (argv[1][0]=='-'))
	{
		cerr << "usage: setbim " << usage_str;
		return -1;
	}

	char *name_bim=new char[strlen(argv[1])];
	strcpy(name_bim,argv[1]);

	argc--;
	argv++;

	unsigned char trace_stat=0;
	int setkiss=0;
	struct bim_params kisspar;
	while((ret=getopt(argc,argv,"bBdk:t:s:p:m:Khu")) != -1)
	{
		switch(ret)
		{
			case 'b':
			{
				trace_stat=2;
				break;
			}
			case 'B':
			{
				trace_stat=3;
				break;
			}
			case 'd':
			{
				trace_stat=1;
				break;
			}
			case 'k':
			{
				setkiss|=SET_KEYUP;
				if((kisspar.tx_delay=strtoul(optarg,NULL,0)) > 255)
				{
					cerr << "Invalid keyup delay.\n";
					return -1;
				}
				break;
			}
			case 't':
			{
				setkiss|=SET_TAIL;
				if((kisspar.tx_tail=strtoul(optarg,NULL,0)) > 255)
				{
					cerr << "Invalid transmition tail.\n";
					return -1;
				}
				break;
			}
			case 's':
			{
				setkiss|=SET_SLOT;
				if((kisspar.slottime=strtoul(optarg,NULL,0)) > 255)
				{
					cerr << "Invalid slot interval.\n";
					return -1;
				}
				break;
			}
			case 'p':
			{
				setkiss|=SET_P;
				if((kisspar.ppersist=strtoul(optarg,NULL,0)) > 255)
				{
					cerr << "Invalid p persistence.\n";
					return -1;
				}
				break;
			}
			case 'm':
			{
				setkiss|=SET_MODE;
				if(!strcasecmp(optarg,"half")) kisspar.fulldup=0;
				else if(!strcasecmp(optarg,"full")) kisspar.fulldup=1;
				else
				{
					cerr << "Invalid mode.\n";
					return -1;
				}
				break;
			}
			case 'K':
			{
				setkiss|=SET_DISPLAY;
				break;
			}
			case 'u':
			{
				getsetparams=1;
				break;
			}
			default:
			{
				cerr << "usage: setbim " << usage_str;
				return -1;
			}
		}
	}


	if((fd_bim=open(name_bim,O_RDWR)) < 0)
	{
		cerr << "setbim: Error: " << strerror(errno) << " (" << errno << "), cannot open " << name_bim << "\n";
		delete[] name_bim;
		return -1;
	}
	delete[] name_bim;

	ret=0;
	if(setkiss)
	{
		do_set_kiss(setkiss,&kisspar);
		ret=1;
	}
	if(getsetparams)
	{
		do_set_params(argc-optind,argv+optind);
		ret=1;
	}
	if(ret) return 0;

	for(;;)
	{
		fd_set fds_read;
		fd_set fds_write;
		FD_ZERO(&fds_read);
		FD_ZERO(&fds_write);
		FD_SET(fd_bim,&fds_read);

		struct timeval tm;
		tm.tv_usec=500000L;
		tm.tv_sec=0L;
		ret=select(fd_bim+1,&fds_read,&fds_write,NULL,&tm);
		if(ret < 0)
		{
			cerr << "setbim: Error: " << strerror(errno) << " (" << errno << ") in select\n";
			return -2;
		}
		if(FD_ISSET(fd_bim,&fds_read)) kiss_input();
		switch(trace_stat)
		{
			case 1:
			{
				int stat;
				ret=ioctl(fd_bim,TIOCMGET,&stat);
				if(ret < 0)
				{	
					cerr << "setbim: Warning: " << strerror(errno) << " (" << errno << ")\n";
					cerr << "\tioctl(TIOCMGET) failed\n";
				}

				struct bim_statistics statistics;
				ret=ioctl(fd_bim,BIMCTL_GETSTAT,&statistics);
				if(ret < 0)
				{
					cerr << "setbim: Warning: " << strerror(errno) << " (" << errno << ")\n";
					cerr << "\tioctl(BIMCTL_GETSTAT) failed\n";
				}

#ifdef BIM_DEBUG
				unsigned long debug1;
				ret=ioctl(fd_bim,BIMCTL_DEBUG1,&debug1);
				if(ret < 0)
				{
					cerr << "setbim: Warning: " << strerror(errno) << " (" << errno << ")\n";
					cerr << "\tioctl(BIMCTL_DEBUG1) failed\n";
				}

				unsigned long debug2;
				ret=ioctl(fd_bim,BIMCTL_DEBUG2,&debug2);
				if(ret < 0)
				{
					cerr << "setbim: Warning: " << strerror(errno) << " (" << errno << ")\n";
					cerr << "\tioctl(BIMCTL_DEBUG2) failed\n";
				}

#endif // BIM_DEBUG
				cout << ((stat & TIOCM_CAR)? 'D' : '-') << ((stat & TIOCM_RTS)? 'P' : '-') << " ";
#ifdef BIM_DEBUG
				cout << "dbg1: " << debug1 << "  dbg2: " << debug2;
#endif // BIM_DEBUG
				cout << "  rx: " << dec << statistics.rx_packets;
				cout << "  tx: " << dec << statistics.tx_packets;
				cout << "  ptt: " << dec << statistics.ptt_keyed;
				cout << "  rxerr: " << dec << statistics.rx_bufferoverrun;
				cout << "  txerr: " << dec << statistics.tx_bufferoverrun << "\n";
				break;
			}
			case 2:
			{

#ifdef BIM_DEBUG
				print_bits(BIMCTL_GETBITS);
#endif // BIM_DEBUG
				cout << "\n";
				break;
			}
			case 3:
			{

#ifdef BIM_DEBUG
				print_bytes(BIMCTL_GETBITS);
#endif // BIM_DEBUG
				cout << "\n";
				break;
			}
		}
	}
}

