/*****************************************************************************
 *
 *	dxlmodem.c  --  oe5dxl 9k6 FSK radio modem driver.
 *
 *	Copyright (C) 1997  Hansi Reiser  (dl9rdz)
 *
 *      Derived from DOS-driver L2PCX       (C) 1996 Chris Rabler, OE5DXL
 *      Based on the linux driver baycom.c  (C) 1996 Thomas Sailer, HB9JNX
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Please note that the GPL allows you to use the driver, NOT the radio.
 *  In order to use the radio, you need a license from the communications
 *  authority of your country.
 *
 *
 *  Command line options:
 *  There arn't any, use sethdlc to configure driver.
 *
 *
 *  mode     driver mode string. (well.... not yet specified, but probably
 *           baudrate in L2PCX-format (+1=scramble, +2= DTR=+12V for AFSK)
 *  iobase   base address of the port; common values are 0x3f8, 0x2f8, 
 *           0x3e8, 0x2e8
 *  irq      interrupt line of the port; common values are 4 and 3
 *
 *
 *  Remember to disable the standard COM driver for those ports which you
 *  are going to use with dxlmodem!
 *  (e.g.by  setserial /dev/cuaX uart none)
 *
 *  History:
 *   0.1  20.03.97  started
 *        21.03.97  first usable version...
 *   0.1b 19.04.97  just improved readability and really only minor changes
 *   0.2  18.01.98  support for multiple devices and hdlcdrv-ioctl-support
 *                  for setting io-port parameters
 */

/*****************************************************************************/

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/types.h>
#include <linux/fcntl.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/in.h>
#include <linux/string.h>
#include <asm/system.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <linux/delay.h>
#include <linux/errno.h>
#include <linux/netdevice.h>
#include <linux/hdlcdrv.h>

#include <linux/version.h>

#if LINUX_VERSION_CODE >= 0x20100
#include <asm/uaccess.h>
#else
#include <asm/segment.h>
#include <linux/mm.h>

#undef put_user
#undef get_user

#define put_user(x,ptr) ({ __put_user((unsigned long)(x),(ptr),sizeof(*(ptr))); 0; })
#define get_user(x,ptr) ({ x = ((__typeof__(*(ptr)))__get_user((ptr),sizeof(*(ptr)))); 0; })

extern inline int copy_from_user(void *to, const void *from, unsigned long n)
{
        int i = verify_area(VERIFY_READ, from, n);
        if (i)
                return i;
        memcpy_fromfs(to, from, n);
        return 0;
}

extern inline int copy_to_user(void *to, const void *from, unsigned long n)
{
        int i = verify_area(VERIFY_WRITE, to, n);
        if (i)
                return i;
        memcpy_tofs(to, from, n);
        return 0;
}
#endif


#include "dxlmodem.h"

/* --------------------------------------------------------------------- */

/* (This is a comment from Thomas Sailers driver baycom.c. I didn't
 * change much on the module interface, so this comment should still be
 * appropriate. Myself, I have tested this driver only with 2.0.29,
 * so I can't tell anything about 2.1.x-kernels):
 *
 * currently this module is supposed to support both module styles, i.e.
 * the old one present up to about 2.1.9, and the new one functioning
 * starting with 2.1.21. The reason is I have a kit allowing to compile
 * this module also under 2.0.x which was requested by several people.
 * This will go in 2.2
 */
#ifdef MODULE_PARM
#define NEW_MODULE_CODE 1
#else /* MODULE_PARM */
#define NEW_MODULE_CODE 0
#define MODULE_PARM(x,y)
#define MODULE_PARM_DESC(x,y)
#define MODULE_AUTHOR(x)
#define MODULE_DESCRIPTION(x)
#endif /* MODULE_PARM */

/* --------------------------------------------------------------------- */

static const char dxl_drvname[] = "dxlmodem";
static const char dxl_drvinfo[] = KERN_INFO "dxlmodem: (C) 1997-1998 Hansi Reiser, dl9rdz\n"
KERN_INFO "dxlmodem: version 0.2 compiled " __TIME__ " " __DATE__ "\n";

/* --------------------------------------------------------------------- */

/* It should be possible to support as many ports as there are
 * physical COM-ports. Just configuration probably wont work yet, so
 * it currently would be useless....
 */
#define NR_PORTS 4

static struct device dxlmodem_device[NR_PORTS];

static struct {
	char *mode;
	int iobase, irq;
} dxlmodem_ports[NR_PORTS] = { { NULL, 0, 0 }, };

/* --------------------------------------------------------------------- */
#if LINUX_VERSION_CODE >= 0x20123
#include <linux/init.h>
#else
#define __init
#define __initdata
#define __initfunc(x) x
#endif

/* --------------------------------------------------------------------- */

#define RBR(iobase) (iobase+0)
#define THR(iobase) (iobase+0)
#define IER(iobase) (iobase+1)
#define IIR(iobase) (iobase+2)
#define FCR(iobase) (iobase+2)
#define LCR(iobase) (iobase+3)
#define MCR(iobase) (iobase+4)
#define LSR(iobase) (iobase+5)
#define MSR(iobase) (iobase+6)
#define SCR(iobase) (iobase+7)
#define DLL(iobase) (iobase+0)
#define DLM(iobase) (iobase+1)

#define DXL_EXTENT 8              /* occupied io range */

#define SLOWER 0x0C
#define FASTER 0x04
#define NORMAL 0x08
#define SEND   0x01
#define MAIN_DCD  50
#define PLLL 3

#define ARB_DIVIDER  48          /* 48 / 4800Hz = 10ms */

/* state-machine-tables */
struct state {
	int dpll;
	unsigned char sq, dd, lastd;
};

#include "dxl-tables.h"


/* ---------------------------------------------------------------------- */
/*
 * Information that need to be kept for each port.
 */

struct dxlmodem_state {
	/* hdlcdrc_state must be at the top (located in dev->priv) */
	struct hdlcdrv_state hdrv;

	/* anything thereafter I can use as I like to... */
	unsigned int options;
	struct modem_state {
		short arb_divider;

		unsigned int txdata,rxdata,descram,scram;
		unsigned char squelch,outbyte,ptton,pttoff,pttgo;
		unsigned char scrambl,send;
		int dpll,only_with_dcd,lostbyte,losttest;

		struct state state;

		union {
		     int lng;
		     unsigned char b[4];
		} txscramb;

		unsigned char flags;
	} modem;

};

/* --------------------------------------------------------------------- */

/*
 * ======================== DXLMODEM routines ==========================
 */

static inline void switch_to_receive( struct device *dev, struct dxlmodem_state *dxl )
{
	outb( dxl->modem.pttoff, MCR(dev->base_addr) );
	/* FIXME: Bugreport von DH5RAE: Manchmal wird bei ihm nicht von
	 * TX->RX richtig umgeschaltet. Evtl brauchen wir zwischen diesen
	 * beiden outb ein kleines delay, damit der Port wirklich auch Zeit
	 * hat das pttoff auszugeben? 
	 * oder kurz was ins LCR schreiben? (z.B. 0 oder NORMAL)
	 * Bei nchster Gelegenheit mal erforschen - im moment hab ich
	 * leider keine Test-Mglichkeit hier :-(
	 */
	dxl->modem.send=0;
	dxl->modem.dpll=-64; /* thats from l2pcx. don't know why it is done. */
	outb( dxl->modem.pttgo, MCR(dev->base_addr) );
}
static inline void switch_to_send( struct device *dev, struct dxlmodem_state *dxl )
{
	outb( dxl->modem.ptton, MCR(dev->base_addr) );
	outb( SEND, LCR(dev->base_addr) );
	dxl->modem.send=1;
	outb( dxl->modem.pttgo, MCR(dev->base_addr) );
}

static void dxl_set_hdlcbaud( struct device *dev, struct dxlmodem_state *dxl, int hdlcbaud )
{
	int d;

	dxl->modem.scrambl = hdlcbaud & 1;
	dxl->modem.ptton = 11; 
	dxl->modem.pttoff = 8; 
	dxl->modem.pttgo = 9;
	hdlcbaud = (hdlcbaud&0xFFFC)>>1;
	d = 14400 / hdlcbaud;
	cli();
	outb(0x80, LCR(dev->base_addr));
	outb( (d&0xFF), DLL(dev->base_addr));
	outb( (d>>8)&0xFF, DLM(dev->base_addr) );
	outb(NORMAL, LCR(dev->base_addr));
	sti();
}

/* --------------------------------------------------------------------- */

#define DXL96_DESCRAM_TAPSH1 17
#define DXL96_DESCRAM_TAPSH2 12
#define DXL96_DESCRAM_TAPSH3 0

#define DXL96_SCRAM_TAP1 0x20000 /* X^17 */
#define DXL96_SCRAM_TAPN 0x00021 /* X^0+X^5 */


static inline void dxl_tx( struct device *dev, struct dxlmodem_state *dxl )
{
	int i, h;
	outb(dxl->modem.outbyte, THR(dev->base_addr));
	if(dxl->modem.txdata<=1) {
		dxl->modem.txdata = hdlcdrv_getbits(&dxl->hdrv) | 0x10000;
	}
	/* could be speed up by doing the scrambling byte-wise instead of
	 * bit-wise, like oe5dxl does it... */
	if(dxl->modem.scrambl) {
#if 0
		/* das hier funktioniert schon mal... */
		/* HB9JNX-scrambling, stolen from par96-treiber */
		/* (IMHO this one is easier to understand) */
		for(i=0; i<2; i++) {
			dxl->modem.scram = (dxl->modem.scram<<1) | 
			     (dxl->modem.scram&1);
			if( !(dxl->modem.txdata & 1) ) 
			     dxl->modem.scram ^= 1;
			if (dxl->modem.scram & (DXL96_SCRAM_TAP1 << 1))
			     dxl->modem.scram ^= (DXL96_SCRAM_TAPN << 1);
			if (dxl->modem.scram & (DXL96_SCRAM_TAP1 << 2))
			     dxl->modem.outbyte = (dxl->modem.outbyte<<1) | 1;
			else
			     dxl->modem.outbyte<<=1;
			dxl->modem.txdata >>=1;
		}
		dxl->modem.outbyte = 38 + !(dxl->modem.outbyte&0x02) 
		     + 16*!(dxl->modem.outbyte&0x01);
#else
		/* ... and now I finally got this to work as well :).. */
		/* OE5DXL-scrambling, stolen from L2PCX-driver */
		/* a bit tricky to understand, but migth be saving some
		 * CPU cycles... */
		dxl->modem.outbyte = (dxl->modem.outbyte<<2) 
		     | ((dxl->modem.txdata&1)<<1) 
		     | ((dxl->modem.txdata&2)>>1);
		dxl->modem.txdata>>=2;
		i = dxl->modem.txscramb.b[1] >> 2;
		dxl->modem.txscramb.lng <<= 1;
		h = dxl->modem.txscramb.b[2];
		dxl->modem.txscramb.lng <<= 1;
		dxl->modem.txscramb.b[0] += (h^dxl->modem.outbyte^i) & 0x03;
		dxl->modem.outbyte = TX2[ (h&0x03)|(dxl->modem.outbyte&0x40) ];
#endif
	}
	else {
		/* FIXME: this DOES NOT WORK JET!. */
		/* (thats the part for operating without scrambler.
		 * I thing NRZI coding is missing as I moved it out of
		 * the TX2-table? not sure. just be told that it does
		 * not work. most people (including myself) don't need
		 * unscrambled FSK anyway...
		 */
		dxl->modem.outbyte=dxl->modem.txdata&3; 
		dxl->modem.txdata>>=2;
		dxl->modem.outbyte=TX2[dxl->modem.outbyte&0x03];
	}
}


static inline void dxl_rx( struct device *dev, struct dxlmodem_state *dxl )
{
	int i;
	unsigned int descx;
      
	/* Something seems to get wrong here. in setmode I'm setting
	 * modem.losttest to 1, so it be 1 in this line. But is isn't.
	 * Either not set properly or cleared afterwards to 0. I don't
	 * understand it currently. Have to track this down somewhen later.
	 * For now, I just put a 1|| to get the lostint-counting done...
	 */
	if( (inb(LSR(dev->base_addr))&0x20)==0 ) return;

	if( 1 || dxl->modem.losttest ) {
		if( inb(LSR(dev->base_addr))&0x40 ) dxl->modem.lostbyte++;
	}
	outb(0, THR(dev->base_addr));
	outb(NORMAL, LCR(dev->base_addr));
	dxl->modem.state = 
	     s_table[ inb(RBR(dev->base_addr)) + dxl->modem.state.lastd ];
	
	/* dpll */
	dxl->modem.dpll += dxl->modem.state.dpll;
	if(dxl->modem.dpll < -PLLL) { 
	     outb(SLOWER, LCR(dev->base_addr)); dxl->modem.dpll=0; 
	}
	else if(dxl->modem.dpll > PLLL) { 
	     outb(FASTER, LCR(dev->base_addr)); dxl->modem.dpll=0; 
	}
	
	/* squelch */
	dxl->modem.squelch=
	     squelch_table[dxl->modem.squelch+dxl->modem.state.sq];

	/* Should I stop decoding when DCD goes off? L2PCX is working
	 * this way. But you'll have to relay 100% on the DCD! */
	if( dxl->modem.only_with_dcd )
	  if(dxl->modem.squelch<MAIN_DCD) return; 

	/*
	 * do receiver; differential decode and descramble on the fly
	 *
	 * for simplicity, just taken from hb9jnx's par96-driver
	 */
	for(i = 0; i < 2; i++) {
		dxl->modem.descram = dxl->modem.descram<<1;
		if( !(dxl->modem.state.dd&2) ) dxl->modem.descram |= 1;
		dxl->modem.state.dd<<=1;

		descx = dxl->modem.descram;
		/* now the diff decoded data is inverted in descram */
		descx ^= ((descx >> DXL96_DESCRAM_TAPSH1) ^
			  (descx >> DXL96_DESCRAM_TAPSH2));
		dxl->modem.rxdata >>= 1;
		if (!(descx & 1))
			dxl->modem.rxdata |= 0x10000;
	}
	if( dxl->modem.rxdata&1 ) {
		hdlcdrv_putbits(&dxl->hdrv, dxl->modem.rxdata>>1);
		dxl->modem.rxdata=0x10000;
	}
}



static void dxl_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	int doptt;
        struct device *dev = (struct device *)dev_id;
	struct dxlmodem_state *dxl = (struct dxlmodem_state *)dev->priv;

	if (!dev || !dxl || dxl->hdrv.magic != HDLCDRV_MAGIC)
		return;

	if( dxl->modem.send ) {
	     dxl_tx( dev, dxl );
	}
	else {
	     dxl_rx( dev, dxl );
	     if( --dxl->modem.arb_divider <= 0 ) {
		  dxl->modem.arb_divider = ARB_DIVIDER;
		  sti();
		  hdlcdrv_setdcd(&dxl->hdrv, (dxl->modem.squelch>=MAIN_DCD));
		  hdlcdrv_arbitrate(dev, &dxl->hdrv); /* alle 10ms! */
	     }
	}
	doptt = hdlcdrv_ptt(&dxl->hdrv);
	if( dxl->modem.send && !doptt)
	     switch_to_receive(dev, dxl);
	else if (!dxl->modem.send && doptt)
	     switch_to_send(dev, dxl);

	sti();
	hdlcdrv_transmitter(dev, &dxl->hdrv);
	hdlcdrv_receiver(dev, &dxl->hdrv);
}

/* --------------------------------------------------------------------- */

enum uart { c_uart_unknown, c_uart_8250,
	c_uart_16450, c_uart_16550, c_uart_16550A};
static const char *uart_str[] =
	{ "unknown", "8250", "16450", "16550", "16550A" };

static enum uart dxlmodem_check_uart(unsigned int iobase)
{
	unsigned char b1,b2,b3;
	enum uart u;
	enum uart uart_tab[] =
		{ c_uart_16450, c_uart_unknown, c_uart_16550, c_uart_16550A };

	b1 = inb(MCR(iobase));
	outb(b1 | 0x10, MCR(iobase));	/* loopback mode */
	b2 = inb(MSR(iobase));
	outb(0x1a, MCR(iobase));
	b3 = inb(MSR(iobase)) & 0xf0;
	outb(b1, MCR(iobase));			/* restore old values */
	outb(b2, MSR(iobase));
	if (b3 != 0x90)
		return c_uart_unknown;
	inb(RBR(iobase));
	inb(RBR(iobase));
	outb(0x01, FCR(iobase));		/* enable FIFOs */
	u = uart_tab[(inb(IIR(iobase)) >> 6) & 3];
	if (u == c_uart_16450) {
		outb(0x5a, SCR(iobase));
		b1 = inb(SCR(iobase));
		outb(0xa5, SCR(iobase));
		b2 = inb(SCR(iobase));
		if ((b1 != 0x5a) || (b2 != 0xa5))
			u = c_uart_8250;
	}
	return u;
}



/* --------------------------------------------------------------------- */

static int dxlmodem_open(struct device *dev)
{
	struct dxlmodem_state *dxl = (struct dxlmodem_state *)dev->priv;
	enum uart u;

	if (!dev || !dxl)
		return -ENXIO;
	if (!dev->base_addr || dev->base_addr > 0x1000-DXL_EXTENT ||
	    dev->irq < 2 || dev->irq > 15)
		return -ENXIO;
	if (check_region(dev->base_addr, DXL_EXTENT)) {
	        printk(KERN_ERR "%s: access denied on io region at 0x%x, already in use\n", dxl_drvname, (int)dev->base_addr);
		return -EACCES;
	}
	memset(&dxl->modem, 0, sizeof(dxl->modem));
	dxl->hdrv.par.bitrate = 9600;
	if ((u = dxlmodem_check_uart(dev->base_addr)) == c_uart_unknown)
		return -EIO;
	outb(0, FCR(dev->base_addr));  /* disable FIFOs */
	outb(0x0d, MCR(dev->base_addr));
	outb(0x0d, MCR(dev->base_addr));
	outb(0, IER(dev->base_addr));
	if (request_irq(dev->irq, dxl_interrupt, SA_INTERRUPT,
			"dxlmodem", dev)) {
	        printk(KERN_ERR "%s: request of irq %d failed\n", dxl_drvname,
		       dev->irq);
		return -EBUSY;
	}
	request_region(dev->base_addr, DXL_EXTENT, "dxlmodem");
	/*
	 * enable transmitter empty interrupt
	 */
#if 0
	/* the next two lines are include in the msdos driver l2pcx 
	 * are the usefull/necessary? Methinks not...
	 */
	outb(0, LCR(dev->base_addr));
	switch_to_receive();
#endif
	outb(2, IER(dev->base_addr));

	dxl_set_hdlcbaud(dev, dxl, 9601);

	printk(KERN_INFO "%s: dxlmodem at iobase 0x%lx irq %u options "
	       "0x%x uart %s\n", dxl_drvname, dev->base_addr, dev->irq,
	       dxl->options, uart_str[u]);
	MOD_INC_USE_COUNT;
	return 0;
}

/* --------------------------------------------------------------------- */

static int dxlmodem_close(struct device *dev)
{
	struct dxlmodem_state *dxl = (struct dxlmodem_state *)dev->priv;

	if (!dev || !dxl)
		return -EINVAL;
	/*
	 * disable interrupts
	 */
	outb(0, IER(dev->base_addr));
	outb(1, MCR(dev->base_addr));
	free_irq(dev->irq, dev);
	release_region(dev->base_addr, DXL_EXTENT);
	printk(KERN_INFO "%s: close dxlmodem at iobase 0x%lx irq %u\n",
	       dxl_drvname, dev->base_addr, dev->irq);
	MOD_DEC_USE_COUNT;
	return 0;
}

/* --------------------------------------------------------------------- */

/*
 * ===================== hdlcdrv driver interface =========================
 */

/* --------------------------------------------------------------------- */

static int dxlmodem_ioctl(struct device *dev, struct ifreq *ifr,
			struct hdlcdrv_ioctl *hi, int cmd);

/* --------------------------------------------------------------------- */

static struct hdlcdrv_ops dxlmodem_ops = {
	dxl_drvname,
	dxl_drvinfo,
	dxlmodem_open,
	dxlmodem_close,
	dxlmodem_ioctl
};

/* --------------------------------------------------------------------- */

/* --------------------------------------------------------------------- */

static struct hdlcdrv_ops dummy_ops = {
	dxl_drvname,
	dxl_drvinfo,
	NULL,
	NULL,
	dxlmodem_ioctl
};

/* --------------------------------------------------------------------- */

static int dxlmodem_setmode(struct dxlmodem_state *dxl, char *modestr)
{
	struct hdlcdrv_ops *newops = NULL;
	unsigned long flags;

	if (!strncmp(modestr, "off", 3))
		newops = &dummy_ops;
	else if (!strncmp(modestr, "96", 2))
		newops = &dxlmodem_ops;
	else
		return -EINVAL;
	save_flags(flags);
	cli();
	dxl->hdrv.ops = newops;
	dxl->options = !!strchr(modestr, '*');
	restore_flags(flags);

	dxl->modem.arb_divider = ARB_DIVIDER;
	dxl->modem.losttest = 1;
	dxl->modem.only_with_dcd = 0;
	return 0;
}

/* --------------------------------------------------------------------- */

static int dxlmodem_ioctl(struct device *dev, struct ifreq *ifr,
			  struct hdlcdrv_ioctl *hi, int cmd)
{
	struct dxlmodem_state *dxl;
	struct dxlmodem_ioctl bi;
	int cmd2;

	if (!dev || !dev->priv ||
	    ((struct dxlmodem_state *)dev->priv)->hdrv.magic != HDLCDRV_MAGIC) {
		printk(KERN_ERR "dxlmodem_ioctl: invalid device struct\n");
		return -EINVAL;
	}
	dxl = (struct dxlmodem_state *)dev->priv;

	if (cmd != SIOCDEVPRIVATE)
		return -ENOIOCTLCMD;
	if (get_user(cmd2, (int *)ifr->ifr_data))
		return -EFAULT;
	switch (hi->cmd) {
	default:
		break;
		
	case HDLCDRVCTL_GETMODE:
		if (dxl->hdrv.ops == &dxlmodem_ops) {
			strcpy(hi->data.modename, "dxlmodem");
			sprintf(hi->data.modename+strlen(hi->data.modename),
				" mode=%d  lostint=%d ",
				9601, dxl->modem.lostbyte);
		}
		else if (dxl->hdrv.ops == &dummy_ops)
			strcpy(hi->data.modename, "off");
		else
			strcpy(hi->data.modename, "invalid");

		if (dxl->options & 1)
			strcat(hi->data.modename, "*");

		if (copy_to_user(ifr->ifr_data, hi, sizeof(struct hdlcdrv_ioctl)))
			return -EFAULT;
		return 0;

	case HDLCDRVCTL_SETMODE:
		if (!suser() || dev->start)
			return -EACCES;
		hi->data.modename[sizeof(hi->data.modename)-1] = '\0';
		return dxlmodem_setmode(dxl, hi->data.modename);

	case HDLCDRVCTL_MODELIST:
		strcpy(hi->data.modename, "9600,off");
		if (copy_to_user(ifr->ifr_data, hi, sizeof(struct hdlcdrv_ioctl)))
			return -EFAULT;
		return 0;

	case HDLCDRVCTL_MODEMPARMASK:
		return HDLCDRV_PARMASK_IOBASE | HDLCDRV_PARMASK_IRQ;
		
	}

	if (copy_from_user(&bi, ifr->ifr_data, sizeof(bi)))
		return -EFAULT;
	switch (bi.cmd) {
	default:
		return -ENOIOCTLCMD;

	}
	if (copy_to_user(ifr->ifr_data, &bi, sizeof(bi)))
		return -EFAULT;
	return 0;

}

/* --------------------------------------------------------------------- */

__initfunc(int dxlmodem_init(void))
{
	int i, j, found = 0;
	char set_hw = 1;
	struct dxlmodem_state *dxl;
	char ifname[HDLCDRV_IFNAMELEN];


	printk(dxl_drvinfo);
	/*
	 * register net devices
	 */
	for (i = 0; i < NR_PORTS; i++) {
		struct device *dev = dxlmodem_device+i;
		sprintf(ifname, "dxl%d", i);

		if (!dxlmodem_ports[i].mode)
			set_hw = 0;
		if (!set_hw)
			dxlmodem_ports[i].iobase = dxlmodem_ports[i].irq = 0;
		j = hdlcdrv_register_hdlcdrv(dev, &dummy_ops,
					     sizeof(struct dxlmodem_state),
					     ifname, dxlmodem_ports[i].iobase,
					     dxlmodem_ports[i].irq, 0);
		if (!j) {
			dxl = (struct dxlmodem_state *)dev->priv;
			if (set_hw && dxlmodem_setmode(dxl, dxlmodem_ports[i].mode))
				set_hw = 0;
			found++;
		} else {
			printk(KERN_WARNING "%s: cannot register net device\n",
			       dxl_drvname);
		}
	}
	if (!found)
		return -ENXIO;
	return 0;
}

/* --------------------------------------------------------------------- */

#ifdef MODULE

MODULE_AUTHOR("Hansi Reiser, hsreiser@cip.informatik.uni-erlangen.de, dl9rdz@oe5xbl.#oe5.aut.eu");
MODULE_DESCRIPTION("9K6 FSK amateur radio modem driver for the oe5dxl-l2pcx-modem");

/* Command line settable parameters */
/* No, we don't have command line parameters -- use sethdlc instead... */

__initfunc(int init_module(void))
{
	/* Put in some default values for devices dxl0 and dxl1.
	 * You can change them via sethdlc -p before ifconfig up */
	dxlmodem_ports[0].mode = "9601";
	dxlmodem_ports[0].iobase = 0x3f8;
	dxlmodem_ports[0].irq = 4;
#if 0
	dxlmodem_ports[1].mode = "9601";
	dxlmodem_ports[1].iobase = 0x2f8;
	dxlmodem_ports[1].irq = 3;
	dxlmodem_ports[2].mode = NULL;
#else
	dxlmodem_ports[1].mode = NULL;
#endif

	return dxlmodem_init();
}

/* --------------------------------------------------------------------- */

void cleanup_module(void)
{
	int i;

	for(i = 0; i < NR_PORTS; i++) {
		struct device *dev = dxlmodem_device+i;
		struct dxlmodem_state *dxl =(struct dxlmodem_state *)dev->priv;

		if (dxl) {
			if (dxl->hdrv.magic != HDLCDRV_MAGIC)
				printk(KERN_ERR "dxlmodem: invalid magic in "
				       "cleanup_module\n");
			else
				hdlcdrv_unregister_hdlcdrv(dev);
		}
	}
}

#else /* MODULE */

/* --------------------------------------------------------------------- */
/*
 * This part is for a non-modularized version.
 * Never tried it myself so probably it just doesn't work
 * I think I should call dxlmodem_init before returning, but 
 * I coded this just to look like Tom's baycom.c, which doesnt
 * call baycom_init as well in that case. Maybe anything labeled
 * __initfunc will be called automaticall? In that case you would
 * have to make sure that _init is called before _setup...
 * I don't care about that in the moment 
 *  -- just use the loadable modul version of the driver :-)
 */

__initfunc(void dxlmodem_setup(char *str, int *ints))
{
	int i;

	for (i = 0; (i < NR_PORTS) && (dxlmodem_ports[i].mode); i++);
	if ((i >= NR_PORTS) || (ints[0] < 3)) {
		printk(KERN_INFO "%s: too many or invalid interface "
		       "specifications\n", dxl_drvname);
		return;
	}
	dxlmodem_ports[i].mode = str;
	dxlmodem_ports[i].iobase = ints[1];
	dxlmodem_ports[i].irq = ints[2];
	if (i < NR_PORTS-1)
		dxlmodem_ports[i+1].mode = NULL;
}

#endif /* MODULE */
/* --------------------------------------------------------------------- */
