/*
 *	cwbeacon.c
 *
 *	cwbeacon is part of the beaconProject, ea4rj@amsat.org
 *	and it is based on the original:
 *
 *	GW4PTS Morse tutor for Linux, by Alan Cox (morse.c 1.00, 21/09/93)
 *
 *	This software is placed under the GNU software license
 *
 *	rev. 19970723
 *	trimmed cw speed
 *	added sign marks: (,) (.) (-)
 *	added speaker on/off (default is on)
 *	added a steady tone at the end of the message
 */

#include <ctype.h>
#include <signal.h>
#include <stdio.h>
#include <unistd.h>

#include <asm/io.h>
#include <linux/kd.h>
#include <sys/ioctl.h>

/* Configuration globals */

#define REV	0.4
#define PORT	0x378		/* lp0=0x3bc, lp1=0x378, lp2=0x278 */

int morse_gap = 0;		/* Set with -d delay */
int morse_interval = 1;		/* Set with -i interval */
int morse_speed = 15;		/* Set with -s speed */
int morse_tone = 800;		/* Set with -t tone */
int morse_att = 0;		/* Set with -x */
int morse_sound = 1;		/* Unset with -z (sound not played) */

long int dot = 1000000;		/* dot length = unit */
long int dash = 3000000;	/* dash length = 3-dot */

typedef struct
{
	char code;
	enum
	{
		NIL,
		DIH,
		DAH,
	} data[7];
} Morse;

Morse MorseTable[]=
{
	' ',NIL,NIL,NIL,NIL,NIL,NIL,NIL,
	'A',DIH,DAH,NIL,NIL,NIL,NIL,NIL,
	'B',DAH,DIH,DIH,DIH,NIL,NIL,NIL,
	'C',DAH,DIH,DAH,DIH,NIL,NIL,NIL,
	'D',DAH,DIH,DIH,NIL,NIL,NIL,NIL,
	'E',DIH,NIL,NIL,NIL,NIL,NIL,NIL,
	'F',DIH,DIH,DAH,DIH,NIL,NIL,NIL,
	'G',DAH,DAH,DIH,NIL,NIL,NIL,NIL,
	'H',DIH,DIH,DIH,DIH,NIL,NIL,NIL,
	'I',DIH,DIH,NIL,NIL,NIL,NIL,NIL,
	'J',DIH,DAH,DAH,DAH,NIL,NIL,NIL,
	'K',DAH,DIH,DAH,NIL,NIL,NIL,NIL,
	'L',DIH,DAH,DIH,DIH,NIL,NIL,NIL,
	'M',DAH,DAH,NIL,NIL,NIL,NIL,NIL,
	'N',DAH,DIH,NIL,NIL,NIL,NIL,NIL,
	'O',DAH,DAH,DAH,NIL,NIL,NIL,NIL,
	'P',DIH,DAH,DAH,DIH,NIL,NIL,NIL,
	'Q',DAH,DAH,DIH,DAH,NIL,NIL,NIL,
	'R',DIH,DAH,DIH,NIL,NIL,NIL,NIL,
	'S',DIH,DIH,DIH,NIL,NIL,NIL,NIL,
	'T',DAH,NIL,NIL,NIL,NIL,NIL,NIL,
	'U',DIH,DIH,DAH,NIL,NIL,NIL,NIL,
	'V',DIH,DIH,DIH,DAH,NIL,NIL,NIL,
	'W',DIH,DAH,DAH,NIL,NIL,NIL,NIL,
	'X',DAH,DIH,DIH,DAH,NIL,NIL,NIL,
	'Y',DAH,DIH,DAH,DAH,NIL,NIL,NIL,
	'Z',DAH,DAH,DIH,DIH,NIL,NIL,NIL,
	'1',DIH,DAH,DAH,DAH,DAH,NIL,NIL,
	'2',DIH,DIH,DAH,DAH,DAH,NIL,NIL,
	'3',DIH,DIH,DIH,DAH,DAH,NIL,NIL,
	'4',DIH,DIH,DIH,DIH,DAH,NIL,NIL,
	'5',DIH,DIH,DIH,DIH,DIH,NIL,NIL,
	'6',DAH,DIH,DIH,DIH,DIH,NIL,NIL,
	'7',DAH,DAH,DIH,DIH,DIH,NIL,NIL,
	'8',DAH,DAH,DAH,DIH,DIH,NIL,NIL,
	'9',DAH,DAH,DAH,DAH,DIH,NIL,NIL,
	'0',DAH,DAH,DAH,DAH,DAH,NIL,NIL,
	'/',DAH,DIH,DIH,DAH,DIH,NIL,NIL,
	'*',DIH,DAH,DIH,DAH,DIH,NIL,NIL,	/* 'AR', end of message */
	'.',DIH,DAH,DIH,DAH,DIH,DAH,NIL,
	'-',DAH,DIH,DIH,DIH,DIH,DAH,NIL,
	',',DAH,DAH,DIH,DIH,DAH,DAH,NIL,
	'?',DIH,DIH,DAH,DAH,DIH,DIH,NIL,
	0,  NIL,NIL,NIL,NIL,NIL,NIL,NIL		/* END MARKER */
};

Morse *CharToMorse(char c)
{
	int ct=0;
	while(MorseTable[ct].code)
	{
		if(MorseTable[ct].code==c)
			return(&MorseTable[ct]);
		ct++;
	}

	return(NULL);
}

void PlayBeep(long duration)
{
	ioperm(PORT, 1, 1);	/* port lp1 */
	
	if(morse_sound)
	{
		signal(SIGINT, SIG_IGN);
		signal(SIGQUIT, SIG_IGN);

		if(ioctl(0, KIOCSOUND, morse_tone) == -1)
			perror("kiocsound");
	}

	outb(1 + morse_att, PORT);	/* bit D0 (pin2 at parallel port) */

	usleep(duration / morse_speed);

	if(morse_sound)
	{
		if(ioctl(0, KIOCSOUND, 0) == -1)
			perror("kiocsound");

		signal(SIGINT, SIG_DFL);
		signal(SIGQUIT, SIG_DFL);
	}

	outb(0 + morse_att, PORT);
}

void PlayMorse(Morse *m)
{
	int d = 0;
	while(d < 6 && m->data[d] != NIL)
	{
		if(m->data[d] ==DIH)
			PlayBeep(dot);
		if(m->data[d] == DAH)
			PlayBeep(dash);
		d++;
		usleep(dot / morse_speed);	/* dot-dash delay */
	}

	usleep(dash / morse_speed +100000 * morse_gap);	/* morse signs delay */
}

void PlayMorseString(char *x)
{
	while(*x)
	{
		char c = islower(*x) ? toupper(*x) : *x;
		Morse *m = CharToMorse(c);

		if(m != NULL)
		{
			PlayMorse(m);
		}

		x++;
	}

	PlayBeep(30e6);		/* steady tone, abt 3s */
}

void PlayFile(char *x)
{
	FILE *f = fopen(x, "r");
	char buf[512];
	if(f == NULL)
	{
		perror(x);
		return;
	}
	while(fgets(buf, 511, f) != NULL)
	{
/*		printf("%s", buf);
*/
		buf[strlen(buf) - 1] = 0;
		PlayMorseString(buf);
	}

	fclose(f);
}

void main(int argc, char *argv[])
{
	char **argp;

	int num1;

	if(argc==1 || argv[1][0]!='-' || argv[1][1] == 'h')
	{
		fprintf(stderr, "\
Send file:	 %s -f file\n\
Send message:	 %s -m \"message\"\n\
Show version:	 %s -v\n\
\n\
Additional options (place first):\n\
Extra delay:	 %s -d delay (default: 0)\n\
Beacon interval: %s -i interval (default: 1min)\n\
Relative speed:  %s -s speed (default: 15wpm)\n\
Set tone:	 %s -t tonevalue (default: 800Hz)\n\
Set attenuator:  %s -x att is on (default: off)\n\
Speaker sound:   %s -z sound not played (default: on)\n",
		argv[0], argv[0], argv[0], argv[0], argv[0], argv[0], argv[0],
		argv[0], argv[0]);

		exit(1);
	}

	argp = argv;

	while(argc > 2)
	{
		if(strcmp(argp[1], "-d") == 0)
		{
			if(sscanf(argp[2], "%d", &num1) == 0)
			{
				fprintf(stderr, "Delay value must be a number.\n");
				exit(1);
			}

			if(num1 < 0 || num1 > 5)
			{
				fprintf(stderr, "Delay range is 0-5.\n");
				exit(1);
			}

			morse_gap = num1;
			argc -= 2;
			argp += 2;
		}

		else if(strcmp(argp[1], "-i") == 0)
		{
			if(sscanf(argp[2], "%d", &num1) == 0)
			{
				fprintf(stderr, "Delay value must be a number.\n");
				exit(1);
			}

			if(num1 < 1 || num1 > 10)
			{
				fprintf(stderr, "Interval range is 1-10min.\n");
				exit(1);
			}

			morse_interval = num1;
			argc -= 2;
			argp += 2;
		}

		else if(strcmp(argp[1], "-s") == 0)
		{
			if(sscanf(argp[2], "%d", &num1) == 0)
			{
				fprintf(stderr, "Speed value must be a number.\n");
				exit(1);
			}

			if(num1 < 5 || num1 > 50)
			{
				fprintf(stderr, "Speed range is 5-50.\n");
				exit(1);
			}

			morse_speed = num1;
			argc -= 2;
			argp += 2;
		}

		else if(strcmp(argp[1], "-t") == 0)
		{
			if(sscanf(argp[2], "%d", &num1) == 0)
			{
				fprintf(stderr, "Tone value must be a number.\n");
				exit(1);
			}
			if(num1 < 300 || num1 > 3000)
			{
				fprintf(stderr, "Tone range is 300-3000.\n");
				exit(1);
			}
			morse_tone = num1;
			argc -= 2;
			argp += 2;
		}

		else if(strcmp(argp[1], "-x") == 0)
		{
			morse_att = 2;	/* D1 on parallel port to be active */
			argc -= 1;
			argp += 1;
		}
		else if(strcmp(argp[1], "-z") == 0)
		{
			morse_sound = 0;
			argc -= 1;
			argp += 1;
		}

		else
			break;
	}

	if(argc < 2)
	{
		fprintf(stderr, "One of -f, -m, or -v must be specified.\n");
		exit(1);
	}

	/* What do we do */
	switch(argp[1][1])
	{
		case 'm':
			if(argc != 3)
			{
				fprintf(stderr, "%s -m \"text\".\n", argv[0]);
				exit(1);
			}

			for (;;)
			{
				PlayMorseString(argp[2]);
				sleep(morse_interval * 60);
			}

			break;

		case 'v':
			printf("EA4RJ cwbeacon for Linux: Rev. %1.1f\n", REV);
			break;

		case 'f':
			if(argc != 3)
			{
				fprintf(stderr, "%s -f filename.\n", argv[0]);
				exit(1);
			}

			for (;;)
			{
				PlayFile(argp[2]);
				sleep(morse_interval * 60);
			}

			break;

		default:
			fprintf(stderr, "Unknown option '%c', %s -h for help.\n",argp[1][1],argv[0]);
			exit(1);
	}
}
