###############################################################################
###############################################################################
##                               Descargar.tcl
###############################################################################
###############################################################################
## Includes the procedures needed for downloading a link
###############################################################################
###############################################################################
## (c) 2001-2003 Andrs Garca Garca. fandom@retemail.es
## You may distribute the contents of this file under the terms of the GPL v2
###############################################################################
###############################################################################

namespace eval Descargar {

###############################################################################
# CheckDir
#    Creates, if needed, the directory in which the link will be saved.
#
# Parameter:
#    fileName: file in which the url is going to be downloaded.
#
# Returns:
#    1: In case of error
###############################################################################
proc CheckDir {fileName} {
    global labelTitles labelMessages

    set newDirectory [file dirname $fileName]
    set tmpDir $newDirectory

    # This is a bit weird, so please be patient.
    # There seems to be an Apache module that is a bit to helpful,
    # if you request a file called 'http://www.domain.com/abc'
    # and 'abc' is a directory, it returns 'abc/index.html' without
    # telling anything, up until this 'Getleft' would create a file
    # 'abc', which was nice and good until you had to download
    # 'abc/something.html', so in that case we now delete the 
    # 'helpful' file and continue.
    while {[catch {file mkdir $newDirectory}]} {
        if {![file exists $tmpDir]} {
            set tmpDir [file dirname $tmpDir]
        } else {
            if {[catch {file delete $tmpDir}]} {
                return 1
            }
        }
    }
    return
}

###############################################################################
# FileExists
#    Checks whether the file we are going to download has already been 
#    downloaded.
#
# Parameter:
#    link: The url about to be downloaded
#    fileName: Name of the file to check
#    sufix: '1' if the fileName ends in 'html?' '0' otherwise.
#
# Returns:
#    - '-1' the file does not exist.
#    - '0' file exists and is not a Html page.
#    - '1' file exists and is a Html page
#    - '4' if it exists and the referrer page will have to change for it.
###############################################################################
proc FileExists {link fileName sufix} {
    global downOptions
    variable relocated
    variable downloaded

    if {[file exists $fileName]} {
        if {($downOptions(update)==0)||[info exists "downloaded($link)"]} {
            if {$sufix==0} {
                # It may so happen with CGI scripts that there is a directory
                # and a file with the same name, I wish webmasters stopped 
                # being clever about the names of CGI scripts, but there is
                # nothing I can do to stop them being stupid.
                if {([file isdirectory $fileName])&&(![regexp {/$} $fileName])} {
                    if {[file exists $fileName/index.html]} {
                        set relocated(url) $link/index.html
                    } elseif {[file exists $fileName.html]} {
                        set relocated(url) $link.html
                    } else {
                        set relocated(url) $link/index.html
                        return -1
                    }
                    return 4
                } elseif {[file exists $fileName.html]} {
                    return 1
                } else {
                    return 0
                }
            } else {
                return 1
            }
        }
    }
    return -1
}

###############################################################################
# ConnectTimeout
#    This procedure is invoked when Getleft fails to connect to the server
#    in 60 seconds. Here it will be decided where to retry or abort.
#
# Returns:
#    - '0' we try again.
#    - '1' we cancel the download.
###############################################################################
proc ConnectTimeout {link referer code} {
    global getleftState getleftOptions labelCurlCodes labelMessages
    global siteUrl

    incr getleftState(noConnect,$siteUrl(www))
    getLog::FileLogEnter $code "$link" "$referer"
    set getleftState(waiting) 1
    set tmp $::Ventana::Rizo::curlReport(nextFile)
    Ventana::ShowWindow wait
    tkwait variable getleftState(waiting)
    if {$getleftState(downloading)==0} {
        return 1
    }
    if {$tmp!=$::Ventana::Rizo::curlReport(nextFile)} {
        return 1
    }
    if {$getleftState(noConnect,$siteUrl(www))>10} {
        if {$getleftState(autoDown)!=1} {
            set what [tk_messageBox  -title $labelCurlCodes($code)          \
                      -type yesno    -message $labelMessages(timeoutCont) \
                      -icon question -parent $::Ventana::window(top)]
            if {$what=="yes"} {
                set getleftState(noConnect,$siteUrl(www)) 0
                return 0
            }
        }
        set getleftOptions(stopFile) 1
        return 1
    }
    return 0
}

###############################################################################
# RelocatedLink
#    This procedure is invoked when the server tells us that a link has
#    been moved to another place. The link is filtered according to the
#    options set by the user and, if it is the very first file in a
#    download, the user is asked whether to follow or not.
#
# Parameter:
#    link: The original link.
#    newLink: The url after the relocation.
#    referer: Referrer of the link.
#
# Returns:
#    - '0' we follow the redirection.
#    - '1' we don't follow, the files is put in the 'filesNotFound' array, so
#      we don't lose time trying again.
###############################################################################
proc RelocatedLink {link newLink referer} {
    global getleftState labelTitles labelMessages labelDialogs siteUrl
    global directories
    variable relocated
    variable relocatedLink
    variable filesNotFound
    variable filesRelocated

    set newLink [::HtmlParser::CompleteUrl $newLink $referer ""]
    set relocated(url) $newLink
    set parsedUrl [HtmlParser::ParseUrl $newLink]
    if {$parsedUrl==1} {
        return 1
    }

    if {($getleftState(filesChosen)==0)&&($getleftState(autoDown)==0)} {
        set what [tk_messageBox -title $labelTitles(relocation) -type yesno \
                -icon question -parent $::Ventana::window(top)              \
                -message "$labelMessages(follow)\n$newLink"]
        if {$what=="no"} {
            return 1
        }
        set siteUrl(prot) [lindex $parsedUrl 0]
        set siteUrl(www)  [lindex $parsedUrl 1]
        set siteUrl(dir)  [lindex $parsedUrl 2]

        return 0
    }
    set relocatedLink(1,url)  $newLink
    set relocatedLink(1,ok)   1
    set relocatedLink(1,file) [::Commands::UrlToFile $newLink $directories(base)]

    ::HtmlParser::FilterLinks $referer Descargar::relocatedLink

    set filesRelocated($link,url)  $newLink
    set filesRelocated($link,ok)   $relocatedLink(1,ok)
    set filesRelocated($link,file) $relocatedLink(1,file)
    if {[regexp {\.html?} $relocatedLink(1,file)]} {
        set filesRelocated($link,html) 1
    } else {
        set filesRelocated($link,html) 0
    }

    if {$relocatedLink(1,ok)==1} {
        ::getLog::FileLogEnter                                              \
                $labelDialogs(relocation) $link $referer
        ::getLog::FileLogEnter "" "$newLink" "$link"
        return 0
    }
    ::getLog::FileLogEnter                                                  \
            "$labelMessages(noFollow)" "$link" "$referer"
    ::getLog::FileLogEnter "" "$newLink" "$link"

    return 1
}

###############################################################################
# UpdateFile
#    Checks whether the file in the server is newer than the one we have.
#
# Parameter:
#    fileName: Name of the file to check
#
# Returns:
#    - '1' if we have to update
#    - '0' if not
###############################################################################
proc UpdateFile {fileName} {
    global ::Ventana::Rizo::meta
    global getleftState getleftOptions
    global labelTitles labelMessages

    if {[file exists $fileName.orig]} {
        set oldFile [file mtime $fileName.orig]
    } else {
        set oldFile [file mtime $fileName]
    }
    if {[catch "set meta(lastModified)"]} {
        if {$getleftState(filesChosen)==0} {
            set which [tk_messageBox -icon info -type yesno \
                    -title $labelTitles(error) \
                    -message $labelMessages(noDate)]
            if {$which=="no"} {
                set getleftOptions(stopFile) 1
                return 0
            }
        }
        set lastChange 0
    } else {
        set lastChange [clock scan $meta(lastModified)]
    }
    if {$lastChange<$oldFile} {
        return 0
    }
    return 1
}

###############################################################################
# DownloadHead
#    Downloads the headers of the link
#
# Parameter:
#    link:    Link to download
#    referer: Referrer page
#    sufix: '1' if the link has a 'html' extension.
#
# Returns:
#    - '4' in case of a relocation and a HTML page
#    - '3' in case of a relocation and not a HTML page
#    - '2' in case of error
#    - '1' if it is a HTML page
#    - '0' if it is not
###############################################################################
proc DownloadHead {link referer sufix} {
    global Ventana::Rizo::meta
    global siteUrl getleftState  directories
    global errorCode labelTitles labelMessages
    variable filesNotFound
    variable relocated

    for {set relocated(ok) 0} {1==1} {} {
        for {} {1==1} {} {
            Ventana::HeadDownloading $link $referer
            if {($::Ventana::Rizo::curlError==7)||($::Ventana::Rizo::curlError==28)||($::Ventana::Rizo::curlError==52)} {
                if {[ConnectTimeout $link $referer $::Ventana::Rizo::curlError]==1} {
                    return 2
                }
            } else {
                break
            }
        }
        if {($::Ventana::Rizo::curlError!=0)||($::Ventana::Rizo::errorMessage!="")} {
#            if {$::DEBUG==1} {
#                tk_messageBox -type ok -icon error -title "cURL error" \
                    -message "cURL reported error: $::Ventana::Rizo::curlError\
                              \n$::Ventana::Rizo::errorMessage\n$link"
#            }
            if {$::Ventana::Rizo::errorMessage=="Proxy requires authorization!"} {
                tk_messageBox -type ok -icon error -title $labelTitles(error) \
                        -message $labelMessages(proxyAuthFail)
                return 2
            }    
            set code [lindex $errorCode 2]
            ::getLog::FileLogEnter $code "$link" "$referer"
            if {$code>=400} {
                set filesNotFound($link) 1
            }
            if {($code!=550)&&($code!=450)} {	    
                return 2
            }	    
        }
        if {$meta(relocate)!=""} {
#            set filesNotFound($link) 1
            if {$::DEBUG==1} {
                puts "Relocated: $meta(relocate)"
            }
            set returnCode [RelocatedLink $link $meta(relocate) $referer]
            if {$returnCode==1} {
                return 2
            }

            set parsedUrl [HtmlParser::ParseUrl $relocated(url)]
            set newWWW    [lindex $parsedUrl 1]
            set newDir    [lindex $parsedUrl 2]

            set fileName  [::Commands::UrlToFile $relocated(url) $directories(base)]
            set link      $relocated(url)
            set relocated(ok) 1
            if {[file exists $fileName]} {
                if {([regexp -nocase {\.html?$} $fileName])\
                        ||([file exists $fileName.html])} {
                    set isHtml 1
                } else {
                    set isHtml 0
                }
                return [expr {3 + $isHtml}]
            }
        } else {
            break
        }
    }
# Some servers don't return a mime time, that's why we also check the sufix
    if {($meta(content)!="text/html")&&(![regexp {html?$} $link])} { 
        return 0
    }
    return 1
}


###############################################################################
# DownloadFile
#    Downloads the link
#
# Parameter:
#    link:     Link to download
#    referer:  referrer page
#    fileName: file is which the link will be saved.
#
# Returns:
#    - '1' in case of error
#    - '0' all is well
###############################################################################
proc DownloadFile {link referer fileName} {
    global labelMessages getleftOptions errorCode

    set Ventana::Rizo::curlReport(parar)    0
    set Ventana::Rizo::curlReport(nextFile) 0
    for {} {1==1} {} {
        Ventana::FileDownloading $fileName.$labelMessages(downSuffix) \
                $link $referer
        tkwait variable Ventana::Rizo::curlReport(nextFile)
        if {($::Ventana::Rizo::curlError==7)||($::Ventana::Rizo::curlError==28)||($::Ventana::Rizo::curlError==52)} {
            if {[ConnectTimeout $link $referer $::Ventana::Rizo::curlError]==1} {
                return 1
            } else {
                continue
            }
        }
        break
    }

    if {$::Ventana::Rizo::curlError!=0} {
        if {$::DEBUG==1} {
            tk_messageBox -type ok -icon error -message "$::errorCode\n$::errorInfo"
        }
        ::getLog::FileLogEnter [lindex $errorCode 2] "$link" "$referer"
        return 1
    }

    # Maybe we should say something in the error log
    if {$getleftOptions(cancelDown)==1} {
        return 1
    }

    return 0
}

###############################################################################
# Download
#    Downloads a link
#
# Parameter:
#    link:     Link to download
#    referer:  referrer page
#    sufix:    Whether the file ends in 'htm?' (1) or not (0)
#
# Returns:
#    - '4' in case of a relocation and a HTML page
#    - '3' in case of a relocation and not a HTML page
#    - '2' in case of error
#    - '1' if it is a HTML page
#    - '0' if it is not
###############################################################################
proc Download {link referer sufix} {
    global siteUrl downOptions getleftOptions errorCode getleftState
    global Ventana::Rizo::meta
    global directories
    global labelDialogs labelMessages
    variable downloaded
    variable filesNotFound
    variable filesRelocated
    variable relocated

    if {[info exists filesNotFound($link)]} {
        return 2
    }

    if {[info exists filesRelocated($link,url)]} {
        set relocated(ok)  1
        set relocated(url) $filesRelocated($link,url)
        return [expr $filesRelocated($link,html)+3]
    }

    set fileName [::Commands::UrlToFile $link $directories(base)]

    set returnCode [FileExists $link $fileName $sufix]
    if {$returnCode!=-1} {
        return $returnCode
    } else {
        set downloaded($fileName) 1
    }

    set returnCode [DownloadHead $link $referer $sufix]
    switch -regexp -- $returnCode {
        {0|1}   "set isHtml $returnCode"
        {2|3|4} "return $returnCode"
    }        
    if {$getleftOptions(cancelDown)} {
        return 2
    }
    if {$relocated(ok)==1} {
        set link $relocated(url)
        set fileName [::Commands::UrlToFile $link $directories(base)]

    }
    if {($isHtml==0)&&($downOptions(onlyHtml)==1)} {
        getLog::FileLogEnter "$labelMessages(noFollow)" "$link" "$referer"
        return 0
    }
    if {($getleftState(filesChosen)==0)&&($isHtml==0)} {
        set fileName $directories(base)/[file tail $fileName]
    }

    set returnCode [CheckDir $fileName]
    if {$returnCode==1} {
        getLog::FileLogEnter "Bad dir" "$link" "$referer"
        return 2
    }
    regexp {(.*)(://)} $link nada prot

    if {($downOptions(update)==1)&&([file exists $fileName])} {
        if {![UpdateFile $fileName]} {
            return $isHtml
        }
    }
    if {[DownloadFile $link $referer $fileName]} {
        return 2
    }
    # With CGI links it may happen that a file and a directory have the same
    # name.
    if {[file isdirectory $fileName]} {
        file rename -force $fileName.$labelMessages(downSuffix) $fileName.html
    } else {
        catch {file rename -force $fileName.$labelMessages(downSuffix) $fileName}
    }
    if {$isHtml==1} {
        file delete $fileName.orig
    }

    if {$relocated(ok)==1} {
        return [expr {3 + $isHtml}]
    }
    if {[catch {file size $fileName} size]} {
        set size 0
    }
    getLog::FileLogEnter 0 $link "$referer" "$size"

    return $isHtml
}

###############################################################################
# DownloadLinks
#    Download all the links taken from a referrer page.
#
# Parameters:
#    urlList: list with the urls to be downloaded.
#    referer: html page from which the links come from.
#
# Returns
#    The link list with only the Html links.
###############################################################################
proc DownloadLinks {urlList referer} {
    global downOptions getleftOptions directories
    variable relocated

    set invokeSed 0

    set urlListTemp ""
    foreach urlToDownload $urlList {
        update
        if {$downOptions(onlyHtml)==1} {
            if {[regexp -nocase -expanded {(jpg)|(jpeg)|(gif)|(gz) |(tar) |(zip)|     \
                                           (exe)|(ps)  |(pdf)|(txt)|(text)|(avi)|     \
                                           (ram)|(wav) |(png)|(tif)|(mov) |(qt) |     \
                                           (js) |(rpm)} \
                [file extension $urlToDownload]]} {
                continue
            }
        }
        set fileName [Commands::UrlToFile $urlToDownload $directories(base)]
        set changedPageLink [Commands::RelativePath $referer $urlToDownload]

        set tag ""
        regexp {(#.*)} $urlToDownload tag

        if {[regexp {/$} $urlToDownload]} {
            lappend urlListTemp ${urlToDownload}index.html
        }

        if {[regexp -nocase {\.html?$} [file extension $fileName]]} {
            set sufix 1
        } else {
            set sufix 0
        }
        set isHtml [Download $urlToDownload $referer $sufix]
        switch -regexp -- $isHtml {
            1 {
                if {$sufix==0} {
                    catch {file rename -force $fileName $fileName.html}
                    Commands::Touch $fileName
                    lappend urlListTemp $urlToDownload.html
                    Commands::SedChangeEnter $changedPageLink$tag          \
                                             ${changedPageLink}.html$tag
                    set invokeSed 1
                } else {
                    lappend urlListTemp $urlToDownload 
                }
            }
            {3|4} {
                set newLink [::Commands::RelativePath $referer $relocated(url)]
                set sufix [regexp -nocase {\.html?$} [file extension $newLink]]
                if {$isHtml==4} {
                    if {$sufix==0} {
                        lappend urlListTemp $relocated(url).html
                        set localFileName [::Commands::UrlToFile $relocated(url) $directories(base)]
                        if {(![regexp -nocase {\.html?$} $localFileName])&&(![file exists $localFileName.html])} {
                            file rename -force $localFileName $localFileName.html
                            Commands::Touch $localFileName
                        }
                        set newLink $newLink.html
                    } else {
                        lappend urlListTemp $relocated(url)
                    }
                }
                Commands::SedChangeEnter $changedPageLink$tag $newLink$tag
                set invokeSed 1
            }
        }
        if {$getleftOptions(stopFile)==1} {
            break
        }
        if {$getleftOptions(pauseFile)==1} {
            Ventana::Pause file
        }
    }
    if {$invokeSed==1} {
        Commands::Sed [::Commands::UrlToFile $referer $directories(base)]
    }
    return $urlListTemp
}

###############################################################################
# Preprocessing
#    Reads the Web page passed as a parameter and proccess it to extract
#    all links to local files and images it has
#
# Parameters:
#    url: url of the page where are going to process for links.
#    level: Level of recursion in which the file is processed.
###############################################################################
proc Preprocessing {url level} {
    global directories getleftState charSets filesDone

    if {[string match $url ""]} return
    set file [::Commands::UrlToFile $url $directories(base)]
    set filesDone($file) $level
    if {![regexp -nocase {\.html?$} $file]} return

#    if {$::DEBUG==1} {
#        puts "Fichero a preprocesar: $file"
#    }

    set directories(local) [file dirname $file]

    if {$getleftState(filesChosen)==1} {
        Ventana::ShowWindow  process
        Ventana::FileStrings $url
    }

    if {[file exists $file.orig]} {
        set fileName $file.orig
    } else {
        set fileName $file
    }

    set returnCode [HtmlParser::Parsing $fileName $url]
    if {$returnCode==1} {
        if {$::DEBUG==1} {
            tk_messageBox -type ok -icon error -type ok \
                    -message "Could not process:\n $fileName"
        }
        return
    }

    if {$getleftState(filesChosen)==0} {
        set encoding ""
        catch {set encoding $charSets($Ventana::Rizo::meta(charSet))}
        if {$HtmlParser::pageEncoding!=""} {
            if {[catch {set charSets($HtmlParser::pageEncoding)} encoding]} {
                set encoding ""
            }
        }
        if {$encoding!=""} {
            HtmlParser::ChangeEncoding $encoding
        }
    }

    HtmlParser::FilterLinks $url ::HtmlParser::links

    update
    ::Commands::ChangePage $url

    return
}

###############################################################################
# ProcessUrlList
#    After all links from a referer page have been downloaded,this checks 
#    for the Html pages between them to continue the download of the site.
#
# Parameters:
#    urlList: list with the urls to be processed for links
#    level: current level of recursion for internal links.
#    externalLevel: current level of recursion for external link, defaults to 0
###############################################################################
proc ProcessUrlList {urlList level {externalLevel 0}} {
    global siteUrl directories urlsDownloaded getleftOptions downOptions 
    global filesDone getleftState

    foreach url $urlList {
        set fileToProcess   [::Commands::UrlToFile $url $directories(base)]
        set parsedUrl       [HtmlParser::ParseUrl $url]
        if {[info exists filesDone($fileToProcess)]} {
            if {($level>=$filesDone($fileToProcess))||($downOptions(levels)==-1)\
                ||($level>=$downOptions(levels))||($getleftState(downloading)==0)} {
                continue
            }
        }
        if {[info exists urlsDownloaded($url)]} {
            if {($level>=$urlsDownloaded($url))||($downOptions(levels)==-1)} {
                continue
            }
        }
        set tmpUrl        $siteUrl(dir)
        set tmpLocal      $directories(local)
        set tmpProt       $siteUrl(prot)
        set domain        [lindex $parsedUrl 1]
        set siteUrl(dir)  [lindex $parsedUrl 2]
        set siteUrl(prot) [lindex $parsedUrl 0]
        if {[string compare [string tolower $domain] \
                [string tolower $siteUrl(www)]]} {
            set nextExternalLevel [expr {$externalLevel + 1}]
        } else {
            set nextExternalLevel 0
        }

        if {(($nextExternalLevel<$downOptions(exLevels))||($downOptions(exLevels)==0))&&($level<100)} {
            Preprocessing $url $level
            if {$HtmlParser::nLinks!=1} {
                set nextLevel [expr {$level +1}]
                ControlDownloading $url $nextLevel $nextExternalLevel
                if {($getleftOptions(stopFile)==1)||($getleftOptions(stopPage)==1)} {
                    return
                }
                set urlsDownloaded($url) $level
            }
        }
        set siteUrl(prot)      $tmpProt
        set siteUrl(dir)       $tmpUrl
        set directories(local) $tmpLocal
    }
    return
}

###############################################################################
# PrepareDownloading
#    Prepares the list of files to download
#
# Parameter
#    currentLevel: the current level of recursion
#
# Returns
#    The list with the urls to be downloaded.
###############################################################################
proc PrepareDownloading {currentLevel} {
    global siteIndex siteMap downOptions getleftState

    for {set i 1 ; set urlList ""} {$i<$HtmlParser::nLinks} {incr i} {
        if {$HtmlParser::links($i,ok)==1} {
            lappend urlList $HtmlParser::links($i,url)
        }
        if {$downOptions(map)==1} {
            set siteMap($siteIndex,level)       $currentLevel
            set siteMap($siteIndex,url)         $HtmlParser::links($i,url)
            set siteMap($siteIndex,file)        $HtmlParser::links($i,file)
            set siteMap($siteIndex,descrip)     $HtmlParser::links($i,descrip)
            catch {set siteMap($siteIndex,type) $HtmlParser::links($i,type)}
            set siteMap($HtmlParser::links($i,url)) 1
            incr siteIndex
        }
    }
    if {$getleftState(filesChosen)==0} {
        set getleftState(filesChosen) 1
    }
    return $urlList
}

###############################################################################
# ControlDownloading
#    Sends 'Downloading' all the links one by one.
#
# Parameters:
#    referer: html page from which the links come from.
#    level: current level of recursion for internal links.
#    externalLevel: current level of recursion for external links.
###############################################################################
proc ControlDownloading {referer level {externalLevel 0}} {
    global getleftOptions getleftState downOptions
    variable relocated 

    set urlList [PrepareDownloading $level]
    if {[llength $urlList]==0} return

    if {$getleftState(downloading)==0} {
        return
    }

    if {($getleftOptions(stopFile)!=1)&&($getleftOptions(stopPage)!=1)} {
        if {(($level<=$downOptions(levels))||($downOptions(levels)==-1))&&($level<=99)} {
            set urlList [DownloadLinks $urlList $referer]
            if {$getleftOptions(pausePage)==1} {
                Ventana::Pause page
            }
        }
    }

    ProcessUrlList $urlList $level $externalLevel

    return
}

}
