###############################################################################
###############################################################################
#####                            Ventana.tcl
###############################################################################
##### Creates the window which will provide some information to the user 
##### while a file is being downloaded.
###############################################################################
##### Copyright 1999-2003 Andrs Garca Garca - fandom@retemail.es
##### Distributed under the terms of the GPL v2
###############################################################################

namespace eval Ventana {

source [file join $dirGetleft(scripts) Rizo.tcl]

###############################################################################
# DefineButtonImages
#    Reads the images needed to create the 'downloading' window.
###############################################################################
proc DefineButtonImages {} {
    global   dirGetleft
    variable images

    set images(url)    [image create photo                             \
            -file [file join "$dirGetleft(icons)" html.gif]]
    set images(folder) [image create photo                             \
            -file [file join "$dirGetleft(icons)" folder.gif]]
    set images(files)  [image create photo                             \
            -file [file join "$dirGetleft(icons)" files.gif]]
    set images(clock)  [image create photo                             \
            -file [file join "$dirGetleft(icons)" clock.gif]]

    set images(pause)  [image create photo                             \
            -file  [file join "$dirGetleft(icons)" pauses.gif]]
    set images(resume) [image create photo                             \
            -file  [file join "$dirGetleft(icons)" resumes.gif]]
    set images(stop)   [image create photo                             \
            -file  [file join "$dirGetleft(icons)" stops.gif]]
    set images(conf)   [image create photo                             \
            -file  [file join "$dirGetleft(icons)" confs.gif]]

    return
}

###############################################################################
# ShowResumeButton
#    When the user pauses the download, the 'Pause' menu button is 
#    subtituted by the 'Resume' button
###############################################################################
proc ShowResumeButton {} {
    variable window
    variable menus
    global   tcl_platform

    if {$tcl_platform(platform)=="windows"} {
        set offset -2
    } else {
        set offset -3
    }

    $window(pause) configure -state disabled
    place $window(resume) -in $window(pause) -x $offset -y $offset

    $menus(stop) entryconfigure 3 -state disabled

    return
}

###############################################################################
# HideResumeButton
#    And when the download is resumed it is hidden again.
###############################################################################
proc HideResumeButton {} {
    variable window
    variable menus

    place forget $window(resume)
    $window(pause) configure -state normal

    $menus(stop) entryconfigure 3 -state normal

    return
}

###############################################################################
# PauseDownloading
#    Pauses the downloading according to the menu entries.
#
# Parameter
#    which: either 'file' or 'page'
###############################################################################
proc Pause {which} {
    global getleftOptions labelTitles labelMessages
    variable window

    set windowState [wm state $window(top)]
    if {$windowState=="iconic"} {
        wm deiconify $window(top)
    }

    ShowResumeButton
#    tk_messageBox -title $labelTitles(paused) -icon info -type ok \
         -message $labelMessages(paused) -parent $window(top)
    if {$which=="file"} {
        tkwait variable getleftOptions(pauseFile)
    } else {
        tkwait variable getleftOptions(pausePage)
    }

    return
}

###############################################################################
# CancelDownloading
#    This procedure is invoked from the downloading window when you click on
#    the 'cancel' button or you want to close the window
#
# Parameter:
#    parent: The window over which the asking dialog will appear.
#    skip: if skip is 'skip' the current file will be skipped and the next one
#    will begin to be downloaded. If there is no parameter, the downloading
#    will be stopped.
#
# Returns
#    '0' if we are not going to stop.
#    '1' if we are.
###############################################################################
proc CancelDownloading {parent {skip {} }} {
    global labelTitles labelMessages
    global getleftState getleftOptions
    global ::Ventana::Rizo::curlReport
    variable afterId
    variable waitingPid
    variable window   

    if {$getleftState(delayedDown)==0} {
        set what [tk_messageBox -type yesno -icon question -parent $parent \
                -title $labelTitles(confirm)                               \
                -message $labelMessages(stopDownload)]
        if {$what=="no"} {
            return 0
        }
        Progreso::ProgressBarReset $window(bar)
        set Ventana::Rizo::curlReport(left) ""
    }
    catch {after cancel $afterId}
    catch {after cancel $waitingPid}
    catch {Rizo::Lector stopNow}

    set getleftOptions(cancelDown)     1
    set Ventana::Rizo::curlReport(end) 1

    if {$getleftState(autoDown)==1} {
        mainWin::SetAutoDownload
    }
    if {($skip=="")||($getleftState(filesChosen)==0)} {
        set getleftOptions(stopFile)  1
        set getleftState(downloading) 0
        catch {wm withdraw $window(top)}
    }

    incr curlReport(nextFile)

    if {$getleftState(waiting)==1} {
        set getleftState(waiting) 0
    }

    return 1
}

###############################################################################
# TrimString
#    Trims the given string so that if it is longer than the available space
#    the middle of it gets substituted with '...', and puts the resulting
#    string in the given label.
#
# Parmetros
#    cadena: string to trim.
#    labelPath: the label in which the string is going to be shown
###############################################################################
proc TrimString {cadena labelPath} {

    set font     [$labelPath cget -font]
    set color    [$labelPath cget -bg]
    set tmpColor [$labelPath cget -fg]
    set varName  [$labelPath cget -textvariable]

    upvar #0 $varName textVariable
    $labelPath configure -fg $color

    set textVariable $cadena
    set labelLength  [expr {[winfo width $labelPath]-5}]
    set strLength    [font measure $font -displayof $labelPath $cadena]

    if {$labelLength<=$strLength} {
        set strChars [string length $cadena]
        set middle   [expr {round($strChars/2)-2}]
        set primer   [string range $cadena 0 $middle]
        set segun    [string range $cadena [expr {$middle+1}] end]
        set textVariable ""
        append textVariable $primer "..." $segun
        while {[font measure $font -displayof $labelPath $textVariable]>$labelLength} {
            regsub {(.)(\.\.\.)(.)} $textVariable {...} textVariable
        }
        BalloonHelp::set_balloon $labelPath $cadena
    } else {
        BalloonHelp::delete_balloon $labelPath
    }

    $labelPath configure -fg $tmpColor

    return
}

###############################################################################
# FileStrings
#    Controls the strings that will be put in the downloading window, to
#    show the name and directory of the file being processed or downloaded
#
# Parameter:
#    url to be downloaded or preprocessed
###############################################################################
proc FileStrings {url} {
    global directories labelMessages
    variable window

    set fileName  [::Commands::UrlToFile $url $directories(base)]
    set parsedUrl [HtmlParser::ParseUrl $url]
    set domain    [string tolower [lindex $parsedUrl 1]]

    regsub -all {:}  $domain {} domain
    if {[regexp "(?:$directories(base)/$domain/)(.*)" \
            $fileName nada tmp]==0} {
        set tmp ""
        if {$::DEBUG==1} {
            tk_messageBox -type ok -title Error \
                -message "Bad string to show: $url"
        }
    }
    if {![regexp {(.*)(?:/)(.*)} $tmp nada tmpDir tmpFile]} {
        set tmpDir ""
        set tmpFile $tmp
    }

    TrimString $domain  $window(url)
    TrimString $tmpDir  $window(dir)
    TrimString $tmpFile $window(file)

    update

    return
}

###############################################################################
# DownloadWindow
#    Creates the window, in which some info about the downloading will be
#    displayed.
###############################################################################
proc DownloadWindow {} {
    global labelButtons labelDialogs labelMenus
    global tcl_platform dirGetleft
    variable window
    variable menus
    variable images

    DefineButtonImages

    set win [toplevel .bajando]
    wm geometry  $win 290x160
    wm resizable $win 0 0

    set marco     [frame $win.marco]
    set marco_in  [frame $marco.marco_interno]
    set marco_url [frame $marco_in.url]
    set marco_dir [frame $marco_in.dir]
    set marco_fil [frame $marco_in.fil]
    set marco_que [frame $marco_in.que]
    set marco_bar [frame $marco_in.bar]
    set botones   [frame $win.botones]

    set etq_url   [label $marco_url.etq_url  -image $images(url)]
    set etq_dir   [label $marco_dir.etq_dir  -image $images(folder)]
    set etq_file  [label $marco_fil.etq_file -image $images(files)]
    set etq_que   [label $marco_que.etq_que  -image $images(clock)]

    BalloonHelp::set_balloon $etq_url  $labelDialogs(url)
    BalloonHelp::set_balloon $etq_dir  $labelDialogs(dir)
    BalloonHelp::set_balloon $etq_file $labelDialogs(file)
    BalloonHelp::set_balloon $etq_que  $labelDialogs(left)

    set etq_url2  [label $marco_url.etq_url2  -width 45 -anchor w         \
            -bd 2 -relief sunken -textvariable Ventana::window(labelUrl)]
    set etq_dir2  [label $marco_dir.etq_dir2  -width 45 -anchor w         \
            -textvariable Ventana::window(labelDir)]
    set etq_file2 [label $marco_fil.etq_file2 -width 45 -anchor w         \
            -textvariable Ventana::window(labelFile)]
    set etq_que2  [label $marco_que.etq_que2  -width 30 -anchor w         \
            -textvariable Ventana::Rizo::curlReport(left)]

    set progressBar [Progreso::ProgressBar $marco_bar.progress]

    set window(top)  $win
    set window(url)  $etq_url2
    set window(dir)  $etq_dir2
    set window(file) $etq_file2
    set window(left) $etq_que2
    set window(bar)  $progressBar

    set config   [button $botones.config -width 48 -height 20 -image $images(conf)      \
            -command "mainWin::UrlListChangeOptions $window(top)"]
    set pause    [menubutton $botones.pause -menu $botones.pause.menu                   \
            -width 50 -height 22 -image $images(pause) -relief raised]
    set stop     [menubutton $botones.stop  -menu $botones.stop.menu                    \
            -width 50 -height 22 -image $images(stop)  -relief raised]
    set resume   [button $botones.resume -width 48 -height 20 -image $images(resume)    \
            -command "::Ventana::ResumeDownloading"]

    ::BalloonHelp::set_balloon $config $labelDialogs(options)
    ::BalloonHelp::set_balloon $pause  $labelDialogs(pause)
    ::BalloonHelp::set_balloon $stop   $labelDialogs(stop)
    ::BalloonHelp::set_balloon $resume $labelDialogs(start)

    set menus(pause) [menu $pause.menu -tearoff 0 -fg black -postcommand "
            BalloonHelp::kill_balloon
    "]
    $menus(pause) add command -label $labelMenus(stopNow) -command {
            Ventana::PauseDownloading now
    }
    $menus(pause) add check -variable getleftOptions(pausePage)              \
            -onvalue 1 -offvalue 0 -label $labelMenus(stopPage)
    $menus(pause) add check -variable getleftOptions(pauseFile)              \
            -onvalue 1 -offvalue 0 -label $labelMenus(stopFile)

    set menus(stop) [menu $stop.menu -tearoff 0 -fg black -postcommand "
            BalloonHelp::kill_balloon
    "]
    $menus(stop) add command -label $labelMenus(stopNow) -command {
        if {$getleftState(delayedDown)==1} Delay::DelayedCancelDelay
        ::Ventana::CancelDownloading $Ventana::window(top)
    }
    $menus(stop) add check -variable getleftOptions(stopPage)                \
            -onvalue 1 -offvalue 0 -label $labelMenus(stopPage)              \
            -command {
                if {$getleftState(autoDown)==1}    mainWin::SetAutoDownload
                if {$getleftState(delayedDown)==1} Delay::DelayedCancelDelay
            }
    $menus(stop) add check -variable getleftOptions(stopFile)                \
            -onvalue 1 -offvalue 0 -label $labelMenus(stopFile)              \
            -command {
                if {$getleftState(autoDown)==1}    mainWin::SetAutoDownload
                if {$getleftState(delayedDown)==1} Delay::DelayedCancelDelay
            }
    $menus(stop) add command -label $labelButtons(skip)                      \
            -command "::Ventana::CancelDownloading $Ventana::window(top) skip"

    set window(cancel) $stop
    set window(pause)  $pause
    set window(resume) $resume

    pack $marco     -ipady 5
    pack $botones   -fill both -expand true -side bottom -padx 7
    pack $marco_in  -padx 10 -side bottom
    pack $marco_url $marco_in $marco_dir $marco_fil $marco_que $marco_bar \
            -fill x
    pack $etq_url   $etq_url2  -side left -pady 2 -padx 2
    pack $etq_dir   $etq_dir2  -side left -pady 2 -padx 2
    pack $etq_file  $etq_file2 -side left -pady 2 -padx 2
    pack $etq_que   $etq_que2  -side left -pady 2 -padx 2
    pack $progressBar
    pack $stop $pause $config -side right -padx 3 -pady 5

    wm protocol $win WM_DELETE_WINDOW "::Ventana::CancelDownloading $win"

    return
}

###############################################################################
# ShowWindow
#   If needed, maps the downloading window, and sets its title bar
#
# Parameter
#   which: either 'head', 'down', 'process' or 'wait' depending on the action 
#          taking place.
###############################################################################
proc ShowWindow {which} {
    global   labelTitles labelButtons getleftState
    variable window
    variable action

    if {$::getleftState(downloading)==0} {
        if {$::DEBUG==1} {
            tk_messageBox -title Error -icon error -type ok         \
                    -message "Pringamos en ShowWindow: $which"
        }
        return
    }   

    if {![winfo exists .bajando]} {
        DownloadWindow
    } else {
        set state [wm state $window(top)]
        if {$state=="withdrawn"} {
            wm deiconify $window(top)
        }
    }

    switch -exact $which {
        head {
            wm title $window(top) $labelTitles(connect)
        }
        down {
            wm title $window(top) $labelTitles(download)
        }
        process {
            wm title $window(top) $labelTitles(process)
            set action process
        }
        wait {
            StartWaiting 10
        }
    }

    update

    return
}

###############################################################################
# FileDownloading
#   Creates a nice window to show info about how the downloading goes
#
# Parameters:
#    fichero: file in which the link will be saved.
#    enlace:  file url.
#    madre:   referer page for the link
###############################################################################
proc FileDownloading {fichero enlace madre} {
    global   getleftState directories
    global   labelTitles
    global   Ventana::Rizo::meta
    global   Ventana::Rizo::curlReport
    variable window
    variable file
    variable link
    variable mother
    variable action

    if {$getleftState(downloading)==0} {
        return
    }

    set file   $fichero
#    set link   $enlace
#    set mother $madre
    set action body

    ShowWindow down
    FileStrings $enlace

    set curlReport(end)   0
    set curlReport(long)  1
    set curlReport(pause) 0

    if {([file exists $fichero])&&($meta(versionServer)>=1.1)} {
        set size [file size $fichero]
        if {($size>$meta(totalBytes))&&($meta(totalBytes)!=0)} {
            file delete $fichero
        } elseif {($size==$meta(totalBytes))&&($meta(totalBytes)!=0)} {
            after 1000 "set ::Ventana::Rizo::curlReport(nextFile) 1"
            return
        }
        Ventana::Rizo::DataRequest $fichero $enlace $madre 1
    } else {
        Ventana::Rizo::DataRequest $fichero $enlace $madre
    }
    BarUpdate

    return
}

###############################################################################
# HeadDownloading
#    Gets the headers of the url from the server.
#
# Parameters:
#    enlace: url to download
#    madre:  referer page of the link
###############################################################################
proc HeadDownloading {enlace madre} {
    global siteUrl getleftState
    variable link
    variable mother
    variable action

    set link   $enlace
    set mother $madre
    set action head

    ShowWindow head
    FileStrings $enlace

    HideResumeButton

    if {[regexp {^ftp://} $enlace]} {
        set ::Ventana::Rizo::meta(content)       "nada"
        set ::Ventana::Rizo::meta(totalBytes)    0
        set ::Ventana::Rizo::meta(versionServer) 1.1
        set ::Ventana::Rizo::meta(relocate)      ""
        set ::Ventana::Rizo::curlError           0
        set ::Ventana::Rizo::errorMessage        ""
        return
    }

    set Ventana::Rizo::curlReport(end) 0
    Ventana::Rizo::HeadRequest $enlace $madre
    tkwait variable Ventana::Rizo::curlReport(end)
           
    return
}

###############################################################################
# PauseDownloading
#    Takes control when you click on the 'Pause' button.
#
# Parameter:
#    when: 'now' when we click on the 'Now' menu entry, usuful to pause
#    the downloading while a file is being process for links.
###############################################################################
proc PauseDownloading {{when ""}} {
    global labelButtons getleftOptions
    variable window

    if {$when=="now"} {
        set getleftOptions(pauseNow) 1
        set Ventana::Rizo::curlReport(pause) 1
    }
    set Ventana::Rizo::curlReport(pause) 1
    catch {Rizo::Lector stopNow}
    ShowResumeButton

    return
}

###############################################################################
# ResumeDownloading
#    Takes control when you click on the 'Resume' button
###############################################################################
proc ResumeDownloading {} {
    global labelButtons getleftOptions
    variable window
    variable file
    variable link
    variable mother
    variable action

    HideResumeButton

    if {($getleftOptions(pauseFile)==1)||($getleftOptions(pausePage)==1)} {
        set getleftOptions(pauseFile) 0
        set getleftOptions(pausePage) 0
        return
    }
    set getleftOptions(pauseNow)  0
    set Ventana::Rizo::curlReport(pause) 0

    if {$action=="body"} {
        Progreso::ProgressBarReset $window(bar)
        Ventana::Rizo::DataRequest "$file" "$link" "$mother" 1
        Ventana::BarUpdate
    } else {
        HeadDownloading "$link" "$mother"
    }
    return
}

###############################################################################
# StartWaiting
#    This procedure is invoked when a timeout has ocurred, to wait before
#    trying again.
#
# Parameter
#    seconds: The seconds yet to wait
###############################################################################
proc StartWaiting {seconds} {
    global   getleftState labelTitles
    variable window
    variable waitingPid

    if {$seconds==0} {
        InterruptWaiting
        return
    }

    wm title $window(top) "$labelTitles(waiting) $seconds"

    ShowResumeButton
    $window(resume) configure -command ::Ventana::InterruptWaiting

    set waitingPid [after 1000 ::Ventana::StartWaiting [expr $seconds -1]]

    return
}

###############################################################################
# InterruptWaiting
#    While waiting to retry after a timeout, if you click on resume
#    this procedure will be invoked.
###############################################################################
proc InterruptWaiting {} {
    global   getleftState
    variable window
    variable waitingPid

    after cancel $waitingPid

    HideResumeButton
    $window(resume) configure -command ::Ventana::ResumeDownloading

    set getleftState(waiting) 0

    return
}

###############################################################################
# BarUpdate
#    Updates the progress bar.
###############################################################################
proc BarUpdate {} {
    global Ventana::Rizo::curlReport
    global Ventana::Rizo::curlError
    global Ventana::Rizo::meta
    global labelDialogs downOptions getleftOptions
    variable file
    variable link
    variable mother
    variable window
    variable afterId

    if {[string match $curlReport(percentage) ""]} return
    Progreso::ProgressBarUpdate $window(bar) $curlReport(percentage)

    if {$getleftOptions(pauseNow)==1} {
        return
    }

    if {($curlReport(end)==0) && ($curlReport(stop)!=1) \
            && ($curlReport(pause)==0)} {
        set afterId [after 1000 Ventana::BarUpdate]
    } else {
        set curlReport(left) ""
        if {($::DEBUG==1)&&($curlError!=0)} {
            puts "Error $curlError en $file"
        }
        if {($curlError==18)||($curlError==28)} {
            getLog::FileLogEnter $labelDialogs(fileIncom) \
                    "$link" "$mother"                        
            if {($downOptions(resume)==1)&&($curlReport(stop)!=1)} {
                Progreso::ProgressBarReset $window(bar)
                if {$curlError==18} {
                    FileDownloading "$file" "$link" "$mother"
                } else {
                    incr curlReport(nextFile)
                }
                return
            }
        }
        if {$curlError==0} {
            Progreso::ProgressBarFull $window(bar)
            update
        }
        after 75
        Progreso::ProgressBarReset $window(bar)
        $window(dir)  configure -text ""
        $window(file) configure -text ""
        incr curlReport(nextFile)
    }
    return
}
}
