###############################################################################
###############################################################################
##                                  Delay.tcl
###############################################################################
###############################################################################
## This namespace is used to program delayed downloads into Getleft.
###############################################################################
###############################################################################
## (c) 2001-2003 Andrs Garca Garca. fandom@retemail.es
## You may distribute the contents of this file under the terms of the GPL v2
###############################################################################
###############################################################################

namespace eval Delay {

###############################################################################
# DelayedCancelDelay
#    Cancel the event that would start or/and the end of a delayed download.
###############################################################################
proc DelayedCancelDelay {} {
    variable delayedDownload
    global   getleftState

    catch {after cancel $delayedDownload(startEvent)}
    catch {after cancel $delayedDownload(endEvent)}

    set getleftState(delaySet)    0
    set getleftState(delayedDown) 0

    $mainWin::taskbar(delay) configure -relief flat

    return
}

###############################################################################
# DelayedStartDownload
#    Starts the download at the given time.
###############################################################################
proc DelayedStartDownload {} {
    variable delayedDownload
    variable window
    global getleftState

    if {$delayedDownload(noEnd)==0} {
        set clicksToEnd [expr ($delayedDownload(endTime)-[clock seconds])*1000]
        if {$clicksToEnd<0} {
            set clicksToEnd 0
        }
        set delayedDownload(endEvent) \
            [after $clicksToEnd Delay::DelayedEndDownload]
    }
    if {$delayedDownload(days)==1} {
        DelayedStartTime $delayedDownload(startTime)
    }
    set getleftState(delayedDown) 1
    mainWin::UrlListClearSel
    mainWin::SetAutoDownload

    return
}

###############################################################################
# DelayedEndDownload
#    Ends the download in the given time.
###############################################################################
proc DelayedEndDownload {} {
    variable delayedDownload
    global   getleftOptions

    switch $delayedDownload(how) {
        0 {
            ::Ventana::CancelDownloading .
        }
        1 {
            set getleftOptions(stopFile) 1
            mainWin::SetAutoDownload
        }
        2 {
            set getleftOptions(stopPage) 1
            mainWin::SetAutoDownload
        }
    }
    if {$delayedDownload(days)==0} {
        $mainWin::taskbar(delay) configure -relief flat
    }

    return
}

###############################################################################
# DelayedStartTime
#    Calculates the time the next download will begin.
#
# Parameter
#    The time in seconds from 01-01-1970 when the download will start.
###############################################################################
proc DelayedStartTime {startTime} {
    variable delayedDownload

    set now [clock seconds]
    if {$delayedDownload(days)==1} {
        set hour   [clock format $startTime -format "%H"]
        set minute [clock format $startTime -format "%M"]
        set startTime 9999999999
        foreach day [list monday tuesday  wednesday thursday \
                          friday saturday sunday] {
            if {$delayedDownload($day)==1} {
                set startTmp [clock scan "$day $hour:$minute"]
                if {$startTmp<$now} {
                    set startTmp [expr $startTmp+7*24*3600]
                }
                if {($startTmp<=$startTime)&&($startTmp>$now)} {
                    set startTime $startTmp
                }
            }
        }
    }
    set delayedDownload(startTime) $startTime
    DelayedSetStartTime

    return
}

###############################################################################
# DelayedSetStartTime
#    Given the time when to start, this procedure will create the event to
#    start the download at that time.
###############################################################################
proc DelayedSetStartTime {} {
    variable delayedDownload

    set clicksToStart [expr ($delayedDownload(startTime)-[clock seconds])*1000]
    # This may happen if we closed Getleft and we are running it again.
    if {$clicksToStart<0} {
        if {$delayedDownload(days)==1} {
            DelayedStartTime $delayedDownload(startTime)
        }
        return
    }        
    set delayedDownload(startEvent)  \
            [after $clicksToStart Delay::DelayedStartDownload]

    return 
}

###############################################################################
# DelayedEndTime
#    Calculates the time the next download will stop.
#
# Parametes
#    The hour and the minute of the time and the start time.
#
# Returns:
#    The time in seconds from 01-01-1970
###############################################################################
proc DelayedEndTime {hour minute startTime} {
    variable delayedDownload

    set endTime [clock scan "today $hour:$minute"]
    while {$startTime>=$endTime} {
        set endTime [expr $endTime + 3600*24]
    }
    return $endTime
}

###############################################################################
# DelayedCheckTime
#    Checks whether the given time makes sense or not.
#
# Parametes
#    The hour and the minute of the time to check.
#
# Returns:
#    '0' makes sense, '1' doesn't.
###############################################################################
proc DelayedCheckTime {hour minute} {

    if {(![regexp {^[0-9]+$} $hour])||(![regexp {^[0-9]+$} $minute])} {
        return 1
    }
    if {($hour<0)||($hour>23)||($minute<0)||($minute>59)} {
        return 1
    }

    return 0
}

###############################################################################
# DelayedActivateEndTime
#    Activates or deactivates the spinboxes with the time to finish downloading
###############################################################################
proc DelayedActivateEndTime {} {
    variable delayedDownloadTemp
    variable window

    if {$delayedDownloadTemp(noEnd)==1} {
        CuadroSpin::EnableSpin $window(endHour)   disable
        CuadroSpin::EnableSpin $window(endMinute) disable
    } else {
        CuadroSpin::EnableSpin $window(endHour)   normal
        CuadroSpin::EnableSpin $window(endMinute) normal
    }

    return
}

###############################################################################
# DelayedActivateDays
#    Activates or deactivates the 'days' buttons.
###############################################################################
proc DelayedActivateDays {} {
    variable delayedDownloadTemp
    variable dayButton

    if {$delayedDownloadTemp(days)==0} {
        $dayButton(monday)    configure -state disable
        $dayButton(tuesday)   configure -state disable
        $dayButton(wednesday) configure -state disable
        $dayButton(thursday)  configure -state disable
        $dayButton(friday)    configure -state disable
        $dayButton(saturday)  configure -state disable
        $dayButton(sunday)    configure -state disable
    } else {
        if {[$dayButton(monday) cget -state]!="normal"} {
            $dayButton(monday)    configure -state normal
            $dayButton(tuesday)   configure -state normal
            $dayButton(wednesday) configure -state normal
            $dayButton(thursday)  configure -state normal
            $dayButton(friday)    configure -state normal
            $dayButton(saturday)  configure -state normal
            $dayButton(sunday)    configure -state normal
        }
    }
    return
}

###############################################################################
# DelayedActivateStop
#    Activates or deactivates the 'days' buttons.
###############################################################################
proc DelayedActivateStop {} {
    variable delayedDownloadTemp
    variable stopButton

    if {$delayedDownloadTemp(noEnd)==0} {
        if {[$stopButton(inme) cget -state]!="normal"} {
            $stopButton(inme) configure -state normal
            $stopButton(file) configure -state normal
            $stopButton(page) configure -state normal
        }
    } else {
        $stopButton(inme) configure -state disable
        $stopButton(file) configure -state disable
        $stopButton(page) configure -state disable
    }
    return
}

###############################################################################
# DelayedDownloadWindow
#    Creates the dialog to define when to start a download.
###############################################################################
proc DelayedDownloadWindow {} {
    global labelDialogs labelTitles labelFrames getleftState getleftOptions
    global indexButtons indexDialogs
    variable delayedDownloadTemp
    variable dayButton
    variable stopButton
    variable window

    if {[winfo exists .delayedDown]} {
        raise .delayedDown .
        return
    }

    set win [toplevel .delayedDown]
    wm title $win $labelTitles(delayedDown)
    wm resizable $win 0 0

    set coord(x) [winfo rootx .]
    set coord(y) [winfo rooty .]
    wm geometry $win +[expr {$coord(x)+100}]+[expr {$coord(y)+15}]

    set extFrame [frame $win.extFrame]
    set intFrame [frame $extFrame.intFrame]

    set timeFrame [fl::FrameLabel $intFrame.time -bd 2 -relief groove           \
            -textvariable labelFrames(time)]
    set startLabel  [label $timeFrame.startLabel -textvariable labelDialogs(begin)]
    set startHour   [CuadroSpin::CuadroSpin $timeFrame.startHour                \
            -default 0 -min 0 -max 23 -width 3                                  \
            -bg $getleftOptions(bg) -fg $getleftOptions(fg)]
    set startMinute [CuadroSpin::CuadroSpin $timeFrame.startMinute              \
            -default 0 -min 0 -max 59 -width 3 -incr 5                          \
            -bg $getleftOptions(bg) -fg $getleftOptions(fg)]
    set endLabel    [label $timeFrame.endLabel -textvariable labelDialogs(end)]
    set endHour     [CuadroSpin::CuadroSpin $timeFrame.endHour                  \
            -default 0 -min 0 -max 23 -width 3 -disbg $getleftOptions(disBg)    \
            -bg $getleftOptions(bg) -fg $getleftOptions(fg)                     \
            -disfg $getleftOptions(disFg)]
    set endMinute   [CuadroSpin::CuadroSpin $timeFrame.endMinute                \
            -default 0 -min 0 -max 59 -width 3 -incr 5                          \
            -disbg $getleftOptions(disBg) -disfg $getleftOptions(disFg)         \
            -bg $getleftOptions(bg) -fg $getleftOptions(fg)]
    set noEnd [underButton::UnderButton $timeFrame.noEnd -buttontype checkbutton\
            -textvariable labelDialogs(noLimit) -under $indexDialogs(noLimit)   \
            -variable Delay::delayedDownloadTemp(noEnd) -command "
                Delay::DelayedActivateEndTime
                Delay::DelayedActivateStop
            "]
    if {$delayedDownloadTemp(noEnd)==""} {
        set delayedDownloadTemp(noEnd) 1
    }
    if {![catch {clock format $delayedDownloadTemp(startTime)}]} {
        set CuadroSpin::CuadroSpins($startHour)   \
                [clock format $delayedDownloadTemp(startTime) -format "%H"]
        set CuadroSpin::CuadroSpins($startMinute) \
                [clock format $delayedDownloadTemp(startTime) -format "%M"]
    }
    if {![catch {clock format $delayedDownloadTemp(endTime)}]} {
        set CuadroSpin::CuadroSpins($endHour)                                  \
                [clock format $delayedDownloadTemp(endTime) -format "%H"]
        set CuadroSpin::CuadroSpins($endMinute)                                \
                [clock format $delayedDownloadTemp(endTime) -format "%M"]
    }

    set window(win)         $win
    set window(startHour)   $startHour
    set window(startMinute) $startMinute
    set window(endHour)     $endHour
    set window(endMinute)   $endMinute

    DelayedActivateEndTime

    set howFrame  [fl::FrameLabel $intFrame.how  -bd 2 -relief groove          \
            -textvariable labelFrames(stop)]
    set stopButton(inme) [underButton::UnderButton $howFrame.inme              \
            -textvariable labelDialogs(inme)      -buttontype radiobutton      \
            -variable Delay::delayedDownloadTemp(how) -value 0                 \
            -under $indexDialogs(inme)] 
    set stopButton(file) [underButton::UnderButton $howFrame.file              \
            -textvariable labelDialogs(afterFile) -buttontype radiobutton      \
            -variable Delay::delayedDownloadTemp(how) -value 1                 \
            -under $indexDialogs(afterFile)]
    set stopButton(page) [underButton::UnderButton $howFrame.page              \
            -textvariable labelDialogs(afterPage) -buttontype radiobutton      \
            -variable Delay::delayedDownloadTemp(how) -value 2                 \
            -under $indexDialogs(afterPage)]
    if {$delayedDownloadTemp(how)==""} {
        set delayedDownloadTemp(how) 1
    }
    DelayedActivateStop

    set daysFrame [fl::FrameLabel $intFrame.days -bd 2 -relief groove          \
            -textvariable labelFrames(days)]
    set today [underButton::UnderButton $daysFrame.today -buttontype radiobutton\
            -textvariable labelDialogs(onlyToday)                              \
            -variable Delay::delayedDownloadTemp(days) -value 0                \
            -buttontype radiobutton -under $indexDialogs(onlyToday)            \
            -command Delay::DelayedActivateDays]
    set on    [underButton::UnderButton $daysFrame.on -buttontype radiobutton  \
            -textvariable labelDialogs(on) -under $indexDialogs(on)            \
            -variable Delay::delayedDownloadTemp(days) -value 1                \
            -command Delay::DelayedActivateDays]
    if {$delayedDownloadTemp(days)==""} {
        set delayedDownloadTemp(days) 0
    }

    set dayButton(monday)    [underButton::UnderButton $daysFrame.monday       \
            -textvariable labelDialogs(monday)    -buttontype checkbutton      \
            -variable Delay::delayedDownloadTemp(monday)                       \
            -under $indexDialogs(monday)]
    set dayButton(tuesday)   [underButton::UnderButton $daysFrame.tuesday      \
            -textvariable labelDialogs(tuesday)   -buttontype checkbutton      \
            -variable Delay::delayedDownloadTemp(tuesday)                      \
            -under $indexDialogs(tuesday)]
    set dayButton(wednesday) [underButton::UnderButton $daysFrame.wednesday    \
            -textvariable labelDialogs(wednesday) -buttontype checkbutton      \
            -variable Delay::delayedDownloadTemp(wednesday)                    \
            -under $indexDialogs(wednesday)]
    set dayButton(thursday)  [underButton::UnderButton $daysFrame.thursday     \
            -textvariable labelDialogs(thursday)  -buttontype checkbutton      \
            -variable Delay::delayedDownloadTemp(thursday)                     \
            -under $indexDialogs(thursday)]
    set dayButton(friday)    [underButton::UnderButton $daysFrame.friday       \
            -textvariable labelDialogs(friday)    -buttontype checkbutton      \
            -variable Delay::delayedDownloadTemp(friday)                       \
            -under $indexDialogs(friday)]
    set dayButton(saturday)  [underButton::UnderButton $daysFrame.saturday     \
            -textvariable labelDialogs(saturday)  -buttontype checkbutton      \
            -variable Delay::delayedDownloadTemp(saturday)                     \
            -under $indexDialogs(saturday)]
    set dayButton(sunday)    [underButton::UnderButton $daysFrame.sunday       \
            -textvariable labelDialogs(sunday)    -buttontype checkbutton      \
            -variable Delay::delayedDownloadTemp(sunday)                       \
            -under $indexDialogs(sunday)]
    DelayedActivateDays

    set buttonFrame [frame $win.buttonFrame]
    set accept [underButton::UnderButton $buttonFrame.accept                   \
        -buttontype button -textvariable labelButtons(ok)                      \
        -width 8 -command "Delay::DelayedDownloadControl 1"                    \
        -under $indexButtons(ok)]
    set cancel [underButton::UnderButton $buttonFrame.cancel                   \
        -buttontype button -textvariable labelButtons(cancel)                  \
        -width 8 -command "Delay::DelayedDownloadControl 0"                    \
        -under $indexButtons(cancel)]
    set clear  [underButton::UnderButton $buttonFrame.clear                    \
        -buttontype button -textvariable labelButtons(clear)                   \
        -width 8 -command "Delay::DelayedDownloadControl 2"                    \
        -under $indexButtons(clear)]

    pack $extFrame -padx 5
    pack $intFrame 
    grid $timeFrame $howFrame     -sticky news -ipadx 7 -ipady 7 -padx 5 -pady 10
    grid $daysFrame -columnspan 2 -sticky news -ipadx 7 -ipady 7 -padx 5

    grid $startLabel $startHour $startMinute -padx 5 -pady 3 -sticky w
    grid $endLabel   $endHour   $endMinute   -padx 5 -pady 3 -sticky w
    grid $noEnd -columnspan 3

    grid $stopButton(inme) -sticky w
    grid $stopButton(file) -sticky w
    grid $stopButton(page) -sticky w

    grid $today $on  -sticky w
    grid $dayButton(monday)    $dayButton(tuesday)                     -sticky w
    grid $dayButton(wednesday) $dayButton(thursday) $dayButton(friday) -sticky w
    grid $dayButton(saturday)  -row 3 -column 1 -sticky w
    grid $dayButton(sunday)    -row 3 -column 2 -sticky w

    pack $buttonFrame           -padx 7 -pady 5   -side right
    pack $cancel $accept $clear -padx 3 -side right

    bind $win <Escape> "$cancel invoke"

    if {$getleftState(delaySet)==0} {
        $clear configure -state disable
    } else {
        $clear configure -state normal
    }

    return
}

###############################################################################
# DelayedDownloadControl
#    Checks that the data entered in the dialog is consistent.
#
# Parameter
#    '1' if the user accept the data, '0' if the dialog was cancelled.
###############################################################################
proc DelayedDownloadControl {done} {
    global labelTitles labelMessages getleftState
    variable window
    variable delayedDownload
    variable delayedDownloadTemp

    if {$done==1} {
        set startHour   $CuadroSpin::CuadroSpins($window(startHour))
        set startMinute $CuadroSpin::CuadroSpins($window(startMinute))

        if {[DelayedCheckTime $startHour $startMinute]==1} {
            tk_messageBox -title $labelTitles(error) -parent $window(win)     \
                    -message $labelMessages(wrongStart)     -type ok 
            return
        }
        if {$delayedDownloadTemp(noEnd)==0} {
            set endHour   $CuadroSpin::CuadroSpins($window(endHour))
            set endMinute $CuadroSpin::CuadroSpins($window(endMinute))

            if {[DelayedCheckTime $endHour $endMinute]==1} {
                tk_messageBox -title $labelTitles(error) -parent $window(win) \
                        -message $labelMessages(wrongStart) -type ok 
                return
            }
        }
        if {$delayedDownloadTemp(days)==1} {
            if {($delayedDownloadTemp(monday)==0)      &&($delayedDownloadTemp(tuesday)==0) \
                 &&($delayedDownloadTemp(wednesday)==0)&&($delayedDownloadTemp(thursday)==0)\
                 &&($delayedDownloadTemp(friday)==0)   &&($delayedDownloadTemp(saturday)==0)\
                 &&($delayedDownloadTemp(sunday)==0)} {
                tk_messageBox -title $labelTitles(error)   -parent $window(win)             \
                        -message $labelMessages(whichDays) -type ok 
                return
            }
        }
    }

    destroy $window(win)

    if {$done==0} return

    if {$done==2} {
        DelayedCancelDelay
        return
    }
    DelayedCancelDelay
    array set delayedDownload [array get delayedDownloadTemp]
    set startTime [clock scan "today $startHour:$startMinute"]
    set now [clock seconds]
    if {$startTime<$now} {
        set startTime [expr $startTime+24*3600]
    }
    DelayedStartTime $startTime
    if {$delayedDownload(noEnd)==0} {
        set delayedDownload(endTime)         \
                [DelayedEndTime $endHour $endMinute $delayedDownload(startTime)]
    }

    $mainWin::taskbar(delay) configure -relief sunken
    set getleftState(delaySet) 1

    return
}

###############################################################################
# DelayedDownload
#    Takes care of all the trivia of defining a delayed download.
###############################################################################
proc DelayedDownload {} {
    variable delayedDownloadTemp
    variable delayedDownload

    catch {unset delayedDownloadTemp}
    array set delayedDownloadTemp [array get delayedDownload]

    DelayedDownloadWindow

    return
}

}
