################################################################################
################################################################################
##                             getLog.tcl
################################################################################
################################################################################
## Defines Getleft's log windows.
################################################################################
################################################################################
## (c) 2001-2003 Andrs Garca Garca. fandom@retemail.es
## You may distribute the contents of this file under the terms of the GPL v2
################################################################################
################################################################################

namespace eval getLog {

################################################################################
# DefineTaskbarButtons
#    Defines the buttons in the taskbar.
#
# Parameters
#    The frame for the buttons.
################################################################################
proc DefineTaskbarButtons {frame} {
    global   labelDialogs labelMenus
    variable siteTaskbar
    variable siteLog

    set siteTaskbar(start)  [button $frame.start   -image $iconNS::iconImages(start) \
            -relief flat -bd 1 -command {
                getLog::UrlLogDownload [getLog::UrlLogSelectedUrl]
            }]
    ::BalloonHelp::set_balloon $siteTaskbar(start)  $labelDialogs(start)

    set siteTaskbar(update) [button $frame.update -image $iconNS::iconImages(update) \
            -relief flat -bd 1 -command {
                getLog::UrlLogUpdate [getLog::UrlLogSelectedUrl]
            }]
    ::BalloonHelp::set_balloon $siteTaskbar(update) $labelDialogs(update)

    set siteTaskbar(queue)  [button $frame.queue -image $iconNS::iconImages(queue)   \
            -relief flat -bd 1 -command {
                getLog::UrlLogQueue [getLog::UrlLogSelectedUrl]    
            }]
    ::BalloonHelp::set_balloon $siteTaskbar(queue)  $labelDialogs(queue)

    set siteTaskbar(conf)  [button $frame.conf  -image $iconNS::iconImages(conf) \
            -relief flat -bd 1 -command {
                getLog::UrlLogChangeOptions [getLog::UrlLogSelectedUrl]    
            }]
    ::BalloonHelp::set_balloon $siteTaskbar(conf)   $labelDialogs(options)

    set siteTaskbar(exit) [button $frame.exit  -image $iconNS::iconImages(exit) \
            -relief flat -bd 1 -command "wm withdraw $siteLog(toplevel)"]
    ::BalloonHelp::set_balloon $siteTaskbar(exit)   $labelDialogs(exit)

    foreach {name path} [array get siteTaskbar] {
        bindtags $path "[bindtags $path] logTaskIcon"
    }
    bind logTaskIcon <<ButtonRaise>> {
        %W configure -relief raised
    }
    bind logTaskIcon <<ButtonFlat>> {
        %W configure -relief flat
    }

    pack $siteTaskbar(start) $siteTaskbar(update) $siteTaskbar(queue) $siteTaskbar(conf) \
            -side left
    pack $siteTaskbar(exit) -side right

    return
}

################################################################################
# ActivateSiteTaskbarButtons
#    Activates or disables the buttons in the taskbar depending on whether
#    there is a row selected.
################################################################################
proc ActivateSiteTaskbarButtons {} {
    variable siteTaskbar
    variable siteLog

    set anchor [$siteLog(listbox) curselection]
    if {$anchor==""} {
        $siteTaskbar(start)  configure -state disabled
        $siteTaskbar(update) configure -state disabled
        $siteTaskbar(queue)  configure -state disabled
        $siteTaskbar(conf)   configure -state disabled
    } else {
        $siteTaskbar(start)  configure -state normal
        $siteTaskbar(update) configure -state normal
        $siteTaskbar(queue)  configure -state normal
        $siteTaskbar(conf)   configure -state normal
    }
    return
}

################################################################################
# UrlLogWrite
#    Writes the file with the urls in the log, the file will be saved in the
#    dirGetleft(conf) directory with the name getLog.log.
################################################################################
proc UrlLogWrite {} {
    variable urlLog
    global dirGetleft tcl_platform

    set urlLogFile [file join $dirGetleft(conf) getLog.log]
    if {(![info exists urlLog(urlList)])||([llength $urlLog(urlList)]==0)} {
        catch {file delete $urlLogFile}
        return
    }
    if {$tcl_platform(platform)=="windows"} {
        if {[file exists $urlLogFile]} {
            # It seems that Win2K doesn't like to read hidden files,
            # so I unhide it for a couple of seconds.
            file attributes $urlLogFile -hidden 0
        }
    } 

    set handle [open "$urlLogFile" w]

    puts $handle "# Log of the sites downloaded by Getleft."
    puts $handle "# This file is generated automatically, please do not edit."
    puts $handle "\n\n\n"

    puts $handle "set urlLog(urlList) [list $urlLog(urlList)]"
    puts $handle "\n\n"
    foreach url $urlLog(urlList) {
        regsub -all {\s} $url {\\&} urlTmp
        catch {
            puts $handle "set urlLog($urlTmp,localDir) \"$urlLog($url,localDir)\""
            puts $handle "set urlLog($urlTmp,date)     $urlLog($url,date)"
            puts $handle "set urlLog($urlTmp,options)  [list $urlLog($url,options)]"
            puts $handle "set urlLog($urlTmp,keep)     $urlLog($url,keep)"
#           puts $handle "set urlLog($urlTmp,encoding) $urlLog($url,encoding)"
        }
    }
    puts $handle "\n\n"
    close $handle

    if {$tcl_platform(platform)=="windows"} {
        file attributes $urlLogFile -hidden 1
    } else {
        file attributes $urlLogFile -permissions 0600
    }

    return
}

################################################################################
# UrlLogRead
#    Reads the file with the urls in the log, the file will be saved in the
#    dirGetleft(conf) directory with the name getLog.log.
#
# Side effect
#    The namespace variable urlLog will be set in all its glory.
################################################################################
proc UrlLogRead {} {
    variable urlLog
    global dirGetleft tcl_platform

    set urlLogFile [file join $dirGetleft(conf) getLog.log]
    if {[file exists $urlLogFile]} {
        if {$tcl_platform(platform)=="windows"} {
            file attributes $urlLogFile -hidden 0
        }
        source $urlLogFile
        if {$tcl_platform(platform)=="windows"} {
            file attributes $urlLogFile -hidden 1
        }
    } else {
        set urlLog(urlList) ""
    }

    return
}

################################################################################
# UrlLogGetIndex
#   When the user clicks in the url list, this procedure takes care of selecting
#   the nearest entry and returns its index.
#
# Parameters
#   win: window where you clicked.
#   y: the coordinate where you did it
#
# Returns
#   The index of the url, or -1 if the is none to return
################################################################################
proc UrlLogGetIndex {win y} {
    
    set ::tablelist::win [winfo parent $win]
    set ::tablelist::y [expr "$y + [winfo y $win]"]
    if {[winfo exists $::tablelist::win]} {
        set index [$::tablelist::win nearest $::tablelist::y]
        if {$index!=-1} {
            if {[catch {tkListboxBeginSelect $::tablelist::win $index}]} {
                tk::ListboxBeginSelect $::tablelist::win $index
            }
        }
        return $index
    }
    return -1
}

################################################################################
# UrlLogUrlIndex
#   Given an Url this procedure returns its index in the listbox.
#
# Parameters
#   url: The url to search for.
#
# Returns
#   The index of the url, or -1 if the is none to return
################################################################################
proc UrlLogUrlIndex {url} {
    variable siteLog
    variable urlLog

    if {(![info exists siteLog(listbox)])||(![winfo exists $siteLog(listbox)])} {
        return -1
    }

    set rowNumber [$siteLog(listbox) size]
    for {set i 0} {$i<$rowNumber} {incr i} {
        set site [$siteLog(listbox) cellcget $i,0 -text]
        if {[string match $url $site]} {
            return $i
        }
    }

    return -1
}

################################################################################
# UrlLogEnter
#    Enters one Url and its data into the urlLog. This procedure is invoked
#    after the site has been downloaded.
#
# Parameters
#    url: The url to enter.
#    dir: The directory where it has been saved.
#    options: A list with the options for downloading the site.
#
# Side effect: To begin with the keep bit will be set to '0'.
################################################################################
proc UrlLogEnter {url dir options} {
    variable urlLog
    variable siteLog

    set oldIndex [lsearch $urlLog(urlList) $url]
    if {$oldIndex==-1} {
        set urlNumber [llength $urlLog(urlList)]
        if {$urlNumber>=100} {
            for {set i 0} {$i<$urlNumber} {incr i} {
                set urlTmp [lindex $urlLog(urlList) $i]
                if {(![info exists urlLog($urlTmp,keep)])||($urlLog($urlTmp,keep)==0)} {
                    UrlLogRemove $urlTmp
                    break
                }
            }
            if {$i==$urlNumber} {
                return
            }
        }
        lappend urlLog(urlList) $url
    }

    set urlLog($url,localDir) "$dir"
    set urlLog($url,date)     [clock seconds]
    set urlLog($url,options)  $options
    if {![info exists urlLog($url,keep)]} {
        set urlLog($url,keep) 0
    }

    if {([info exists siteLog])&&([winfo exists $siteLog(listbox)])} {
        set oldIndex [UrlLogUrlIndex $url]
        $siteLog(listbox) delete $oldIndex
        if {$oldIndex==-1} {
            GetLogWindowRow $url $dir
        } else {
            GetLogWindowRow $url $dir $oldIndex
        }
    }

    return
}

################################################################################
# UrlLogSelectedUrl
#   When the user clicks on the icons, this procedure is used to know the url
#   of the row that was selected at the time.
#
# Returns
#   The url or an empty string if no row is selected
################################################################################
proc UrlLogSelectedUrl {} {
    variable siteLog

    set anchor [$siteLog(listbox) curselection]
    if {[catch {$siteLog(listbox) cellcget $anchor,0 -text} url]} {
        return
    }

    return $url
}

################################################################################
# UrlLogDownload
#   Starts downloading one of the sites in the log.
#
# Parameters
#   url: The site we are talking about.
################################################################################
proc UrlLogDownload {url} {
    variable urlLog
    global   getleftState

    if {$url==""} return

    PutUrlInQueue $url "$urlLog($url,localDir)" "$urlLog($url,options)"

    if {$getleftState(downloading)==0} {
        AutomaticDownload $url                
    }

    return
}

################################################################################
# UrlLogUpdate
#   Starts updating one of the sites in the log, actually it only makes sure
#   the 'update' option is set before passing it the 'UrlLogDownload'
#
# Parameters
#   url: The site we are talking about.
################################################################################
proc UrlLogUpdate {url} {
    variable urlLog

    if {$url==""} return

    set optionsTmp $urlLog($url,options)

    regsub {update [0-1]} "$urlLog($url,options)" \
           {update 1}       urlLog($url,options)

    UrlLogDownload $url                

    set urlLog($url,options) $optionsTmp

    return
}

################################################################################
# UrlLogQueue
#   Puts the url and its data into Getleft's url queue for future downloading.
#
# Parameters
#   url: The site we are talking about.
################################################################################
proc UrlLogQueue {url} {
    variable urlLog

    if {$url==""} return

    PutUrlInQueue $url "$urlLog($url,localDir)" "$urlLog($url,options)"

    return
}

################################################################################
# UrlLogChangeOptions
#   Changes the options for downloading an URL in the urlList.
#
# Parameters
#   url: The site we are talking about.
################################################################################
proc UrlLogChangeOptions {url} {
    variable urlLog

    if {$url==""} return

    array set ::downOptionsTemp $urlLog($url,options)
    set done [::Opciones::ChooseOptions ::downOptionsTemp]
    if {$done==1} {
        set urlLog($url,options) [array get ::downOptionsTemp]
    }
    return
}

################################################################################
# UrlLogChangeDir
#   Changes the directory to store the download.
#
# Parameters
#   url: The site we are talking about.
################################################################################
proc UrlLogChangeDir {url} {
    variable urlLog
    variable siteLog

    set baseDir [Dialogos::SelectDirectory [file dirname $urlLog($url,localDir)]]
    if {$baseDir!=""} {
        set urlLog($url,localDir) $baseDir
        set index [UrlLogUrlIndex $url]
        $siteLog(listbox) cellconfigure $index,1 -text [file nativename $baseDir]
    }

    return
}

################################################################################
# UrlLogLaunchLocalCopy
#    If the local copy of the site is still present, this will launch the 
#    browser and open it. 
#
# Parameters
#    url: The downloaded url.
#    dir: The directory in which the site was saved.
################################################################################
proc UrlLogLaunchLocalCopy {url dir} {
    global labelTitles labelMessages
    variable siteLog

    set localFile [Commands::UrlToFile $url $dir]

    if {![file exists $localFile]} {
        tk_messageBox -type ok -icon error -message $labelMessages(noFile) \
                -title $labelTitles(error) -parent $siteLog(toplevel)
        return
    }
    if {[file isdirectory $localFile]} {
        set tmpFileI  [file join $localFile index.html]
        set tmpFileII [file join $localFile index.htm]
        if {[file exists $tmpFileI]} {
            set localFile $tmpFileI
        } elseif {[file exists $tmpFileII]} {
            set localFile $tmpFileII
        }
    }
    Ayuda::InvokeBrowser $localFile $getLog::siteLog(toplevel)

    return
}

################################################################################
# UrlLogRemove
#    Removes one Url from the log 
#
# Parameters
#    url: The url to remove.
################################################################################
proc UrlLogRemove {url} {
    variable urlLog
    variable siteLog

    set index [lsearch $urlLog(urlList) $url]
    if {$index==-1} {
        return
    }

    set urlLog(urlList) [lreplace $urlLog(urlList) $index $index]
    catch {unset urlLog($url,locaDir)}
    catch {unset urlLog($url,date)}
    catch {unset urlLog($url,options)}
    catch {unset urlLog($url,keep)}

    set index [UrlLogUrlIndex $url]
    if {$index==-1} {
        return
    }
    $siteLog(listbox) delete $index
    ActivateSiteTaskbarButtons

    return
}

################################################################################
# UrlLogRightClick
#   This procedure is invoked when the user right-clicks in the multicolumn
#   list with the sites, it makes a context menu pop up with the
#   appropiate options
#
# Parameters
#   win: window where you right-clicked.
#   y,x: the coordinates where you did it for the tablelist widget.
#   X,Y: and for the pop up menu.
################################################################################
proc UrlLogRightClick {win x y X Y} {
    variable siteLog
    variable siteMenus
    variable urlLog

    set index [UrlLogGetIndex $win $y]
    if {$index==-1} return

    ActivateSiteTaskbarButtons

    set url [$siteLog(listbox) cellcget $index,0 -text]
    set dir [$siteLog(listbox) cellcget $index,1 -text]

    $siteMenus(rightClick) entryconfigure 0                                  \
            -command "getLog::UrlLogDownload [list $url]"
    $siteMenus(rightClick) entryconfigure 1                                  \
            -command "getLog::UrlLogUpdate   [list $url]"
    $siteMenus(rightClick) entryconfigure 3                                  \
            -command "getLog::UrlLogQueue    [list $url]"
    $siteMenus(rightClick) entryconfigure 5                                  \
            -command "getLog::UrlLogChangeOptions [list $url]"
    $siteMenus(rightClick) entryconfigure 6                                  \
            -command "getLog::UrlLogChangeDir [list $url]"
    $siteMenus(rightClick) entryconfigure 7                                  \
            -variable getLog::urlLog($url,keep)
    $siteMenus(rightClick) entryconfigure 9                                  \
            -command "clipboard clear
                      clipboard append [list $url]"
    $siteMenus(rightClick) entryconfigure 11                                 \
            -command "Ayuda::InvokeBrowser $url $getLog::siteLog(toplevel)"
    $siteMenus(rightClick) entryconfigure 12                                 \
            -command "getLog::UrlLogLaunchLocalCopy [list $url] [list $dir]"
    $siteMenus(rightClick) entryconfigure 14                                 \
            -command "getLog::UrlLogRemove [list $url]"

    tk_popup $siteMenus(rightClick) $X $Y

    return
}

################################################################################
# UrlLogDoubleClick
#   This procedure is invoked when the user double-clicks in the multicolumn
#   list with the sites, it makes a context menu pop up with the
#   appropiate options
#
# Parameters
#   win: window where you right-clicked.
#   y,x: the coordinates where you did it for the tablelist widget.
#   X,Y: and for the pop up menu.
################################################################################
proc UrlLogDoubleClick {win x y X Y} {
    variable siteLog
    variable siteMenus
    variable urlLog

    set index [UrlLogGetIndex $win $y]
    if {$index==-1} return

    ActivateSiteTaskbarButtons

    set url [$siteLog(listbox) cellcget $index,0 -text]

    $siteMenus(doubleClick) entryconfigure 0                         \
            -command "getLog::UrlLogDownload [list $url]"
    $siteMenus(doubleClick) entryconfigure 1                         \
            -command "getLog::UrlLogUpdate   [list $url]"

    tk_popup $siteMenus(doubleClick) $X $Y

    return
}

###############################################################################
# GetLogSortUrl
#    This procedure is used to sort the list depending on the url.
#
# Parameter:
#    firstUrl, secondUrl: The urls to compare
#
# Returns:
#     '-1', '0' or '1' if the first url is 'smaller' than, equal to or 
#     'greater' than the second.
###############################################################################
proc GetLogSortUrl {firstUrl secondUrl} {
    global labelMonths

    regexp {(?:://)(.*)} $firstUrl  nada firstUrl
    regexp {(?:://)(.*)} $secondUrl nada secondUrl

    set firstUrl  [string tolower $firstUrl]
    set secondUrl [string tolower $secondUrl]

    if {$firstUrl<$secondUrl} {
        return -1
    } elseif {$firstUrl==$secondUrl} {
        return 0
    }
    return 1
}

###############################################################################
# GetLogCanonicalDate
#    This procedure changes the given date to seconds
#
# Parameter:
#    size: The date in human format
#
# Returns:
#     The date in seconds.
###############################################################################
proc GetLogCanonicalDate {date} {
    global labelMonths

    regexp {([^-]*)(?:-)([^-]*)(?:-)([^-]*)(?:--)?(.*)} \
            $date  nada day  month  year  hour

    if {![info exists hour]} {
        set date [clock scan $labelMonths($month)/$day/$year]
    } else {
        set date [clock scan "$labelMonths($month)/$day/$year $hour"]
    }

    return $date
}

###############################################################################
# GetLogSortDate
#    This procedure is used to sort the list depending on the downloading date.
#
# Parameter:
#    firstDate, secondDate: The dates to compare
#
# Returns:
#     '-1', '0' or '1' if the first date is earlier than, equal to or later
#     than the second.
###############################################################################
proc GetLogSortDate {firstDate secondDate} {

    set first  [GetLogCanonicalDate $firstDate]
    set second [GetLogCanonicalDate $secondDate]

    if {$first<$second} {
        return -1
    } elseif {$first==$second} {
        return 0
    }
    return 1
}

###############################################################################
# GetLogCanonicalSize
#    This procedure changes the given size to bytes
#
# Parameter:
#    size: The size in human format (32,45K , 8.23M, ...)
#
# Returns:
#     The size in bytes.
###############################################################################
proc GetLogCanonicalSize {size} {
    global labelDialogs

    regsub "$labelDialogs(decimal)" $size  {.} size

    if {[regexp {(.*)(K)} $size nada size]} {
        set size [expr $size * 1024]
    } elseif {[regexp {(.*)(M)} $size nada size]} {
        set size [expr $size * 1024 * 1024]
    } else {
        regexp {(.*)(B)} $size nada size
    }

    return $size
}

###############################################################################
# GetLogSortSize
#    This procedure is used to sort the list depending on the size of the
#    files.
#
# Parameter:
#    firstSize, secondSize The sizes to compare
#
# Returns:
#     '-1', '0' or '1' if the first size is smaller than, equal to or bigger
#     than the second.
###############################################################################
proc GetLogSortSize {firstSize secondSize} {

    set firstSize  [GetLogCanonicalSize $firstSize]
    set secondSize [GetLogCanonicalSize $secondSize]

    if {$firstSize<$secondSize} {
        return -1
    } elseif {$firstSize==$secondSize} {
        return 0
    }
    return 1
}

###############################################################################
# GetLogWindowRow
#    Puts one row in the log window.
###############################################################################
proc GetLogWindowRow {url dir {index end}} {
    global labelMonths labelDialogs
    variable siteLog
    variable urlLog

    set month $labelMonths([clock format $urlLog($url,date) -format %m])

    # This is a workaround, in Tcl8.4a3 if I write '-format "%d-$month-%Y"'
    # unicode characters in '$month' get mangled.
    set date0 [clock format $urlLog($url,date) -format "%d"]
    set date1 [clock format $urlLog($url,date) -format "%Y"]
    set date  "$date0-$month-$date1"

    set tmp [$siteLog(listbox) yview]
    $siteLog(listbox) insert $index [list $url [file nativename $dir] $date]
    if {([lindex $tmp 1]==1)||([lindex $tmp 0]==0)} {
        $siteLog(listbox) yview  $index
    }

    return
}

###############################################################################
# GetLogWindowPopulate
#    Puts the urls in the log in the listbox widget.
###############################################################################
proc GetLogWindowPopulate {} {
    variable urlLog
    variable siteLog

    foreach url $urlLog(urlList) {
        catch {GetLogWindowRow $url $urlLog($url,localDir)}
    }
    $siteLog(listbox) yview  0
    return
}

###############################################################################
# GetLogWindow
#    Creates the toplevel window with the log in it.
###############################################################################
proc GetLogWindow {} {
    global dirGetleft labelDialogs labelTitles labelMenus getleftOptions
    variable siteLog
    variable siteMenus

    if {[winfo exists .getLogWindow]} {
        set windowState [wm state .getLogWindow]
        if {$windowState=="withdrawn"} {
            wm deiconify .getLogWindow
        }
        raise $siteLog(toplevel) .
        focus $siteLog(listbox)
        return
    }

    set coord(x) [winfo rootx .]
    set coord(y) [winfo rooty .]

    set win [toplevel .getLogWindow]
    wm title $win $labelTitles(getLog)
    wm geometry  $win +[expr {$coord(x)+100}]+[expr {$coord(y)+15}]

    set extFrame [frame $win.extFrame]
    set intFrame [frame $extFrame.intFrame]

    set taskbarFrame [frame $intFrame.taskbarFrame]
    set logFrame     [frame $intFrame.logFrame -bd 2 -relief sunken]

    set scrollbarY [scrollbar $logFrame.scrolly -orient vertical   \
            -command  [list $logFrame.listbox yview]]
    set scrollbarX [scrollbar $logFrame.scrollx -orient horizontal \
            -command  [list $logFrame.listbox xview]]

    set listbox [::tablelist::tablelist $logFrame.listbox       \
            -bd 0 -height 10 -width 90 -stretch all             \
            -fg $getleftOptions(fg) -bg $getleftOptions(bg)     \
            -columns [list 0 $labelDialogs(url)      left       \
                           0 $labelDialogs(localDir) left       \
                           0 $labelDialogs(date)     left]      \
	        -showarrow 1 -labelcommand tablelist::sortByColumn  \
            -xscrollcommand "$scrollbarX set"                   \
            -yscrollcommand "$scrollbarY set"                   \
            -stripebg #eff9fa -exportselection 0]

    $listbox columnconfigure 0 -sortmode command
    $listbox columnconfigure 0 -sortcommand ::getLog::GetLogSortUrl
    $listbox columnconfigure 1 -sortmode command
    $listbox columnconfigure 1 -sortcommand ::getLog::GetLogSortUrl
    $listbox columnconfigure 2 -sortmode command
    $listbox columnconfigure 2 -sortcommand ::getLog::GetLogSortDate

    set siteLog(toplevel) $win
    set siteLog(listbox)  $listbox

    DefineTaskbarButtons $taskbarFrame

    pack $extFrame -padx 10 -pady 10 -expand true -fill both
    pack $intFrame -expand true -fill both
    pack $taskbarFrame -padx 2  -fill x
    pack $logFrame -expand true -fill both
    grid $scrollbarY -in $logFrame -row 0 -column 1 -sticky ns
    grid $scrollbarX -in $logFrame -row 1 -column 0 -sticky ew
    grid $listbox    -in $logFrame -row 0 -column 0 -sticky news \
            -padx 0 -pady 0

    grid columnconfigure $logFrame 0 -weight 1
    grid columnconfigure $logFrame 1 -weight 0
    grid rowconfigure    $logFrame 0 -weight 1
    grid rowconfigure    $logFrame 1 -weight 0

    focus [$listbox bodypath]

    bind [$listbox bodypath] <ButtonRelease-1> "
        getLog::ActivateSiteTaskbarButtons
        focus [$listbox bodypath]
    "
    bind [$listbox bodypath] <Button-3> {
        event generate %W <Button-1>        -x %x -y %y
        event generate %W <ButtonRelease-1> -x %x -y %y
        getLog::UrlLogRightClick %W %x %y %X %Y
    }
    bind [$listbox bodypath] <Double-Button-1> {
        getLog::UrlLogDoubleClick %W %x %y %X %Y
    }
    bind [$listbox bodypath] <FocusIn> {
        $getLog::siteLog(listbox) selection set active
        ActivateSiteTaskbarButtons
    }
    bind [$listbox bodypath] <FocusIn> {
        $getLog::siteLog(listbox) selection clear 0 end
        $getLog::siteLog(listbox) selection set active
    }
    bind [$listbox bodypath] <KeyRelease-Next> {
        $getLog::siteLog(listbox) selection clear 0 end
        $getLog::siteLog(listbox) selection set active
    }
    bind [$listbox bodypath] <KeyRelease-Prior> {
        $getLog::siteLog(listbox) selection clear 0 end
        $getLog::siteLog(listbox) selection set active
    }
    bind $win <Key-Delete> {
        getLog::UrlLogRemove [getLog::UrlLogSelectedUrl]
        $getLog::siteLog(listbox) selection set active
    }
    bind $win <Escape>     "destroy $win"

    set siteMenus(rightClick) [menu $win.menur -tearoff 0]
    $siteMenus(rightClick) add command -label $labelMenus(download)
    $siteMenus(rightClick) add command -label $labelMenus(update)
    $siteMenus(rightClick) add separator
    $siteMenus(rightClick) add command -label $labelMenus(queue)
    $siteMenus(rightClick) add separator
    $siteMenus(rightClick) add command -label $labelMenus(options)
    $siteMenus(rightClick) add command -label $labelMenus(localDir)
    $siteMenus(rightClick) add check   -label $labelMenus(notErase)
    $siteMenus(rightClick) add separator
    $siteMenus(rightClick) add command -label $labelDialogs(copyLink)
    $siteMenus(rightClick) add separator
    $siteMenus(rightClick) add command -label $labelDialogs(launchLink)
    $siteMenus(rightClick) add command -label $labelDialogs(seeDown)
    $siteMenus(rightClick) add separator
    $siteMenus(rightClick) add command -label $labelMenus(delete)

    set siteMenus(doubleClick) [menu $win.menud -tearoff 0]
    $siteMenus(doubleClick) add command -label $labelMenus(download)
    $siteMenus(doubleClick) add command -label $labelMenus(update)

    GetLogWindowPopulate

    return
}

###############################################################################
# GetLog
#    Procedures that controls what happens with the log, it cares care of 
#    reading it into memory, invoking the procedure to draw the window and
#    whatever else needed.
###############################################################################
proc GetLog {} {
    variable urlLog

    if {![info exists urlLog]} {
        UrlLogRead
    }

    GetLogWindow

    return
}

# When we start Getleft we read the log

UrlLogRead

# Now comes the log for individual files

################################################################################
# ActivateFileTaskbarButtons
#    Activates or disables the buttons in the taskbar depending on whether
#    there is a row selected.
################################################################################
proc ActivateFileTaskbarButtons {} {
    variable fileTaskbar
    variable fileLog

    set anchor [$fileLog(listbox) curselection]
    if {$anchor==""} {
        $fileTaskbar(start) configure -state disabled
    } else {
        $fileTaskbar(start) configure -state normal
    }

    set size [$fileLog(listbox) size]
    if {$size==0} {
        $fileTaskbar(clear) configure -state disabled
    } else {
        $fileTaskbar(clear) configure -state normal
    }

    return
}

################################################################################
# FileLogRemove
#    Removes one Url from the log 
#
# Parameters
#    url: The url to remove.
################################################################################
proc FileLogRemove {} {
    variable fileLog

    set anchor [$fileLog(listbox) curselection]
    if {$anchor==""} return

    $fileLog(listbox) delete $anchor
    $fileLog(listbox) selection set $anchor
    ActivateFileTaskbarButtons

    return
}

################################################################################
# FileLogRightClick
#   This procedure is invoked when the user right-clicks in the multicolumn
#   list with the files, it makes a context menu pop up with the
#   appropiate options
#
# Parameters
#   win: window where you right-clicked.
#   y,x: the coordinates where you did it for the tablelist widget.
#   X,Y: and for the pop up menu.
################################################################################
proc FileLogRightClick {win x y X Y} {
    variable fileLog
    variable fileMenus

    set url [FileLogSelectedUrl]
    if {$url==""} return
    set ref [FileLogSelectedRef]

    ActivateFileTaskbarButtons

    $fileMenus(rightClick) entryconfigure 0                                  \
            -command "getLog::FileLogDownload [list $url]"
    $fileMenus(rightClick) entryconfigure 2                                  \
            -command "clipboard clear
                      clipboard append [list $url]"
    $fileMenus(rightClick) entryconfigure 4                                  \
            -command "Ayuda::InvokeBrowser [list $url] $getLog::fileLog(win)"
    $fileMenus(rightClick) entryconfigure 5                                  \
            -command "Ayuda::InvokeBrowser [list $ref] $getLog::fileLog(win)"
    $fileMenus(rightClick) entryconfigure 7                                  \
            -command "getLog::FileLogRemove"

    tk_popup $fileMenus(rightClick) $X $Y

    return
}

################################################################################
# FileLogSelectedUrl
#   When the user clicks on the icons, this procedure is used to know the url
#   of the row that was selected at the time.
#
# Returns
#   The url or an empty string if no row is selected
################################################################################
proc FileLogSelectedUrl {} {
    variable fileLog

    set anchor [$fileLog(listbox) curselection]
    if {[catch {$fileLog(listbox) cellcget $anchor,1 -text} url]} {
        return
    }
    return $url
}

################################################################################
# FileLogSelectedRef
#   When the user clicks on the icons, this procedure is used to know the 
#   referrer of the url of the row that was selected at the time.
#
# Returns
#   The referrer or an empty string if no row is selected
################################################################################
proc FileLogSelectedRef {} {
    variable fileLog

    set anchor [$fileLog(listbox) curselection]
    if {[catch {$fileLog(listbox) cellcget $anchor,3 -text} ref]} {
        return
    }
    if {$ref!="-"} {
        return $ref
    }
    return        
}

################################################################################
# FileLogDownload
#   Ask for a directory and starts downloading the given url.
################################################################################
proc FileLogDownload {url} {
    variable fileLog

    if {$url==""} return

    EnterUrl::EnterUrl 0 $fileLog(win) $url 

    return
}

################################################################################
# FileLogDefineTaskbarButtons
#    Defines the buttons in the taskbar.
#
# Parameters
#    The frame for the buttons.
################################################################################
proc FileLogDefineTaskbarButtons {frame} {
    global   labelDialogs labelMenus labelButtons
    variable fileTaskbar
    variable fileLog

    set fileTaskbar(start)  [button $frame.start -image $iconNS::iconImages(start) \
            -relief flat -bd 1 -command {
                getLog::FileLogDownload [getLog::FileLogSelectedUrl]
            }]
    ::BalloonHelp::set_balloon $fileTaskbar(start)  $labelDialogs(start)

    set fileTaskbar(clear) [button $frame.clear  -image $iconNS::iconImages(clear) \
            -relief flat -bd 1 -command "getLog::FileLogClear"]
    ::BalloonHelp::set_balloon $fileTaskbar(clear)  $labelButtons(clear)

    set fileTaskbar(exit) [button $frame.exit    -image $iconNS::iconImages(exit) \
            -relief flat -bd 1 -command "wm withdraw $fileLog(win)"]
    ::BalloonHelp::set_balloon $fileTaskbar(exit)   $labelDialogs(exit)

    foreach {name path} [array get fileTaskbar] {
        bindtags $path "[bindtags $path] logTaskIcon"
    }
    bind logTaskIcon <<ButtonRaise>> {
        %W configure -relief raised
    }
    bind logTaskIcon <<ButtonFlat>> {
        %W configure -relief flat
    }
    pack $fileTaskbar(start) $fileTaskbar(clear) -side left
    pack $fileTaskbar(exit)  -side right

    return
}

###############################################################################
# FileLogEnter
#    Logs in the file 'fileDown.log' the error messages returned by 'curl'
#
# Parameters
#    message: The error message.
#    link: link that we where trying to download.
#    referer: Referrer page of that link.
#    size:    The size of the downloaded file.
###############################################################################
proc FileLogEnter {message link referer {size ""}} {
    global dirGetleft getleftState tcl_platform
    global labelDialogs labelMonths labelCurlCodes 
    global errorCode supportedLang getleftOptions
    variable fileLog 

    if {[regexp {[0-9]+} $message]} {
        catch {set message $labelCurlCodes($message)}
    }

    set seconds [clock seconds]
    set month $labelMonths([clock format $seconds -format %m])

    # This is a workaround, in Tcl8.4a3 if I write '-format "%d-$month-%Y"'
    # unicode characters in '$month' get mangled.
    set date0 [clock format $seconds -format "%d"]
    set date1 [clock format $seconds -format "%Y -- %H:%M"]
    set date  "$date0-$month-$date1"

    if {$size!=""} {
        if {$size<512} {
            set size ${size}B
        } elseif {$size<1048576} {
            set size "[format "%.2f" [expr {$size/1024.0}]]K"
        } else {
            set size "[format "%.2f" [expr {$size/1048576.0}]]M"
        }
        regsub {\.} $size $labelDialogs(decimal) size
    }

    set fileLog(file) [file join $dirGetleft(conf) fileDown.log]
    if {$tcl_platform(platform)=="windows"} {
        if {[file exists $fileLog(file)]} {
            file attributes $fileLog(file) -hidden 0
        }
    } 
    set handle  [open $fileLog(file) a+]
    fconfigure $handle -encoding $supportedLang($getleftOptions(lang))

    # Some of these variables maybe empty, others may have spaces so we
    # put them between quotes.
    puts $handle "\"$message\" \"$link\" \"$size\" \"$referer\" \"$date\""

    close $handle

    if {$tcl_platform(platform)=="windows"} {
        file attributes $fileLog(file) -hidden 1
    } else {
        file attributes $fileLog(file) -permissions 0600
    }

    if {([info exists fileLog(win)])&&([winfo exists $fileLog(win)])} {
        set tmp [$fileLog(listbox) yview]
        $fileLog(listbox) insert end [list $message $link $size $referer $date]
        if {([lindex $tmp 1]==1)||([lindex $tmp 0]==0)} {
            $fileLog(listbox) yview moveto 1
        }
    }
    if {([info exists fileLog(win)])&&([winfo exists $fileLog(win)])} {
        ActivateFileTaskbarButtons
    }

    return
}

###############################################################################
# EraseErrorClear
#    Deletes the error log and clears the listbox.
###############################################################################
proc FileLogClear {} {
    global dirGetleft labelTitles labelMessages
    variable fileLog

    set what [tk_messageBox -title "$labelTitles(sure)" -parent $fileLog(win)\
            -message "$labelMessages(clearLog)" -type yesno]
    if {$what=="no"} {
        return
    }

    $fileLog(listbox) delete 0 end
    catch {file delete $fileLog(file)}

    ActivateFileTaskbarButtons

    return
}

###############################################################################
# FileLogScroll
#    Scrolls the text up or down a third of its height
#
# Parameters
#    text: full path of the text widget
#    how:  either 'up' or 'down'
###############################################################################
proc FileLogScroll {text how} {

    set height [$text cget -height]
    set scroll [expr {round ($height/2.5)}]
    if {$how=="up"} {
        set scroll [expr {$scroll * -1}]
    }

    $text yview scroll $scroll units

    return
}

###############################################################################
# FileLog
#   Shows a window with  the contents of the file log.
###############################################################################
proc FileLog {} {
    global labelButtons labelTitles labelMessages labelDialogs labelMenus
    global dirGetleft supportedLang getleftOptions
    variable fileLog
    variable fileMenus

    if {[winfo exists .fileLog]} {
        set windowState [wm state .fileLog]
        if {$windowState=="withdrawn"} {
            wm deiconify .fileLog
        }
        raise $fileLog(win)
        focus $fileLog(listbox)
        return
    }

    set fileLog(file) [file join "$dirGetleft(conf)" fileDown.log]
    set formerLog     [file join "$dirGetleft(conf)" Geterror.log]
    catch {file delete -force $formerLog}

    set coord(x) [winfo rootx .]
    set coord(y) [winfo rooty .]

    set fileLog(win) [toplevel .fileLog]
    wm title $fileLog(win) $labelTitles(getLog)
    wm geometry $fileLog(win) +[expr {$coord(x)+100}]+[expr {$coord(y)+50}]

    set extFrame     [frame $fileLog(win).ext]
    set intFrame     [frame $extFrame.int]
    set taskBarFrame [frame $intFrame.taskBar]
    set logFrame     [frame $intFrame.log -bd 2 -relief sunken]

    set scrollBars(y) [scrollbar $intFrame.yscroll -orient vertical     \
            -command [list $logFrame.listbox yview] ]
    set scrollBars(x) [scrollbar $intFrame.xscroll -orient horizontal   \
            -command [list $logFrame.listbox xview] ]

    set fileLog(listbox) [::tablelist::tablelist $logFrame.listbox      \
            -bd 0 -height 10 -width 120 -stretch all                    \
            -fg $getleftOptions(fg) -bg $getleftOptions(bg)             \
            -columns [list 0 $labelDialogs(code)     left               \
                           0 $labelDialogs(url)      left               \
                           0 $labelDialogs(size)     right              \
                           0 $labelDialogs(referer)  left               \
                           0 $labelDialogs(date)     left]              \
            -showarrow 1 -labelcommand tablelist::sortByColumn          \
            -xscrollcommand "$scrollBars(x) set"                        \
            -yscrollcommand "$scrollBars(y) set"                        \
            -stripebg #eff9fa -exportselection 0]

    $fileLog(listbox) columnconfigure 0 -labelalign center
    $fileLog(listbox) columnconfigure 2 -labelalign center

    $fileLog(listbox) columnconfigure 1 -sortmode command
    $fileLog(listbox) columnconfigure 1 -sortcommand ::getLog::GetLogSortUrl
    $fileLog(listbox) columnconfigure 2 -sortmode command
    $fileLog(listbox) columnconfigure 2 -sortcommand ::getLog::GetLogSortSize
    $fileLog(listbox) columnconfigure 3 -sortmode command
    $fileLog(listbox) columnconfigure 3 -sortcommand ::getLog::GetLogSortUrl
    $fileLog(listbox) columnconfigure 4 -sortmode command
    $fileLog(listbox) columnconfigure 4 -sortcommand ::getLog::GetLogSortDate

    pack $extFrame -padx 10 -pady 10 -expand true -fill both
    pack $intFrame -expand true -fill both
    pack $taskBarFrame -padx 2  -fill x
    pack $logFrame -expand true -fill both

    FileLogDefineTaskbarButtons $taskBarFrame

    grid $scrollBars(y)     -in $logFrame -row 0 -column 1 -sticky ns
    grid $scrollBars(x)     -in $logFrame -row 1 -column 0 -sticky ew
    grid $fileLog(listbox)  -in $logFrame -row 0 -column 0 -sticky news \
            -padx 0 -pady 0

    grid columnconfigure $logFrame 0 -weight 1
    grid columnconfigure $logFrame 1 -weight 0
    grid rowconfigure    $logFrame 0 -weight 1
    grid rowconfigure    $logFrame 1 -weight 0

    bind $fileLog(win) <Escape> "destroy $fileLog(win)"

    bind [$fileLog(listbox) bodypath] <ButtonRelease-1> "
        getLog::ActivateFileTaskbarButtons
        focus [$fileLog(listbox) bodypath]
    "
    bind [$fileLog(listbox) bodypath] <Double-Button-1> {
        getLog::FileLogDownload [getLog::FileLogSelectedUrl]
    }
    bind [$fileLog(listbox) bodypath] <Button-3> {
        event generate %W <Button-1>        -x %x -y %y
        event generate %W <ButtonRelease-1> -x %x -y %y
        getLog::FileLogRightClick %W %x %y %X %Y
    }
    bind [$fileLog(listbox) bodypath] <Key-Delete> {
        getLog::FileLogRemove
        $getLog::fileLog(listbox) selection set active
    }
    bind [$fileLog(listbox) bodypath] <FocusIn> {
        $getLog::fileLog(listbox) selection clear 0 end
        $getLog::fileLog(listbox) selection set active
        getLog::ActivateFileTaskbarButtons
    }
    bind [$fileLog(listbox) bodypath] <KeyRelease-Next> {
        $getLog::fileLog(listbox) selection clear 0 end
        $getLog::fileLog(listbox) selection set active
    }
    bind [$fileLog(listbox) bodypath] <KeyRelease-Prior> {
        $getLog::fileLog(listbox) selection clear 0 end
        $getLog::fileLog(listbox) selection set active
    }
    focus [$fileLog(listbox) bodypath]

    if {![file exists $fileLog(file)]} {
        return
    }
    if {[catch {open $fileLog(file) r} handle]} {
        return
    }
    fconfigure $handle -encoding $supportedLang($getleftOptions(lang))

    for {} {![eof $handle]} {} {
        set line [gets $handle]
        if {$line!=""} {
            catch {$fileLog(listbox) insert end $line}
        }
    }

    set fileMenus(rightClick) [menu $fileLog(win).menur -tearoff 0]
    $fileMenus(rightClick) add command -label $labelMenus(download)
    $fileMenus(rightClick) add separator
    $fileMenus(rightClick) add command -label $labelDialogs(copyLink)
    $fileMenus(rightClick) add separator
    $fileMenus(rightClick) add command -label $labelDialogs(launchLink)
    $fileMenus(rightClick) add command -label $labelDialogs(launchRef)
    $fileMenus(rightClick) add separator
    $fileMenus(rightClick) add command -label $labelMenus(delete)        \
            -command "getLog::FileLogRemove"

    close $handle
    ActivateFileTaskbarButtons

    return
}

}


