################################################################################
################################################################################
##                          mainwin.tcl
################################################################################
################################################################################
## Defines the main window of Getleft, the menus and such
################################################################################
################################################################################
## (c) 2001-2003 Andrs Garca Garca. fandom@retemail.es
## You may distribute the contents of this file under the terms of the GPL v2
################################################################################
################################################################################

namespace eval iconNS {

################################################################################
# DefineIconImages
#     Prepares the images for the icons in the taskbar so that Tcl can use them
################################################################################
proc DefineIconImages {} {
    global dirGetleft
    variable iconImages

    set iconNS::iconImages(url)     [image create photo             \
            -file [file join "$dirGetleft(icons)" url.gif]]
    set iconNS::iconImages(start)   [image create photo             \
            -file [file join "$dirGetleft(icons)" start.gif]]
    set iconNS::iconImages(auto)    [image create photo             \
            -file [file join "$dirGetleft(icons)" auto.gif]]
    set iconNS::iconImages(delay)   [image create photo             \
            -file [file join "$dirGetleft(icons)" delay.gif]]
    set iconNS::iconImages(log)     [image create photo             \
            -file [file join "$dirGetleft(icons)" log.gif]]
    set iconNS::iconImages(fileLog) [image create photo             \
            -file [file join "$dirGetleft(icons)" fileLog.gif]]
    set iconNS::iconImages(get)     [image create photo             \
            -file [file join "$dirGetleft(images)" get.gif]]
    set iconNS::iconImages(help)    [image create photo             \
            -file [file join "$dirGetleft(icons)" help.gif]]

    set iconNS::iconImages(update)  [image create photo             \
            -file [file join "$dirGetleft(icons)" update.gif]]
    set iconNS::iconImages(queue)   [image create photo             \
            -file [file join "$dirGetleft(icons)" queue.gif]]
    set iconNS::iconImages(exit)    [image create photo             \
            -file [file join "$dirGetleft(icons)" exit.gif]]

    set iconNS::iconImages(conf)    [image create photo             \
            -file [file join "$dirGetleft(icons)" conf.gif]]
    set iconNS::iconImages(clear)   [image create photo             \
            -file [file join "$dirGetleft(icons)" clear.gif]]

    return
}
}

# Virtual event to raise and flat the buttons in the main window
# and logs.

event add <<ButtonRaise>> <FocusIn>
event add <<ButtonRaise>> <Enter>

event add <<ButtonFlat>>  <FocusOut>
event add <<ButtonFlat>>  <Leave>

namespace eval mainWin {

################################################################################
# ShowScrollX - ShowScrollY
#    They substitute the usual 'set' command of the scrollbar that is invoked
#    when a scrollbar changes visibility. This procedure will hide or show
#    the scrollbars as needed. The after commands are there so to prevent
#    scrollbars appearing and disappering due to the apperance/disaperance
#    of the other, which forms a loop that crashes Tcl/Tk.
#
# Parameter
#    first - last: The same as for the 'set' command in a scrollbar.
################################################################################
proc ShowScrollX {first last} {
    variable mainWin

    if {($first<0.02)&&($last==1)} {
        if {[winfo viewable $mainWin(scrollX)]} {
            grid forget $mainWin(scrollX)
        }
    } else {
        if {![winfo viewable $mainWin(scrollX)]} {
            grid $mainWin(scrollX) -in $mainWin(listFrame)        \
                    -row 1 -column 0 -sticky ew
        }
        $mainWin(scrollX) set $first $last
    }
    return
}

proc ShowScrollY {first last} {
    variable mainWin

    if {($first<0.02)&&($last==1)} {
        if {[winfo viewable $mainWin(scrollY)]} {
            grid forget $mainWin(scrollY)
        }
    } else {
        if {![winfo viewable $mainWin(scrollY)]} {
            grid $mainWin(scrollY) -in $mainWin(listFrame)        \
                    -row 0 -column 1 -sticky ns
        }
        $mainWin(scrollY) set $first $last
     }
    return
}

################################################################################
# ActivateDownloadButtons
#    When invoked activates or disables the taskbar buttons depending on
#    whether we are downloading or not
################################################################################
proc ActivateDownloadButtons {} {
    variable mainWin
    variable taskbar
    global   getleftState

    $mainWin::taskbar(start) configure -state disabled
    $mainWin::taskbar(auto)  configure -state disabled

    if {[$mainWin(listbox) size]} {
        if {$getleftState(downloading)==0} {
            set anchor [$mainWin(listbox) curselection]
            if {$anchor!=""} {
                $mainWin::taskbar(start) configure -state normal
            }
        }
        $mainWin::taskbar(auto)  configure -state normal
    }
    return
}

################################################################################
# ActivateTaskbarButtons
#    When invoked activates or disables the taskbar buttons depending on
#    whether we are downloading or not
################################################################################
proc ActivateTaskbarButtons {} {
    global getleftState labelDialogs labelMessages
    variable mainWin

    if {$getleftState(downloading)==0} {
        set mainWin(ctrFrameLabel)    $labelDialogs(ready)
        set mainWin(tmpCtrFrameLabel) $labelDialogs(ready)
    } else {
        set mainWin(ctrFrameLabel)    "$labelMessages(downloading): $getleftState(url)"
        set mainWin(tmpCtrFrameLabel) "$labelMessages(downloading): $getleftState(url)"
    }

    ActivateDownloadButtons

    return
}

################################################################################
# DefineTaskbarButtons
#    Defines the buttons in the taskbar.
#
# Parameters
#    The frames for the buttons.
################################################################################
proc DefineTaskbarButtons {left right} {
    global   labelDialogs labelMenus getleftState
    variable taskbar

    iconNS::DefineIconImages

    set taskbar(url)   [button $left.url   -image $iconNS::iconImages(url)   \
            -relief flat -bd 1 -command EnterUrl::EnterUrl]
    ::BalloonHelp::set_balloon $taskbar(url) $labelDialogs(enterUrl)

    set taskbar(start) [button $left.start -image $iconNS::iconImages(start) \
            -relief flat -bd 1                                               \
            -command  {AutomaticDownload [mainWin::UrlListSelectedUrl]}]
    ::BalloonHelp::set_balloon $taskbar(start) $labelDialogs(start)

    set taskbar(auto) [button $left.auto -image $iconNS::iconImages(auto)    \
            -relief flat -bd 1 -command mainWin::SetAutoDownload]
    ::BalloonHelp::set_balloon $taskbar(auto) $labelDialogs(auto)

    set taskbar(delay) [button $left.delay -image $iconNS::iconImages(delay) \
            -relief flat -bd 1                                               \
            -command "::Delay::DelayedDownload"]
    if {$getleftState(delaySet)==1} {
        $taskbar(delay) configure -relief sunken
    }    
    ::BalloonHelp::set_balloon $taskbar(delay) $labelDialogs(progDelay)

    set taskbar(log)   [button $left.log   -image $iconNS::iconImages(log)   \
            -relief flat -bd 1                                               \
            -command "::getLog::GetLog"]
    ::BalloonHelp::set_balloon $taskbar(log) $labelDialogs(getLog)

    set taskbar(fileLog) [button $right.fileLog -image $iconNS::iconImages(fileLog) \
            -relief flat -bd 1 -command ::getLog::FileLog]
    ::BalloonHelp::set_balloon $taskbar(fileLog) $labelDialogs(fileLog)
    set taskbar(help)  [button $right.conf -image $iconNS::iconImages(help)  \
            -relief flat -bd 1 -command Ayuda::Manual]
    ::BalloonHelp::set_balloon $taskbar(help) $labelDialogs(help)

    set taskbar(get)   [button $right.get  -image $iconNS::iconImages(get)   \
            -relief flat -bd 1 -command Ayuda::About]
    ::BalloonHelp::set_balloon $taskbar(get) $labelDialogs(about)

    foreach {name path} [array get taskbar] {
        bindtags $path "[bindtags $path] taskIcon"
    }
    bind taskIcon <<ButtonRaise>> {
        %W configure -relief raised
        set mainWin::mainWin(tmpCtrFrameLabel) $mainWin::mainWin(ctrFrameLabel)
        set mainWin::mainWin(ctrFrameLabel)    $BalloonHelp::Bulle(%W)
    }
    bind taskIcon <<ButtonFlat>> {
        if {(("%W"==$mainWin::taskbar(auto))&&($::getleftState(autoDown)==1))
                ||(("%W"==$mainWin::taskbar(delay))&&($::getleftState(delaySet)==1))} {
            %W configure -relief sunken
        } else {
            %W configure -relief flat
        }        
        set mainWin::mainWin(ctrFrameLabel) $mainWin::mainWin(tmpCtrFrameLabel)
    }

    foreach {name path} [array get menus] {
        bind $path <FocusOut> "
            $taskbar($name) configure -relief flat
            BalloonHelp::kill_balloon
        "
    }

    return
}

################################################################################
# NextToAutoDownload
#    When invoked this procedure will start downloading the next site in the
#    queue, the point here is that urls entered to get the site map are only
#    downloaded up to where the map is done and then left in the queue.
################################################################################
proc NextToAutoDownload {} {
    global getleftState urlsToDownload dirGetleft

    while {[llength $getleftState(urlsToAutoDown)]>0} {
        foreach url $getleftState(urlsToAutoDown) {
            if {$getleftState(autoDown)==0} break
            if {![info exists urlsToDownload($url,options)]} {
                set urlIndex [lsearch $getleftState(urlsToAutoDown) $url]
                set getleftState(urlsToAutoDown)                              \
                        [lreplace $getleftState(urlsToAutoDown)               \
                        $urlIndex $urlIndex]
                continue
            }
            array set optionsTemp $urlsToDownload($url,options)
            if {$optionsTemp(map)==1} {
                set domain   [lindex [HtmlParser::ParseUrl $url] 1]
                set domain   [string tolower $domain]
                set fileName [file join $dirGetleft(conf) $domain.gdt]
                if {![catch {open "$fileName" r} handle]} {
                    set found 0
                    for {} {![eof $handle]} {} {
                        set line [gets $handle]
                        if {[regexp {\(mapLinks\)} "$line"]} {
                            set found 1
                            break
                        }
                    }
                    close $handle
                    if {$found} {
                        AutomaticDownload $url
                        continue
                    } else {
                        set urlIndex [lsearch $getleftState(urlsToAutoDown) $url]
                        set getleftState(urlsToAutoDown)                          \
                                [lreplace $getleftState(urlsToAutoDown)           \
                                $urlIndex $urlIndex]
                        set urlIndex [UrlListUrlIndex $url]
                        $main(listbox) rowconfigure $urlIndex -bg "" -fg ""
                    }
                }
            }
            AutomaticDownload $url
        }
    }
    catch {wm withdraw $::Ventana::window(top)}

    if {$getleftState(autoDown)==1} {
        SetAutoDownload
    }

    return
}

################################################################################
# SetAutoDownload
#   When the button in the taskbar is clicked, this procedure enables or
#   disables full atomatic download.
################################################################################
proc SetAutoDownload {} {
    global getleftState
    variable taskbar
    variable mainWin

    if {$getleftState(autoDown)==0} {
        set getleftState(autoDown) 1
        $taskbar(auto) configure -relief sunken

        set getleftState(urlsToAutoDown) ""
        set selectedUrls [$mainWin(listbox) curselection]
        if {$selectedUrls==""} {
            set getleftState(urlsToAutoDown) $getleftState(urlQueue)
            set urlNumber [$mainWin(listbox) size]
            for {set i 0} {$i<$urlNumber} {incr i} {
                $mainWin(listbox) rowconfigure $i -bg blue -fg white
            }
        } else {
            foreach index $selectedUrls {
                $mainWin(listbox) rowconfigure $index -bg blue -fg white
                lappend getleftState(urlsToAutoDown)                 \
                        [$mainWin(listbox) cellcget $index,0 -text]    
            }
        }
        if {($getleftState(downloading)==0)                          \
                &&([llength $getleftState(urlQueue)]>0)} {
            NextToAutoDownload
        }
    } else {
        set getleftState(autoDown) 0
        set getleftState(urlsToAutoDown) ""
        $taskbar(auto) configure -relief flat
        set urlNumber [$mainWin(listbox) size]
        for {set i 0} {$i<$urlNumber} {incr i} {
            $mainWin(listbox) rowconfigure $i -bg "" -fg ""
        }
    }
    return
}

################################################################################
# UrlListUrlIndex
#   Given an Url this procedure returns its index in the url list.
#
# Parameters
#   url: The url to search for.
#
# Returns
#   The index of the url, or -1 if the is none to return
################################################################################
proc UrlListUrlIndex {url} {
    variable mainWin

    set rowNumber [$mainWin(listbox) size]
    for {set i 0} {$i<$rowNumber} {incr i} {
        set site [$mainWin(listbox) cellcget $i,0 -text]
        if {[string match $url $site]} {
            return $i
        }
    }
    return -1
}

################################################################################
# UrlListIncrResume
#   Given an Url this procedure increments the 'Resumed' counter in the
#   main window.
#
# Parameters
#   url: The url that is going to be resumed.
################################################################################
proc UrlListIncrResume {url} {
    variable mainWin

    set index [UrlListUrlIndex $url]
    set i [$mainWin(listbox) cellcget $index,1 -text]
    $mainWin(listbox) cellconfigure $index,1 -text [incr i]

    return
}

################################################################################
# UrlListDeleteUrls
#   Invoked when the user want to delete urls in the queue,
#   checks whether it is the url being downloaded, if it isn't it calls
#   'UrlListDeleteEntry' to get it deleted
#
# Parameters
#   urlIndexList: List with the listbox indices of the urls to delete from the
#   queue.
################################################################################
proc UrlListDeleteUrls {urlIndexList} {
    global getleftState
    variable mainWin

    # We start deleting at the last one, so that the other indices don't
    # get altered.
    set urlIndexList [lsort -decreasing -integer $urlIndexList]
    foreach urlIndex $urlIndexList {
        if {[catch {$mainWin(listbox) cellcget $urlIndex,0 -text} url]} {
            continue
        }
        if {($getleftState(downloading)==1)&&($url==$getleftState(url))} {
            continue
        }
        UrlListDeleteEntry $url
    }

    return
}

################################################################################
# UrlListDeleteEntry
#   Deletes urls from the url list.
#
# Parameters
#   url: Url to delete from the queue.
################################################################################
proc UrlListDeleteEntry {url} {
    variable mainWin
    global getleftState urlsToDownload dirGetleft

    set urlIndex [UrlListUrlIndex $url]
    $mainWin(listbox) delete $urlIndex

    set index [lsearch $getleftState(urlQueue) $url]
    set getleftState(urlQueue)                                             \
            [lreplace $getleftState(urlQueue) $index $index]

    getLog::UrlLogEnter $url $urlsToDownload($url,dir)                     \
            $urlsToDownload($url,options)

    catch {unset urlsToDownload($url)}
    catch {unset urlsToDownload($url,dir)}
    catch {unset urlsToDownload($url,resume)}
    catch {unset urlsToDownload($url,options)}

    set domain [string tolower [lindex [HtmlParser::ParseUrl $url] 1]]
    catch {file delete [file join $dirGetleft(conf) $domain.gdt]}

    if {$getleftState(autoDown)==1} {
        set index [lsearch $getleftState(urlsToAutoDown) $url]
        if {$index!=-1} {
            set getleftState(urlsToAutoDown)                               \
                    [lreplace $getleftState(urlsToAutoDown) $index $index]
        }
    }

    SetResumeSitesMenu
    ActivateDownloadButtons

    return
}

################################################################################
# UrlListCopyLink
#   Copies the selected links into the clipboard.
#
# Parameters
#   urlIndexList: List with the indices of the urls to copy.
################################################################################
proc UrlListCopyLink {urlIndexList} {
    variable mainWin

    clipboard clear
    foreach urlIndex $urlIndexList {
        clipboard append "[$mainWin(listbox) cellcget $urlIndex,0 -text]"
    }

    return
}

################################################################################
# UrlListMoveEntry
#   Moves one of the entry one row up or down.
#
# Parameters
#   urlIndexList: List with the indices of the urls to move.
#   how: '-1' up, '1' down.
################################################################################
proc UrlListMoveEntry {urlIndexList how} {
    variable mainWin
    global getleftState urlsToDownload

    set rowLimit [expr [$mainWin(listbox) size] - 1]
    if {$how==1} {
        set urlIndexList [lsort -decreasing -integer $urlIndexList]
    }
    foreach urlIndex $urlIndexList {
        set url [$mainWin(listbox) cellcget $urlIndex,0 -text]
        set queueIndex [lsearch $getleftState(urlQueue) $url]

        if {$how==-1} {
            if {$urlIndex==0}         continue
        } else {
            if {$rowLimit==$urlIndex} continue
        }

        $mainWin(listbox) delete $urlIndex
        incr urlIndex $how
        $mainWin(listbox) insert $urlIndex [list                     \
                $url $urlsToDownload($url,resume)                    \
                [file nativename $urlsToDownload($url,dir)]]
        $mainWin(listbox) selection set $urlIndex $urlIndex

        set getleftState(urlQueue) [lreplace $getleftState(urlQueue) \
            $queueIndex $queueIndex]
        incr queueIndex $how
        set getleftState(urlQueue) [linsert $getleftState(urlQueue)  \
                $queueIndex $url]
    }
    return
}

################################################################################
# UrlListMoveTopBottom
#   Moves one of the urls to the top or the bottom of the list.
#
# Parameters
#   urlIndexList: The list with the indices of the urls to move.
#   where: Either 'top' or 'bottom', defaults to 'bottom'
################################################################################
proc UrlListMoveTopBottom {urlIndexList {where bottom}} {
    variable mainWin
    global getleftState urlsToDownload

    if {$where=="bottom"} {
        set where end
        set adjust -1
        set moveto 1
    } else {
        set where  0
        set adjust 1
        set moveto 0
        set urlIndexList [lsort -decreasing -integer $urlIndexList]
    }

    set moved 0
    foreach urlIndex $urlIndexList {
        set urlIndex [expr $urlIndex + $adjust * $moved]
        incr moved        
        set url [$mainWin(listbox) cellcget $urlIndex,0 -text]
        set queueIndex [lsearch $getleftState(urlQueue) $url]

        set rowLimit [expr [$mainWin(listbox) size] - 1]
        if {($rowLimit==$urlIndex)&&($where=="end")} {
            return
        }
        if {($urlIndex==0)&&($where==0)} {
            return
        }

        $mainWin(listbox) delete $urlIndex
        $mainWin(listbox) insert $where [list                        \
                $url $urlsToDownload($url,resume)                    \
                [file nativename $urlsToDownload($url,dir)]]
        $mainWin(listbox) selection set $where $where

        set getleftState(urlQueue) [lreplace $getleftState(urlQueue) \
                $queueIndex $queueIndex]
        set getleftState(urlQueue) [linsert $getleftState(urlQueue)  \
                $where $url]
    }
    $mainWin(listbox) yview moveto $moveto
    $mainWin(listbox) selection anchor $urlIndex

    return
}

################################################################################
# UrlListEnterEntry
#   Enters one site into the url list.
#
# Parameters
#   url: The url of the entry to enter.
#   resume: How many times it has been resumed so far.
#   dir: The directory where it is going to be saved.
################################################################################
proc UrlListEnterEntry {url resume dir} {
    variable mainWin

    $mainWin(listbox) insert end [list $url $resume [file nativename $dir]]
    $mainWin(listbox) yview moveto 1

    ActivateDownloadButtons

    return
}

################################################################################
# UrlListSelectedUrl
#   Returns the url of the selected entry in the list.
################################################################################
proc UrlListSelectedUrl {} {
    variable mainWin

    set anchor [lindex [$mainWin(listbox) curselection] 0]
    if {[catch {$mainWin(listbox) cellcget $anchor,0 -text} url]} {
        return
    }

    return $url
}

###############################################################################
# UpdateUrlOptions
#    During a download, if the users changes the options by using the menus,
#    procedure is invoked to make sure, the new chosen options are saved for
#    the site.
###############################################################################
proc UpdateUrlOptions {} {
    global getleftState urlsToDownload downOptions

    if {$getleftState(downloading)==0} return

    set urlsToDownload($getleftState(url),options) [array get downOptions]

    return
}

################################################################################
# UrlListChangeOptions
#   Change the options for downloading an URL in the urlList.
#
# Parameters
#   parent: The window over which the dialog will appear.
#   urlIndexList: List with the indices of the urls whose options we want to
#   change, it defaults to "", which happens when it is invoked with the icon.
################################################################################
proc UrlListChangeOptions {parent {urlIndexList ""}} {
    global urlsToDownload getleftState
    variable mainWin

    if {$urlIndexList==""} {
        ::Opciones::ChooseOptions ::downOptions $parent
        UpdateUrlOptions
        return
    }

    set url [$mainWin(listbox) cellcget [lindex $urlIndexList 0],0 -text]
    array set ::downOptionsTemp $urlsToDownload($url,options)
    set done [::Opciones::ChooseOptions ::downOptionsTemp $parent]

    if {$done==1} {
        foreach urlIndex $urlIndexList {
            set url [$mainWin(listbox) cellcget $urlIndex,0 -text]
            set urlsToDownload($url,options) [array get ::downOptionsTemp]
            if {($getleftState(downloading)==1)                             \
                    &&([string match $getleftState(url) $url])} {
                array set ::downOptions $urlsToDownload($url,options)
            }
        }
    }

    return
}

################################################################################
# UrlListGetIndex
#   When the user clicks in the url list, this procedure takes care of selecting
#   the nearest entry and returns its index.
#
# Parameters
#   win: window where you clicked.
#   y: the coordinate where you did it
#
# Returns
#   The index of the url, or -1 if the is none to return
################################################################################
proc UrlListGetIndex {win y} {
    variable mainWin   

    set y [expr "$y + [winfo y $win]"]
    set index [$mainWin(listbox) nearest $y]
    if {$index!=-1} {
        set anchor [$mainWin(listbox) curselection]
        if {[lsearch $anchor $index]==-1} {
            $mainWin(listbox) selection clear 0 end
        }
        $mainWin(listbox) selection set $index $index
	}
	return $index
}

################################################################################
# UrlListRightClick
#   This procedure is invoked when the user right-clicks in the multicolumn
#   list with the sites to download, it makes a context menu pop up with the
#   appropiate options
#
# Parameters
#   win: window where you right-clicked.
#   y,x: the coordinates where you did it for the tablelist widget.
#   X,Y: and for the pop up menu.
################################################################################
proc UrlListRightClick {win x y X Y} {
    variable mainWin
    variable menus
    global getleftState urlsToDownload

    set index [UrlListGetIndex $win $y]
    if {$index==-1} return

    ActivateDownloadButtons

    $menus(rightClick) entryconfigure 0                                 \
            -command "AutomaticDownload  [mainWin::UrlListSelectedUrl]"
    $menus(rightClick) entryconfigure 2                                 \
            -command {mainWin::UrlListChangeOptions .                   \
                    "[$mainWin::mainWin(listbox) curselection]"}
    $menus(rightClick) entryconfigure 4                                 \
            -command {mainWin::UrlListCopyLink                          \
                    "[$mainWin::mainWin(listbox) curselection]"}
    $menus(rightClick) entryconfigure 6                                 \
            -command "
                Ayuda::InvokeBrowser [mainWin::UrlListSelectedUrl] .
            "
    $menus(rightClick) entryconfigure 8                                 \
            -command {mainWin::UrlListMoveTopBottom                     \
                    "[$mainWin::mainWin(listbox) curselection]" top}
    $menus(rightClick) entryconfigure 9                                 \
            -command {mainWin::UrlListMoveEntry                         \
                    "[$mainWin::mainWin(listbox) curselection]" -1}
    $menus(rightClick) entryconfigure 10                                \
            -command {mainWin::UrlListMoveEntry                         \
                    "[$mainWin::mainWin(listbox) curselection]"  1}
    $menus(rightClick) entryconfigure 11                                \
            -command {mainWin::UrlListMoveTopBottom                     \
                    "[$mainWin::mainWin(listbox) curselection]" bottom}
    $menus(rightClick) entryconfigure 13                                \
            -command {mainWin::UrlListDeleteUrls                        \
                    "[$mainWin::mainWin(listbox) curselection]"}

    if {$getleftState(downloading)==0} {
        $menus(rightClick) entryconfigure 0  -state normal
        $menus(rightClick) entryconfigure 13 -state normal
    } else {
        $menus(rightClick) entryconfigure 0  -state disable
	    set url [$mainWin(listbox) cellcget $index,0 -text]
        if {[string match $url $getleftState(url)]} {
            $menus(rightClick) entryconfigure 13 -state disable
        } else {
            $menus(rightClick) entryconfigure 13 -state normal
        }
    }
    tk_popup $menus(rightClick) $X $Y
    return
}

################################################################################
# UrlListDoubleClick
#   This procedure is invoked when the user double-clicks in the multicolumn
#   list with the sites to download, it will try to start downloading the site.
#
# Parameters
#   win: window where you double-clicked.
#   y: the coordinate where you did it
################################################################################
proc UrlListDoubleClick {win y} {
    variable mainWin
    global getleftState

    if {$getleftState(downloading)!=0} return

    set index [UrlListGetIndex $win $y]
    if {$index==-1} return

    set url [$mainWin(listbox) cellcget $index,0 -text]

    AutomaticDownload $url

    return
}

###############################################################################
# UrlListClearSel
#    Clears the selection in the site listbox.
###############################################################################
proc UrlListClearSel {} {
    variable mainWin

    $mainWin(listbox) selection clear 0 end

    return
}

###############################################################################
# SetResumeSitesMenu
#    Puts the entries in the "Resume Sites" menu.
###############################################################################
proc SetResumeSitesMenu {} {
    global getleftState

    .menus.fichero.resume delete 0 end

    foreach site [lsort $getleftState(urlQueue)] {
        .menus.fichero.resume add command -label $site            \
                -command "AutomaticDownload $site"
    }
    return
}

###############################################################################
# MenusLabels
#    Puts the labels into the menus.
#
# Parameters:
#    labFile: file with the labels in the, hopefully, desired language
###############################################################################
proc MenusLabels {{labFile ""}} {
    global labelButtons labelTitles labelMessages labelDialogs labelMonths
    global labelMenus indexMenus dirGetleft tcl_platform
    variable menus

    if {$labFile==""} {
        set labFile en
    }

    ReadLangFile $labFile

    .menus      entryconfigure 1 -label $labelMenus(file)           \
            -underline $indexMenus(file)
    $menus(1)   entryconfigure 0 -label $labelMenus(enterUrl)       \
            -underline $indexMenus(enterUrl)
    $menus(1)   entryconfigure 1 -label $labelMenus(siteMap)        \
            -underline $indexMenus(siteMap)
    $menus(1)   entryconfigure 2 -label $labelMenus(resumeSite)     \
            -underline $indexMenus(resumeSite)
    $menus(1)   entryconfigure 4 -label $labelMenus(exit)           \
            -underline $indexMenus(exit)

    .menus      entryconfigure 2 -label $labelMenus(options)        \
            -underline $indexMenus(options)
    $menus(2)   entryconfigure 0 -label $labelMenus(all)            \
            -underline $indexMenus(all)
    $menus(2)   entryconfigure 1 -label $labelMenus(uplinks)        \
            -underline $indexMenus(uplinks)
    $menus(2,0) entryconfigure 0 -label $labelMenus(linksFollow)    \
            -underline $indexMenus(linksFollow)
    $menus(2,0) entryconfigure 1 -label $labelMenus(linksIgnore)    \
            -underline $indexMenus(linksIgnore)
    $menus(2)   entryconfigure 2 -label $labelMenus(levels)         \
            -underline $indexMenus(levels)
    $menus(2,0) entryconfigure 2 -label $labelMenus(linksIgnore)    \
            -underline $indexMenus(linksIgnore)
    $menus(2,1) entryconfigure 0 -label $labelMenus(noLimit)        \
            -underline $indexMenus(noLimit)
    $menus(2)   entryconfigure 3 -label $labelMenus(external)       \
            -underline $indexMenus(external)
    $menus(2,2) entryconfigure 0 -label $labelMenus(linksIgnore)    \
            -underline $indexMenus(linksIgnore)
    $menus(2)   entryconfigure 4 -label $labelMenus(fileFilter)     \
            -underline $indexMenus(fileFilter)
    $menus(2,3) entryconfigure 0 -label $labelMenus(onlyHtml)       \
            -underline $indexMenus(onlyHtml)
    $menus(2,3) entryconfigure 1 -label $labelMenus(images)         \
            -underline $indexMenus(images)
    $menus(2,3,1) entryconfigure 0 -label $labelMenus(allImages)    \
            -underline $indexMenus(allImages)
    $menus(2,3,1) entryconfigure 1 -label $labelMenus(onlyThumb)    \
            -underline $indexMenus(onlyThumb)
    $menus(2,3,1) entryconfigure 2 -label $labelMenus(noThumb)      \
            -underline $indexMenus(noThumb)
    $menus(2,3) entryconfigure 2 -label $labelMenus(chooseFilter)   \
            -underline $indexMenus(chooseFilter)
    $menus(2)   entryconfigure 5 -label $labelMenus(resume)         \
            -underline $indexMenus(resume)
    $menus(2)   entryconfigure 6 -label $labelMenus(update)         \
            -underline $indexMenus(update)
    $menus(2)   entryconfigure 7 -label $labelMenus(cgi)            \
            -underline $indexMenus(cgi)
    $menus(2)   entryconfigure 8 -label $labelMenus(useProxy)       \
            -underline $indexMenus(useProxy)

    .menus      entryconfigure 3 -label $labelMenus(tools)          \
            -underline $indexMenus(tools)
    $menus(3)   entryconfigure 0 -label $labelMenus(purgeFiles)     \
            -underline $indexMenus(purgeFiles)
    $menus(3)   entryconfigure 1 -label $labelMenus(restoreOrig)    \
            -underline $indexMenus(restoreOrig)
    $menus(3)   entryconfigure 2 -label $labelMenus(configureProxy) \
            -underline $indexMenus(configureProxy)
    $menus(3)   entryconfigure 3 -label $labelMenus(getLog)         \
            -underline $indexMenus(getLog)
    $menus(3)   entryconfigure 4 -label $labelMenus(languages)      \
            -underline $indexMenus(languages)
    $menus(3)   entryconfigure 5 -label Iconos                      \
            -underline $indexMenus(languages)
    if {$tcl_platform(platform)=="unix"} {
        $menus(3)   entryconfigure 6 -label $labelMenus(browser)    \
                -underline $indexMenus(browser)
    }

    .menus      entryconfigure 4 -label $labelMenus(help)           \
            -underline $indexMenus(help)
    $menus(4)   entryconfigure 0 -label $labelMenus(manual)         \
            -underline $indexMenus(manual)
    $menus(4)   entryconfigure 2 -label $labelMenus(changes)        \
            -underline $indexMenus(changes)
    $menus(4)   entryconfigure 3 -label $labelMenus(license)        \
            -underline $indexMenus(license)
    $menus(4)   entryconfigure 5 -label $labelMenus(about)          \
            -underline $indexMenus(about)

    return
}

###############################################################################
# ShowMenu1
#    Before posting the file menu, this procedure decides which entries
#    should be active and which will have to wait
###############################################################################
proc ShowMenu1 {} {
    global getleftState getleftOptions

    if {$getleftState(downloading)==0} {
        .menus.fichero entryconfigure 2 -state normal
        if {[llength $getleftState(urlQueue)]==0} {
            .menus.fichero entryconfigure 2 -state disabled
        }
    } else {
        .menus.fichero entryconfigure 2 -state disabled
    }

    return
}

###############################################################################
# CreateMenus
#   Creates the menus, without putting the labels
###############################################################################
proc CreateMenus {} {
    global getleftOptions dirGetleft supportedLang tcl_platform
    variable menus

    menu .menus -relief flat
    . config -menu .menus

    set menus(1) [menu .menus.fichero      -tearoff 0                    \
            -postcommand mainWin::ShowMenu1]
    .menus add cascade -menu $menus(1)
    set menus(2) [menu .menus.editar       -tearoff 0]
    .menus add cascade -menu $menus(2)
    set menus(3) [menu .menus.herramientas -tearoff 0]
    .menus add cascade -menu $menus(3)
    set menus(4) [menu .menus.help         -tearoff 0]
    .menus add cascade -menu $menus(4)

    $menus(1) add command -command  EnterUrl::EnterUrl
    $menus(1) add command -command "EnterUrl::EnterUrl 1"
    $menus(1) add cascade -menu $menus(1).resume
    $menus(1) add separator
    $menus(1) add command -command ExitGetleft

    set menus(1,2) [menu $menus(1).resume -tearoff 0]
    SetResumeSitesMenu

    $menus(2) add command -command "mainWin::UrlListChangeOptions ."
    $menus(2) add cascade -menu $menus(2).directorios
    $menus(2) add cascade -menu $menus(2).levels
    $menus(2) add cascade -menu $menus(2).exLevels
    $menus(2) add cascade -menu $menus(2).filters
    $menus(2) add check   -variable downOptions(resume)   -onvalue 1 -offvalue 0 
    $menus(2) add check   -variable downOptions(update)   -onvalue 1 -offvalue 0 \
        -command {
            mainWin::UpdateUrlOptions
            catch {unset urlsDownloaded}
        }
    $menus(2) add check   -variable downOptions(cgi)      -onvalue 1 -offvalue 0 \
        -command {
            mainWin::UpdateUrlOptions
            catch {unset urlsDownloaded}
        }
    $menus(2) add check   -variable getleftOptions(proxy) -onvalue 1 -offvalue 0

    set menus(2,0) [menu $menus(2).directorios -tearoff 0]
    $menus(2,0) add radio -variable downOptions(dir) -value 1             \
            -command {
                mainWin::UpdateUrlOptions
                catch {unset urlsDownloaded}
            }
    $menus(2,0) add radio -variable downOptions(dir) -value 0             \
            -command {
                mainWin::UpdateUrlOptions
                catch {unset urlsDownloaded}
            }

    set menus(2,1) [menu $menus(2).levels -tearoff 0]
    $menus(2,1) add radio -variable downOptions(levels) -value -1
    for {set i 0} {$i<6} {incr i} {
        $menus(2,1) add radio -label $i -variable downOptions(levels)     \
                -value $i -command {
                              mainWin::UpdateUrlOptions
                              catch {unset urlsDownloaded}
                          }
    }

    set menus(2,2) [menu $menus(2).exLevels -tearoff 0]
    $menus(2,2) add radio -variable downOptions(exLevels) -value 0        \
            -command {
                set downOptions(external) 0
                mainWin::UpdateUrlOptions
                catch {unset urlsDownloaded}
            }
    for {set i 1} {$i<4} {incr i} {
        $menus(2,2) add radio -label $i -variable downOptions(exLevels)   \
            -value $i -command {
                set downOptions(external) 1
                mainWin::UpdateUrlOptions
                catch {unset urlsDownloaded}
            }
    }

    set menus(2,3) [menu $menus(2).filters -tearoff 0]
    $menus(2,3) add check -variable downOptions(onlyHtml)                 \
                -onvalue 1 -offvalue 0 -command {
                    mainWin::UpdateUrlOptions
                    catch {unset urlsDownloaded}
                }
    $menus(2,3) add cascade -menu $menus(2,3).images
    $menus(2,3) add command -command {
                Herramientas::FilterFiles $downOptions(filter)
                mainWin::UpdateUrlOptions
            }
    set menus(2,3,1) [menu $menus(2,3).images -tearoff 0]
    $menus(2,3,1) add radio -variable downOptions(images) -value 0        \
            -command {
                mainWin::UpdateUrlOptions
                catch {unset urlsDownloaded}
            }
    $menus(2,3,1) add radio -variable downOptions(images) -value 1        \
            -command {
                mainWin::UpdateUrlOptions
                catch {unset urlsDownloaded}
            }
    $menus(2,3,1) add radio -variable downOptions(images) -value 2        \
            -command {
                mainWin::UpdateUrlOptions
                catch {unset urlsDownloaded}
            }

    $menus(3) add command -command Herramientas::PurgeFiles
    $menus(3) add command -command Herramientas::RestoreOriginals
    $menus(3) add command -command Herramientas::ConfProxy
    $menus(3) add command -command getLog::FileLog
    $menus(3) add cascade -menu $menus(3).idiomas
    $menus(3) add cascade -menu $menus(3).icons

    if {$tcl_platform(platform)=="unix"} {
        $menus(3) add command -command "Ayuda::ChooseLinuxBrowser"
    }

    set menus(3,0) [menu $menus(3).idiomas -tearoff 0]
    set i 0
    foreach lang $supportedLang(langList) {
        if {[file exists [file join $dirGetleft(languages)                \
                menus.$supportedLang($lang)]]} {
            $menus(3,0) add command                                       \
                    -command "ChangeLanguage $supportedLang($lang)"
            $menus(3,0) entryconfigure $i -label $lang
            incr i
        }
    }

    set menus(3,1) [menu $menus(3).icons -tearoff 0]
    foreach iconSet [lsort [glob $dirGetleft(images)/*/]] {
        $menus(3,1) add radio -variable dirGetleft(icons)                 \
            -value [string trimright $iconSet /]                          \
            -label [file tail $iconSet] -command "
                ChangeIconSet
            "
    }    

    $menus(4) add command -command Ayuda::Manual
    $menus(4) add separator
    $menus(4) add command -command Ayuda::Changes
    $menus(4) add command -command Ayuda::Licence
    $menus(4) add separator
    $menus(4) add command -command Ayuda::About

    MenusLabels $getleftOptions(lang)

    return
}

###############################################################################
# MainWindow
#    Defines how the main window of Getleft looks.
###############################################################################
proc MainWindow {} {
    global   getleftState getleftOptions urlsToDownload labelMenus labelDialogs
    global   dirGetleft   tcl_platform   tcl_patchLevel
    variable menus
    variable taskbar
    variable mainWin

    wm title    . "Getleft v1.1"
    wm minsize  . 300 165
    wm geometry . 660x280

    if {[winfo exists .menus]} {
        destroy .menus
        destroy .menu
        destroy .extFrame
    }

    CreateMenus

    set extFrame  [frame .extFrame]
    set iconFrame [frame $extFrame.iconFrame]
    set listFrame [frame $extFrame.listFrame -bd 2 -relief sunken]
    set ctrFrame  [frame $extFrame.ctrFrame]

    set mainWin(listFrame) $listFrame

    set leftIconFrame  [frame $iconFrame.left ]
    set rightIconFrame [frame $iconFrame.right]
    DefineTaskbarButtons $leftIconFrame $rightIconFrame

    set mainWin(scrollY) [scrollbar $listFrame.scrolly -orient vertical    \
            -command  [list $listFrame.listbox yview]]
    set mainWin(scrollX) [scrollbar $listFrame.scrollx -orient horizontal  \
            -command  [list $listFrame.listbox xview]]

    set mainWin(listbox) [::tablelist::tablelist $listFrame.listbox        \
            -bd 0 -height 10 -width 90 -stretch 3 -selectmode extended     \
            -fg $getleftOptions(fg) -bg $getleftOptions(bg)                \
            -columns [list 44 $labelDialogs(url)      left                 \
                            0 $labelDialogs(resumed)  center               \
                           20 $labelDialogs(localDir) left]                \
            -xscrollcommand mainWin::ShowScrollX                           \
            -yscrollcommand mainWin::ShowScrollY                           \
            -exportselection 0]

    set intCtrFrame [frame $ctrFrame.int -bd 2 -relief sunken]
    set ctrLabel    [label $intCtrFrame.label                              \
        -textvariable mainWin::mainWin(ctrFrameLabel)]

    set backColor [$intCtrFrame cget -background]
    $ctrLabel          configure -bg      $backColor
    $mainWin(listbox)  configure -labelbg $backColor

    grid $extFrame       -row 0 -sticky news
    grid $iconFrame             -sticky ew   -padx 5
    grid $rightIconFrame -row 0 -sticky e
    grid $leftIconFrame  -row 0 -sticky w
    grid $taskbar(url)     $taskbar(start) $taskbar(auto) $taskbar(delay)  \
         $taskbar(log)   
    grid $taskbar(fileLog) $taskbar(help)  $taskbar(get)  -sticky e
    grid columnconfigure   $iconFrame 0 -weight 1
    grid $listFrame -sticky news -pady 2
    grid $mainWin(listbox) -in $listFrame -row 0 -column 0 -sticky news    \
            -padx 0 -pady 0

    grid rowconfigure    .          0 -weight 1
    grid columnconfigure .          0 -weight 1
    grid columnconfigure $extFrame  0 -weight 1
    grid columnconfigure $listFrame 0 -weight 1
    grid columnconfigure $listFrame 1 -weight 0 -minsize 0
    grid rowconfigure    $listFrame 0 -weight 1
    grid rowconfigure    $listFrame 1 -weight 0 -minsize 0

    grid $ctrFrame       -sticky news
    grid $intCtrFrame    -sticky news
    grid $ctrLabel       -sticky w
    grid rowconfigure    $extFrame 0    -weight 0
    grid rowconfigure    $extFrame 1    -weight 1
    grid rowconfigure    $extFrame 2    -weight 0
    grid columnconfigure $ctrFrame 0    -weight 1
    grid columnconfigure $intCtrFrame 0 -weight 1

    grid propagate $listFrame 0

    if {![info exists getleftState(urlQueue)]} {
        return
    }

    focus [$mainWin(listbox) bodypath]

    bind [$mainWin(listbox) bodypath] <ButtonRelease-1> "
        focus [$mainWin(listbox) bodypath]
    "
    bind [$mainWin(listbox) bodypath] <Double-Button-1> {
        mainWin::UrlListDoubleClick %W %y
    }
    bind [$mainWin(listbox) bodypath] <Button-3> {
        event generate %W <Button-1>        -x %x -y %y
        event generate %W <ButtonRelease-1> -x %x -y %y
        mainWin::UrlListRightClick %W %x %y %X %Y
    }
    bind [$mainWin(listbox) bodypath] <Return> {
        mainWin::UrlListDoubleClick %W %y
    }
    bind [$mainWin(listbox) bodypath] <KP_Enter> {
        mainWin::UrlListDoubleClick %W %y
    }
    bind [$mainWin(listbox) bodypath] <FocusIn> {
        $mainWin::mainWin(listbox) selection clear 0 end
        $mainWin::mainWin(listbox) selection set active
        mainWin::ActivateDownloadButtons
    }
    bind [$mainWin(listbox) bodypath] <KeyRelease-Next> {
        $mainWin::mainWin(listbox) selection clear 0 end
        $mainWin::mainWin(listbox) selection set active
    }
    bind [$mainWin(listbox) bodypath] <KeyRelease-Prior> {
        $mainWin::mainWin(listbox) selection clear 0 end
        $mainWin::mainWin(listbox) selection set active
    }
    bind . <Escape> {
        $mainWin::mainWin(listbox) selection clear 0 end
    }
    bind . <Key-Delete> {
        mainWin::UrlListDeleteUrls [$mainWin::mainWin(listbox) curselection]
        $mainWin::mainWin(listbox) selection set active
    }
    bind . <<ListboxSelect>> {
        mainWin::ActivateDownloadButtons
    }

    wm protocol . WM_DELETE_WINDOW {
        ExitGetleft
    }

    bind Button <Return>   {%W invoke}
    bind Button <KP_Enter> {%W invoke}

    foreach site $getleftState(urlQueue) {
        UrlListEnterEntry $site $urlsToDownload($site,resume)              \
                                $urlsToDownload($site,dir)
    }
    $mainWin(listbox) yview moveto 0

    ActivateTaskbarButtons

    set menus(rightClick) [menu .menu -tearoff 0]
    $menus(rightClick) add command -label $labelMenus(resume)
    $menus(rightClick) add separator
    $menus(rightClick) add command -label $labelMenus(options)
    $menus(rightClick) add separator
    $menus(rightClick) add command -label $labelDialogs(copyLink)
    $menus(rightClick) add separator
    $menus(rightClick) add command -label $labelDialogs(launchLink)
    $menus(rightClick) add separator
    $menus(rightClick) add command -label $labelMenus(top)
    $menus(rightClick) add command -label $labelMenus(up)
    $menus(rightClick) add command -label $labelMenus(down)
    $menus(rightClick) add command -label $labelMenus(bottom)
    $menus(rightClick) add separator
    $menus(rightClick) add command -label $labelMenus(delete)
 
    
    if {$tcl_platform(platform)=="windows"} {
        if {[regexp {((8\.3\.)(3|4))||(8.4)} $tcl_patchLevel]} {
            catch {wm iconbitmap . -default [file join $dirGetleft(images) icon.ico]}
        }
    }
    return
}
}
