/*
 * CrazySumForPasswords v.3
 *
 * AUTHOR:
 *  Robert Fahy <rfahy@ymail.com>
 *
 * DATE:
 *  December 2010
 *
 * ABOUT:
 *  - the old versions were poorly written and commented, and the executables were insecure
 *  - this version is cleaner, and slightly more professional (more secure and more maintainable)
 *  - this version produces output similar to old CSFP-P, but they are not compatible!
 *  - my style of writing has also changed a lot since August '09 (mostly for the better, I think)
 *
 * TODO:
 *  - maybe incorporate SHA512 algorithm into own program... or maybe not
 *  - maybe use something other (better) than pipes (like sockets?)
 *  - add a `-b' binary option?
 *
 */

// include files
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/wait.h>

// preprocessor definitions with obvious purposes
#define CSFP_NAME_VERSION				"CSFP v.3 Final"
#define CSFP_GENERIC_PROGRAM_FILENAME	"<csfp3.elf>"
#define CSFP_GENERIC_MESSAGE			"<Error text is unavailable.>"
#define CSFP_DEFAULT_PASS_SIZE			128
#define CSFP_MAX_PASS_SIZE				4096

// ABOUT: un-comment to allow printing a newline character after password
//#define CSFP_PRINT_NEWLINE

// ABOUT: un-comment to allow clearing of environment variables
// (has effect only inside this program and its children)
#define CSFP_CLEAR_ENVIRON

// ABOUT: maximum length to accept in csfp_strlen() function
#define CSFP_STRLEN_MAXLENGTH			CSFP_MAX_PASS_SIZE

// declarations without definitions (aka prototypes)
char * csfp_strcpy(char *dest, const char *src);
size_t csfp_strlen(const char *str);
size_t csfp_hasletters(const char *str);
void print_usage(const char *program_filename, const char *message);
int compute(const char *argument1, const char *argument2);
void encrypt(char *result, const char *original, const unsigned int size);
int main(int argc, char **argv); // for completeness of this "index"

char * csfp_strcpy(char *dest, const char *src)
// ABOUT: a stupider version of strcpy
// WARN: requires length of `dest' to be one unit larger than length of `src'!
{
	size_t
		i=0;

	const size_t // NOTE: variable added for performance
		src_length_ = csfp_strlen(src);

	if (src != NULL &&
		*src != '\0' &&
		dest != NULL)
	{
		while (i < src_length_)
		{
			dest[i] = src[i];
			++i;
		}

		dest[src_length_] = '\0';

		return dest;
	}
	else
		return NULL;
} // ### int csfp_strcpy(char *dest, const char *src)

size_t csfp_strlen(const char *str)
// ABOUT: a slightly more secure version of strlen(), which stops counting bytes after a predefined limit
{
#ifndef CSFP_STRLEN_MAXLENGTH
#define CSFP_STRLEN_MAXLENGTH			4096
#endif

	size_t i=0;

	if (str != NULL)	// WARN: safer but possibly redundant!
		while (str[i] != '\0')
			if (i < CSFP_STRLEN_MAXLENGTH)
				++i;
			else
				break;

	return i;
} // ### size_t csfp_strlen(const char *str)

size_t csfp_hasletters(const char *str)
// ABOUT: returns non-zero value if finds a non-digit character in `str' (that includes symbols)
{
	size_t i=0;

	if (str != NULL)	// WARN: safer but possibly redundant!
		while (str[i] != '\0')
			if (!isdigit(str[i++]))
				return i;

	return 0;
} // ### size_t csfp_hasletters(const char *str)

void print_usage(const char *program_filename, const char *message)
// ABOUT: prints usage help plus additional `message'
{
	char
		*program_filename_ = NULL,
		*message_ = NULL;

	// check against bogus `program_filename'
	if (program_filename != NULL &&
		*program_filename != '\0')
	{
		if ((program_filename_ = (char *)malloc(csfp_strlen(program_filename) + 1))
			!= NULL)
			csfp_strcpy(program_filename_, program_filename);
		else
			return;
	}
	else	// bad `program_filename'
		if ((program_filename_ = (char *)malloc(csfp_strlen(CSFP_GENERIC_PROGRAM_FILENAME) + 1))
			!= NULL)
			csfp_strcpy(program_filename_, CSFP_GENERIC_PROGRAM_FILENAME);
		else
			return;

	// check against bogus `message'
	if (message != NULL &&
		*message != '\0')
	{
		if ((message_ = (char *)malloc(csfp_strlen(message) + 1))
			!= NULL)
			csfp_strcpy(message_, message);
		else
			return;
	}
	else	// bad `message'
		if ((message_ = (char *)malloc(csfp_strlen(CSFP_GENERIC_MESSAGE) + 1))
			!= NULL)
			csfp_strcpy(message_, CSFP_GENERIC_MESSAGE);
		else
			return;

	printf ("\n%s usage:\n %s <[file_to_be_checksumed] [desired_length_of_result]>\n"
		"Also accepting standard input (stdin) and providing length is not compulsory\n"
		" (default is \"%d\", maximum is \"%d\").\n\nERROR: %s\n\n\n",
		CSFP_NAME_VERSION, program_filename_, CSFP_DEFAULT_PASS_SIZE, CSFP_MAX_PASS_SIZE, message_);

	free(program_filename_);
	program_filename_ = NULL;
	free(message_);
	message_ = NULL;
} // ### void print_usage(const char *program_filename, const char *message)

int compute(const char *argument1, const char *argument2)
// ABOUT: verifies `argument1' and `argument2' and computes the password data accordingly
{
	int
		chksum2prog_[2],	// needed for "pipe" data transfer from checksum utility to our program
		exit_status_;		// needed for returning EXIT_FAILURE or EXIT_SUCCESS status

	pid_t pid_;				// needed to determine which process we're in after forking

	unsigned int desired_length_ = CSFP_DEFAULT_PASS_SIZE;	// desired length of the computed password

	char
		sha512_checksum_[128],		// will store the first 128 characters of sha512sum's output
		*csfp_checksum_,			// will store the CSFP password-checksum
		*execlp_argument_ = NULL;	// file argument for execlp's call to `sha512sum' (NULL for stdin)

	if (argument1 != NULL)
	{
		// optimistically presume that `argument1' is filename
		if ((execlp_argument_ = (char *)malloc(csfp_strlen(argument1) + 1))
			!= NULL)
			csfp_strcpy(execlp_argument_, argument1);
		else
		{
			perror("compute() -> malloc(execlp_argument_)");
			exit(EXIT_FAILURE);
		}

		if (argument2 == NULL)
		{
			if (csfp_hasletters(argument1))
			// we have letters, consider `argument1' to be a filename
				desired_length_ = CSFP_DEFAULT_PASS_SIZE;
			else // `argument1' is all-digit, consider it to be desired length
			{
				free(execlp_argument_);
				execlp_argument_ = NULL;
				desired_length_ = (unsigned int)atoi(argument1);
			}
		}
		else // consider `argument1' to be filename, `argument2' to be desired length
			desired_length_ = (unsigned int)atoi(argument2);

		// make sure we stay within bounds
		if (desired_length_ > CSFP_MAX_PASS_SIZE)
			desired_length_ = CSFP_MAX_PASS_SIZE;
	}

	if (pipe(chksum2prog_) == -1)
	// initialise pipe and check success
	{
		perror("compute() -> pipe(chksum2prog_)");
		exit(EXIT_FAILURE);
	}

	if ((pid_ = fork()) < 0)
	// check for unsuccessful fork
	{
		perror("compute() -> fork()");
		exit(EXIT_FAILURE);
	}
	else
	{
		if (!pid_) // child process, calls `/usr/bin/sha512sum' and returns checksum through pipe to parent
		{
			exit_status_ = EXIT_SUCCESS;
#ifdef CSFP_CLEAR_ENVIRON
			clearenv();
#endif
			close(chksum2prog_[0]);		// close input end of the pipe
			dup2(chksum2prog_[1], 1);	// bind stdout to output end of the pipe

			if (execlp("/usr/bin/sha512sum", "/usr/bin/sha512sum", "--", execlp_argument_, NULL) < 0)
			// check for unsuccessful exec
			{
				perror("compute() -> execlp(\"/usr/bin/sha512sum\")");
				exit_status_ = EXIT_FAILURE;
			}
		}
		else // parent process, gets checksum through pipe from child, mangles and outputs it
		{
			exit_status_ = EXIT_SUCCESS;
			close(chksum2prog_[1]);	// close output end of the pipe
			read(chksum2prog_[0], sha512_checksum_, sizeof sha512_checksum_);

			if ((csfp_checksum_ = (char *)malloc(desired_length_ + 1))
				!= NULL)
			{
				//csfp_checksum_[desired_length_] = '\0'; // NOTE: this is redundant
				encrypt(csfp_checksum_, sha512_checksum_, desired_length_);
				fputs(csfp_checksum_, stdout);
#ifdef CSFP_PRINT_NEWLINE
				putchar('\n');
#endif
				free(csfp_checksum_);
				csfp_checksum_ = NULL;
			}
			else
			{
				perror("compute() -> malloc(csfp_checksum_)");
				exit_status_ = EXIT_FAILURE;
			}

			wait(NULL);	// wait for child process to terminate
		}

		if (execlp_argument_ != NULL)
		{
			free(execlp_argument_);
			execlp_argument_ = NULL;
		}

		return exit_status_;
	}
} // ### int compute(const char *argument1, const char *argument2)

void encrypt(char *result, const char *original, const unsigned int size)
// ABOUT: changes the `result' array by processing `original'
{
	// TODO: change this array to better suit your needs (be careful not to lose your program afterwards)
	//  also, be careful not to break the special `\\' and `\"' characters
	const char symbol_pool_[] =
		"0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ%/\\'`\"[]{}()<>*+-=?!_^&$#@~|;:.,";

	// NOTE: these variables vastly increase performance
	const size_t
		symbol_pool_length_ = csfp_strlen(symbol_pool_),
		original_length_ = csfp_strlen(original);

	unsigned int
		i=0,
		k=0,
		salt_ = 0;	// this is probably not what specialists would call "salt"

	if (original != NULL &&
		*original != '\0' &&
		result != NULL)
	{
		while (i < size)
		{
			salt_ += (unsigned int)original[k++];
			salt_ += size * k;

			while (salt_ >= symbol_pool_length_)
				salt_ -= symbol_pool_length_;

			result[i++] = symbol_pool_[salt_];

			if (k == original_length_)
				k=0;
		}

		result[size] = '\0'; // make sure our array ends
	}
	else
		exit(EXIT_FAILURE);
} // ### void encrypt(char *result, const char *original, const unsigned int size)

// ### MAIN FUNCTION ###
int main(int argc, char **argv)
// ### MAIN FUNCTION ###
{
	if (argc > 3)
	{
		print_usage(argv[0], "Too many arguments.");
		return EXIT_FAILURE;
	}
	else
		if (argc < 1)
		// NOTE: this shouldn't normally be possible except with a
		//  specially crafted exec() call from another program
		{
			print_usage(CSFP_GENERIC_PROGRAM_FILENAME, "Nice try, Joe.");
			return EXIT_FAILURE;
		}
		else
			switch (argc)
			{
				case 1: return compute(NULL, NULL);			// case 1: standard input
				case 2: return compute(argv[1], NULL);		// case 2a and 2b: standard input and desired length, or filename
				case 3: return compute(argv[1], argv[2]);	// case 3: filename and desired length
				default:
					print_usage(argv[0], "Unknown. Email <rfahy@ymail.com> and include detail, please.");
					return EXIT_FAILURE;
			}
} // ### MAIN FUNCTION ###
