/*
 * coreop.c
 *
 * AUTHORS: Robert Fahy <rfahy@ymail.com>
 * CREDITS: n/a
 * VERSION: 1.01
 *
 */

/*
 * TASK:
 * - define core operations designed strictly for XOR-FD
 *
 */

/*
 * TODO:
 * - output user-friendly error messages
 * - should use `print_usage()' instead of `PRINT_ERROR_NOERRNO()' in some cases
 * - check against soft and hard links
 * - maybe should revert to old style variable declarations?
 *
 */

// standard include files
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>
#include <string.h>

// environment-specific include files
#include <sys/types.h>

// custom include files
#include "globals.h"
#include "defaults.h"
#include "baseop.h"
#include "error.h"
#include "coreop.h"

char * encrypt_static(char *data, size_t length)
// ABOUT: uses XOR operation to pad contents of `data' with contents of `_global_padding_' for given `length'
// WARN: this version loops `_global_padding_' and it assumes that it doesn't change.
//  it also takes into account `_global_padding_length_'. this is a design flaw I'm aware of.
{
	static size_t i=0;

	if (data != NULL)
		while (length--)
		{
			if (i >= _global_padding_length_)
			// reset `i' to keep within bounds
				i=0;

			*data++ ^= _global_padding_[i++];
		}
	else
		return NULL;

	return data;
} // ### char * encrypt_static(char *data, size_t length)

char * encrypt_normal(char *data, const char *padding, size_t length)
// ABOUT: uses XOR operation to pad contents of `data' with contents of `padding' for given `length'
// WARN: both arrays need to have same `length'
{
	if (data != NULL &&
		padding != NULL)
		while (length--)
			*data++ ^= *padding++;
	else
		return NULL;

	return data;
} // ### char * encrypt_normal(char *data, const char *padding, size_t length)

int compute_case1(const char *argument1)
// ABOUT: manages "case 1", `argument1' is input/output filename and key is read
//  from the Standard Input stream (stdin)
{
	// check filename/pathname for basic sanity
	if (argument1 == NULL ||
		*argument1 == '\0')
	{
		print_usage(DEFAULT_PROGRAM_FILENAME, "Must provide a valid filename for input/output file.");
		return EXIT_FAILURE;
	}

	FILE *input_file_ = NULL;

	// attempt to open the input/output file
	if ((input_file_ = fopen(argument1, "r+b"))
		== NULL)
	{
		PRINT_ERROR("fopen()");
		return EXIT_FAILURE;
	}

	off_t input_file_size_ = 0; // needed for percentage calculus

	// check if input file size is 0 (zero) or less (which is bogus)
	if ((input_file_size_ = get_file_size(input_file_))
		<= 0)
	{
		PRINT_ERROR_NOERRNO("Input file `%s' does not contain any data or cannot be accessed.", argument1);
		FCLOSE_VERBOSE(input_file_);

		return EXIT_FAILURE;
	}

	// NOTE: we don't check this for errors because the program will jam waiting anyway,
	//  if no data is fed through standard input (stdin)
	_global_padding_length_ = fread(_global_padding_, sizeof (char), STDIN_BUFFER_SIZE, stdin);

	printf("\n\n%s\n\n* Input/Output file:\t%s\n* Progress:\t  0%%",
		DEFAULT_NAME_VERSION, argument1);

	size_t rw_length_ = 0; // keep track of how many bytes we read or wrote
	char rw_buffer_[BUFFER_SIZE]; // read/write file buffer

	off_t // keep track of the total number of bytes we read or wrote
		total_bytes_read_ = 0,
		total_bytes_written_ = 0;

	while ((rw_length_ = fread(rw_buffer_, sizeof (char), BUFFER_SIZE, input_file_)))
	// did we read something? yes...
		if (_global_run_flag_ == XORFD_CONTINUE) // check if we're not halting
		{
			// check the error indicator in `input_file_' stream
			if (ferror(input_file_))
			{
				PRINT_ERROR("fread() + ferror()");
				FCLOSE_VERBOSE(input_file_);

				return EXIT_FAILURE;
			}

			total_bytes_read_ += rw_length_;

			// check if chunk encryption failed
			if (encrypt_static(rw_buffer_, rw_length_) == NULL)
			{
				PRINT_ERROR_NOERRNO("encrypt_static()");
				FCLOSE_VERBOSE(input_file_);

				return EXIT_FAILURE;
			}

			// we must seek to previous position for to overwrite,
			// so check the seek for errors
			if (fseek(input_file_, 0 - (long)rw_length_, SEEK_CUR) == -1)
			{
				PRINT_ERROR("fseek()");
				FCLOSE_VERBOSE(input_file_);

				return EXIT_FAILURE;
			}

			total_bytes_written_ += fwrite(rw_buffer_, sizeof (char), rw_length_, input_file_);

			// check the error indicator in `input_file_' stream, again
			if (ferror(input_file_))
			{
				PRINT_ERROR("fwrite() + ferror()");
				FCLOSE_VERBOSE(input_file_);

				return EXIT_FAILURE;
			}

			// flush stream and check for error
			if (fflush(input_file_) == EOF)
			{
				PRINT_ERROR("fflush()");
				FCLOSE_VERBOSE(input_file_);

				return EXIT_FAILURE;
			}

			// print progress percentage
			printf("\b\b\b\b%3d%%", get_percentage(input_file_size_, total_bytes_written_));
		}
		else // _global_run_flag_ != XORFD_CONTINUE
		{
			printf("\n* Caught a signal to abort.\n");
			break;
		}

	FCLOSE_VERBOSE(input_file_);

	if (_global_run_flag_ == XORFD_CONTINUE)
		printf(" - operation completed.");
	else
		printf("* Released the resources.");

	printf("\n* Total bytes read:\t%ld\n"
		"* Total bytes written:\t%ld\n\n",
		(long int)total_bytes_read_,
		(long int)total_bytes_written_);

	return EXIT_SUCCESS;
} // ### int compute_case1(const char *argument1)

int compute_case2(const char *argument1, const char *argument2)
// ABOUT: manages "case 2", `argument1' is input filename, `argument2' is output filename
//  and key is read from stdin
{
	// check filename/pathname for basic sanity
	if (argument1 == NULL ||
		*argument1 == '\0')
	{
		print_usage(DEFAULT_PROGRAM_FILENAME, "Must provide a valid filename for input file.");
		return EXIT_FAILURE;
	}

	// check filename/pathname for basic sanity
	if (argument2 == NULL ||
		*argument2 == '\0')
	{
		print_usage(DEFAULT_PROGRAM_FILENAME, "Must provide a valid filename for output file.");
		return EXIT_FAILURE;
	}

	// check against same filename
	if (!strncmp(argument1, argument2, MAX_PATHNAME_LENGTH))
	{
		PRINT_ERROR_NOERRNO("Input file and output file cannot be the same.\n\n"
			"(NOTE: This program does not recognize pathnames longer than %u characters.)",
			MAX_PATHNAME_LENGTH);

		return EXIT_FAILURE;
	}

	FILE *input_file_ = NULL;

	// attempt to open the input file
	if ((input_file_ = fopen(argument1, "rb"))
		== NULL)
	{
		PRINT_ERROR("fopen()");
		return EXIT_FAILURE;
	}

	off_t input_file_size_ = 0; // needed for percentage calculus

	// check if input file size is 0 (zero) or less (which is bogus)
	if ((input_file_size_ = get_file_size(input_file_))
		<= 0)
	{
		PRINT_ERROR_NOERRNO("Input file `%s' does not contain any data or cannot be accessed.", argument1);
		FCLOSE_VERBOSE(input_file_);

		return EXIT_FAILURE;
	}

	FILE *output_file_ = NULL;

	// attempt to open the output file
	if ((output_file_ = fopen(argument2, "wb"))
		== NULL)
	{
		PRINT_ERROR("fopen()");
		FCLOSE_VERBOSE(input_file_);

		return EXIT_FAILURE;
	}

	// NOTE: we don't check this for errors because the program will jam waiting anyway,
	//  if no data is fed through standard input (stdin)
	_global_padding_length_ = fread(_global_padding_, sizeof (char), STDIN_BUFFER_SIZE, stdin);

	printf("\n\n%s\n\n* Input file:\t%s\n* Output file:\t%s\n* Progress:\t  0%%",
		DEFAULT_NAME_VERSION, argument1, argument2);

	size_t rw_length_ = 0; // keep track of how many bytes we read or wrote
	char rw_buffer_[BUFFER_SIZE]; // read/write file buffer

	off_t // keep track of the total number of bytes we read or wrote
		total_bytes_read_ = 0,
		total_bytes_written_ = 0;

	while (!feof(input_file_))
		if (_global_run_flag_ == XORFD_CONTINUE) // check if we're not halting
		{
			total_bytes_read_ +=
			rw_length_ = fread(rw_buffer_, sizeof (char), BUFFER_SIZE, input_file_);

			// check the error indicator in `input_file_' stream
			if (ferror(input_file_))
			{
				PRINT_ERROR("fread() + ferror()");
				FCLOSE_VERBOSE(input_file_);
				FCLOSE_VERBOSE(output_file_);

				return EXIT_FAILURE;
			}

			// check if chunk encryption failed
			if (encrypt_static(rw_buffer_, rw_length_) == NULL)
			{
				PRINT_ERROR_NOERRNO("encrypt_static()");
				FCLOSE_VERBOSE(input_file_);
				FCLOSE_VERBOSE(output_file_);

				return EXIT_FAILURE;
			}

			total_bytes_written_ += fwrite(rw_buffer_, sizeof (char), rw_length_, output_file_);

			// check the error indicator in `output_file_' stream
			if (ferror(output_file_))
			{
				PRINT_ERROR("fwrite() + ferror()");
				FCLOSE_VERBOSE(input_file_);
				FCLOSE_VERBOSE(output_file_);

				return EXIT_FAILURE;
			}

			// print progress percentage
			printf("\b\b\b\b%3d%%", get_percentage(input_file_size_, total_bytes_written_));
		}
		else // _global_run_flag_ != XORFD_CONTINUE
		{
			printf("\n* Caught a signal to abort.\n");
			break;
		}

	FCLOSE_VERBOSE(input_file_);
	FCLOSE_VERBOSE(output_file_);

	if (_global_run_flag_ == XORFD_CONTINUE)
		printf(" - operation completed.");
	else
		printf("* Released the resources.");

	printf("\n* Total bytes read:\t%ld\n"
		"* Total bytes written:\t%ld\n\n",
		(long int)total_bytes_read_,
		(long int)total_bytes_written_);

	return EXIT_SUCCESS;
} // ### int compute_case2(const char *argument1, const char *argument2)

int compute_case3(const char *argument1, const char *argument2, const char *argument3)
// ABOUT: manages "case 3", `argument1' is input filename, `argument2' is output filename
//  and `argument3' is key filename
{
	// check filename/pathname for basic sanity
	if (argument1 == NULL ||
		*argument1 == '\0')
	{
		print_usage(DEFAULT_PROGRAM_FILENAME, "Must provide a valid filename for input file.");
		return EXIT_FAILURE;
	}

	// check filename/pathname for basic sanity
	if (argument2 == NULL ||
		*argument2 == '\0')
	{
		print_usage(DEFAULT_PROGRAM_FILENAME, "Must provide a valid filename for output file.");
		return EXIT_FAILURE;
	}

	// check filename/pathname for basic sanity
	if (argument3 == NULL ||
		*argument3 == '\0')
	{
		print_usage(DEFAULT_PROGRAM_FILENAME, "Must provide a valid filename for key file.");
		return EXIT_FAILURE;
	}

	// check against same filename (input and output files)
	if (!strncmp(argument1, argument2, MAX_PATHNAME_LENGTH))
	{
		PRINT_ERROR_NOERRNO("Input file and output file cannot be the same.\n\n"
			"(NOTE: This program does not recognize pathnames longer than %u characters.)",
			MAX_PATHNAME_LENGTH);

		return EXIT_FAILURE;
	}

	// check against same filename (input and key files)
	if (!strncmp(argument1, argument3, MAX_PATHNAME_LENGTH))
	{
		PRINT_ERROR_NOERRNO("Input file and key file cannot be the same.\n\n"
			"(NOTE: This program does not recognize pathnames longer than %u characters.)",
			MAX_PATHNAME_LENGTH);

		return EXIT_FAILURE;
	}

	// check against same filename (output and key files)
	if (!strncmp(argument2, argument3, MAX_PATHNAME_LENGTH))
	{
		PRINT_ERROR_NOERRNO("Output file and key file cannot be the same.\n\n"
			"(NOTE: This program does not recognize pathnames longer than %u characters.)",
			MAX_PATHNAME_LENGTH);

		return EXIT_FAILURE;
	}

	FILE *input_file_ = NULL;

	// attempt to open the input file
	if ((input_file_ = fopen(argument1, "rb"))
		== NULL)
	{
		PRINT_ERROR("fopen()");
		return EXIT_FAILURE;
	}

	off_t input_file_size_ = 0; // needed for percentage calculus

	// check if input file size is 0 (zero) or less (which is bogus)
	if ((input_file_size_ = get_file_size(input_file_))
		<= 0)
	{
		PRINT_ERROR_NOERRNO("Input file `%s' does not contain any data or cannot be accessed.", argument1);
		FCLOSE_VERBOSE(input_file_);

		return EXIT_FAILURE;
	}

	FILE *output_file_ = NULL;

	// attempt to open the output file
	if ((output_file_ = fopen(argument2, "wb"))
		== NULL)
	{
		PRINT_ERROR("fopen()");
		FCLOSE_VERBOSE(input_file_);

		return EXIT_FAILURE;
	}

	FILE *key_file_ = NULL;

	// attempt to open the key file
	if ((key_file_ = fopen(argument3, "rb"))
		== NULL)
	{
		PRINT_ERROR("fopen()");
		FCLOSE_VERBOSE(input_file_);
		FCLOSE_VERBOSE(output_file_);

		return EXIT_FAILURE;
	}

	// check if key file size is 0 (zero) or less (which is bogus)
	if (get_file_size_tame(key_file_) <= 0)
	{
		PRINT_ERROR_NOERRNO("Key file `%s' does not contain any data or cannot be accessed.", argument3);
		FCLOSE_VERBOSE(input_file_);
		FCLOSE_VERBOSE(output_file_);
		FCLOSE_VERBOSE(key_file_);

		return EXIT_FAILURE;
	}

	printf("\n\n%s\n\n* Input file:\t%s\n* Output file:\t%s\n* Key file:\t%s\n* Progress:\t  0%%",
		DEFAULT_NAME_VERSION, argument1, argument2, argument3);

	char
		rw_buffer_[BUFFER_SIZE],	// read/write file buffer
		key_buffer_[BUFFER_SIZE];	// key file buffer

	off_t // to keep track of the total number of bytes we read or wrote
		total_bytes_read_ = 0,
		total_bytes_written_ = 0;

	size_t // to keep track of how many bytes we read or wrote
		rw_length_ = 0,
		key_length_ = 0,
		buffer_free_ = BUFFER_SIZE; // how much of the buffer is free

	while (!feof(input_file_))
		if (_global_run_flag_ == XORFD_CONTINUE)
		{
			key_length_ = 0;

			total_bytes_read_ +=
			buffer_free_ =
			rw_length_ = fread(rw_buffer_, sizeof (char), BUFFER_SIZE, input_file_);

			// check the error indicator in `input_file_' stream
			if (ferror(input_file_))
			{
				PRINT_ERROR("fread() + ferror()");
				FCLOSE_VERBOSE(input_file_);
				FCLOSE_VERBOSE(output_file_);
				FCLOSE_VERBOSE(key_file_);

				return EXIT_FAILURE;
			}

			while (key_length_ < rw_length_)
			{
				key_length_ += fread(key_buffer_ + key_length_, sizeof (char), buffer_free_, key_file_);

				// check the error indicator in `key_file_' stream
				if (ferror(key_file_))
				{
					PRINT_ERROR("fread() + ferror()");
					FCLOSE_VERBOSE(input_file_);
					FCLOSE_VERBOSE(output_file_);
					FCLOSE_VERBOSE(key_file_);

					return EXIT_FAILURE;
				}

				buffer_free_ = rw_length_ - key_length_;

				if (feof(key_file_))
					rewind(key_file_); // FIXME: use fseeko() instead for error-checking?
			}

			// check if chunk encryption failed
			if (encrypt_normal(rw_buffer_, key_buffer_, rw_length_) == NULL)
			{
				PRINT_ERROR_NOERRNO("encrypt_normal()");
				FCLOSE_VERBOSE(input_file_);
				FCLOSE_VERBOSE(output_file_);
				FCLOSE_VERBOSE(key_file_);

				return EXIT_FAILURE;
			}

			total_bytes_written_ += fwrite(rw_buffer_, sizeof (char), rw_length_, output_file_);

			// check the error indicator in `output_file_' stream
			if (ferror(output_file_))
			{
				PRINT_ERROR("fwrite() + ferror()");
				FCLOSE_VERBOSE(input_file_);
				FCLOSE_VERBOSE(output_file_);
				FCLOSE_VERBOSE(key_file_);

				return EXIT_FAILURE;
			}

			// print progress percentage
			printf("\b\b\b\b%3d%%", get_percentage(input_file_size_, total_bytes_written_));
		}
		else // _global_run_flag_ != XORFD_CONTINUE
		{
			printf("\n* Caught a signal to abort.\n");
			break;
		}

	FCLOSE_VERBOSE(input_file_);
	FCLOSE_VERBOSE(output_file_);
	FCLOSE_VERBOSE(key_file_);

	if (_global_run_flag_ == XORFD_CONTINUE)
		printf(" - operation completed.");
	else
		printf("* Cleanup OK.");

	printf("\n* Total bytes read:\t%ld\n"
		"* Total bytes written:\t%ld\n\n",
		(long int)total_bytes_read_,
		(long int)total_bytes_written_);

	return EXIT_SUCCESS;
} // ### int compute_case3(const char *argument1, const char *argument2, const char *argument3)
