/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.sling.graphql.core.it;

import javax.inject.Inject;
import javax.script.ScriptEngineFactory;

import static com.jayway.jsonpath.matchers.JsonPathMatchers.hasJsonPath;

import org.apache.sling.graphql.api.SchemaProvider;
import org.apache.sling.graphql.core.mocks.ReplacingSchemaProvider;
import org.apache.sling.resource.presence.ResourcePresence;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.ops4j.pax.exam.Configuration;
import org.ops4j.pax.exam.Option;
import org.ops4j.pax.exam.junit.PaxExam;
import org.ops4j.pax.exam.spi.reactors.ExamReactorStrategy;
import org.ops4j.pax.exam.spi.reactors.PerClass;
import org.ops4j.pax.exam.util.Filter;
import org.osgi.framework.BundleContext;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.ops4j.pax.exam.cm.ConfigurationAdminOptions.factoryConfiguration;

@RunWith(PaxExam.class)
@ExamReactorStrategy(PerClass.class)
public class ServerSideQueryIT extends GraphQLCoreTestSupport {

    @Inject
    @Filter(value = "(names=graphql)")
    protected ScriptEngineFactory scriptEngineFactory;

    @Inject
    @Filter(value = "(path=/apps/graphql/test/one/json.gql)")
    private ResourcePresence resourcePresence;

    @Inject
    private BundleContext bundleContext;

    @Inject
    private SchemaProvider defaultSchemaProvider;

    @Configuration
    public Option[] configuration() {
        return new Option[]{
            baseConfiguration(),
            testDataFetchers(),
            factoryConfiguration("org.apache.sling.resource.presence.internal.ResourcePresenter")
                .put("path", "/apps/graphql/test/one/json.gql")
                .asOption(),
        };
    }

    private void assertDefaultContent(String selector, String fieldName) throws Exception {
        final String path = "/graphql/one";
        final String json = getContent(path + selector + ".json");
        assertThat(json, hasJsonPath("$.data." + fieldName));
        assertThat(json, hasJsonPath("$.data." + fieldName + ".path", equalTo("/content/graphql/one")));
        assertThat(json, hasJsonPath("$.data." + fieldName + ".resourceType", equalTo("graphql/test/one")));
    }

    @Test
    public void testJsonContent() throws Exception {
        assertDefaultContent("", "oneSchemaResource");
    }

    @Test
    public void testMultipleSchemaProviders() throws Exception {
        new ReplacingSchemaProvider("oneSchemaResource", "REPLACED").register(bundleContext, defaultSchemaProvider, Integer.MAX_VALUE);
        new ReplacingSchemaProvider("oneSchemaResource", "NOT_THIS_ONE").register(bundleContext, defaultSchemaProvider, 1);
        assertDefaultContent(".REPLACED", "REPLACED");
    }

    @Test
    public void testPaginatedQuery() throws Exception {
        final String json = getContent("graphql/one.query.json");
        assertThat(json, hasJsonPath("$.data.oneSchemaQuery.pageInfo.startCursor"));
        assertThat(json, hasJsonPath("$.data.oneSchemaQuery.edges[0].node.path", equalTo("/content/graphql/one")));
        assertThat(json, hasJsonPath("$.data.oneSchemaQuery.edges[1].node.path", equalTo("/content/graphql")));
        assertThat(json, hasJsonPath("$.data.oneSchemaQuery.edges[1].node.resourceType", equalTo("graphql/test/root")));
        assertThat(json, hasJsonPath("$.data.oneSchemaQuery.edges[2].node.path", equalTo("/content")));
    }
}
