/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.hono.client.kafka;

import io.vertx.core.Future;
import io.vertx.core.Promise;
import io.vertx.core.Vertx;
import io.vertx.kafka.admin.KafkaAdminClient;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.config.ConfigException;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KafkaClientFactory {
    public static final Duration UNLIMITED_RETRIES_DURATION = Duration.ofSeconds(-1L);
    public static final int CLIENT_CREATION_RETRY_DELAY_MILLIS = 1000;
    private static final Logger LOG = LoggerFactory.getLogger(KafkaClientFactory.class);
    private static final Pattern COMMA_WITH_WHITESPACE = Pattern.compile("\\s*,\\s*");
    private final Vertx vertx;
    private Clock clock = Clock.systemUTC();

    public KafkaClientFactory(Vertx vertx) {
        this.vertx = Objects.requireNonNull(vertx);
    }

    void setClock(Clock clock) {
        this.clock = Objects.requireNonNull(clock);
    }

    public <T> Future<T> createClientWithRetries(Supplier<T> clientSupplier, String bootstrapServersConfig, Duration retriesTimeout) {
        return this.createClientWithRetries(clientSupplier, () -> true, bootstrapServersConfig, retriesTimeout);
    }

    public <T> Future<T> createClientWithRetries(Supplier<T> clientSupplier, Supplier<Boolean> keepTrying, String bootstrapServersConfig, Duration retriesTimeout) {
        Objects.requireNonNull(clientSupplier);
        Objects.requireNonNull(keepTrying);
        Promise resultPromise = Promise.promise();
        this.createClientWithRetries(clientSupplier, keepTrying, this.getRetriesTimeLimit(retriesTimeout), () -> KafkaClientFactory.containsValidServerEntries(bootstrapServersConfig), resultPromise);
        return resultPromise.future();
    }

    public Future<KafkaAdminClient> createKafkaAdminClientWithRetries(Map<String, String> clientConfig, Supplier<Boolean> keepTrying, Duration retriesTimeout) {
        Objects.requireNonNull(clientConfig);
        Objects.requireNonNull(keepTrying);
        Promise resultPromise = Promise.promise();
        this.createClientWithRetries(() -> KafkaAdminClient.create(this.vertx, clientConfig), keepTrying, this.getRetriesTimeLimit(retriesTimeout), () -> KafkaClientFactory.containsValidServerEntries((String)clientConfig.get("bootstrap.servers")), resultPromise);
        return resultPromise.future();
    }

    private Instant getRetriesTimeLimit(Duration retriesTimeout) {
        if (retriesTimeout == null || retriesTimeout.isNegative()) {
            return Instant.MAX;
        }
        if (retriesTimeout.isZero()) {
            return Instant.MIN;
        }
        return Instant.now(this.clock).plus(retriesTimeout);
    }

    private <T> void createClientWithRetries(Supplier<T> clientSupplier, Supplier<Boolean> keepTrying, Instant retriesTimeLimit, Supplier<Boolean> serverEntriesValid, Promise<T> resultPromise) {
        if (!keepTrying.get().booleanValue()) {
            resultPromise.fail("client code has canceled further attempts to create Kafka client");
            return;
        }
        try {
            T client = clientSupplier.get();
            LOG.debug("successfully created client [type: {}]", (Object)client.getClass().getName());
            resultPromise.complete(client);
        }
        catch (Exception e) {
            if (!retriesTimeLimit.equals(Instant.MIN) && e instanceof KafkaException && KafkaClientFactory.isBootstrapServersConfigException(e.getCause()) && serverEntriesValid.get().booleanValue()) {
                if (!keepTrying.get().booleanValue()) {
                    LOG.debug("client code has canceled further attempts to create Kafka client");
                    resultPromise.fail(e);
                } else if (Instant.now(this.clock).isBefore(retriesTimeLimit)) {
                    LOG.debug("error creating Kafka client, will retry in {}ms: {}", (Object)1000, (Object)e.getCause().getMessage());
                    this.vertx.setTimer(1000L, tid -> this.createClientWithRetries(clientSupplier, keepTrying, retriesTimeLimit, () -> true, resultPromise));
                } else {
                    LOG.warn("error creating Kafka client (no further attempts will be done, timeout for retries reached): {}\n", (Object)e.getCause().getMessage());
                    resultPromise.fail(e);
                }
            }
            LOG.warn("failed to create client due to terminal error (won't retry)", e);
            resultPromise.fail(e);
        }
    }

    public static boolean isRetriableClientCreationError(Throwable exception, String bootstrapServersConfig) {
        return exception instanceof KafkaException && KafkaClientFactory.isBootstrapServersConfigException(exception.getCause()) && KafkaClientFactory.containsValidServerEntries(bootstrapServersConfig);
    }

    private static boolean isBootstrapServersConfigException(Throwable ex) {
        return ex instanceof ConfigException && ex.getMessage() != null && ex.getMessage().contains("bootstrap.servers");
    }

    private static boolean containsValidServerEntries(String bootstrapServersConfig) {
        List urlList = Optional.ofNullable(bootstrapServersConfig).map(serversString -> {
            String trimmed = serversString.trim();
            if (trimmed.isEmpty()) {
                return List.of();
            }
            return Arrays.asList(COMMA_WITH_WHITESPACE.split(trimmed, -1));
        }).orElseGet(List::of);
        return !urlList.isEmpty() && urlList.stream().allMatch(KafkaClientFactory::containsHostAndPort);
    }

    private static boolean containsHostAndPort(String url) {
        try {
            return Utils.getHost(url) != null && Utils.getPort(url) != null;
        }
        catch (IllegalArgumentException e) {
            return false;
        }
    }
}

