/**
 * Copyright (c) 2010-2012, Abel Hegedus, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Abel Hegedus - initial API and implementation
 */
package org.eclipse.incquery.testing.core;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import com.google.inject.Injector;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.incquery.patternlanguage.emf.eMFPatternLanguage.PatternModel;
import org.eclipse.incquery.runtime.api.IMatchProcessor;
import org.eclipse.incquery.runtime.api.IPatternMatch;
import org.eclipse.incquery.runtime.api.IQuerySpecification;
import org.eclipse.incquery.runtime.api.IncQueryEngine;
import org.eclipse.incquery.runtime.api.IncQueryMatcher;
import org.eclipse.incquery.snapshot.EIQSnapshot.IncQuerySnapshot;
import org.eclipse.incquery.snapshot.EIQSnapshot.MatchRecord;
import org.eclipse.incquery.snapshot.EIQSnapshot.MatchSetRecord;
import org.eclipse.incquery.snapshot.EIQSnapshot.MatchSubstitutionRecord;
import org.eclipse.incquery.testing.core.ModelLoadHelper;
import org.eclipse.incquery.testing.core.SnapshotHelper;
import org.eclipse.incquery.testing.core.TestingLogAppender;
import org.eclipse.incquery.testing.queries.UnexpectedMatchRecordMatch;
import org.eclipse.incquery.testing.queries.UnexpectedMatchRecordMatcher;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.junit.Assert;

/**
 * Primitive methods for executing a functional test for EMF-IncQuery.
 */
@SuppressWarnings("all")
public class TestExecutor {
  public final static String CORRECTRESULTS = "Correct result set";
  
  public final static String CORRECT_SINGLE = "Correct single match for parameterless pattern";
  
  public final static String CORRECT_EMPTY = "Correct empty match set";
  
  public final static String UNEXPECTED_MATCH = "Unexpected match";
  
  public final static String EXPECTED_NOT_FOUND = "Expected match not found";
  
  public final static String MULTIPLE_FOR_EXPECTED = "Multiple matches found for expected match";
  
  public final static String MATCHSETRECORD_NOT_IN_SNAPSHOT = "Expected match set record is not part of snapshot";
  
  public final static String PATTERNNAME_DIFFERENT = "Expected pattern qualified name different from actual";
  
  @Inject
  @Extension
  private ModelLoadHelper _modelLoadHelper;
  
  @Inject
  @Extension
  private SnapshotHelper _snapshotHelper;
  
  @Inject
  private Injector injector;
  
  @Inject
  private Logger logger;
  
  /**
   * Checks the pattern name of the matcher against the one stored in
   *  the record and checks parameterless patterns as well.
   * 
   * Returns true if further comparison is allowed, false otherwise.
   */
  public boolean validateMatcherBeforeCompare(final IncQueryMatcher matcher, final MatchSetRecord expected, final Set diff) {
    String _patternName = matcher.getPatternName();
    String _patternQualifiedName = expected.getPatternQualifiedName();
    boolean _equals = _patternName.equals(_patternQualifiedName);
    boolean _not = (!_equals);
    if (_not) {
      String _patternQualifiedName_1 = expected.getPatternQualifiedName();
      String _plus = ((TestExecutor.PATTERNNAME_DIFFERENT + " (") + _patternQualifiedName_1);
      String _plus_1 = (_plus + "!=");
      String _patternName_1 = matcher.getPatternName();
      String _plus_2 = (_plus_1 + _patternName_1);
      String _plus_3 = (_plus_2 + ")");
      diff.add(_plus_3);
      return false;
    }
    List _parameterNames = matcher.getParameterNames();
    int _size = _parameterNames.size();
    boolean _equals_1 = (_size == 0);
    if (_equals_1) {
      EList<MatchRecord> _matches = expected.getMatches();
      int _size_1 = _matches.size();
      boolean _equals_2 = (_size_1 == 1);
      if (_equals_2) {
        int _countMatches = matcher.countMatches();
        boolean _equals_3 = (_countMatches == 1);
        if (_equals_3) {
          diff.add(TestExecutor.CORRECT_SINGLE);
          return true;
        } else {
          int _countMatches_1 = matcher.countMatches();
          boolean _equals_4 = (_countMatches_1 == 0);
          if (_equals_4) {
            diff.add(TestExecutor.CORRECT_EMPTY);
            return true;
          }
        }
      }
    }
    return true;
  }
  
  /**
   * Compares the match set of a given matcher with the given match record
   *  using EMF-IncQuery as a compare tool.
   * Therefore the comparison depends on correct EMF-IncQuery query evaluation
   *  (for a given limited pattern language feature set).
   */
  public HashSet<String> compareResultSetsAsRecords(final IncQueryMatcher matcher, final MatchSetRecord expected) {
    try {
      final HashSet<String> diff = CollectionLiterals.<String>newHashSet();
      boolean correctResults = this.validateMatcherBeforeCompare(matcher, expected, diff);
      if ((!correctResults)) {
        return diff;
      }
      EObject _eContainer = expected.eContainer();
      if ((!(_eContainer instanceof IncQuerySnapshot))) {
        diff.add(TestExecutor.MATCHSETRECORD_NOT_IN_SNAPSHOT);
        return diff;
      }
      EObject _eContainer_1 = expected.eContainer();
      final IncQuerySnapshot snapshot = ((IncQuerySnapshot) _eContainer_1);
      Notifier _eMFRootForSnapshot = this._snapshotHelper.getEMFRootForSnapshot(snapshot);
      final IncQueryEngine engine = IncQueryEngine.on(_eMFRootForSnapshot);
      IQuerySpecification<UnexpectedMatchRecordMatcher> _querySpecification = UnexpectedMatchRecordMatcher.querySpecification();
      final UnexpectedMatchRecordMatcher unexpectedMatcher = _querySpecification.getMatcher(engine);
      MatchRecord _filter = expected.getFilter();
      final IPatternMatch partialMatch = this._snapshotHelper.createMatchForMachRecord(matcher, _filter);
      final MatchSetRecord actual = this._snapshotHelper.saveMatchesToSnapshot(matcher, partialMatch, snapshot);
      final IMatchProcessor<UnexpectedMatchRecordMatch> _function = new IMatchProcessor<UnexpectedMatchRecordMatch>() {
        @Override
        public void process(final UnexpectedMatchRecordMatch it) {
          String _prettyPrint = it.prettyPrint();
          String _plus = ((TestExecutor.UNEXPECTED_MATCH + " (") + _prettyPrint);
          String _plus_1 = (_plus + ")");
          diff.add(_plus_1);
        }
      };
      unexpectedMatcher.forEachMatch(actual, expected, null, _function);
      final IMatchProcessor<UnexpectedMatchRecordMatch> _function_1 = new IMatchProcessor<UnexpectedMatchRecordMatch>() {
        @Override
        public void process(final UnexpectedMatchRecordMatch it) {
          String _prettyPrint = it.prettyPrint();
          String _plus = ((TestExecutor.EXPECTED_NOT_FOUND + " (") + _prettyPrint);
          String _plus_1 = (_plus + ")");
          diff.add(_plus_1);
        }
      };
      unexpectedMatcher.forEachMatch(expected, actual, null, _function_1);
      return diff;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Compares the match set of a given matcher with the given match record using the
   *  records as partial matches on the matcher.
   * Therefore the comparison does not depend on correct EMF-IncQuery query evaluation.
   */
  public HashSet<String> compareResultSets(final IncQueryMatcher matcher, final MatchSetRecord expected) {
    final HashSet<String> diff = CollectionLiterals.<String>newHashSet();
    boolean correctResults = this.validateMatcherBeforeCompare(matcher, expected, diff);
    if ((!correctResults)) {
      return diff;
    }
    final ArrayList<IPatternMatch> foundMatches = CollectionLiterals.<IPatternMatch>newArrayList();
    EList<MatchRecord> _matches = expected.getMatches();
    for (final MatchRecord matchRecord : _matches) {
      {
        final IPatternMatch partialMatch = this._snapshotHelper.createMatchForMachRecord(matcher, matchRecord);
        final int numMatches = matcher.countMatches(partialMatch);
        if ((numMatches == 0)) {
          StringBuilder _printMatchRecord = this.printMatchRecord(matchRecord);
          String _plus = ((TestExecutor.EXPECTED_NOT_FOUND + " (") + _printMatchRecord);
          String _plus_1 = (_plus + ")");
          diff.add(_plus_1);
          correctResults = false;
        } else {
          if ((numMatches == 1)) {
            foundMatches.add(partialMatch);
          } else {
            StringBuilder _printMatchRecord_1 = this.printMatchRecord(matchRecord);
            String _plus_2 = ((TestExecutor.MULTIPLE_FOR_EXPECTED + " (") + _printMatchRecord_1);
            String _plus_3 = (_plus_2 + ")");
            diff.add(_plus_3);
            correctResults = false;
          }
        }
      }
    }
    MatchRecord _filter = expected.getFilter();
    IPatternMatch _createMatchForMachRecord = this._snapshotHelper.createMatchForMachRecord(matcher, _filter);
    final IMatchProcessor<IPatternMatch> _function = new IMatchProcessor<IPatternMatch>() {
      @Override
      public void process(final IPatternMatch it) {
        boolean _contains = foundMatches.contains(it);
        boolean _not = (!_contains);
        if (_not) {
          String _prettyPrint = it.prettyPrint();
          String _plus = ((TestExecutor.UNEXPECTED_MATCH + " (") + _prettyPrint);
          String _plus_1 = (_plus + ")");
          diff.add(_plus_1);
        }
      }
    };
    matcher.forEachMatch(_createMatchForMachRecord, _function);
    return diff;
  }
  
  public StringBuilder printMatchRecord(final MatchRecord record) {
    StringBuilder _xblockexpression = null;
    {
      final StringBuilder sb = new StringBuilder();
      EObject _eContainer = record.eContainer();
      final MatchSetRecord matchSet = ((MatchSetRecord) _eContainer);
      EList<MatchSubstitutionRecord> _substitutions = record.getSubstitutions();
      final Procedure1<MatchSubstitutionRecord> _function = new Procedure1<MatchSubstitutionRecord>() {
        @Override
        public void apply(final MatchSubstitutionRecord it) {
          int _length = sb.length();
          boolean _greaterThan = (_length > 0);
          if (_greaterThan) {
            sb.append(",");
          }
          String _parameterName = it.getParameterName();
          StringBuilder _append = sb.append(_parameterName);
          StringBuilder _append_1 = _append.append("=");
          Object _derivedValue = it.getDerivedValue();
          _append_1.append(_derivedValue);
        }
      };
      IterableExtensions.<MatchSubstitutionRecord>forEach(_substitutions, _function);
      String _patternQualifiedName = matchSet.getPatternQualifiedName();
      String _plus = (_patternQualifiedName + "(");
      sb.insert(0, _plus);
      _xblockexpression = sb.append(")");
    }
    return _xblockexpression;
  }
  
  /**
   * Compares match set of each matcher initialized from the given pattern model
   *  based on the input specification of the snapshot.
   * If any of the matchers return incorrect results, the assert fails.
   */
  public void assertMatchResults(final PatternModel patternModel, final IncQuerySnapshot snapshot) {
    try {
      final HashSet<Object> diff = CollectionLiterals.<Object>newHashSet();
      final Notifier input = this._snapshotHelper.getEMFRootForSnapshot(snapshot);
      final IncQueryEngine engine = IncQueryEngine.on(input);
      this.registerLogger(engine);
      EList<MatchSetRecord> _matchSetRecords = snapshot.getMatchSetRecords();
      final Procedure1<MatchSetRecord> _function = new Procedure1<MatchSetRecord>() {
        @Override
        public void apply(final MatchSetRecord matchSet) {
          String _patternQualifiedName = matchSet.getPatternQualifiedName();
          final IncQueryMatcher<? extends IPatternMatch> matcher = TestExecutor.this._modelLoadHelper.initializeMatcherFromModel(patternModel, engine, _patternQualifiedName);
          boolean _notEquals = (!Objects.equal(matcher, null));
          if (_notEquals) {
            final HashSet<String> result = TestExecutor.this.compareResultSets(matcher, matchSet);
            boolean _or = false;
            boolean _or_1 = false;
            boolean _equals = Objects.equal(result, null);
            if (_equals) {
              _or_1 = true;
            } else {
              HashSet<String> _newHashSet = CollectionLiterals.<String>newHashSet(TestExecutor.CORRECT_EMPTY);
              boolean _equals_1 = _newHashSet.equals(result);
              _or_1 = _equals_1;
            }
            if (_or_1) {
              _or = true;
            } else {
              HashSet<String> _newHashSet_1 = CollectionLiterals.<String>newHashSet(TestExecutor.CORRECT_SINGLE);
              boolean _equals_2 = _newHashSet_1.equals(result);
              _or = _equals_2;
            }
            boolean _not = (!_or);
            if (_not) {
              diff.addAll(result);
            }
          }
        }
      };
      IterableExtensions.<MatchSetRecord>forEach(_matchSetRecords, _function);
      String _logDifference = this.logDifference(diff, engine);
      boolean _isEmpty = diff.isEmpty();
      Assert.assertTrue(_logDifference, _isEmpty);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Compares match set of each matcher initialized from the given pattern model
   *  based on the input specification of the snapshot (specified as a platform URI).
   * If any of the matchers return incorrect results, the assert fails.
   */
  public void assertMatchResults(final PatternModel patternModel, final String snapshotUri) {
    final IncQuerySnapshot snapshot = this._modelLoadHelper.loadExpectedResultsFromUri(snapshotUri);
    this.assertMatchResults(patternModel, snapshot);
  }
  
  /**
   * Compares match set of each matcher initialized from the given pattern model (specified as a platform URI)
   *  based on the input specification of the snapshot (specified as a platform URI).
   * If any of the matchers return incorrect results, the assert fails.
   */
  public void assertMatchResults(final String patternUri, final String snapshotUri) {
    final PatternModel patternModel = this._modelLoadHelper.loadPatternModelFromUri(patternUri, this.injector);
    this.assertMatchResults(patternModel, snapshotUri);
  }
  
  public void registerLogger(final IncQueryEngine engine) {
    TestingLogAppender _testingLogAppender = new TestingLogAppender();
    this.logger.addAppender(_testingLogAppender);
  }
  
  public String retrieveLoggerOutput(final IncQueryEngine engine) {
    final Logger logger = this.logger;
    final Enumeration appers = logger.getAllAppenders();
    while (appers.hasMoreElements()) {
      {
        final Object apper = appers.nextElement();
        if ((apper instanceof TestingLogAppender)) {
          StringBuilder _output = ((TestingLogAppender) apper).getOutput();
          return _output.toString();
        }
      }
    }
    return "Logger output not recorded";
  }
  
  public String logDifference(final Set<?> diff) {
    String _xblockexpression = null;
    {
      final StringBuilder stringBuilder = new StringBuilder();
      this.logDifference(diff, stringBuilder);
      _xblockexpression = stringBuilder.toString();
    }
    return _xblockexpression;
  }
  
  public String logDifference(final Set<Object> diff, final IncQueryEngine engine) {
    String _xblockexpression = null;
    {
      final StringBuilder stringBuilder = new StringBuilder();
      this.logDifference(diff, stringBuilder);
      String _retrieveLoggerOutput = this.retrieveLoggerOutput(engine);
      stringBuilder.append(_retrieveLoggerOutput);
      _xblockexpression = stringBuilder.toString();
    }
    return _xblockexpression;
  }
  
  private void logDifference(final Set<?> diff, final StringBuilder stringBuilder) {
    final Procedure1<Object> _function = new Procedure1<Object>() {
      @Override
      public void apply(final Object it) {
        stringBuilder.append(("\n" + it));
      }
    };
    IterableExtensions.forEach(diff, _function);
  }
}
