package org.eclipse.incquery.testing.core;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import com.google.inject.Inject;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.incquery.patternlanguage.emf.eMFPatternLanguage.PatternModel;
import org.eclipse.incquery.runtime.api.IMatchProcessor;
import org.eclipse.incquery.runtime.api.IPatternMatch;
import org.eclipse.incquery.runtime.api.IQuerySpecification;
import org.eclipse.incquery.runtime.api.IncQueryEngine;
import org.eclipse.incquery.runtime.api.IncQueryMatcher;
import org.eclipse.incquery.snapshot.EIQSnapshot.IncQuerySnapshot;
import org.eclipse.incquery.snapshot.EIQSnapshot.MatchRecord;
import org.eclipse.incquery.snapshot.EIQSnapshot.MatchSetRecord;
import org.eclipse.incquery.snapshot.EIQSnapshot.MatchSubstitutionRecord;
import org.eclipse.incquery.testing.core.ModelLoadHelper;
import org.eclipse.incquery.testing.core.SnapshotHelper;
import org.eclipse.incquery.testing.core.TestingLogAppender;
import org.eclipse.incquery.testing.queries.UnexpectedMatchRecordMatch;
import org.eclipse.incquery.testing.queries.UnexpectedMatchRecordMatcher;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.junit.Assert;

/**
 * Primitive methods for executing a functional test for EMF-IncQuery.
 */
@SuppressWarnings("all")
public class TestExecutor {
  public final static String CORRECTRESULTS = "Correct result set";
  
  public final static String CORRECT_SINGLE = "Correct single match for parameterless pattern";
  
  public final static String CORRECT_EMPTY = "Correct empty match set";
  
  public final static String UNEXPECTED_MATCH = "Unexpected match";
  
  public final static String EXPECTED_NOT_FOUND = "Expected match not found";
  
  public final static String MULTIPLE_FOR_EXPECTED = "Multiple matches found for expected match";
  
  public final static String MATCHSETRECORD_NOT_IN_SNAPSHOT = "Expected match set record is not part of snapshot";
  
  public final static String PATTERNNAME_DIFFERENT = "Expected pattern qualified name different from actual";
  
  @Inject
  private ModelLoadHelper _modelLoadHelper;
  
  @Inject
  private SnapshotHelper _snapshotHelper;
  
  /**
   * Checks the pattern name of the matcher against the one stored in
   *  the record and checks parameterless patterns as well.
   * 
   * Returns true if further comparison is allowed, false otherwise.
   */
  public boolean validateMatcherBeforeCompare(final IncQueryMatcher matcher, final MatchSetRecord expected, final Set diff) {
    String _patternName = matcher.getPatternName();
    String _patternQualifiedName = expected.getPatternQualifiedName();
    boolean _equals = _patternName.equals(_patternQualifiedName);
    boolean _not = (!_equals);
    if (_not) {
      String _plus = (TestExecutor.PATTERNNAME_DIFFERENT + " (");
      String _patternQualifiedName_1 = expected.getPatternQualifiedName();
      String _plus_1 = (_plus + _patternQualifiedName_1);
      String _plus_2 = (_plus_1 + "!=");
      String _patternName_1 = matcher.getPatternName();
      String _plus_3 = (_plus_2 + _patternName_1);
      String _plus_4 = (_plus_3 + ")");
      diff.add(_plus_4);
      return false;
    }
    List<String> _parameterNames = matcher.getParameterNames();
    int _size = _parameterNames.size();
    boolean _equals_1 = (_size == 0);
    if (_equals_1) {
      EList<MatchRecord> _matches = expected.getMatches();
      int _size_1 = _matches.size();
      boolean _equals_2 = (_size_1 == 1);
      if (_equals_2) {
        int _countMatches = matcher.countMatches();
        boolean _equals_3 = (_countMatches == 1);
        if (_equals_3) {
          diff.add(TestExecutor.CORRECT_SINGLE);
          return true;
        } else {
          int _countMatches_1 = matcher.countMatches();
          boolean _equals_4 = (_countMatches_1 == 0);
          if (_equals_4) {
            diff.add(TestExecutor.CORRECT_EMPTY);
            return true;
          }
        }
      }
    }
    return true;
  }
  
  /**
   * Compares the match set of a given matcher with the given match record
   *  using EMF-IncQuery as a compare tool.
   * Therefore the comparison depends on correct EMF-IncQuery query evaluation
   *  (for a given limited pattern language feature set).
   */
  public HashSet<Object> compareResultSetsAsRecords(final IncQueryMatcher matcher, final MatchSetRecord expected) {
    try {
      final HashSet<Object> diff = CollectionLiterals.<Object>newHashSet();
      boolean correctResults = this.validateMatcherBeforeCompare(matcher, expected, diff);
      boolean _not = (!correctResults);
      if (_not) {
        return diff;
      }
      EObject _eContainer = expected.eContainer();
      boolean _not_1 = (!(_eContainer instanceof IncQuerySnapshot));
      if (_not_1) {
        diff.add(TestExecutor.MATCHSETRECORD_NOT_IN_SNAPSHOT);
        return diff;
      }
      EObject _eContainer_1 = expected.eContainer();
      final IncQuerySnapshot snapshot = ((IncQuerySnapshot) _eContainer_1);
      Notifier _eMFRootForSnapshot = this._snapshotHelper.getEMFRootForSnapshot(snapshot);
      final IncQueryEngine engine = IncQueryEngine.on(_eMFRootForSnapshot);
      IQuerySpecification<UnexpectedMatchRecordMatcher> _querySpecification = UnexpectedMatchRecordMatcher.querySpecification();
      final UnexpectedMatchRecordMatcher unexpectedMatcher = _querySpecification.getMatcher(engine);
      MatchRecord _filter = expected.getFilter();
      final IPatternMatch partialMatch = this._snapshotHelper.createMatchForMachRecord(matcher, _filter);
      final MatchSetRecord actual = this._snapshotHelper.saveMatchesToSnapshot(matcher, partialMatch, snapshot);
      final Procedure1<UnexpectedMatchRecordMatch> _function = new Procedure1<UnexpectedMatchRecordMatch>() {
          public void apply(final UnexpectedMatchRecordMatch it) {
            String _plus = (TestExecutor.UNEXPECTED_MATCH + " (");
            String _prettyPrint = it.prettyPrint();
            String _plus_1 = (_plus + _prettyPrint);
            String _plus_2 = (_plus_1 + ")");
            diff.add(_plus_2);
          }
        };
      unexpectedMatcher.forEachMatch(actual, expected, null, new IMatchProcessor<UnexpectedMatchRecordMatch>() {
          public void process(UnexpectedMatchRecordMatch p0) {
            _function.apply(p0);
          }
      });
      final Procedure1<UnexpectedMatchRecordMatch> _function_1 = new Procedure1<UnexpectedMatchRecordMatch>() {
          public void apply(final UnexpectedMatchRecordMatch it) {
            String _plus = (TestExecutor.EXPECTED_NOT_FOUND + " (");
            String _prettyPrint = it.prettyPrint();
            String _plus_1 = (_plus + _prettyPrint);
            String _plus_2 = (_plus_1 + ")");
            diff.add(_plus_2);
          }
        };
      unexpectedMatcher.forEachMatch(expected, actual, null, new IMatchProcessor<UnexpectedMatchRecordMatch>() {
          public void process(UnexpectedMatchRecordMatch p0) {
            _function_1.apply(p0);
          }
      });
      return diff;
    } catch (Exception _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Compares the match set of a given matcher with the given match record using the
   *  records as partial matches on the matcher.
   * Therefore the comparison does not depend on correct EMF-IncQuery query evaluation.
   */
  public HashSet<Object> compareResultSets(final IncQueryMatcher matcher, final MatchSetRecord expected) {
    final HashSet<Object> diff = CollectionLiterals.<Object>newHashSet();
    boolean correctResults = this.validateMatcherBeforeCompare(matcher, expected, diff);
    boolean _not = (!correctResults);
    if (_not) {
      return diff;
    }
    final ArrayList<Object> foundMatches = CollectionLiterals.<Object>newArrayList();
    EList<MatchRecord> _matches = expected.getMatches();
    for (final MatchRecord matchRecord : _matches) {
      {
        final IPatternMatch partialMatch = this._snapshotHelper.createMatchForMachRecord(matcher, matchRecord);
        final int numMatches = matcher.countMatches(partialMatch);
        boolean _equals = (numMatches == 0);
        if (_equals) {
          String _plus = (TestExecutor.EXPECTED_NOT_FOUND + " (");
          StringBuilder _printMatchRecord = this.printMatchRecord(matchRecord);
          String _plus_1 = (_plus + _printMatchRecord);
          String _plus_2 = (_plus_1 + ")");
          diff.add(_plus_2);
          correctResults = false;
        } else {
          boolean _equals_1 = (numMatches == 1);
          if (_equals_1) {
            foundMatches.add(partialMatch);
          } else {
            String _plus_3 = (TestExecutor.MULTIPLE_FOR_EXPECTED + " (");
            StringBuilder _printMatchRecord_1 = this.printMatchRecord(matchRecord);
            String _plus_4 = (_plus_3 + _printMatchRecord_1);
            String _plus_5 = (_plus_4 + ")");
            diff.add(_plus_5);
            correctResults = false;
          }
        }
      }
    }
    MatchRecord _filter = expected.getFilter();
    IPatternMatch _createMatchForMachRecord = this._snapshotHelper.createMatchForMachRecord(matcher, _filter);
    final Procedure1<IPatternMatch> _function = new Procedure1<IPatternMatch>() {
        public void apply(final IPatternMatch it) {
          boolean _contains = foundMatches.contains(it);
          boolean _not = (!_contains);
          if (_not) {
            String _plus = (TestExecutor.UNEXPECTED_MATCH + " (");
            String _prettyPrint = it.prettyPrint();
            String _plus_1 = (_plus + _prettyPrint);
            String _plus_2 = (_plus_1 + ")");
            diff.add(_plus_2);
          }
        }
      };
    matcher.forEachMatch(_createMatchForMachRecord, new IMatchProcessor<IPatternMatch>() {
        public void process(IPatternMatch p0) {
          _function.apply(p0);
        }
    });
    return diff;
  }
  
  public StringBuilder printMatchRecord(final MatchRecord record) {
    StringBuilder _xblockexpression = null;
    {
      StringBuilder _stringBuilder = new StringBuilder();
      final StringBuilder sb = _stringBuilder;
      EObject _eContainer = record.eContainer();
      final MatchSetRecord matchSet = ((MatchSetRecord) _eContainer);
      EList<MatchSubstitutionRecord> _substitutions = record.getSubstitutions();
      final Procedure1<MatchSubstitutionRecord> _function = new Procedure1<MatchSubstitutionRecord>() {
          public void apply(final MatchSubstitutionRecord it) {
            int _length = sb.length();
            boolean _greaterThan = (_length > 0);
            if (_greaterThan) {
              sb.append(",");
            }
            String _parameterName = it.getParameterName();
            StringBuilder _append = sb.append(_parameterName);
            StringBuilder _append_1 = _append.append("=");
            Object _derivedValue = it.getDerivedValue();
            _append_1.append(_derivedValue);
          }
        };
      IterableExtensions.<MatchSubstitutionRecord>forEach(_substitutions, _function);
      String _patternQualifiedName = matchSet.getPatternQualifiedName();
      String _plus = (_patternQualifiedName + "(");
      sb.insert(0, _plus);
      StringBuilder _append = sb.append(")");
      _xblockexpression = (_append);
    }
    return _xblockexpression;
  }
  
  /**
   * Compares match set of each matcher initialized from the given pattern model
   *  based on the input specification of the snapshot.
   * If any of the matchers return incorrect results, the assert fails.
   */
  public void assertMatchResults(final PatternModel patternModel, final IncQuerySnapshot snapshot) {
    try {
      final HashSet<Object> diff = CollectionLiterals.<Object>newHashSet();
      final Notifier input = this._snapshotHelper.getEMFRootForSnapshot(snapshot);
      final IncQueryEngine engine = IncQueryEngine.on(input);
      this.registerLogger(engine);
      EList<MatchSetRecord> _matchSetRecords = snapshot.getMatchSetRecords();
      final Procedure1<MatchSetRecord> _function = new Procedure1<MatchSetRecord>() {
          public void apply(final MatchSetRecord matchSet) {
            String _patternQualifiedName = matchSet.getPatternQualifiedName();
            final IncQueryMatcher<? extends IPatternMatch> matcher = TestExecutor.this._modelLoadHelper.initializeMatcherFromModel(patternModel, engine, _patternQualifiedName);
            boolean _notEquals = (!Objects.equal(matcher, null));
            if (_notEquals) {
              final HashSet<Object> result = TestExecutor.this.compareResultSets(matcher, matchSet);
              boolean _or = false;
              boolean _or_1 = false;
              boolean _equals = Objects.equal(result, null);
              if (_equals) {
                _or_1 = true;
              } else {
                HashSet<String> _newHashSet = CollectionLiterals.<String>newHashSet(TestExecutor.CORRECT_EMPTY);
                boolean _equals_1 = _newHashSet.equals(result);
                _or_1 = (_equals || _equals_1);
              }
              if (_or_1) {
                _or = true;
              } else {
                HashSet<String> _newHashSet_1 = CollectionLiterals.<String>newHashSet(TestExecutor.CORRECT_SINGLE);
                boolean _equals_2 = _newHashSet_1.equals(result);
                _or = (_or_1 || _equals_2);
              }
              boolean _not = (!_or);
              if (_not) {
                Iterables.<Object>addAll(diff, result);
              }
            }
          }
        };
      IterableExtensions.<MatchSetRecord>forEach(_matchSetRecords, _function);
      String _logDifference = this.logDifference(diff, engine);
      boolean _isEmpty = diff.isEmpty();
      Assert.assertTrue(_logDifference, _isEmpty);
    } catch (Exception _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  /**
   * Compares match set of each matcher initialized from the given pattern model
   *  based on the input specification of the snapshot (specified as a platform URI).
   * If any of the matchers return incorrect results, the assert fails.
   */
  public void assertMatchResults(final PatternModel patternModel, final String snapshotUri) {
    final IncQuerySnapshot snapshot = this._modelLoadHelper.loadExpectedResultsFromUri(snapshotUri);
    this.assertMatchResults(patternModel, snapshot);
  }
  
  /**
   * Compares match set of each matcher initialized from the given pattern model (specified as a platform URI)
   *  based on the input specification of the snapshot (specified as a platform URI).
   * If any of the matchers return incorrect results, the assert fails.
   */
  public void assertMatchResults(final String patternUri, final String snapshotUri) {
    final PatternModel patternModel = this._modelLoadHelper.loadPatternModelFromUri(patternUri);
    this.assertMatchResults(patternModel, snapshotUri);
  }
  
  public void registerLogger(final IncQueryEngine engine) {
    Logger _logger = engine.getLogger();
    TestingLogAppender _testingLogAppender = new TestingLogAppender();
    _logger.addAppender(_testingLogAppender);
  }
  
  public String retrieveLoggerOutput(final IncQueryEngine engine) {
    final Logger logger = engine.getLogger();
    final Enumeration appers = logger.getAllAppenders();
    boolean _hasMoreElements = appers.hasMoreElements();
    boolean _while = _hasMoreElements;
    while (_while) {
      {
        final Object apper = appers.nextElement();
        if ((apper instanceof TestingLogAppender)) {
          StringBuilder _output = ((TestingLogAppender) apper).getOutput();
          return _output.toString();
        }
      }
      boolean _hasMoreElements_1 = appers.hasMoreElements();
      _while = _hasMoreElements_1;
    }
    return "Logger output not recorded";
  }
  
  public String logDifference(final Set<Object> diff) {
    String _xblockexpression = null;
    {
      StringBuilder _stringBuilder = new StringBuilder();
      final StringBuilder stringBuilder = _stringBuilder;
      this.logDifference(diff, stringBuilder);
      String _string = stringBuilder.toString();
      _xblockexpression = (_string);
    }
    return _xblockexpression;
  }
  
  public String logDifference(final Set<Object> diff, final IncQueryEngine engine) {
    String _xblockexpression = null;
    {
      StringBuilder _stringBuilder = new StringBuilder();
      final StringBuilder stringBuilder = _stringBuilder;
      this.logDifference(diff, stringBuilder);
      String _retrieveLoggerOutput = this.retrieveLoggerOutput(engine);
      stringBuilder.append(_retrieveLoggerOutput);
      String _string = stringBuilder.toString();
      _xblockexpression = (_string);
    }
    return _xblockexpression;
  }
  
  private void logDifference(final Set<Object> diff, final StringBuilder stringBuilder) {
    final Procedure1<Object> _function = new Procedure1<Object>() {
        public void apply(final Object it) {
          String _plus = ("\n" + it);
          stringBuilder.append(_plus);
        }
      };
    IterableExtensions.<Object>forEach(diff, _function);
  }
}
