/**
 * Copyright (c) 2014,2019 Contributors to the Eclipse Foundation
 * 
 * See the NOTICE file(s) distributed with this work for additional
 * information regarding copyright ownership.
 * 
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.smarthome.model.script.jvmmodel;

import com.google.inject.Inject;
import java.util.Arrays;
import java.util.Collection;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.smarthome.core.items.Item;
import org.eclipse.smarthome.core.items.ItemRegistry;
import org.eclipse.smarthome.core.types.Type;
import org.eclipse.smarthome.model.script.scoping.StateAndCommandProvider;
import org.eclipse.smarthome.model.script.script.Script;
import org.eclipse.xtext.common.types.JvmField;
import org.eclipse.xtext.common.types.JvmGenericType;
import org.eclipse.xtext.common.types.JvmMember;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.xbase.jvmmodel.AbstractModelInferrer;
import org.eclipse.xtext.xbase.jvmmodel.IJvmDeclaredTypeAcceptor;
import org.eclipse.xtext.xbase.jvmmodel.JvmTypesBuilder;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.StringExtensions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * <p>Infers a JVM model from the source model.</p>
 * 
 * <p>The JVM model should contain all elements that would appear in the Java code
 * which is generated from the source model. Other models link against the JVM model rather than the source model.</p>
 * 
 * @author Oliver Libutzki - Xtext 2.5.0 migration
 */
@SuppressWarnings("all")
public class ScriptJvmModelInferrer extends AbstractModelInferrer {
  private final static Logger logger = LoggerFactory.getLogger(ScriptJvmModelInferrer.class);
  
  /**
   * conveninence API to build and initialize JvmTypes and their members.
   */
  @Inject
  @Extension
  private JvmTypesBuilder _jvmTypesBuilder;
  
  @Inject
  private ItemRegistry itemRegistry;
  
  @Inject
  private StateAndCommandProvider stateAndCommandProvider;
  
  /**
   * Is called for each instance of the first argument's type contained in a resource.
   * 
   * @param element - the model to create one or more JvmDeclaredTypes from.
   * @param acceptor - each created JvmDeclaredType without a container should be passed to the acceptor in order get attached to the
   *                   current resource.
   * @param isPreLinkingPhase - whether the method is called in a pre linking phase, i.e. when the global index isn't fully updated. You
   *        must not rely on linking using the index if iPrelinkingPhase is <code>true</code>
   */
  protected void _infer(final Script script, final IJvmDeclaredTypeAcceptor acceptor, final boolean isPreIndexingPhase) {
    String _firstUpper = StringExtensions.toFirstUpper(IterableExtensions.<String>head(((Iterable<String>)Conversions.doWrapArray(script.eResource().getURI().lastSegment().split("\\.")))));
    final String className = (_firstUpper + "Script");
    final Procedure1<JvmGenericType> _function = (JvmGenericType it) -> {
      final Set<String> fieldNames = CollectionLiterals.<String>newHashSet();
      final Iterable<Type> types = this.stateAndCommandProvider.getAllTypes();
      final Consumer<Type> _function_1 = (Type type) -> {
        final String name = type.toString();
        boolean _add = fieldNames.add(name);
        if (_add) {
          EList<JvmMember> _members = it.getMembers();
          final Procedure1<JvmField> _function_2 = (JvmField it_1) -> {
            it_1.setStatic(true);
          };
          JvmField _field = this._jvmTypesBuilder.toField(script, name, this._jvmTypesBuilder.newTypeRef(script, type.getClass()), _function_2);
          this._jvmTypesBuilder.<JvmField>operator_add(_members, _field);
        } else {
          ScriptJvmModelInferrer.logger.warn("Duplicate field: \'{}\'. Ignoring \'{}\'.", name, type.getClass().getName());
        }
      };
      types.forEach(_function_1);
      Collection<Item> _items = null;
      if (this.itemRegistry!=null) {
        _items=this.itemRegistry.getItems();
      }
      if (_items!=null) {
        final Consumer<Item> _function_2 = (Item item) -> {
          final String name = item.getName();
          boolean _add = fieldNames.add(name);
          if (_add) {
            EList<JvmMember> _members = it.getMembers();
            final Procedure1<JvmField> _function_3 = (JvmField it_1) -> {
              it_1.setStatic(true);
            };
            JvmField _field = this._jvmTypesBuilder.toField(script, item.getName(), this._jvmTypesBuilder.newTypeRef(script, item.getClass()), _function_3);
            this._jvmTypesBuilder.<JvmField>operator_add(_members, _field);
          } else {
            ScriptJvmModelInferrer.logger.warn("Duplicate field: \'{}\'. Ignoring \'{}\'.", item.getName(), item.getClass().getName());
          }
        };
        _items.forEach(_function_2);
      }
      EList<JvmMember> _members = it.getMembers();
      final Procedure1<JvmOperation> _function_3 = (JvmOperation it_1) -> {
        it_1.setStatic(true);
        this._jvmTypesBuilder.setBody(it_1, script);
      };
      JvmOperation _method = this._jvmTypesBuilder.toMethod(script, "_script", null, _function_3);
      this._jvmTypesBuilder.<JvmOperation>operator_add(_members, _method);
    };
    acceptor.<JvmGenericType>accept(this._jvmTypesBuilder.toClass(script, className)).initializeLater(_function);
  }
  
  public void infer(final EObject script, final IJvmDeclaredTypeAcceptor acceptor, final boolean isPreIndexingPhase) {
    if (script instanceof Script) {
      _infer((Script)script, acceptor, isPreIndexingPhase);
      return;
    } else if (script != null) {
      _infer(script, acceptor, isPreIndexingPhase);
      return;
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(script, acceptor, isPreIndexingPhase).toString());
    }
  }
}
