#!/bin/bash

# Live kernel build script from Tomas M. modified for Zenlive, 2008-02-26

# Please install the latest Zenwalk's kernelscsi, kernelheaders and kernelsource,
# against which you want to build kernelive (identical version)
#
# Before running this script, please ensure that the latest aufs2
# --which you should get by using git as instructed on aufs website-- 
# as well as the latest squashfs & lzma are located in the same 
# directory as this script. Your kernelive config file & other 
# eventual patches should also be there.
#
# If for some reasons, you have to re-run this script again after a
# failed attempt, You should first remove the patches applied to the
# kernel by running the remove_patches.sh script first or else
# delete /usr/src/(uname -r) & install kernel, headers & sources again.
# 
# Last build date: 20090402

softname='kernelive'
version=`uname -r`
pkgversion='60.2'
arch='i486'
cpu='i686'

buildir=`pwd`
sources=/usr/src
srcdir="$sources/linux-${version}"
newdir="${softname}-${version}-${arch}-${pkgversion}"
dest=${buildir}/${newdir}
liveconfig=$buildir/kernelive.config

SQUASHFS=squashfs3.4
LZMA=lzma457
AUFS=aufs2-standalone
SQUASHFS_LATEST_KERNEL_PATCH=linux-2.6.27-rc4-next

# log starts
(

# ----------------------

eecho()
{
   echo "-------------------------------------"
   echo " * " "$@"
   echo "-------------------------------------"
}

set -e

rm -Rf ${dest}
rm -f ${dest}.tgz
mkdir -p ${dest}
cd $sources
rm -Rf $AUFS
rm -Rf $LZMA
rm -Rf $SQUASHFS
K=`echo $srcdir | sed -r s/'\/'/'\\\\\/'/g`

eecho "Unpacking archives"

mkdir -p $LZMA
echo $LZMA
tar -C $LZMA -xf $buildir/$LZMA.tar.*
echo $AUFS
tar -xf $buildir/$AUFS.tar.gz
echo $SQUASHFS
tar -xf $buildir/$SQUASHFS.tar.*
echo "Squashfs LZMA patches"
tar -xf $buildir/sqlzma*.tar.*

eecho "Preparing kernel: aufs, squashfs and lzma patches"

cd $srcdir
make mrproper
cp $liveconfig .config

patch -p1 < $sources/$AUFS/aufs2-standalone.patch
eecho "First patch applied succesfully"

patch -p1 < $sources/$SQUASHFS/kernel-patches/$SQUASHFS_LATEST_KERNEL_PATCH/*patch
eecho "Second patch applied succesfully"

cd $sources
patch -p0 < $sources/sqlzma2u-3.4.patch
eecho "Third patch applied succesfully"

cd $sources/$LZMA
patch -p1 < $sources/sqlzma1-457.patch
eecho "Fourth patch applied succesfully"

cd $sources/$SQUASHFS/squashfs-tools
patch -p0 < $buildir/mksquashfs.quiet.patch
eecho "Fifth patch applied succesfully"

patch -p0 < $buildir/unsquashfs.quiet.patch
eecho "Sixth patch applied succesfully"

cd $srcdir
patch -p1 < $sources/sqlzma2k-3.4.patch
eecho "seventh patch applied succesfully"

cd $sources/$AUFS
cat config.mk \
 | grep -v "^export CONFIG_AUFS_DEBUG =" \
 | grep -v "^export CONFIG_AUFS_BRANCH_MAX_127 =" \
 | sed "s/CONFIG_AUFS_BRANCH_MAX_1023 =.*/CONFIG_AUFS_BRANCH_MAX_1023 = y/"\
 | sed "s/CONFIG_AUFS_BDEV_LOOP =.*/CONFIG_AUFS_BDEV_LOOP = y/" >Makefile
 
 
eecho "aufs makefile edited succesfully"

cp -r fs/aufs $srcdir/fs
cp include/linux/aufs_type.h $srcdir/include/linux

cd $sources
cat Makefile | sed -r "s/KDir = .*/KDir = $K/" \
             | sed -r "s/SqFs = .*/SqFs = $K\/fs\/squashfs\//" \
             | sed -r "s/LzmaVer =.*/LzmaVer = $LZMA/" \
             | sed -r "s/C=1/C=0/" >Makefile2
cat Makefile2 >Makefile
eecho "lzma makefile edited succesfully"

cd $srcdir
eecho "configuring kernel"
# squashfs will be compiled separately, so make sure it's not in the kernel
# configuration if you are prompted with a choice.
# sed '/EXTRAVERSION/ s/$/-live/' -i Makefile # ToDo -> edit the rest of the build script to include extraversion
make oldconfig
make scripts
make prepare
sed 's/Windows/Ndiswrapper/' -i Makefile
eecho "kernel configured succesfully"

# Compiling the kernel
eecho "compiling kernel"
cd $srcdir
make -j 3 modules
INSTALL_MOD_PATH=${dest} make modules_install
make -j 3 bzImage
cp .config $liveconfig

# Compiling squashfs+lzma modules and tools
eecho "compiling squashfs+lzma modules and tools"
cd $sources
make
mkdir -p ${dest}/usr/bin
cp $sources/$SQUASHFS/squashfs-tools/{mk,un}squashfs ${dest}/usr/bin
strip --strip-unneeded ${dest}/usr/bin/*
cd ${dest}/lib/modules/*
rm -f modules.*
mkdir -p kernel/fs/squashfs/
cp $sources/$LZMA/C/Compress/Lzma/kmod/{un,sq}lzma.ko kernel/fs/squashfs/
cp $srcdir/fs/squashfs/squashfs.ko kernel/fs/squashfs/


# Copying the kernel
eecho "copying kernel"
cd $srcdir
mkdir -p ${dest}/boot
cp .config ${dest}/boot/config-$version
cp arch/x86/boot/bzImage ${dest}/boot/vmlinuz-$version
cp System.map ${dest}/boot/System.map-$version
(
cd ${dest}/boot
ln -s vmlinuz-$version vmlinuz
ln -s System.map-$version System.map
ln -s config-$version config
)

# Creating rc.modules
eecho "creating rc.modules"
mkdir -p ${dest}/etc/rc.d

cat << EOF > ${dest}/etc/rc.d/rc.modules
#!/bin/sh
# rc.modules   $(date) <Generated by kernelive>
EOF


cat << "EOF" >> ${dest}/etc/rc.d/rc.modules
#
# This file loads extra drivers into the Linux kernel.
# As kmod and udev handle module loading, this file will be mostly
# commented out, but is still useful for loading selected modules at boot time.
#

RELEASE=$(uname -r)

echo "Updating module dependencies for Linux $RELEASE"

/sbin/depmod -A


EOF

modprobe -l | sed -e "s/^.*\/\(.*\)\.ko$/# \/sbin\/modprobe \1/m ; s/# \(.* ac$\)/\1/ ; s/# \(.* battery$\)/\1/ ; s/# \(.* button$\)/\1/ ; s/# \(.* fan$\)/\1/ ; s/# \(.* fuse$\)/\1/ ; s/# \(.* processor$\)/\1/ ; s/# \(.* thermal$\)/\1/ ; s/# \(.* video$\)/\1/" | sort \
>> ${dest}/etc/rc.d/rc.modules

chmod 755 ${dest}/etc/rc.d/rc.modules
mv ${dest}/etc/rc.d/rc.modules ${dest}/etc/rc.d/rc.modules.new


# copy files necessary for later builds, like nvidia and vmware
(
	cd ${srcdir}/include
	
	mkdir -p ${dest}/usr/src/linux-${version}/include   
	for i in acpi asm-generic asm-x86 config linux math-emu media net pcmcia scsi sound video; do
		cp -rf $i ${dest}/usr/src/linux-${version}/include/   
	done
	
	( cd ${dest}/usr/src/linux-${version}/include ; ln -sf asm-x86 asm )
	
	( cd ${dest}/usr/src/linux-${version}/include/linux ; ln -sf autoconf.h config.h )
  
	cd ${srcdir}
	cp .config ${dest}/usr/src/linux-${version}
	cp Makefile ${dest}/usr/src/linux-${version}
	cp Module.symvers ${dest}/usr/src/linux-${version}
	cp -a scripts ${dest}/usr/src/linux-${version}
	
	# fix permissions on scripts dir
	chmod 755 -R ${dest}/usr/src/linux-${version}/scripts
	mkdir -p ${dest}/usr/src/linux-${version}/.tmp_versions
  
	mkdir -p ${dest}/usr/src/linux-${version}/arch/x86/kernel
	cp arch/x86/Makefile* ${dest}/usr/src/linux-${version}/arch/x86/
	cp arch/x86/kernel/asm-offsets.s ${dest}/usr/src/linux-${version}/arch/x86/kernel/
	
	# add headers for lirc package
	mkdir -p ${dest}/usr/src/linux-${version}/drivers/media/video
	cp drivers/media/video/*.h  ${dest}/usr/src/linux-${version}/drivers/media/video/
	for i in bt8xx cpia2 cx25840 cx88 em28xx et61x251 pwc saa7134 sn9c102 usbvideo zc0301; do
		mkdir -p ${dest}/usr/src/linux-${version}/drivers/media/video/$i
		cp -a drivers/media/video/$i/*.h ${dest}/usr/src/linux-${version}/drivers/media/video/$i
	done

	# add dm headers
	mkdir -p ${dest}/usr/src/linux-${version}/drivers/md
	cp drivers/md/*.h  ${dest}/usr/src/linux-${version}/drivers/md
  
	# add inotify.h
	mkdir -p ${dest}/usr/src/linux-${version}/include/linux
	cp include/linux/inotify.h ${dest}/usr/src/linux-${version}/include/linux/
    
	# add wireless headers
	mkdir -p ${dest}/usr/src/linux-${version}/net/mac80211/
	cp net/mac80211/*.h ${dest}/usr/src/linux-${version}/net/mac80211/
  
	# add CLUSTERIP file for iptables
	mkdir -p ${dest}/usr/src/linux-${version}/net/ipv4/netfilter/
	cp net/ipv4/netfilter/ipt_CLUSTERIP.c ${dest}/usr/src/linux-${version}/net/ipv4/netfilter/
	
	# add xfs and shmem for aufs building
	mkdir -p ${dest}/usr/src/linux-${version}/fs/xfs
	mkdir -p ${dest}/usr/src/linux-${version}/mm
	cp fs/xfs/xfs_sb.h ${dest}/usr/src/linux-${version}/fs/xfs/xfs_sb.h
	cp mm/shmem.c ${dest}/usr/src/linux-${version}/mm/shmem.c
	
	# add vmlinux
	cp vmlinux ${dest}/usr/src/linux-${version}
  
	# copy in Kconfig files
	for i in `find . -name "Kconfig*"`; do 
		mkdir -p ${dest}/usr/src/linux-${version}/`echo $i | sed 's|/Kconfig.*||'`
		cp $i ${dest}/usr/src/linux-${version}/$i
	done
)

# copy files necessary for modules to be loaded without a reboot:
cp /lib/modules/$version/modules.* ${dest}/lib/modules/$version/

# Creating kernel package
eecho "creating kernel package"

chown -R root:root ${dest}

mkdir -p ${dest}/install

cat << "EOF" >  ${dest}/install/doinst.sh
config() {
  NEW="$1"
  OLD="`dirname $NEW`/`basename $NEW .new`"
  # If there's no config file by that name, mv it over:
  if [ ! -r $OLD ]; then
    mv $NEW $OLD
  elif [ "`cat $OLD | md5sum`" = "`cat $NEW | md5sum`" ]; then # toss the redundant copy
    rm $NEW
  fi
  # Otherwise, we leave the .new copy for the admin to consider...
}
config etc/rc.d/rc.modules.new

lilo 

EOF

cat << "EOF" > ${dest}/install/slack-desc
kernelive: kernelive
kernelive:
kernelive: Linux kernel for Zenwalk Live is compiled with support for
kernelive: squashfs, lzma & aufs as well as for most hardware, 
kernelive: including SCSI, USB booting devices, etc.. 
kernelive: 
kernelive: Drivers (such as for CD-ROM drives on proprietary interfaces,
kernelive: or ethernet cards) may be loaded as modules.
kernelive: See /etc/rc.d/rc.modules for examples
kernelive:
kernelive: 
EOF

mkdir -p ${dest}/usr/src/${softname}-${version}
cp ${buildir}/build-${softname}.sh ${dest}/usr/src/${softname}-${version}

cd ${dest}
makepkg -p -l y -c n ${softname}-${version}-${arch}-${pkgversion}.tgz

eecho "your kernel is ready in ${dest}"

# log ends
) 2>&1 | tee "${softname}-${version}-${arch}-${pkgversion}-build.log"
