#include <ctype.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/wait.h>
#include <unistd.h>
#include "types.h"
#include "macros.h"

struct sched_param {
        unsigned int priority;
        int fork_penalty_threshold;
        unsigned int starvation_threshold;
        unsigned int ts_max;
        unsigned int run_q, run_q_min, run_q_max;
};

/* because I can't include libc's sched.h */
extern int sched_setparam __P((pid_t __pid,
		const struct sched_param *__param));
extern int sched_getparam __P((pid_t __pid,
		struct sched_param *__param));
extern int sched_setscheduler __P((pid_t __pid, int __policy,
	const struct sched_param *__param));
extern int sched_getscheduler __P((pid_t __pid));

const char *policy_name[3] = { "Adaptive", "FIFO", "Round_Robin" };

void	qsched(pid_t pid, int verbose);

int     opt_policy = -1,
        opt_priority = -1,
        opt_fork_penalty_threshold = -2,
        opt_starvation_threshold = -1,
        opt_ts_max = -1,
        opt_run_q_min = -1, opt_run_q_max = -1,
        opt_nohup = 0;

int main(int argc, char **argv) {
  int    i;
  pid_t  pid;

  for (optind = 1; (optind < argc) && (argv[optind][0] == '-'); ) {
    i = argv[optind++][1];
    switch (i) {
    case 'p': opt_policy = atoi(argv[optind++]); break;
    case 't': opt_priority = atoi(argv[optind++]); break;
    case 'f': opt_fork_penalty_threshold = atoi(argv[optind++]); break;
    case 's': opt_starvation_threshold = atoi(argv[optind++]); break;
    case 'm': opt_ts_max = atoi(argv[optind++]); break;
    case 'l': opt_run_q_min = atoi(argv[optind++]); break;
    case 'u': opt_run_q_max = atoi(argv[optind++]); break;
    case 'n': opt_nohup = TRUE; break;
    }
  }

  if (optind >= argc) {
    printf("\
usage: qsched [-ptfslu] [pid [pid...]]\n\
       qsched [-ptfslun] command [arg [arg...]]\n\
   -p   scheduling policy\n\
          0 = Adaptive\n\
          1 = FIFO\n\
          2 = Round-Robin\n\
   -t   timeslice in milliseconds\n\
   -f   fork penalty threshold in seconds\n\
   -s   starvation threshold in milliseconds\n\
   -m   consecutive timeslices before demotion\n\
   -l   minimum run-queue (0-31)\n\
   -u   maximum run-queue (0-31)\n\
   -n   ignore SIGHUP\n");
    return 0;
  }

  while (optind < argc) {
    if (!isdigit(argv[optind][0])) {
      pid = getpid();
      if (opt_nohup) signal(SIGHUP, SIG_IGN);
      qsched(pid, 0);
      if (execvp(argv[optind], argv + optind)) {
        perror("execvp");
        return 1;
      }
    } else {
      pid = atoi(argv[optind++]);
      qsched(pid, 1);
    }
  }

  return 0;
}

void qsched(pid_t pid, int verbose) {
  int                    policy_in, policy_out, err;
  struct sched_param     param_in, param_out;
  FILE			*f;
  char                  *s, path[64], cmd[128];

  policy_in = sched_getscheduler(pid);
  if (policy_in < 0) {
    perror("sched_getscheduler");
    return;
  }
  policy_out = policy_in;
  err = sched_getparam(pid, &param_in);
  if (err < 0) {
    perror("sched_getparam");
    return;
  }
  param_out = param_in;

  if (opt_policy != -1)
    policy_out = opt_policy;
  if (opt_priority != -1)
    param_out.priority = opt_priority;
  if (opt_fork_penalty_threshold != -2)
    param_out.fork_penalty_threshold = opt_fork_penalty_threshold;
  if (opt_starvation_threshold != -1)
    param_out.starvation_threshold = opt_starvation_threshold;
  if (opt_ts_max != -1)
    param_out.ts_max = opt_ts_max;
  if (opt_run_q_min != -1)
    param_out.run_q_min = opt_run_q_min;
  if (opt_run_q_max != -1)
    param_out.run_q_max = opt_run_q_max;

  /* ensure run_q_min <= run_q_max */
  if ((param_in.run_q_min != param_out.run_q_min) &&
      (param_in.run_q_max == param_out.run_q_max))
  {
    param_out.run_q_max = max(param_out.run_q_max, param_in.run_q_min);
  }
  else
  if ((param_in.run_q_min == param_out.run_q_min) &&
      (param_in.run_q_max != param_out.run_q_max))
  {
    param_out.run_q_min = min(param_out.run_q_min, param_in.run_q_max);
  }

  if (policy_out)
    param_out.run_q_max = param_out.run_q_min;

  if ((policy_in != policy_out) ||
      memcmp(&param_in, &param_out, sizeof(struct sched_param)))
  {
    err = sched_setscheduler(pid, policy_out, &param_out);
    if (err < 0) {
      perror("sched_setscheduler");
      return;
    }

    /* allow time for the settings to take effect */
    usleep(250000);
  }

  policy_in = sched_getscheduler(pid);
  if (policy_in < 0) {
    perror("sched_getscheduler");
    return;
  }
  err = sched_getparam(pid, &param_in);
  if (err < 0) {
    perror("sched_getparam");
    return;
  }

  /* get the command-line */
  sprintf(path, "/proc/%d/cmdline", pid);
  f = fopen(path, "r");
  if (!f) {
    perror("fopen");
    return;
  }
  bzero(cmd, 128);
  fgets(cmd, 128, f);
  fclose(f);
  s = cmd;
  for (;;) {
    if (*s) s++;
    else if (*(s + 1)) {
      *s = ' ';
      s += 2;
    } else break;
  }

  if (verbose) printf("\
pid: %d\n\
cmd: %s\n\
     policy:       %u (%s)\n\
     timeslice:    %ums\n\
     fork_thresh:  %ds\n\
     strv_thresh:  %ums\n\
     ts_max:       %u\n\
     run_q:        %u\n\
     run_q_min:    %u\n\
     run_q_max:    %u\n",
                   pid, cmd,
	           policy_in,
                   policy_name[policy_in],
                   param_in.priority,
                   param_in.fork_penalty_threshold,
                   param_in.starvation_threshold,
                   param_in.ts_max,
                   param_in.run_q,
                   param_in.run_q_min,
                   param_in.run_q_max);
}
