/*
 * SCSI Media Changer device driver for Linux
 *
 * supported to be source code compatible with NetBSD's driver
 *
 *     (c) 1996 Gerd Knorr <kraxel@goldbach.in-berlin.de>
 *
 */

#include <linux/module.h>

#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/interrupt.h>
#include <asm/system.h>

#include <linux/version.h>
#if LINUX_VERSION_CODE >= 0x020100
  /* 2.1.x kernels */
  #include <asm/uaccess.h>
  #define WRITE_TO_USER(user,kernel,size) \
	{ if (copy_to_user((user), (kernel), (size))) return -EFAULT; }
  #define READ_FROM_USER(kernel,user,size) \
	{ if (copy_from_user((kernel), (user), (size))) return -EFAULT; }
#else
  /* 2.0.x kernels */
  #include <asm/segment.h>
  #define WRITE_TO_USER(user,kernel,size) \
	{ int err; \
	  err = verify_area (VERIFY_WRITE, (user), (size)); \
	  if (err) return err; \
	  memcpy_tofs ((user), (kernel), (size)); }
  #define READ_FROM_USER(kernel,user,size) \
	{ int err; \
	  err = verify_area (VERIFY_READ, (user), (size)); \
	  if (err) return err; \
	  memcpy_fromfs((kernel), (user), (size)); }
#endif

#if LINUX_VERSION_CODE < 0x02014b
#define scsi_allocate_device         allocate_device
#define scsi_release_command(SCpnt)  (SCpnt->request.rq_status = RQ_INACTIVE)
#endif


#define MAJOR_NR	86
#define DEVICE_NAME	"SCSI Changer"
#define DEVICE_ON(device)
#define DEVICE_OFF(device)
#define TYPE_CHANGER	0x08            /* not in scsi.h (yet) */
#define CH_EXTRA_DEVS   2

#include <linux/blk.h>
#include "scsi.h"
#include "hosts.h"
#include "constants.h"

#include "chio.h"

/* insmod options */
int check_busy = 1;     /* busy check, can be turned off using this */
int debug = 0;          /* you can turn on misc debug messages with this */

#if LINUX_VERSION_CODE >= 0x020117
MODULE_SUPPORTED_DEVICE("sch");
MODULE_DESCRIPTION("device driver for scsi media changer devices");
MODULE_AUTHOR("Gerd Knorr <kraxel@goldbach.in-berlin.de>");

MODULE_PARM(check_busy,"i");
MODULE_PARM_DESC(check_busy, \
    "enable/disable busy check for data transfer elements (default: on)");

MODULE_PARM(debug,"i");
MODULE_PARM_DESC(debug,"enable/disable debug messages (default: off)");
#endif

/* ------------------------------------------------------------------- */
/* this should go to a header file later */

typedef struct {
    Scsi_Device  	*device;
    u_int               firsts[4];
    u_int               counts[4];
    Scsi_Device         **dt;        /* ptrs to data transfer elements */
    u_int               unit_attention;
    u_int		voltags;
} Scsi_Changer;

/* ------------------------------------------------------------------- */

#define MAX_RETRIES   1
#define CH_TIMEOUT    (600 * HZ)

static int  ch_init(void);
static void ch_finish(void);
static int  ch_attach(Scsi_Device *);
static int  ch_detect(Scsi_Device *);
static void ch_detach(Scsi_Device *);
static int  ch_open(struct inode * inode, struct file * filp);
static int  ch_release(struct inode * inode, struct file * filp);
static int  ch_ioctl(struct inode * inode, struct file * filp,
		     unsigned int cmd, unsigned long arg);

struct Scsi_Device_Template ch_template = {
    NULL, "media changer", "sch", NULL, TYPE_CHANGER /* scsi.h */, 
    MAJOR_NR, 0, 0, 0, 1,
    ch_detect, ch_init,
    ch_finish, ch_attach, ch_detach
};

static struct file_operations changer_fops =
{
    NULL,                   /* lseek */
    NULL,                   /* read */
    NULL,                   /* write */
    NULL,                   /* readdir */
    NULL,                   /* select */
    ch_ioctl,               /* ioctl */
    NULL,                   /* mmap */
    ch_open,                /* special open code */
#if LINUX_VERSION_CODE >= 0x020176 /* 2.1.118 */
    NULL,
#endif
    ch_release,             /* release */
    NULL,                   /* fsync */
    NULL,                   /* fasync */
    NULL,                   /* Disk change */
    NULL                    /* revalidate */
};

#if 0
static struct {
    char *vendor;
    char *model;
    int  ft,fs,fe,fd; /* firsts */
    int  nt,ns,ne,nd; /* counts */
} broken_devices[] = {
    { "HP      ", "C1713T          ",  0x00,0x0b,0x0a,0x01,  1,16,1,1 },
    { NULL,       NULL,                0,0,0,0,              0,0,0,0 }
};
#endif

static struct {
    unsigned char  sense;
    unsigned char  asc;
    unsigned char  ascq;
    int		   errno;
} error_codes[] = {
/* Just filled in what looks right. Hav'nt checked any standard paper for
   these errno assignments, so they may be wrong... */
{ ILLEGAL_REQUEST, 0x21,0x01/*Invalid element address*/,           EBADSLT },
{ ILLEGAL_REQUEST, 0x28,0x01/*Import or export element accessed*/, EBADE   },
{ ILLEGAL_REQUEST, 0x3B,0x0D/*Medium destination element full*/,   EXFULL  },
{ ILLEGAL_REQUEST, 0x3B,0x0E/*Medium source element empty*/,       EBADE   },

{ ILLEGAL_REQUEST, 0x20,0x00/*Invalid command operation code*/,    EBADRQC },
{ 0,0,0,0 }
};

Scsi_Changer * scsi_Changers = NULL;

/* ------------------------------------------------------------------- */

static void
ch_request_done (Scsi_Cmnd * SCpnt)
{
    struct request * req;
    
    req = &SCpnt->request;
    req->rq_status = RQ_SCSI_DONE; /* Busy, but indicate request done */
    
    if (req->sem != NULL) {
	up(req->sem);
    }
}

static int
ch_do_cmd(int target, unsigned char * sr_cmd,
	  void * buffer, unsigned buflength)
{
    Scsi_Cmnd * SCpnt;
    int errno, retries = 0;

#if LINUX_VERSION_CODE >= 0x020175
    if(!scsi_block_when_processing_errors(scsi_Changers[target].device))
        return -ENXIO;
#endif

    SCpnt = scsi_allocate_device(NULL, scsi_Changers[target].device, 1);
retry:
    errno = 0;
    {
	struct semaphore sem = MUTEX_LOCKED;
	SCpnt->request.sem = &sem;
	scsi_do_cmd(SCpnt,
		    (void *) sr_cmd, buffer, buflength, ch_request_done, 
		    CH_TIMEOUT, MAX_RETRIES);
	down(&sem);
	SCpnt->request.sem = NULL;
    }
    
    if(driver_byte(SCpnt->result) != 0) {
	if (debug) {
	    printk(DEVICE_NAME " (debug) error, command: ");
	    print_command(sr_cmd);
	    print_sense(DEVICE_NAME " (debug)", SCpnt);
	}
        /* Check to see if additional sense information is available */
        if(SCpnt->sense_buffer[7] > 5 && SCpnt->sense_buffer[12] != 0) {
	    int i;
	    for (i = 0; error_codes[i].errno != 0; i++)
		if (error_codes[i].sense == SCpnt->sense_buffer[ 2] &&
		    error_codes[i].asc   == SCpnt->sense_buffer[12] &&
		    error_codes[i].ascq  == SCpnt->sense_buffer[13]) {
			errno = -error_codes[i].errno;
			break;
		}
	}
	if (errno == 0) {
	    errno = -EIO;
	    switch(SCpnt->sense_buffer[2] & 0xf) {
	    case UNIT_ATTENTION:
		if (debug)
		    printk(DEVICE_NAME ": got UNIT ATTENTION\n");
		scsi_Changers[target].unit_attention = 1;
		if (retries++ < 3)
		    goto retry;
		break;
	    case NOT_READY:
		printk(DEVICE_NAME " reports NOT READY.\n");
		break;
	    case ILLEGAL_REQUEST:
		printk(DEVICE_NAME " reports ILLEGAL REQUEST.\n");
		break;
	    default:
		printk(DEVICE_NAME " error, command: ");
		print_command(sr_cmd);
		print_sense(DEVICE_NAME, SCpnt);
	    }
	}
    }
    
    /* Wake up a process waiting for device */
    wake_up(&SCpnt->device->device_wait);
    scsi_release_command(SCpnt);
    return errno;
}

/* ------------------------------------------------------------------------ */

static Scsi_Device*
find_device(struct Scsi_Host *host, u_char channel, u_char id, u_char lun)
{
    Scsi_Device *ret;
#if LINUX_VERSION_CODE >= 0x02014b
    ret = host->host_queue;
#else
    ret = scsi_devices;
#endif

    while (ret != NULL &&
	   (ret->host    != host || ret->channel != channel ||
	    ret->id      != id   || ret->lun     != lun))
	ret = ret->next;

    return ret;
}

static int
ch_read_element_status(int target, u_int elem, char *data)
{
    u_char  cmd[12];
    u_char  *buffer;
    int     result;

    buffer = (unsigned char *) scsi_malloc(512);
    if(!buffer) return -ENOMEM;

retry:
    memset(cmd,0,sizeof(cmd));
    cmd[0] = 0xb8;                      /* READ ELEMENT STATUS */
    cmd[1] = scsi_Changers[target].device->lun << 5 | 
	(scsi_Changers[target].voltags ? 0x10 : 0);
    cmd[2] = (elem >> 8) & 0xff;
    cmd[3] = elem        & 0xff;
    cmd[5] = 1;                         /* # of elements, currently only one */
    cmd[9] = 255;                       /* alloc length  */
    if (0 == (result = ch_do_cmd(target, cmd, buffer, 256))) {
	if (((buffer[16] << 8) | buffer[17]) != elem) {
	    printk(KERN_DEBUG DEVICE_NAME
		   "Oops: asked for element 0x%02x, got 0x%02x\n",
		   elem,(buffer[16] << 8) | buffer[17]);
	    scsi_free(buffer, 512);
	    return -1;
	}
	memcpy(data,buffer+16,16);
    } else {
	if (scsi_Changers[target].voltags) {
	    scsi_Changers[target].voltags = 0;
	    goto retry;
	}
	printk(KERN_WARNING DEVICE_NAME
	       ":  READ ELEMENT STATUS for element 0x%x failed\n",elem);
    }
    
    scsi_free(buffer, 512);
    return result;
}

static int 
ch_init_elem(int target)
{
	u_char cmd[6];

	memset(cmd,0,sizeof(cmd));
	cmd[0] = 0x07 ;                 /* INITIALIZE ELEMENT STATUS */
	cmd[1] = scsi_Changers[target].device->lun << 5;
	return ch_do_cmd(target, cmd, NULL, 0);
}	

static int
ch_readconfig(int target)
{
    u_char  cmd[10], data[16];
    u_char  *buffer;
    int     result,i;
    u_int   elem;

    buffer = (unsigned char *) scsi_malloc(512);
    memset(buffer,0,512);
    if(!buffer) return -ENOMEM;

    memset(cmd,0,sizeof(cmd));
    cmd[0] = MODE_SENSE;
    cmd[1] = scsi_Changers[target].device->lun << 5;
    cmd[2] = 0x1d;
    cmd[4] = 255;
    result = ch_do_cmd(target, cmd, buffer, 255);
    if (debug) {
	printk("element address assignment page: ");
	for (i = 0; i < buffer[0]+buffer[3]; i++)
	    printk("%02x ",buffer[i]);
	printk("\n");
    }
    if (0 == result) {
	scsi_Changers[target].firsts[CHET_MT] =
		(buffer[buffer[3]+ 6] << 8) | buffer[buffer[3]+ 7];
	scsi_Changers[target].counts[CHET_MT] =
		(buffer[buffer[3]+ 8] << 8) | buffer[buffer[3]+ 9];
	scsi_Changers[target].firsts[CHET_ST] =
		(buffer[buffer[3]+10] << 8) | buffer[buffer[3]+11];
	scsi_Changers[target].counts[CHET_ST] =
		(buffer[buffer[3]+12] << 8) | buffer[buffer[3]+13];
	scsi_Changers[target].firsts[CHET_IE] =
		(buffer[buffer[3]+14] << 8) | buffer[buffer[3]+15];
	scsi_Changers[target].counts[CHET_IE] =
		(buffer[buffer[3]+16] << 8) | buffer[buffer[3]+17];
	scsi_Changers[target].firsts[CHET_DT] =
		(buffer[buffer[3]+18] << 8) | buffer[buffer[3]+19];
	scsi_Changers[target].counts[CHET_DT] =
		(buffer[buffer[3]+20] << 8) | buffer[buffer[3]+21];
	if (debug) {
	    printk(DEVICE_NAME
		   ": mt:0x%x+%d st:0x%x+%d ie:0x%x+%d dt:0x%x+%d\n",
		   scsi_Changers[target].firsts[CHET_MT],
		   scsi_Changers[target].counts[CHET_MT],
		   scsi_Changers[target].firsts[CHET_ST],
		   scsi_Changers[target].counts[CHET_ST],
		   scsi_Changers[target].firsts[CHET_IE],
		   scsi_Changers[target].counts[CHET_IE],
		   scsi_Changers[target].firsts[CHET_DT],
		   scsi_Changers[target].counts[CHET_DT]);
	}
    } else {
	printk(KERN_WARNING DEVICE_NAME
	       ": reading element address assigment page failed!\n");
#if 0
		for (i = 0; broken_devices[i].vendor != NULL; i++) {
	    if (0 == strncmp(scsi_Changers[target].device->vendor,
			     broken_devices[i].vendor,8) &&
		0 == strncmp(scsi_Changers[target].device->model,
			     broken_devices[i].model,16))
		{
		    printk(KERN_WARNING DEVICE_NAME
			   "... using values from internal table instead\n");
		    scsi_Changers[target].firsts[CHET_MT] =
			broken_devices[i].ft;
		    scsi_Changers[target].counts[CHET_MT] = 
			broken_devices[i].nt;
		    scsi_Changers[target].firsts[CHET_ST] = 
			broken_devices[i].fs;
		    scsi_Changers[target].counts[CHET_ST] = 
			broken_devices[i].ns;
		    scsi_Changers[target].firsts[CHET_IE] = 
			broken_devices[i].fe;
		    scsi_Changers[target].counts[CHET_IE] = 
			broken_devices[i].ne;
		    scsi_Changers[target].firsts[CHET_DT] = 
			broken_devices[i].fd;
		    scsi_Changers[target].counts[CHET_DT] = 
			broken_devices[i].nd;
		}
	}
#endif
    }
    
    /* look up the devices of the data transfer elements */
    scsi_Changers[target].dt =
	kmalloc(scsi_Changers[target].counts[CHET_DT]*sizeof(Scsi_Device*),
		GFP_ATOMIC);
    for (elem = 0; elem < scsi_Changers[target].counts[CHET_DT]; elem++) {
	if (0 != ch_read_element_status
	    (target,elem+scsi_Changers[target].firsts[CHET_DT],data)) {
	    printk(DEVICE_NAME " Huh?: READ ELEMENT STATUS for "
		   "data transfer element 0x%x failed\n",
		   elem+scsi_Changers[target].firsts[CHET_DT]);
	} else {
	    printk(KERN_INFO DEVICE_NAME " data transfer element 0x%x: ",
		   elem+scsi_Changers[target].firsts[CHET_DT]);
	    if (data[6] & 0x80) {
		printk("not this SCSI bus\n");
		scsi_Changers[target].dt[elem] = NULL;
	    } else if (0 == (data[6] & 0x30)) {
		printk("ID/LUN unknown\n");
		scsi_Changers[target].dt[elem] = NULL;
	    } else {
		int                  id,lun;
		
		id  = scsi_Changers[target].device->id;
		lun = 0;
		if (data[6] & 0x20) id  = data[7];
		if (data[6] & 0x10) lun = data[6] & 7;
		printk("ID %i, LUN %i, ",id,lun);
		scsi_Changers[target].dt[elem] =
		    find_device(scsi_Changers[target].device->host,
				scsi_Changers[target].device->channel,
				id,lun);
		if (!scsi_Changers[target].dt[elem]) {
		    /* should not happen */
		    printk("Huh? device not found !\n");
		} else {
		    printk("name: %8.8s %16.16s %4.4s\n",
			   scsi_Changers[target].dt[elem]->vendor,
			   scsi_Changers[target].dt[elem]->model,
			   scsi_Changers[target].dt[elem]->rev);
		}
	    }
	}
    }
    scsi_Changers[target].voltags = 1;
    scsi_free(buffer, 512);
    return 0;
}

/* ------------------------------------------------------------------------ */

static int
ch_position(int target, u_int trans, u_int elem, int rotate)
{
    u_char  cmd[10];

    if (debug)
	printk(KERN_DEBUG DEVICE_NAME " position: 0x%x\n",elem);
    if (0 == trans)
	trans = scsi_Changers[target].firsts[CHET_MT];
    memset(cmd,0,sizeof(cmd));
    cmd[0]  = 0x2b;                  /* POSITION TO ELEMENT */
    cmd[1]  = scsi_Changers[target].device->lun << 5;
    cmd[2]  = (trans >> 8) & 0xff;
    cmd[3]  =  trans       & 0xff;
    cmd[4]  = (elem  >> 8) & 0xff;
    cmd[5]  =  elem        & 0xff;
    cmd[8]  = rotate ? 1 : 0;
    return ch_do_cmd(target, cmd, NULL,0);
}

static int
ch_move(int target, u_int trans, u_int src, u_int dest, int rotate)
{
    u_char  cmd[12];

    if (debug)
	printk(KERN_DEBUG DEVICE_NAME " move: 0x%x => 0x%x\n",src,dest);
    if (0 == trans)
	trans = scsi_Changers[target].firsts[CHET_MT];
    memset(cmd,0,sizeof(cmd));
    cmd[0]  = 0xa5;                  /* MOVE MEDIUM */
    cmd[1]  = scsi_Changers[target].device->lun << 5;
    cmd[2]  = (trans >> 8) & 0xff;
    cmd[3]  =  trans       & 0xff;
    cmd[4]  = (src   >> 8) & 0xff;
    cmd[5]  =  src         & 0xff;
    cmd[6]  = (dest  >> 8) & 0xff;
    cmd[7]  =  dest        & 0xff;
    cmd[10] = rotate ? 1 : 0;
    return ch_do_cmd(target, cmd, NULL,0);
}

static int
ch_exchange(int target, u_int trans, u_int src,
	    u_int dest1, u_int dest2, int rotate1, int rotate2)
{
    u_char  cmd[12];

    if (debug)
	printk(KERN_DEBUG DEVICE_NAME " exchange: 0x%x => 0x%x => 0x%x\n",
	       src,dest1,dest2);
    if (0 == trans)
	trans = scsi_Changers[target].firsts[CHET_MT];
    memset(cmd,0,sizeof(cmd));
    cmd[0]  = 0xa6;                  /* EXCHANGE MEDIUM */
    cmd[1]  = scsi_Changers[target].device->lun << 5;
    cmd[2]  = (trans >> 8) & 0xff;
    cmd[3]  =  trans       & 0xff;
    cmd[4]  = (src   >> 8) & 0xff;
    cmd[5]  =  src         & 0xff;
    cmd[6]  = (dest1 >> 8) & 0xff;
    cmd[7]  =  dest1       & 0xff;
    cmd[8]  = (dest2 >> 8) & 0xff;
    cmd[9]  =  dest2       & 0xff;
    cmd[10] = (rotate1 ? 1 : 0) | (rotate2 ? 2 : 0);

    return ch_do_cmd(target, cmd, NULL,0);
}

/* ------------------------------------------------------------------------ */

static int
ch_release(struct inode * inode, struct file * filp)
{
    int minor = MINOR(inode->i_rdev);
    
    scsi_Changers[minor].device->access_count--;
#if LINUX_VERSION_CODE >= 0x020117
    if (scsi_Changers[minor].device->host->hostt->module)
	__MOD_DEC_USE_COUNT(scsi_Changers[minor].device->host->hostt->module);
    if(ch_template.module)
	__MOD_DEC_USE_COUNT(ch_template.module);
#else
    if (scsi_Changers[minor].device->host->hostt->usage_count)
	(*scsi_Changers[minor].device->host->hostt->usage_count)--;
    if(ch_template.usage_count)
	(*ch_template.usage_count)--;
#endif
    return 0;
}

static int
ch_open(struct inode * inode, struct file * filp)
{    
    int minor = MINOR(inode->i_rdev);

    if(minor >= ch_template.nr_dev || !scsi_Changers[minor].device)
	return -ENXIO;

    scsi_Changers[minor].device->access_count++;
#if LINUX_VERSION_CODE >= 0x020117
    if (scsi_Changers[minor].device->host->hostt->module)
	__MOD_INC_USE_COUNT(scsi_Changers[minor].device->host->hostt->module);
    if(ch_template.module)
	__MOD_INC_USE_COUNT(ch_template.module);
#else
    if (scsi_Changers[minor].device->host->hostt->usage_count)
	(*scsi_Changers[minor].device->host->hostt->usage_count)++;
    if(ch_template.usage_count)
	(*ch_template.usage_count)++;
#endif
    
    return 0;
}

static int
ch_checkrange(int target, int type, int unit)
{
    if (type < 0 || type > 3 || unit < 0 ||
	unit >= scsi_Changers[target].counts[type])
	return -1;
    return 0;    
}

/* for data transfer elements: check if they are busy */
static int
ch_is_busy(int target, int type, int unit)
{
    if (!check_busy)
	return 0;
    if (type != CHET_DT)
	return 0;
    if (!scsi_Changers[target].dt[unit])
	return 0;
    return scsi_Changers[target].dt[unit]->access_count;
}

static int
ch_ioctl(struct inode * inode, struct file * filp,
	 unsigned int cmd, unsigned long arg)
{
    int target = MINOR(inode->i_rdev);

    if (target >= ch_template.nr_dev ||
	!scsi_Changers[target].device) return -ENXIO;
	
    switch (cmd) {
    case CHIOGPARAMS:
    {
	struct changer_params params;

	params.cp_curpicker = 0;
	params.cp_npickers  = scsi_Changers[target].counts[CHET_MT];
	params.cp_nslots    = scsi_Changers[target].counts[CHET_ST];
	params.cp_nportals  = scsi_Changers[target].counts[CHET_IE];
	params.cp_ndrives   = scsi_Changers[target].counts[CHET_DT];

	WRITE_TO_USER ((void *) arg, &params, sizeof (struct changer_params));
	return 0;
    }
    break;
    
    case CHIOPOSITION:
    {
	struct changer_position pos;

	READ_FROM_USER (&pos, (void*)arg, sizeof (pos));

	if (0 != ch_checkrange(target, pos.cp_type, pos.cp_unit)) {
	    if (debug)
		printk(DEVICE_NAME ": CHIOPOSITION: invalid parameter\n");
	    return -EBADSLT;
	}
	return ch_position(target,0,
		    scsi_Changers[target].firsts[pos.cp_type] + pos.cp_unit,
		    pos.cp_flags & CP_INVERT);
    }
    break;
    
    case CHIOMOVE:
    {
	struct changer_move mv;

	READ_FROM_USER (&mv, (void*)arg, sizeof (mv));

	if (0 != ch_checkrange(target, mv.cm_fromtype, mv.cm_fromunit) ||
	    0 != ch_checkrange(target, mv.cm_totype,   mv.cm_tounit  )) {
	    if (debug)
		printk(DEVICE_NAME ": CHIOMOVE: invalid parameter\n");
	    return -EBADSLT;
	}
	if (ch_is_busy(target, mv.cm_fromtype, mv.cm_fromunit) ||
	    ch_is_busy(target, mv.cm_totype,   mv.cm_tounit  ))
	    return -EBUSY;

	return ch_move(target,0,
		scsi_Changers[target].firsts[mv.cm_fromtype] + mv.cm_fromunit,
		scsi_Changers[target].firsts[mv.cm_totype]   + mv.cm_tounit,
		mv.cm_flags & CM_INVERT);
    }
    break;

#if 0
    case CHIOINIT:
    {
    	return ch_init_changer(target);
    }
    break;
#endif

    case CHIOEXCHANGE:
    {
	struct changer_exchange mv;

	READ_FROM_USER (&mv, (void*)arg, sizeof (mv));

	if (0 != ch_checkrange(target, mv.ce_srctype,  mv.ce_srcunit ) ||
	    0 != ch_checkrange(target, mv.ce_fdsttype, mv.ce_fdstunit) ||
	    0 != ch_checkrange(target, mv.ce_sdsttype, mv.ce_sdstunit)) {
	    if (debug)
		printk(DEVICE_NAME ": CHIOEXCHANGE: invalid parameter\n");
	    return -EBADSLT;
	}
	if (0 != ch_is_busy(target, mv.ce_srctype,  mv.ce_srcunit ) ||
	    0 != ch_is_busy(target, mv.ce_fdsttype, mv.ce_fdstunit) ||
	    0 != ch_is_busy(target, mv.ce_sdsttype, mv.ce_sdstunit))
	    return -EBUSY;

	return ch_exchange
	    (target,0,
	     scsi_Changers[target].firsts[mv.ce_srctype]  + mv.ce_srcunit,
	     scsi_Changers[target].firsts[mv.ce_fdsttype] + mv.ce_fdstunit,
	     scsi_Changers[target].firsts[mv.ce_sdsttype] + mv.ce_sdstunit,
	     mv.ce_flags & CE_INVERT1, mv.ce_flags & CE_INVERT2);
    }
    break;
    
    case CHIOGSTATUS:
    {
	struct changer_element_status ces;
        int i;
	u_char data[16];
	
	READ_FROM_USER (&ces, (void*)arg, sizeof (ces));

	if (ces.ces_type < 0 || ces.ces_type > 3)
	    return -EINVAL;

#if LINUX_VERSION_CODE < 0x020100
	{ int err;
	  err = verify_area (VERIFY_WRITE, (void *) ces.ces_data,
			     scsi_Changers[target].counts[ces.ces_type]);
	  if (err) return err; }
#endif
	
	for (i = 0; i < scsi_Changers[target].counts[ces.ces_type]; i++) {
	    if (0 != ch_read_element_status
		(target, scsi_Changers[target].firsts[ces.ces_type]+i,data))
		return -EIO;
	    put_user(data[2], ces.ces_data+i);
	    if (data[2] & CESTATUS_EXCEPT)
		printk(DEVICE_NAME ": element 0x%x: asc=0x%x, ascq=0x%x\n",
		       scsi_Changers[target].firsts[ces.ces_type]+i,
		       (int)data[4],(int)data[5]);
	    if (0 != ch_read_element_status
		(target, scsi_Changers[target].firsts[ces.ces_type]+i,data))
		return -EIO;
	}
    }
    break;
    
    case CHIOGELEM:
    {
	struct changer_get_element cge;
	u_char  cmd[12];
	u_char  *buffer;
	int     elem,result,i;
	
	READ_FROM_USER (&cge, (void*)arg, sizeof (cge));

	if (0 != ch_checkrange(target, cge.cge_type, cge.cge_unit))
	    return -EINVAL;
	elem = scsi_Changers[target].firsts[cge.cge_type] + cge.cge_unit;

	buffer = (unsigned char *) scsi_malloc(512);
	if(!buffer) return -ENOMEM;

voltag_retry:
	memset(cmd,0,sizeof(cmd));
	cmd[0] = 0xb8;                      /* READ ELEMENT STATUS */
	cmd[1] = scsi_Changers[target].device->lun << 5 |
	    (scsi_Changers[target].voltags ? 0x10 : 0);
	cmd[2] = (elem >> 8) & 0xff;
	cmd[3] = elem        & 0xff;
	cmd[5] = 1;                   /* # of elements, currently only one */
	cmd[9] = 255;                 /* alloc length  */

	if (0 == (result = ch_do_cmd(target, cmd, buffer, 256))) {
	    cge.cge_status = buffer[18];
	    cge.cge_flags = 0;
	    if (buffer[18] & CESTATUS_EXCEPT) {
		/* XXX cge_errno */
	    }
	    if (buffer[25] & 0x80) {
		cge.cge_flags |= CGE_SRC;
		if (buffer[25] & 0x40)
		    cge.cge_flags |= CGE_INVERT;
		elem = (buffer[26]<<8) | buffer[27];
		for (i = 0; i < 4; i++) {
		    if (elem >= scsi_Changers[target].firsts[i] &&
			elem < scsi_Changers[target].firsts[i]+
			scsi_Changers[target].counts[i]) {
			cge.cge_srctype = i;
			cge.cge_srcunit = elem-scsi_Changers[target].firsts[i];
		    }
		}
	    }
	    if ((buffer[22] & 0x30) == 0x30) {
		cge.cge_flags |= CGE_IDLUN;
		cge.cge_id  = buffer[23];
		cge.cge_lun = buffer[22] & 7;
	    }
	    if (buffer[9] & 0x80) {
		cge.cge_flags |= CGE_PVOLTAG;
		memcpy(cge.cge_pvoltag,buffer+28,36);
	    }
	    if (buffer[9] & 0x40) {
		cge.cge_flags |= CGE_AVOLTAG;
		memcpy(cge.cge_avoltag,buffer+64,36);
	    }
	} else if (scsi_Changers[target].voltags) {
	    scsi_Changers[target].voltags = 0;
	    goto voltag_retry;
	}
	
	scsi_free(buffer, 512);

	WRITE_TO_USER ((void*)arg, &cge, sizeof (cge));
	return result;
	break;
    }
    
    default:
	return scsi_ioctl(scsi_Changers[target].device, cmd, arg);

    }
    return 0;
}

/* ------------------------------------------------------------------------ */

static int
ch_detect(Scsi_Device * SDp)
{
    if(SDp->type != TYPE_CHANGER)
	return 0;
    
    printk("Detected scsi changer ch%d at scsi%d, channel %d, id %d, lun %d\n", 
	   ch_template.dev_noticed++,
	   SDp->host->host_no, SDp->channel, SDp->id, SDp->lun); 
    
    return 1;
}

static int ch_attach(Scsi_Device * SDp){
    Scsi_Changer * cpnt;
    int i;
    
    if(SDp->type != TYPE_CHANGER)
	return 1;
    
    if (ch_template.nr_dev >= ch_template.dev_max)
    {
	SDp->attached--;
	return 1;
    }
    
    for(cpnt = scsi_Changers, i=0; i<ch_template.dev_max; i++, cpnt++) 
	if(!cpnt->device) break;
    
    if(i >= ch_template.dev_max) panic ("scsi_devicelist corrupt (ch)");

#if 0
    SDp->scsi_request_fn = do_ch_request;
#endif
    scsi_Changers[i].device = SDp;
    
    ch_template.nr_dev++;
    if(ch_template.nr_dev > ch_template.dev_max)
	panic ("scsi_devicelist corrupt (ch)");
    return 0;
}

static int ch_registered = 0;

static int
ch_init()
{
    if(ch_template.dev_noticed == 0) return 0;

    if(!ch_registered) {
	if (register_chrdev(MAJOR_NR,"ch",&changer_fops)) {
	    printk("Unable to get major %d for SCSI-Changer\n",MAJOR_NR);
	    return 1;
	}
	ch_registered++;
    }
    
    if (scsi_Changers) return 0;
    ch_template.dev_max =
            ch_template.dev_noticed + CH_EXTRA_DEVS;
    scsi_Changers = (Scsi_Changer *)
	scsi_init_malloc(ch_template.dev_max * sizeof(Scsi_Changer),
			 GFP_ATOMIC);
    memset(scsi_Changers, 0, ch_template.dev_max * sizeof(Scsi_Changer));

    return 0;
}

void
ch_finish()
{
    int i;
    
    for (i = 0; i < ch_template.nr_dev; ++i)
    {
	if (scsi_Changers[i].device) {
	    ch_readconfig(i);
	    ch_init_elem(i);
	}
    }

    return;
}	

static void ch_detach(Scsi_Device * SDp)
{
    Scsi_Changer * cpnt;
    int i;
    
    for(cpnt = scsi_Changers, i=0; i<ch_template.dev_max; i++, cpnt++) 
	if(cpnt->device == SDp) {
	    /*
	     * Reset things back to a sane state so that one can re-load a new
	     * driver (perhaps the same one).
	     */
	    kfree(cpnt->dt);
	    cpnt->device = NULL;
	    SDp->attached--;
	    ch_template.nr_dev--;
	    ch_template.dev_noticed--;
	    return;
	}
    return;
}

#ifdef MODULE

int init_module(void) {
    printk(KERN_INFO "SCSI Media Changer driver v" VERSION
	   " for Linux " UTS_RELEASE " loaded\n");
#if LINUX_VERSION_CODE >= 0x020117
    ch_template.module = &__this_module;
#else
    ch_template.usage_count = &mod_use_count_;
#endif
    return scsi_register_module(MODULE_SCSI_DEV, &ch_template);
}

void cleanup_module( void) 
{
    scsi_unregister_module(MODULE_SCSI_DEV, &ch_template);
    unregister_chrdev(MAJOR_NR, "ch");
    ch_registered--;
    if(scsi_Changers != NULL) {
	scsi_init_free((char *) scsi_Changers,
		       (ch_template.dev_noticed + CH_EXTRA_DEVS) 
		       * sizeof(Scsi_Changer));
    }
    
    ch_template.dev_max = 0;
}
#endif /* MODULE */
