/*****************************************************************************/

/*
 *	stlshow.c  -- display driver internal port state (for debug).
 *
 *	Copyright (C) 1996-1997  Stallion Technologies (support@stallion.oz.au).
 *	Copyright (C) 1996  Greg Ungerer (gerg@stallion.oz.au).
 *	Copyright (C) 1996  Nick Andrew (nick@zeta.org.au).
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*****************************************************************************/

#define __KERNEL__

#include <fcntl.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <sys/stat.h>
#include <linux/types.h>
#include <linux/termios.h>
#include <linux/tqueue.h>
#include <linux/serial.h>
#include <linux/sched.h>

#include "cdk.h"
#include "comstats.h"
#include "istallion.h"
#include "stallion.h"

/*****************************************************************************/

char	*version = "5.4.0";
char	*defdevice="/dev/staliomem0";
char	*ctrldevice;
char	*progname;

unsigned char	data[1024];

/*
 *	List of all options used.
 */
struct option longops[] = {
	{ "version", 0, 0, 'V' },
	{ "help", 0, 0, 'h' },
	{ "port", 1, 0, 'p' },
	{ "board", 0, 0, 'b' },
	{ "panel", 0, 0, 'P' },
	{ "device", 0, 0, 'd' },
	{ "control-device", 0, 0, 'c' },
	{ 0, 0, 0, 0 }
};

/*****************************************************************************/

/*
 *	Function prototypes.
 */
void	usage(void);
void	useportdevice(int fd, char *devname, int *brdnrp, int *panelnrp, int *portnrp);
int	stlshowbrd(int fd, int brdnr);
int	stlshowport(int fd, int brdnr, int panelnr, int portnr);
int	stlishowbrd(int fd, int brdnr);
int	stlishowport(int fd, int brdnr, int panelnr, int portnr);

/*****************************************************************************/

void usage(void)
{
	fprintf(stderr, "Usage: %s [OPTION]\n\n", progname);
	fprintf(stderr, "  -h, --help            print this information\n");
	fprintf(stderr, "  -V, --version         show version information and exit\n");
	fprintf(stderr, "  -b, --board           dislpay board\n");
	fprintf(stderr, "  -P, --panel           dislpay panel\n");
	fprintf(stderr, "  -p, --port            dislpay port\n");
	fprintf(stderr, "  -d, --device          dislpay port device state\n");
	fprintf(stderr, "  -c, --control-device  specify control device to use\n");
}

/*****************************************************************************/

void useportdevice(int fd, char *devname, int *brdnrp, int *panelnrp, int *portnrp)
{
	struct stat	statinfo;
	stlibrd_t	*brdp;
	int		panel, port, portcnt, rc;

	if (stat(devname, &statinfo) < 0) {
		fprintf(stderr, "%s: port device %s does not exist\n",
			progname, devname);
		exit(1);
	}
	if ((statinfo.st_mode & S_IFMT) != S_IFCHR) {
		fprintf(stderr, "%s: port device %s is not a char device\n",
			progname, devname);
		exit(1);
	}

	*brdnrp = (statinfo.st_rdev & 0xc0) >> 6;

	brdp = (stlibrd_t *) &data[0];
	brdp->brdnr = *brdnrp;
	rc = ioctl(fd, COM_READBOARD, brdp);
	if (rc < 0) {
		fprintf(stderr, "%s: ioctl(COM_READBOARD) failed!\n", progname);
		perror("showboard");
		exit(-1);
	}

	portcnt = 0;
	port = statinfo.st_rdev & 0x3f;
	for (panel = 0; (panel < brdp->nrpanels); panel++) {
		if ((portcnt + brdp->panels[panel]) > port)
			break;
		portcnt += brdp->panels[panel];
	}
	*panelnrp = panel;
	*portnrp = port - portcnt;
}

/*****************************************************************************/

int stlishowbrd(int fd, int brdnr)
{
	stlibrd_t	*brdp;
	int		panel, port;
	int		rc;

	brdp = (stlibrd_t *) &data[0];
	brdp->brdnr = brdnr;
	rc = ioctl(fd, COM_READBOARD, brdp);
	if (rc < 0) {
		fprintf(stderr, "%s: ioctl(COM_READBOARD) failed!\n", progname);
		perror("showboard");
		return(-1);
	}
	if (brdp->magic != STLI_BOARDMAGIC)
		return(-1);

	fprintf(stdout, "STLI BOARD STRUCTURE FOR board=%d\n\n", brdnr);
	fprintf(stdout, "brdnr = %d\n", brdp->brdnr);
	fprintf(stdout, "brdtype = %d\n", brdp->brdtype);
	fprintf(stdout, "magic = 0x%x\n", brdp->magic);
	fprintf(stdout, "state = %d\n", brdp->state);
	fprintf(stdout, "nrpanels = %d\n", brdp->nrpanels);
	fprintf(stdout, "nrports = %d\n", brdp->nrports);
	fprintf(stdout, "nrdevs = %d\n", brdp->nrdevs);
	fprintf(stdout, "iobase = %u\n", brdp->iobase);
	fprintf(stdout, "memaddr = %lu\n", brdp->memaddr);
	fprintf(stdout, "membase = %p\n", brdp->membase);
	fprintf(stdout, "memsize = %d\n", brdp->memsize);
	fprintf(stdout, "pagesize = %d\n", brdp->pagesize);
	fprintf(stdout, "hostoffset = %d\n", brdp->hostoffset);
	fprintf(stdout, "slaveoffset = %d\n", brdp->slaveoffset);
	fprintf(stdout, "bitsize = %d\n", brdp->bitsize);

	for (panel = 0; panel < STL_MAXPANELS; ++panel)
		fprintf(stdout, "panels[%d] = %d\n", panel,
			brdp->panels[panel]);

	fprintf(stdout, "init = %p\n", brdp->init);
	fprintf(stdout, "enable = %p\n", brdp->enable);
	fprintf(stdout, "reenable = %p\n", brdp->reenable);
	fprintf(stdout, "disable = %p\n", brdp->disable);
	fprintf(stdout, "getmemptr = %p\n", brdp->getmemptr);
	fprintf(stdout, "intr = %p\n", brdp->intr);
	fprintf(stdout, "reset = %p\n", brdp->reset);

	for (port = 0; port < STL_MAXPORTS; ++port)
		fprintf(stdout, "ports[%d] = %p\n", port, brdp->ports[port]);
	return(0);
}

/*****************************************************************************/

int stlishowport(int fd, int brdnr, int panelnr, int portnr)
{
	stliport_t	*portp;
	int		rc;

	portp = (stliport_t *) &data[0];
	portp->brdnr = brdnr;
	portp->panelnr = panelnr;
	portp->portnr = portnr;
	
	rc = ioctl(fd, COM_READPORT, portp);
	if (rc < 0) {
		fprintf(stderr, "%s: ioctl(COM_READPORT) failed!\n", progname);
		perror("ERROR");
		return(-1);
	}
	if (portp->magic != STLI_PORTMAGIC)
		return(-1);

	fprintf(stdout, "STLI PORT STRUCTURE FOR board=%d panel=%d port=%d\n\n",
		brdnr, panelnr, portnr);
	fprintf(stdout, "brdnr = %d\n", portp->brdnr);
	fprintf(stdout, "panelnr = %d\n", portp->panelnr);
	fprintf(stdout, "portnr = %d\n", portp->portnr);
	fprintf(stdout, "magic = 0x%x\n", portp->magic);
	fprintf(stdout, "state = 0x%lx\n", portp->state);
	fprintf(stdout, "devnr = %d\n", portp->devnr);
	fprintf(stdout, "flags = 0x%x\n", portp->flags);
	fprintf(stdout, "baud_base = %d\n", portp->baud_base);
	fprintf(stdout, "custom_divisor = %d\n", portp->custom_divisor);
	fprintf(stdout, "close_delay = %d\n", portp->close_delay);
	fprintf(stdout, "closing_wait = %d\n", portp->closing_wait);
	fprintf(stdout, "refcount = %d\n", portp->refcount);
	fprintf(stdout, "openwaitcnt = %d\n", portp->openwaitcnt);
	fprintf(stdout, "rc = %d\n", portp->rc);
	fprintf(stdout, "argsize = %d\n", portp->argsize);
	fprintf(stdout, "argp = %p\n", portp->argp);
	fprintf(stdout, "session = %ld\n", portp->session);
	fprintf(stdout, "pgrp = %ld\n", portp->pgrp);
	fprintf(stdout, "rxmarkmsk = %u\n", portp->rxmarkmsk);
	fprintf(stdout, "tty = %p\n", portp->tty);
	fprintf(stdout, "open_wait = %p\n", portp->open_wait);
	fprintf(stdout, "close_wait = %p\n", portp->close_wait);
	fprintf(stdout, "raw_wait = %p\n", portp->raw_wait);
#ifdef	JUNK
	fprintf(stdout, "tqhangup = %d\n", portp->tqhangup);
	fprintf(stdout, "normaltermios = %d\n", portp->normaltermios);
	fprintf(stdout, "callouttermios = %d\n", portp->callouttermios);
	fprintf(stdout, "asig = %d\n", portp->asig);
#endif
	fprintf(stdout, "addr = %lu\n", portp->addr);
	fprintf(stdout, "rxoffset = %lu\n", portp->rxoffset);
	fprintf(stdout, "txoffset = %lu\n", portp->txoffset);
	fprintf(stdout, "sigs = 0x%lx\n", portp->sigs);
	fprintf(stdout, "pflag = %lu\n", portp->pflag);
	fprintf(stdout, "rxsize = %u\n", portp->rxsize);
	fprintf(stdout, "txsize = %u\n", portp->txsize);
	fprintf(stdout, "reqbit = 0x%02x\n", portp->reqbit);
	fprintf(stdout, "portidx = 0x%02x\n", portp->portidx);
	fprintf(stdout, "portbit = 0x%02x\n", portp->portbit);
	return(0);
}

/*****************************************************************************/

int stlshowbrd(int fd, int brdnr)
{
	stlbrd_t	*brdp;
	int		panel, port;
	int		rc;

	brdp = (stlbrd_t *) &data[0];
	brdp->brdnr = brdnr;
	rc = ioctl(fd, COM_READBOARD, brdp);
	if (rc < 0) {
		fprintf(stderr, "%s: ioctl(COM_READBOARD) failed!\n", progname);
		perror("showboard");
		return(-1);
	}
	if (brdp->magic != STL_BOARDMAGIC)
		return(-1);

	fprintf(stdout, "STL BOARD STRUCTURE FOR board=%d\n\n", brdnr);
	fprintf(stdout, "brdnr = %d\n", brdp->brdnr);
	fprintf(stdout, "brdtype = %d\n", brdp->brdtype);
	fprintf(stdout, "magic = 0x%x\n", brdp->magic);
	fprintf(stdout, "state = %d\n", brdp->state);
	fprintf(stdout, "nrpanels = %d\n", brdp->nrpanels);
	fprintf(stdout, "nrports = %d\n", brdp->nrports);
	fprintf(stdout, "irq = %d\n", brdp->irq);
	fprintf(stdout, "irqtype = %d\n", brdp->irqtype);
	fprintf(stdout, "ioaddr1 = %u\n", brdp->ioaddr1);
	fprintf(stdout, "ioaddr2 = %u\n", brdp->ioaddr2);
	fprintf(stdout, "iostatus = %u\n", brdp->iostatus);
	fprintf(stdout, "ioctrl = %u\n", brdp->ioctrl);
	fprintf(stdout, "ioctrlval = %u\n", brdp->ioctrlval);
	fprintf(stdout, "hwid = %u\n", brdp->hwid);
	fprintf(stdout, "clk = %u\n", brdp->clk);

	for (panel = 0; panel < STL_MAXPANELS; ++panel)
		fprintf(stdout, "panels[%d] = %p\n", panel,
			brdp->panels[panel]);
	return(0);
}

/*****************************************************************************/

int stlshowport(int fd, int brdnr, int panelnr, int portnr)
{
	stlport_t	*portp;
	int		rc;

	portp = (stlport_t *) &data[0];
	portp->brdnr = brdnr;
	portp->panelnr = panelnr;
	portp->portnr = portnr;
	
	rc = ioctl(fd, COM_READPORT, portp);
	if (rc < 0) {
		fprintf(stderr, "%s: ioctl(COM_READPORT) failed!\n", progname);
		perror("ERROR");
		return(-1);
	}
	if (portp->magic != STL_PORTMAGIC)
		return(-1);

	fprintf(stdout, "STL PORT STRUCTURE FOR board=%d panel=%d port=%d\n\n",
		brdnr, panelnr, portnr);
	fprintf(stdout, "brdnr = %d\n", portp->brdnr);
	fprintf(stdout, "panelnr = %d\n", portp->panelnr);
	fprintf(stdout, "portnr = %d\n", portp->portnr);
	fprintf(stdout, "magic = 0x%x\n", portp->magic);
	fprintf(stdout, "ioaddr = %d\n", portp->ioaddr);
	fprintf(stdout, "uartaddr = %d\n", portp->uartaddr);
	fprintf(stdout, "pagenr = %d\n", portp->pagenr);
	fprintf(stdout, "istate = 0x%lx\n", portp ->istate);
	fprintf(stdout, "flags = 0x%x\n", portp->flags);
	fprintf(stdout, "baud_base = %d\n", portp->baud_base);
	fprintf(stdout, "custom_divisor = %d\n", portp->custom_divisor);
	fprintf(stdout, "close_delay = %d\n", portp->close_delay);
	fprintf(stdout, "closing_wait = %d\n", portp->closing_wait);
	fprintf(stdout, "refcount = %d\n", portp->refcount);
	fprintf(stdout, "openwaitcnt = %d\n", portp->openwaitcnt);
	fprintf(stdout, "session = %ld\n", portp->session);
	fprintf(stdout, "pgrp = %ld\n", portp->pgrp);
	fprintf(stdout, "sigs = 0x%lx\n", portp->sigs);
	fprintf(stdout, "rxignoremsk = %u\n", portp->rxignoremsk);
	fprintf(stdout, "rxmarkmsk = %u\n", portp->rxmarkmsk);
	fprintf(stdout, "clk = %u\n", portp->clk);
	fprintf(stdout, "hwid = %u\n", portp->hwid);
	fprintf(stdout, "tty = %p\n", portp->tty);
	fprintf(stdout, "open_wait = %p\n", portp->open_wait);
	fprintf(stdout, "close_wait = %p\n", portp->close_wait);
#ifdef	JUNK
	fprintf(stdout, "normaltermios = %d\n", portp->normaltermios);
	fprintf(stdout, "callouttermios = %d\n", portp->callouttermios);
	fprintf(stdout, "tqueue = %d\n", portp->tqueue);
#endif
	fprintf(stdout, "tx.buf = %p\n", portp->tx.buf);
	fprintf(stdout, "tx.head = %p\n", portp->tx.head);
	fprintf(stdout, "tx.tail = %p\n", portp->tx.tail);
	return(0);
}

/*****************************************************************************/

int main(int argc, char *argv[])
{
	int	fd;
	char	*portdev;
	int	c, useport;
	int	brdnr, panelnr, portnr;

	progname = argv[0];
	ctrldevice = defdevice;
	brdnr  = 0;
	panelnr = 0;
	portnr = -1;
	useport = 0;

	while ((c = getopt_long(argc, argv, "?hVb:p:P:d:c:", longops, &optind)) != -1) {
		switch(c) {
		case 'V':
			fprintf(stdout, "%s version %s\n", progname, version);
			return(0);
			break;
		case 'b':
			brdnr = atoi(optarg);
			break;
		case 'p':
			portnr = atoi(optarg);
			break;
		case 'P':
			panelnr = atoi(optarg);
			break;
		case 'd':
			useport++;
			portdev = optarg;
			break;
		case 'c':
			ctrldevice = optarg;
			break;
		case '?':
		case 'h':
			usage();
			return(0);
		default:
			usage();
			return(2);
		}
	}

	fd = open(ctrldevice, O_RDWR);
	if (fd < 0) {
		fprintf(stderr, "%s: Cannot open control device %s\n",
			progname, ctrldevice);
		perror("ERROR");
		return(2);
	}

	if (useport)
		useportdevice(fd, portdev, &brdnr, &panelnr, &portnr);

	if (portnr == -1) {
		if (stlshowbrd(fd, brdnr) < 0)
			stlishowbrd(fd, brdnr);
	} else {
		if (stlshowport(fd, brdnr, panelnr, portnr) < 0)
			stlishowport(fd, brdnr, panelnr, portnr);
	}

	exit(0);
}

/*****************************************************************************/
