use strict;
use warnings;

use FindBin;
use lib "$FindBin::Bin/../lib";

use Test::More;

use_ok('TestApp');

my $dispatcher = TestApp->dispatcher;

#
#   Private Action
#
my $private_action = $dispatcher->get_action_by_path(
                       '/class_forward_test_method'
                     );

ok(!defined($dispatcher->uri_for_action($private_action)),
   "Private action returns undef for URI");

#
#   Path Action
#
my $path_action = $dispatcher->get_action_by_path(
                    '/action/testrelative/relative'
                  );

is($dispatcher->uri_for_action($path_action), "/action/relative/relative",
   "Public path action returns correct URI");

ok(!defined($dispatcher->uri_for_action($path_action, [ 'foo' ])),
   "no URI returned for Path action when snippets are given");

#
#   Index Action
#
my $index_action = $dispatcher->get_action_by_path(
                     '/action/index/index'
                   );

ok(!defined($dispatcher->uri_for_action($index_action, [ 'foo' ])),
   "no URI returned for index action when snippets are given");

is($dispatcher->uri_for_action($index_action),
   "/action/index",
   "index action returns correct path");

#
#   Chained Action
#
my $chained_action = $dispatcher->get_action_by_path(
                       '/action/chained/endpoint',
                     );

ok(!defined($dispatcher->uri_for_action($chained_action)),
   "Chained action without captures returns undef");

ok(!defined($dispatcher->uri_for_action($chained_action, [ 1, 2 ])),
   "Chained action with too many captures returns undef");

is($dispatcher->uri_for_action($chained_action, [ 1 ]),
   "/chained/foo/1/end",
   "Chained action with correct captures returns correct path");

#
#   Tests with Context
#
my $request = Catalyst::Request->new( {
                _log => Catalyst::Log->new,
                base => URI->new('http://127.0.0.1/foo')
              } );

my $context = TestApp->new( {
                request => $request,
                namespace => 'yada',
              } );




# JNAP: I'm going to todo these tests, calling uri_for as a class method
# should work, but its not really useful so I think theres not much harm
# if someone needs this for a business case they are welcome to figure out
# what is going

TODO: {
    local $TODO = "Need to fix using uri_for and uri_for_action as a class method";


# this works, using $ctx
is($context->uri_for($context->controller('Action::Chained')->action_for('endpoint')),
   "http://127.0.0.1/foo/yada/chained/foo/end",
   "uri_for a controller and action");


# this fails, uri_for returns undef, why isn't this one working??
  is( $context->uri_for_action( '/action/chained/endpoint' ),
          'http://127.0.0.1/chained/foo/end',
     "uri_for a controller and action as string");

# this fails, uri_for returns undef
  is(TestApp->uri_for_action($context->controller('Action::Chained')->action_for('endpoint')),
     "/chained/foo/end",
     "uri_for a controller and action, called with only class name");

# this fails, uri_for returns undef
  is(TestApp->uri_for_action('/action/chained/endpoint' ),
     "/chained/foo/end",
     "uri_for a controller and action as string, called with only class name");

# this fails, uri_for returns undef
  is(TestApp->uri_for_action(  $chained_action),
     "/chained/foo/end",
     "uri_for action via dispatcher, called with only class name");
}


is($context->uri_for($context->controller('Action')),
   "http://127.0.0.1/foo/yada/action/",
   "uri_for a controller");

is($context->uri_for($path_action),
   "http://127.0.0.1/foo/action/relative/relative",
   "uri_for correct for path action");

is($context->uri_for($path_action, qw/one two/, { q => 1 }),
   "http://127.0.0.1/foo/action/relative/relative/one/two?q=1",
   "uri_for correct for path action with args and query");

ok(!defined($context->uri_for($path_action, [ 'blah' ])),
   "no URI returned by uri_for for Path action with snippets");

is($context->uri_for($chained_action, [ 1 ], 2, { q => 1 }),
   "http://127.0.0.1/foo/chained/foo/1/end/2?q=1",
   "uri_for correct for chained with captures, args and query");

#
#   More Chained with Context Tests
#
{
    is( $context->uri_for_action( '/action/chained/endpoint2', [1,2], (3,4), { x => 5 } ),
        'http://127.0.0.1/foo/chained/foo2/1/2/end2/3/4?x=5',
        'uri_for_action correct for chained with multiple captures and args' );

    is( $context->uri_for_action( '/action/chained/endpoint2', [1,2,3,4], { x => 5 } ),
        'http://127.0.0.1/foo/chained/foo2/1/2/end2/3/4?x=5',
        'uri_for_action correct for chained with multiple captures and args combined' );

    is( $context->uri_for_action( '/action/chained/three_end', [1,2,3], (4,5,6) ),
        'http://127.0.0.1/foo/chained/one/1/two/2/3/three/4/5/6',
        'uri_for_action correct for chained with multiple capturing actions' );

    is( $context->uri_for_action( '/action/chained/three_end', [1,2,3,4,5,6] ),
        'http://127.0.0.1/foo/chained/one/1/two/2/3/three/4/5/6',
        'uri_for_action correct for chained with multiple capturing actions and args combined' );

    my $action_needs_two = '/action/chained/endpoint2';

    ok( ! defined( $context->uri_for_action($action_needs_two, [1],     (2,3)) ),
        'uri_for_action returns undef for not enough captures' );

    is( $context->uri_for_action($action_needs_two,            [1,2],   (2,3)),
        'http://127.0.0.1/foo/chained/foo2/1/2/end2/2/3',
        'uri_for_action returns correct uri for correct captures' );

    is( $context->uri_for_action($action_needs_two,            [1,2,2,3]),
        'http://127.0.0.1/foo/chained/foo2/1/2/end2/2/3',
        'uri_for_action returns correct uri for correct captures and args combined' );

    ok( ! defined( $context->uri_for_action($action_needs_two, [1,2,3], (2,3)) ),
        'uri_for_action returns undef for too many captures' );

    is( $context->uri_for_action($action_needs_two, [1,2],   (3)),
        'http://127.0.0.1/foo/chained/foo2/1/2/end2/3',
        'uri_for_action returns uri with lesser args than specified on action' );

    is( $context->uri_for_action($action_needs_two, [1,2,3]),
        'http://127.0.0.1/foo/chained/foo2/1/2/end2/3',
        'uri_for_action returns uri with lesser args than specified on action with captures combined' );

    is( $context->uri_for_action($action_needs_two, [1,2],   (3,4,5)),
        'http://127.0.0.1/foo/chained/foo2/1/2/end2/3/4/5',
        'uri_for_action returns uri with more args than specified on action' );

    is( $context->uri_for_action($action_needs_two, [1,2,3,4,5]),
        'http://127.0.0.1/foo/chained/foo2/1/2/end2/3/4/5',
        'uri_for_action returns uri with more args than specified on action with captures combined' );

    is( $context->uri_for_action($action_needs_two, [1,''], (3,4)),
        'http://127.0.0.1/foo/chained/foo2/1//end2/3/4',
        'uri_for_action returns uri with empty capture on undef capture' );

    is( $context->uri_for_action($action_needs_two, [1,'',3,4]),
        'http://127.0.0.1/foo/chained/foo2/1//end2/3/4',
        'uri_for_action returns uri with empty capture on undef capture and args combined' );

    is( $context->uri_for_action($action_needs_two, [1,2], ('',3)),
        'http://127.0.0.1/foo/chained/foo2/1/2/end2//3',
        'uri_for_action returns uri with empty arg on undef argument' );

    is( $context->uri_for_action($action_needs_two, [1,2,'',3]),
        'http://127.0.0.1/foo/chained/foo2/1/2/end2//3',
        'uri_for_action returns uri with empty arg on undef argument and args combined' );

    is( $context->uri_for_action($action_needs_two, [1,2], (3,'')),
        'http://127.0.0.1/foo/chained/foo2/1/2/end2/3/',
        'uri_for_action returns uri with empty arg on undef last argument' );

    is( $context->uri_for_action($action_needs_two, [1,2,3,'']),
        'http://127.0.0.1/foo/chained/foo2/1/2/end2/3/',
        'uri_for_action returns uri with empty arg on undef last argument with captures combined' );

    my $complex_chained = '/action/chained/empty_chain_f';
    is( $context->uri_for_action( $complex_chained, [23], (13), {q => 3} ),
        'http://127.0.0.1/foo/chained/empty/23/13?q=3',
        'uri_for_action returns correct uri for chain with many empty path parts' );
    is( $context->uri_for_action( $complex_chained, [23,13], {q => 3} ),
        'http://127.0.0.1/foo/chained/empty/23/13?q=3',
        'uri_for_action returns correct uri for chain with many empty path parts with captures and args combined' );

    eval { $context->uri_for_action( '/does/not/exist' ) };
    like $@, qr{^Can't find action for path '/does/not/exist'},
        'uri_for_action croaks on nonexistent path';

}

done_testing;

