package Net::FreeDB2;

# Copyright 2002, Vincenzo Zocca.

# See LICENSE section for usage and distribution rights.

require 5.005_62;
use strict;
use warnings;

require Exporter;
use AutoLoader qw(AUTOLOAD);
use Error qw (:try);

our @ISA = qw(Exporter);

# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.

# This allows declaration	use Net::FreeDB2 ':all';
# If you do not need this, moving things directly into @EXPORT or @EXPORT_OK
# will save memory.
our %EXPORT_TAGS = ( 'all' => [ qw(
	
) ] );

our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

our @EXPORT = qw(
	
);
our ( $VERSION ) = '$Revision: 0.8.2.6 $ ' =~ /\$Revision:\s+([^\s]+)/;

sub connection {
	# Allow both styles Net::FreeDB2->connection and Net::FreeDB2::connection
	$_[0] eq 'Net::FreeDB2' && shift;

	# Get options
	my $opt;
	if (ref ($_[0]) eq 'HASH') {
		$opt = shift;
	} else {
		$opt = {};
		foreach my $av (split (/\s*;+\s*/, shift)) {
			my ($a, $v) = split (/\s*=\s*/, $av);
			$a && $v || next;
			$opt->{$a} = $v;
		}
	}

	if (defined ($opt->{protocol}) && $opt->{protocol} eq 'CDDBP') {
		require Net::FreeDB2::Connection::CDDBP;
		import Net::FreeDB2::Connection::CDDBP;
		return (Net::FreeDB2::Connection::CDDBP->new ($opt));
	} else {
		require Net::FreeDB2::Connection::HTTP;
		import Net::FreeDB2::Connection::HTTP;
		return (Net::FreeDB2::Connection::HTTP->new ($opt));
	}
}


1;
__END__

=head1 NAME

Net::FreeDB2 - FreeDB generic connection class

=head1 SYNOPSIS

 # quick and dirty CD ripper
 
 use strict;
 
 # Make device name
 use Getopt::Std;
 $::opt_d = '';
 getopts ('d:');
 my $dev = $::opt_d || '/dev/cdrom';
 
 # Make a Net::FreeDB2::connection
 use Net::FreeDB2;
 my $conn = Net::FreeDB2->connection ({
 	client_name => 'quickrip',
 	client_version => '1.0',
 });
 
 # Create a Net::FreeDB2::Entry object with the information in it from the CD
 # in $dev
 use Net::FreeDB2::Entry;
 my $entry = Net::FreeDB2::Entry->new ({dev => $dev});
 
 # Query the FreeDB/CDDB database
 my $res = $conn->query ($entry);
 $res->hasError () && die ('Oops, error quering FreeDB');
 
 # Check for matches
 if (! scalar ($res->getMatches ())) {
 	warn ('No matches found');
 	exit (0);
 }
 
 # Get the first match
 my $match = ($res->getMatches ())[0];
 
 # Read the match into an entry
 my $res = $conn->read ($match);
 $res->hasError () && die ('Oops, error reading FreeDB');
 my $entry = $res->getEntry ();
 
 # Make cdparanoia commands to rip the tracks
 my $tracks = scalar ($entry->getFrameOffset ());
 my @cmd = qw (cdparanoia);
 push (@cmd, '-d', $dev);
 push (@cmd, '-B');
 for (my $i = 1; $i <= $tracks; $i++) {
 	my $title = $entry->getTtitlen ($i);
 	$title =~ s/\s+/-/g;
 	my @cmd = (@cmd, $i);
 
 	# Execute command
 	print STDERR "@cmd\n";
 	system (@cmd);
 	die ('Oops, command failed') if ($?>>8);
 
 	# Rename output file
 	my $src = sprintf ("track%02d.cdda.wav", $i);
 	my $dest = sprintf ("%02d-%s.wav", $i, $title);
 	print STDERR "mv $src $dest\n";
 	rename ($src, $dest);
 }

=head1 DESCRIPTION

C<Net::FreeDB2> is another FreeDB/CDDB access class hierarchy. The basics are similar to the other FreeDB/CDDB Perl modules.

However, in the writers opinion, the details differ enough for it to have a certain right of existance. It's organization is inspired by C<Net::LDAP>.

=over Information is captured into sensible objects.

=item Net::FreeDB

FreeDB/CDDB connection factory.

=item Net::FreeDB2::Connection

Abstract connection

=item Net::FreeDB2::Connection::HTTP

Concrete connection for the HTTP protocol

=item Net::FreeDB2::Connection::CDDBP

Concrete connection for the CDDBP protocol

=item Net::FreeDB2::Entry

A FreeDB/CDDB entry. Can be read/written from/to a FreeDB/CDDB connection or a file.

=item Net::FreeDB2::Match

Match obtained from a FreeDB/CDDB query comand

=item Net::FreeDB2::Site

Site obtained from a FreeDB/CDDB sites command

=item Net::FreeDB2::Response*

Responses from FreeDB/CDDB commands

=back

=head1 CONSTRUCTOR

none

=head1 METHODS

=over

=item connection (OPT_HASH_REF || OPT_STRING)

Returns an instanciated C<Net::FreeDB2::Connection> object. C<Error::Simple> exceptions generated by instanciated classes are passed through.

Allowed/mandatory options for C<OPT_HASH_REF> are:

=over

=item protocol

Protocol to use. Allowed vaulues: C<HTTP> and C<CDDBP>. Defaults to C<HTTP>. C<HTTP> causes a C<Net::FreeDB2::Connection::HTTP> object to be instanciated. C<CDDBP> causes a C<Net::FreeDB2::Connection::CDDBP> object to be instanciated.

=item client_name

Mandatory option to name the connecting client software.

=item client_version

Mandatory option with the client software version string.

=item client_host

The hostname of the client. Defaults to C<&Sys::Hostname::hostname ()>.

=item client_user

The user of the client. Defaults to C<scalar (getpwuid ($E<gt>))>;

=item freedb_host

The FreeDB/CDDB host to use (e.g. C<www.freedb.org>). If defined together with C<freedb_cgi> the connection is actually made during costruction.

=item freedb_port

The port on the FreeDB/CDDB host to use.

=item freedb_cgi

The FreeDB/CDDB host to use (e.g. C<~cddb/cddb.cgi>). If defined together with C<freedb_host> the connection is actually made during costruction. Only for C<HTTP> protocol. Will not cause error if set for protocol <CDDBP>.

=item proxy_host

Proxy host to use. Only supported for C<HTTP> protocol. Will not cause error if set for protocol <CDDBP>.

=item proxy_port

Proxy port to use. Defaults to C<8080>. Only supported for C<HTTP> protocol. Will not cause error if set for protocol <CDDBP>.

=item proxy_user

Proxy user name to use. Only supported for C<HTTP> protocol. Will not cause error if set for protocol <CDDBP>.

=item proxy_passwd

Proxy password to use. Only supported for C<HTTP> protocol. Will not cause error if set for protocol <CDDBP>.

=back

Options are allowed to be passed as strings. In this case, C<OPT_STRING> must be in the format:
 <attribute1>=<value1>;<attribute2>=<value2>;...
Example:
 client_name=my-client;client_version=0.0.1

=back

=head1 SEE ALSO

L<Net::FreeDB2::Entry>, L<Net::FreeDB2::Match>, L<Net::FreeDB2::Response>, L<Net::FreeDB2::Response::Query> and L<Net::FreeDB2::Response::Read>

=head1 BUGS

None known (yet).

=head1 HISTORY

First development: September 2002

=head1 AUTHOR

Vincenzo Zocca E<lt>Vincenzo@Zocca.comE<gt>

=head1 COPYRIGHT

Copyright 2002, Vincenzo Zocca.

=head1 LICENSE

This file is part of the C<Net::FreeDB2> module hierarchy for Perl by
Vincenzo Zocca.

The Net::FreeDB2 module hierarchy is free software; you can redistribute it
and/or modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2 of
the License, or (at your option) any later version.

The Net::FreeDB2 module hierarchy is distributed in the hope that it will
be useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with the Net::FreeDB2 module hierarchy; if not, write to
the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
Boston, MA 02111-1307 USA

=cut
