/*
 * LDAPGetEntries.java    1.1
 *
 * Copyright (c) 1997 Netscape Communications Corporation
 *
 * Netscape grants you a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Netscape.
 *
 * This software is provided "AS IS," without a warranty of any kind.
 * See the CDK License Agreement for additional terms and conditions.
 */
package netscape.ldap.beans;

import netscape.ldap.*;
import netscape.ldap.beans.LDAPBasePropertySupport;
import java.util.Enumeration;
import java.util.Vector;


/**
 * Invisible Bean that just takes a host, port, directory base,
 * search string, and optional authentication name and password,
 * and returns a list of all matching DNs. The search has the scope
 * "SUB", which means that it will find an entry anywhere at or
 * below the directory base, unless a different scope is specified.
 * <BR><BR>
 * Optionally, a client can register as a PropertyChangeListener
 * and will be notified when the values are available.
 *<BR><BR>
 * A null result means no matching DNs were found. The reason is
 * available through getErrorCode(), which returns one of
 * the following:
 *<PRE>
 *     OK
 *     INVALID_PARAMETER
 *     CONNECT_ERROR
 *     AUTHENTICATION_ERROR
 *     PROPERTY_NOT_FOUND
 *     AMBIGUOUS_RESULTS
 *</PRE>
 *
 *<pre>
 * History:
 *  v1.0: Rob Weltman
 *  v1.1: cdk_team
 *</pre>
 */
public class LDAPGetEntries extends LDAPBasePropertySupport {

	/**
	 * Constructor with no parameters
	 */
    public LDAPGetEntries() {
		super();
	}

	/**
	* Constructor with host, port, and base initializers
	* @param theHost host string
	* @param thePort port number
	* @param theBase directory base string
	*/
    public LDAPGetEntries( String theHost, int thePort, String theBase ) {
	    setHost( theHost );
		setPort( thePort );
		setBase( theBase );
	}

	/**
	* Constructor with host, port, base, and scope initializers
	* @param theHost host string
	* @param thePort port number
	* @param theBase directory base string
	* @param theScope one of LDAPConnection.SCOPE_BASE,
	* LDAPConnection.SCOPE_SUB, LDAPConnection.SCOPE_ONE
	*/
    public LDAPGetEntries( String theHost,
						   int thePort,
						   String theBase,
						   int theScope ) {
	    setHost( theHost );
		setPort( thePort );
		setBase( theBase );
		setScope( theScope );
	}

    private String convertToStrings( String[] aResult ) {
        String sResult = "";
        if ( null != aResult ) {
            for ( int i = 0; i < aResult.length; i++ ) {
                sResult += aResult[i] + "\n";
            }
        }
        return sResult;
    }
    
    private void notifyResult( String[] newResult ) {
	    String sNewResult = convertToStrings( newResult );
	    firePropertyChange( "result", result, newResult );
	    firePropertyChange( "resultString", _sResult, sNewResult );
		_sResult = sNewResult;
		result = newResult;
	}
	
	public void setResultString( String sNewValue ) {
	    _sResult = sNewValue;
	}

	public String getResultString() {
	    return _sResult;
	}
	
	/**
	 * Searches and returns values for a specified attribute
	 * @param host host string
	 * @param port port number
	 * @param base directory base string
	 * @param scope one of LDAPConnection.SCOPE_BASE,
	 * LDAPConnection.SCOPE_SUB, LDAPConnection.SCOPE_ONE
	 * @param filter search filter
	 * @param attribute name of property to return values for
	 * @return Array of values for the property
	 */
    public String[] getEntries( String host,
								int port,
								String base,
								int scope,
								String filter,
								String attribute) {
		setHost( host );
		setPort( port );
		setBase( base );
		setScope( scope );
		setFilter( filter );
		return getEntries();
	}

	/**
	 * Searches and returns values of a previously registered property,
	 * using previously set parameters
	 * @return Array of values for the property
	 */
    public String[] getEntries() {
		boolean invalid = false;
		if ( getFilter().length() < 1 ) {
		    printDebug( "Invalid filter" );
			invalid = true;
		} else if ( (getHost().length() < 1) || (getBase().length() < 1) ) {
		    printDebug( "Invalid filter" );
			invalid = true;
		}
		if ( invalid ) {
			setErrorCode( INVALID_PARAMETER );
			notifyResult( null );
			return null;
		}

		String[] res = null;
		LDAPConnection m_ldc = new LDAPConnection();
		try {
			try {
				printDebug("Connecting to " + getHost() +
						   " " + getPort());
				connect( m_ldc, getHost(), getPort());
			} catch (Exception e) {
				printDebug( "Failed to connect to " + getHost() + ": " +
							e.toString() );
				setErrorCode( CONNECT_ERROR );
				notifyResult( null );
				m_ldc = null;
				throw( new Exception() );
			}

			// Authenticate?
			if ( (!getAuthDN().equals("")) &&
				 (!getAuthPassword().equals("")) ) {

				printDebug( "Authenticating " + getAuthDN() );
				try {
					m_ldc.authenticate( getAuthDN(), getAuthPassword() );
				} catch (Exception e) {
					printDebug( "Failed to authenticate: " + e.toString() );
					setErrorCode( AUTHENTICATION_ERROR );
					notifyResult( null );
					throw( new Exception() );
				}
			}

			// Search
			try {
				printDebug("Searching " + getBase() +
						   " for " + getFilter() + ", scope = " + getScope());
				String[] attrs = new String[1];
				// Specify non-attribute, since we don't care about them
				attrs[0] = "cn";
				LDAPSearchResults results = m_ldc.search( getBase(),
														  getScope(),
														  getFilter(),
														  attrs,
														  false);

				// Create a vector for the results
				Vector v = new Vector();
				while ( results.hasMoreElements() ) {
					String dn = ((LDAPEntry)results.next()).getDN();
					v.addElement( dn );
					printDebug( "... " + dn );
				}
				// Pull out the DNs and create a string array
				if ( v.size() > 0 ) {
					res = new String[v.size()];
					for( int i = 0; i < v.size(); i++ )
						res[i] = (String)v.elementAt( i );
					setErrorCode( OK );
				} else {
					printDebug( "No entries found for " + getFilter() );
					setErrorCode( PROPERTY_NOT_FOUND );
				}
			} catch (Exception e) {
				printDebug( "Failed to search for " + getFilter() + ": " +
							e.toString() );
				setErrorCode( PROPERTY_NOT_FOUND );
			}
		} catch (Exception e) {
		}

		try {
			if ( (m_ldc != null) && m_ldc.isConnected() )
				m_ldc.disconnect();
		} catch ( Exception e ) {
		}

		notifyResult( res );
		return res;
	}

  /**
   * The main body if we run it as application instead of applet.
   * @param args list of arguments
   */
    public static void main(String args[]) {
        if (args.length != 5) {
		    System.out.println( "Usage: LDAPGetEntries host port base" +
								" scope filter" );
			System.exit(1);
        }
        LDAPGetEntries app = new LDAPGetEntries();
		app.setHost( args[0] );
		app.setPort( java.lang.Integer.parseInt( args[1] ) );
		app.setBase( args[2] );
		app.setScope( args[3] );
		app.setFilter( args[4] );
        String[] response = app.getEntries();
		if ( response != null ) {
			for( int i = 0; i < response.length; i++ )
				System.out.println( "\t" + response );
		}
        System.exit(0);
    }

    /*
	 * Variables
	 */
	public static final int OK = 0;
	public static final int INVALID_PARAMETER = 1;
	public static final int CONNECT_ERROR = 2;
	public static final int AUTHENTICATION_ERROR = 3;
	public static final int PROPERTY_NOT_FOUND = 4;
	public static final int AMBIGUOUS_RESULTS = 5;
    private String attribute = new String("cn");
    private String[] result;
    private String _sResult = null;
}
