/*
 * LDAPGetProperty.java    1.1
 *
 * Copyright (c) 1997 Netscape Communications Corporation
 *
 * Netscape grants you a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Netscape.
 *
 * This software is provided "AS IS," without a warranty of any kind.
 * See the CDK License Agreement for additional terms and conditions.
 */
package netscape.ldap.beans;

import netscape.ldap.*;
import netscape.ldap.beans.LDAPBasePropertySupport;
import java.util.Enumeration;
import java.util.Vector;


/**
 * Invisible Bean that just takes a name and password, host and
 * port, and directory base and attribute name, and returns an
 * attribute's values from a Directory Server. The values are
 * assumed to be strings, and are returned as an array. The
 * search has the scope "SUB", which means that it will find
 * an entry anywhere at or below the directory base.
 * <BR><BR>
 * Optionally, a client can register as a PropertyChangeListener
 * and will be notified when the values are available.
 *<BR><BR>
 * A null result means the property fetch failed. The reason is
 * available through getErrorCode(), which returns one of
 * the following:
 *<PRE>
 *     OK
 *     INVALID_PARAMETER
 *     CONNECT_ERROR
 *     AUTHENTICATION_ERROR
 *     PROPERTY_NOT_FOUND
 *     AMBIGUOUS_RESULTS
 *</PRE>
 *
 *<pre>
 * History:
 *  v1.0: Rob Weltman
 *  v1.1: cdk_team
 *</pre>
 */
public class LDAPGetProperty extends LDAPBasePropertySupport {

	/**
	 * Constructor with no parameters
	 */
    public LDAPGetProperty() {}

	/**
	* Constructor with host, port, and base initializers
	* @param theHost host string
	* @param thePort port number
	* @param theBase directory base string
	*/
    public LDAPGetProperty( String theHost, int thePort, String theBase ) {
	    setHost( theHost );
		setPort( thePort );
		setBase( theBase );
	}

	/**
	 * Returns the name of the attribute to retrieve
	 * @return attribute name to retrieve
	 */
    public String getAttribute() { 
		return attribute; 
	}

	/**
	 * Sets the attribute to retrieve
	 */
    public void setAttribute( String attr ) { 
		attribute = attr;
	}

    private String convertToStrings( String[] aResult ) {
        String sResult = "";
        if ( null != aResult ) {
            for ( int i = 0; i < aResult.length; i++ ) {
                sResult += aResult[i] + "\n";
            }
        }
        return sResult;
    }
    
    private void notifyResult( String[] newResult ) {
	    String sNewResult = convertToStrings( newResult );
	    firePropertyChange( "result", result, newResult );
	    firePropertyChange( "resultString", _sResult, sNewResult );
		_sResult = sNewResult;
		result = newResult;
	}
	
	/**
	 * Searches and returns values for a specified attribute
	 * @param host host string
	 * @param port port number
	 * @param base directory base string
	 * @param filter search filter
	 * @param attribute name of property to return values for
	 * @return Array of values for the property
	 */
    public String[] getProperty( String host, int port, String base,
								 String filter, String attribute) {
		setHost( host );
		setPort( port );
		setBase( base );
		setFilter( filter );
		setAttribute( attribute );
		return getProperty();
	}

	/**
	 * Searches and returns values of a previously registered property,
	 * using previously set parameters
	 * @return Array of values for the property
	 */
    public String[] getProperty() {
		if ( (attribute.length() < 1) || (getFilter().length() < 1) ) {
		    printDebug( "Invalid attribute name or filter" );
			setErrorCode( INVALID_PARAMETER );
			notifyResult( null );
			return null;
		}

	    String[] res = null;
		LDAPConnection m_ldc;
		try {
			m_ldc = new LDAPConnection();
			printDebug("Connecting to " + getHost() +
							   " " + getPort());
			connect( m_ldc, getHost(), getPort());
		} catch (Exception e) {
		    printDebug( "Failed to connect to " + getHost() + ": "
						+ e.toString() );
			setErrorCode( CONNECT_ERROR );
			notifyResult( null );
			return null;
		}

		// Authenticate?
		if ( (!getAuthDN().equals("")) && (!getAuthPassword().equals("")) ) {
		    printDebug( "Authenticating " + getAuthDN() + " - " +
								getAuthPassword() );
			try {
			    m_ldc.authenticate( getAuthDN(), getAuthPassword() );
			} catch (Exception e) {
			    printDebug( "Failed to authenticate to " +
									getHost() + ": " + e.toString() );
				setErrorCode( AUTHENTICATION_ERROR );
				notifyResult( null );
				return null;
			}
		}

		// Search
		try {
		    String[] attrs = new String[1];
			attrs[0] = attribute;
		    LDAPSearchResults results = m_ldc.search(getBase(),
													 LDAPv2.SCOPE_SUB,
													 getFilter(),
													 attrs, false);

			// Should be only one result, at most
			LDAPEntry entry;
			if ( results.hasMoreElements() ) {
			    entry = results.next();
				printDebug( "... " + entry.getDN() );
				// Shouldn't be any more
				if ( results.hasMoreElements() ) {
					printDebug( "More than one entry found for " +
								getFilter() );
					setErrorCode( AMBIGUOUS_RESULTS );
				} else {
                    // Good - exactly one entry found; get the attribute
					// Treat DN as a special case
					if ( attribute.equalsIgnoreCase( "dn" ) ) {
						res = new String[1];
						res[0] = entry.getDN();
						setErrorCode( OK );
					} else {
						LDAPAttributeSet attrset = entry.getAttributeSet();
						Enumeration attrsenum = attrset.getAttributes();
						if (attrsenum.hasMoreElements()) {
							LDAPAttribute attr =
								(LDAPAttribute)attrsenum.nextElement();
							printDebug( attr.getName() + " = " );
							// Get the values as strings
							Enumeration valuesenum = attr.getStringValues();
							// Create a string array for the results
							Vector v = new Vector();
							while (valuesenum.hasMoreElements()) {
								String val = (String)valuesenum.nextElement();
								v.addElement( val );
								printDebug( "\t\t" + val );
							}
							res = new String[v.size()];
							for( int i = 0; i < v.size(); i++ )
								res[i] = (String)v.elementAt( i );
							setErrorCode( OK );
						} else {
							printDebug( "No properties found for " +
										attribute );
							setErrorCode( PROPERTY_NOT_FOUND );
						}
					}
				}
            } else {
				printDebug( "No entries found for " + getFilter() );
				setErrorCode( PROPERTY_NOT_FOUND );
			}
		} catch (Exception e) {
		    printDebug( "Failed to search for " + getFilter() + ": "
                                    + e.toString() );
			setErrorCode( PROPERTY_NOT_FOUND );
		}

		notifyResult( res );
		return res;
	}

  /**
   * The main body if we run it as application instead of applet.
   * @param args list of arguments
   */
    public static void main(String args[]) {
        if (args.length != 5) {
		    System.out.println( "Usage: LDAPGetProperty host port base" +
								" filter attribute" );
			System.exit(1);
        }
        LDAPGetProperty app = new LDAPGetProperty();
		app.setHost( args[0] );
		app.setPort( java.lang.Integer.parseInt( args[1] ) );
		app.setBase( args[2] );
		app.setFilter( args[3] );
		app.setAttribute( args[4] );
        String[] response = app.getProperty();
		if ( response != null ) {
			for( int i = 0; i < response.length; i++ )
				System.out.println( "\t" + response );
		}
        System.exit(0);
    }

    /*
	 * Variables
	 */
	public static final int OK = 0;
	public static final int INVALID_PARAMETER = 1;
	public static final int CONNECT_ERROR = 2;
	public static final int AUTHENTICATION_ERROR = 3;
	public static final int PROPERTY_NOT_FOUND = 4;
	public static final int AMBIGUOUS_RESULTS = 5;
    private String attribute = new String("cn");
    private String[] result;
    private String _sResult = null;
}
