/*
 * LDAPIsMember.java    1.1
 *
 * Copyright (c) 1997 Netscape Communications Corporation
 *
 * Netscape grants you a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Netscape.
 *
 * This software is provided "AS IS," without a warranty of any kind.
 * See the CDK License Agreement for additional terms and conditions.
 */
package netscape.ldap.beans;

import netscape.ldap.*;
import java.util.Enumeration;
import java.util.StringTokenizer;

/**
 * Invisible Bean that just takes a host and port, optional
 * authentication name and password, and DN of a group and another DN
 * which might be a member of the group, and returns true or
 * false, depending on whether the second DN is a member of the first.
 * <BR><BR>
 * A false result means the member could not be identified as
 * belonging to the group. The exact reason is
 * available through getErrorCode(), which returns one of
 * the following:
 *<PRE>
 *     OK
 *     INVALID_PARAMETER
 *     CONNECT_ERROR
 *     AUTHENTICATION_ERROR
 *     PROPERTY_NOT_FOUND
 *     AMBIGUOUS_RESULTS
 *     NO_SUCH_OBJECT
 *</PRE>
 *<p>
 *<pre>
 * History:
 *  v1.0: Rob Weltman
 *  v1.1: cdk_team
 *</pre>
 */
public class LDAPIsMember extends LDAPBasePropertySupport {

	/**
	 * Constructor with no parameters
	 */
    public LDAPIsMember() {}

	/**
	* Constructor with host, port, and group DN initializers
	* @param host host string
	* @param port port number
	* @param group distinguished name of the group
	*/
    public LDAPIsMember( String host, int port, String group ) {
	    setHost( host );
		setPort( port );
		setGroup( group );
	}

	/**
	* Constructor with host, port, authentication DN and password
	* and group DN initializers
	* @param host host string
	* @param port port number
	* @param dn fully qualified distinguished name to authenticate
	* @param password password for authenticating the dn
	* @param group distinguished name of the group
	*/
    public LDAPIsMember( String host, int port,
						 String dn, String password, String theGroup ) {
	    setHost( host );
		setPort( port );
		setGroup( group );
		setAuthDN( dn );
		setAuthPassword( password );
	}

	/**
	 * Checks if an entity (specified by distinguished name) is a
	 * member of a particular group (specified by distinguished name)
	 * @param host host string
	 * @param port port number
	 * @param dn fully qualified distinguished name to authenticate;
	 * can be null or ""
	 * @param password password for authenticating the dn; can be null
	 * or ""
	 * @param group distinguished name of the group
	 * @param member distinguished name of member to be checked
	 * @return true if the specified member belongs to the group
	 */
    public boolean isMember( String host, int port,
							 String dn, String password,
							 String group, String member ) {
	    setHost( host );
		setPort( port );
		setAuthDN( dn );
		setAuthPassword( password );
		setGroup( group );
		setMember( member );

	    if ( (host == null) || (host.length() < 1)  ) {
		    printDebug( "Invalid host name" );
			setErrorCode( INVALID_PARAMETER );
			return false;
		}

		if ( (member == null) || (group == null) ||
			 (member.length() < 1) || (group.length() < 1) ) {
		    printDebug( "Invalid member or group name" );
			setErrorCode( INVALID_PARAMETER );
			return false;
		}

		LDAPConnection m_ldc;
		boolean isMember = false;
		try {
			m_ldc = new LDAPConnection();
			printDebug("Connecting to " + host +
							   " " + port);
			connect( m_ldc, getHost(), getPort());
		} catch (Exception e) {
		    printDebug( "Failed to connect to " + host + ": "
						+ e.toString() );
			setErrorCode( CONNECT_ERROR );
			return false;
		}

		// Authenticate?
		if ( (dn != null) && (password != null) &&
			 (dn.length() > 0) && (password.length() > 0) ) {
		    printDebug( "Authenticating " + dn + " - " + password );
			try {
			    m_ldc.authenticate( dn, password );
			} catch (Exception e) {
			    printDebug( "Failed to authenticate to " +
									host + ": " + e.toString() );
				setErrorCode( AUTHENTICATION_ERROR );
				return false;
			}
		}

		// Search
		try {
		    String[] attrs = new String[3];
			attrs[0] = "member";
			attrs[1] = "uniqueMember";
			attrs[2] = "memberOfGroup";
		    LDAPSearchResults results = m_ldc.search( group,
													 LDAPConnection.SCOPE_BASE,
													 "objectclass=*",
													 attrs, false);

			// Should be only one result, at most
			LDAPEntry entry;
			if ( results.hasMoreElements() ) {
			    entry = (LDAPEntry)results.nextElement();
				printDebug( "... " + entry.getDN() );
				// Shouldn't be any more
				if ( results.hasMoreElements() ) {
					printDebug( "More than one entry found for " +
								getFilter() );
					setErrorCode( AMBIGUOUS_RESULTS );
				} else {
					String normMember = normalizeDN( member );
                    // Good - exactly one entry found; get the attributes
					LDAPAttributeSet attrset = entry.getAttributeSet();
					Enumeration attrsenum = attrset.getAttributes();
					while ( attrsenum.hasMoreElements() && !isMember ) {
						LDAPAttribute attr =
							(LDAPAttribute)attrsenum.nextElement();
						printDebug( attr.getName() + " = " );
						// Get the values as strings
						Enumeration valuesenum = attr.getStringValues();
						while (valuesenum.hasMoreElements()) {
							String val = (String)valuesenum.nextElement();
							printDebug( "\t\t" + val );
							String normFound = normalizeDN( val );
							if ( normMember.equalsIgnoreCase( normFound ) ) {
								isMember = true;
								setErrorCode( OK );
								break;
							}
						}
					}
					if ( !isMember )
						setErrorCode( PROPERTY_NOT_FOUND );
				}
            } else {
				printDebug( "No entries found for " + group );
				setErrorCode( NO_SUCH_OBJECT );
			}
		} catch (Exception e) {
		    printDebug( "Failed to search for " + group + ": "
                                    + e.toString() );
			setErrorCode( NO_SUCH_OBJECT );
		}

		return isMember;
	}

	/**
	 * Checks if an entity (specified by distinguished name) is a
	 * member of a particular group (specified by distinguished name)
	 * @return true if the specified member belongs to the group
	 */
    public boolean isMember() {
	    return isMember( getHost(), getPort(),
						 getAuthDN(), getAuthPassword(),
						 getGroup(), getMember() );
	}

	/**
	 * Returns the distinguished name of the group
	 * @return group name
	 */
    public String getGroup() { 
		return group; 
	}

	/**
	 * Sets the distinguished name of the group
	 * @param group group name
	 */
    public void setGroup( String group ) { 
		this.group = group;
	}

	/**
	 * Returns the distinguished name of the member
	 * @return member name
	 */
    public String getMember() { 
		return member;
	}

	/**
	 * Sets the distinguished name of the member
	 * @param member member name
	 */
    public void setMember( String member ) { 
		this.member = member;
	}

    private String normalizeDN( String dn ) {
		StringTokenizer st = new StringTokenizer( dn, "," );
		String norm = "";
		if( st.hasMoreTokens() ) {
			norm = st.nextToken();
			while( st.hasMoreTokens() )
				norm = norm + "," + st.nextToken().trim();
		}
		return norm;
	}

  /**
   * The main body if we run it as application instead of applet.
   * @param args list of arguments
   */
    public static void main(String args[]) {
        if (args.length != 4) {
		    System.out.println( "Usage: LDAPIsMember host port group" +
								" member" );
			System.exit(1);
        }
        LDAPIsMember app = new LDAPIsMember();
		app.setHost( args[0] );
		app.setPort( java.lang.Integer.parseInt( args[1] ) );
		app.setGroup( args[2] );
		app.setMember( args[3] );
        boolean response = app.isMember();
		if ( response == false )
			System.out.println( "Not a member" );
		else
			System.out.println( "Is a member" );
        System.exit(0);
    }

    /*
	 * Variables
	 */
	public static final int OK = 0;
	public static final int INVALID_PARAMETER = 1;
	public static final int CONNECT_ERROR = 2;
	public static final int AUTHENTICATION_ERROR = 3;
	public static final int PROPERTY_NOT_FOUND = 4;
	public static final int AMBIGUOUS_RESULTS = 5;
	public static final int NO_SUCH_OBJECT = 5;
    private String group = "";
	private String member = "";
}
