/*
 * LDAPSimpleAuth.java    1.1
 *
 * Copyright (c) 1997 Netscape Communications Corporation
 *
 * Netscape grants you a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Netscape.
 *
 * This software is provided "AS IS," without a warranty of any kind.
 * See the CDK License Agreement for additional terms and conditions.
 */
package netscape.ldap.beans;

import netscape.ldap.*;
import netscape.ldap.beans.LDAPBasePropertySupport;

/**
 * Invisible Bean that just authenticates a user with a Directory
 * Server and returns Y or N. It takes a host and port, and then either
 * a full distinguished name and password, an RDN and directory base, or
 * a cn value and directory base.
 * <BR><BR>
 * Optionally, a client can register as
 * a PropertyChangeListener and will be notified when an authentication
 * completes.
 * <BR><BR>
 * The Bean can be used from JavaScript, as in the following example
 * where the parameters are taken from HTML text fields in an HTML
 * form called "input":
 * <PRE>
 * <XMP>
 * <SCRIPT LANGUAGE="JavaScript">
 * function checkAuthentication() {
 *     auth = new Packages.netscape.ldap.beans.LDAPSimpleAuth();
 *     auth.setHost( document.input.host.value );
 *     auth.setPort( parseInt(document.input.port.value) );
 *     auth.setBase( document.input.base.value );
 *     auth.setAuthDN( document.input.username.value );
 *     auth.setAuthPassword( document.input.password.value );
 *     result = auth.authenticate();
 *     alert( "The response is: " + result );
 * }
 * </SCRIPT>
 * </XMP>
 *</PRE>
 *
 *<pre>
 * History:
 *  v1.0: Rob Weltman
 *  v1.1: cdk_team
 *</pre>
 */
public class LDAPSimpleAuth extends LDAPBasePropertySupport {

	/**
	 * Constructor with no parameters
	 */
    public LDAPSimpleAuth() {}

	/**
	* Constructor with host and port initializers
	* @param theHost host string
	* @param thePort port number
	*/
    public LDAPSimpleAuth( String theHost,
						   int thePort ) {
	    setHost( theHost );
		setPort( thePort );
	}

	/**
	* Constructor with host, port, and base initializers - use this if
	* if an RDN is to be used to form a complete DN for authentication
	* @param theHost host string
	* @param thePort port number
	* @param theBase directory base string
	*/
    public LDAPSimpleAuth( String theHost,
						   int thePort,
						   String theBase ) {
	    setHost( theHost );
		setPort( thePort );
		setBase( theBase );
	}

	/**
	* Constructor with all required authentication parameters
	* @param theHost host string
	* @param thePort port number
	* @param dn fully qualified distinguished name to authenticate
	* @param password password for authenticating the dn
	*/
    public LDAPSimpleAuth( String theHost,
						   int thePort,
						   String dn,
						   String password ) {
	    setHost( theHost );
		setPort( thePort );
		setAuthDN( dn );
		setAuthPassword( password );
	}

    private void notifyResult( String newResult ) {
	    firePropertyChange( "result", result, newResult );
		result = newResult;
	}

	/**
	 * Connect to LDAP server using parameters specified in
	 * constructor and/or by setting properties and attempt to
	 * authenticate.
	 * @return "Y" on successful authentication, "N" otherwise
	 */
    public String authenticate() {
		LDAPConnection m_ldc = null;
		String result = "N";
		try {
			m_ldc = new LDAPConnection();
			System.out.println("Connecting to " + getHost() +
							   " " + getPort());
			connect( m_ldc, getHost(), getPort());
		} catch (Exception e) {
			System.out.println( "Failed to connect to " + getHost() +
								": " + e.toString() );
		}
		if ( m_ldc.isConnected() ) {
			System.out.println( "Authenticating " + getAuthDN() );
			try {
				m_ldc.authenticate( getAuthDN(), getAuthPassword() );
				result = "Y";
			} catch (Exception e) {
				System.out.println( "Failed to authenticate to " +
									getHost() + ": " + e.toString() );
			}
		}
		notifyResult( result );
		return result;
	}

	/**
	 * Connect to LDAP server using parameters specified in
	 * constructor and/or by setting properties and attempt to
	 * authenticate.
	 * @param dn fully qualified distinguished name to authenticate
	 * @param password password for authenticating the dn
	 * @return "Y" on successful authentication, "N" otherwise
	 */
    public String authenticate( String dn,
								String password ) {
	    setAuthDN( dn );
		setAuthPassword( password );
		return authenticate();
	}

  /**
   * The main body if we run it as stand-alone application.
   * @param args list of arguments
   */
    public static void main(String args[]) {
        if ( (args.length != 5) && (args.length != 4) ) {
		    System.out.println( "Usage: LDAPSimpleAuth " +
								"host port base userName password" );
		    System.out.println( "       LDAPSimpleAuth " +
								"host port DN password" );
			System.exit(1);
        }
        LDAPSimpleAuth app = new LDAPSimpleAuth();
		app.setHost( args[0] );
		app.setPort( java.lang.Integer.parseInt( args[1] ) );
		if ( args.length == 5 ) {
		    app.setBase( args[2] );
			app.setUserName( args[3] );
			app.setAuthPassword( args[4] );
		} else {
			app.setAuthDN( args[2] );
			app.setAuthPassword( args[3] );
		}
        String response = app.authenticate();
		System.out.println( "Response: " + response );
        System.exit(0);
    }

    /*
	 * Variables
	 */
    private String result = new String("");
}
