#!perl
use strict;
use warnings;
use Test::More;
use File::Spec;
use Sereal::Decoder;
our %corpus;

plan tests => 5 + keys %corpus;

my $decoder= Sereal::Decoder->new(); 
use Data::Dumper;

my %by_dumper;
my $last;
$Data::Dumper::Sortkeys= 1;
$Data::Dumper::Indent= 1;
foreach my $ver (sort { $a <=> $b } keys %corpus) {
    my $dat= unpack "u*", $corpus{$ver};
    my $decoded= $decoder->decode($dat);
    ok($decoded,"Decoding version $ver data did not die");
    my $dumper= Dumper($decoded);
    print $dumper unless $last;
    $last = $decoded;
    push @{$by_dumper{$dumper}}, $ver;
}
is(0+keys(%by_dumper),1, "Expecting the same result from all versions")
    or do {
        foreach my $dump (sort { @{$by_dumper{$b}} <=> @{$by_dumper{$a}} 
                          || $a cmp $b } keys %by_dumper) 
        {
            print STDERR "Versions: ", join " ", @{$by_dumper{$dump}}, "\n";
            print STDERR $dump,"\n";
        }
    };
$last->{alias}= "check";
is($last->{alias_array}[0], "check", "Aliasing worked idx 0");
is($last->{alias_array}[1], "check", "Aliasing worked idx 1");
is($last->{alias_array}[2], "check", "Aliasing worked idx 2");
like("THIS IS A CASE INSENSITIVE REGEXP", $last->{re2},
     "re2 matches as expected");

BEGIN { 
    %corpus = (
'2.01' => <<'EOF',
M/7-R;`(`7V)H,E%B:#-F9G5R8FQE9'-R968HZW=H>2!O:"!W:'D_:V%L:6%S
M7V%R<F%Y0^5A;&EA<RXO+B]F<F-O;G-T*"!B:7-R<G)R<F5F<R@H*"@I%F5C
M<F]S<T(HJ*E;*5QC<F4Q+&9296=E>'`H,7!T:&ES(&ES(&$@<F5G97AP87AE
M86QI87,N+V)R,:BIC`%C<F4R+64H,28A=&AI<R!I<R!A(&-A<V4@:6YS96YS
M:71I=F4@<F5G97AP86EC87(R**L#9'1H:7,H8FES+64H,69A('1E<W1@:V%L
M:6%S7V-R;W-S0BY<+EMD;G5M<T<A4R`J(^%Z%*Y'X=H_9#`P,#%G,2XP,#`Q
:,&4@(#$@(&4@(#`@(&-A<C$IP@%B<C(NC0$`
EOF
'2.02' => <<'EOF',
M/7-R;`(`7V-A<C$HJP-D=&AI<RAB:7,L9E)E9V5X<"@Q9F$@=&5S=&!B<C&H
MJ2=E86QI87/E86QI87-B<C(N*&1S<F5F*.MW:'D@;V@@=VAY/V-R93$M$R@Q
M<'1H:7,@:7,@82!R96=E>'!A>&MA;&EA<U]A<G)A>4,N,"XP+C!B:#)18F@S
M9F9U<F)L961N=6US1R%3("HCX7H4KD?AVC]D,#`P,6<Q+C`P,#$P92`@,2`@
M92`@,"`@9G)C;VYS="@@8F-A<C(I!V5C<F]S<T(HJ*G-`2G.`6-R93(M$R@Q
M)B%T:&ES(&ES(&$@8V%S92!I;G-E;G-I=&EV92!R96=E>'!A:6ES<G)R<G)E
;9G,H*"@H*4%K86QI87-?8W)O<W-"+LX!+LT!
EOF
'2.03' => <<'EOF',
M/7-R;`(`7V5A;&EA<^5A;&EA<V1N=6US1R%3("HCX7H4KD?AVC]D,#`P,6<Q
M+C`P,#$P92`@,2`@92`@,"`@9'-R968HZW=H>2!O:"!W:'D_96-R;W-S0BBH
MJ50I56)H,E%B:#-F9G5R8FQE8G(QJ*EJ:V%L:6%S7V-R;W-S0BY5+E1F<F-O
M;G-T*"!B8G(R+FMC87(R**L#9'1H:7,H8FES+&9296=E>'`H,69A('1E<W1@
M8W)E,BV>`2@Q)B%T:&ES(&ES(&$@8V%S92!I;G-E;G-I=&EV92!R96=E>'!A
M:6ES<G)R<G)E9G,H*"@H*4!C<F4Q+9X!*#%P=&AI<R!I<R!A(')E9V5X<&%X
:8V%R,2F2`6MA;&EA<U]A<G)A>4,N""X(+@@`
EOF
'2.04' => <<'EOF',
M/7-R;`(`7V-A<C$HJP-D=&AI<RAB:7,L9E)E9V5X<"@Q9F$@=&5S=&!E86QI
M87/E86QI87-E8W)O<W-"**BI."DY9'-R968HZW=H>2!O:"!W:'D_9&YU;7-'
M(5,@*B/A>A2N1^':/V0P,#`Q9S$N,#`P,3!E("`Q("!E("`P("!B:#)18F@S
M9F9U<F)L96MA;&EA<U]C<F]S<T(N.2XX8W)E,BT3*#$F(71H:7,@:7,@82!C
M87-E(&EN<V5N<VET:79E(')E9V5X<&%I8W)E,2T3*#%P=&AI<R!I<R!A(')E
M9V5X<&%X8G(QJ*GE`6ES<G)R<G)E9G,H*"@H*4-F<F-O;G-T*"!B:V%L:6%S
97V%R<F%Y0RXJ+BHN*F-A<C(I!V)R,B[F`0``
EOF
'2.06' => <<'EOF',
M/7-R;`(`7V5A;&EA<^5A;&EA<V-R93$L9E)E9V5X<"@Q<'1H:7,@:7,@82!R
M96=E>'!A>&)R,:BI,F-A<C(HJP-D=&AI<RAB:7,M$R@Q9F$@=&5S=&!F<F-O
M;G-T*"!B9'-R968HZW=H>2!O:"!W:'D_:V%L:6%S7V-R;W-S0JBI>BY[8F@R
M46)H,V9F=7)B;&5I<W)R<G)R969S*"@H*"EA8W)E,BT3*#$F(71H:7,@:7,@
M82!C87-E(&EN<V5N<VET:79E(')E9V5X<&%I:V%L:6%S7V%R<F%Y0RX(+@@N
M"&5C<F]S<T(I>REZ9&YU;7-'(5,@*B/A>A2N1^':/V0P,#`Q9S$N,#`P,3!E
6("`Q("!E("`P("!C87(Q*3IB<C(N,P``
EOF
'2.08' => <<'EOF',
M/7-R;`(`7V-A<C$HJP-D=&AI<RAB:7,L9E)E9V5X<"@Q9F$@=&5S=&!C<F4R
M+1,H,28A=&AI<R!I<R!A(&-A<V4@:6YS96YS:71I=F4@<F5G97AP86EK86QI
M87-?87)R87E#Y6%L:6%S+EXN7F9R8V]N<W0H(&)E8W)O<W-"**BI>BE[9&YU
M;7-'(5,@*B/A>A2N1^':/V0P,#`Q9S$N,#`P,3!E("`Q("!E("`P("!B<C*H
MJ:X!:7-R<G)R<F5F<R@H*"@HZW=H>2!O:"!W:'D_9'-R968IP0%C<F4Q+1,H
M,7!T:&ES(&ES(&$@<F5G97AP87AB:#)18F@S9F9U<F)L96MA;&EA<U]C<F]S
:<T(N>RYZ8G(Q+J\!96%L:6%S+EYC87(R*0<`
EOF
'2.09' => <<'EOF',
M/7-R;`(`7V-A<C(HJP-D=&AI<RAB:7,L9E)E9V5X<"@Q9F$@=&5S=&!K86QI
M87-?87)R87E#Y6%L:6%S+C$N,65A;&EA<RXQ96-R;W-S0BBHJ4LI3&)R,:BI
M4V)R,BY49'-R968HZW=H>2!O:"!W:'D_9G)C;VYS="@@8F-A<C$I!V)H,E%B
M:#-F9G5R8FQE8W)E,BT3*#$F(71H:7,@:7,@82!C87-E(&EN<V5N<VET:79E
M(')E9V5X<&%I8W)E,2T3*#%P=&AI<R!I<R!A(')E9V5X<&%X9&YU;7-'(5,@
M*B/A>A2N1^':/V0P,#`Q9S$N,#`P,3!E("`Q("!E("`P("!K86QI87-?8W)O
7<W-"+DPN2VES<G)R<G)E9G,H*"@H*6$`
EOF
'2.1' => <<'EOF',
M/7-R;`(`7V-R93$L9E)E9V5X<"@Q<'1H:7,@:7,@82!R96=E>'!A>&1S<F5F
M*.MW:'D@;V@@=VAY/V)R,JBI.&MA;&EA<U]C<F]S<T*HJ4@N265C<F]S<T(I
M22E(8W)E,BT'*#$F(71H:7,@:7,@82!C87-E(&EN<V5N<VET:79E(')E9V5X
M<&%I:7-R<G)R<F5F<R@H*"@I*69R8V]N<W0H(&)K86QI87-?87)R87E#Y6%L
M:6%S+JP!+JP!9&YU;7-'(5,@*B/A>A2N1^':/V0P,#`Q9S$N,#`P,3!E("`Q
M("!E("`P("!C87(Q**L#9'1H:7,H8FES+0<H,69A('1E<W1@96%L:6%S+JP!
:8G(Q+CEB:#)18F@S9F9U<F)L96-A<C(IZ0$`
EOF
'2.11' => <<'EOF',
M/7-R;`(`7V5A;&EA<^5A;&EA<V5C<F]S<T(HJ*D6*1=K86QI87-?87)R87E#
M+@@N""X(8W)E,2QF4F5G97AP*#%P=&AI<R!I<R!A(')E9V5X<&%X9G)C;VYS
M="@@8F-A<C(HJP-D=&AI<RAB:7,M,R@Q9F$@=&5S=&!D<W)E9BCK=VAY(&]H
M('=H>3]K86QI87-?8W)O<W-"+A<N%FES<G)R<G)E9G,H*"@H*7MC87(Q*5YB
M:#)18F@S9F9U<F)L96)R,JBIOP%D;G5M<T<A4R`J(^%Z%*Y'X=H_9#`P,#%G
M,2XP,#`Q,&4@(#$@(&4@(#`@(&)R,2[``6-R93(M,R@Q)B%T:&ES(&ES(&$@
98V%S92!I;G-E;G-I=&EV92!R96=E>'!A:0``
EOF
'2.12' => <<'EOF',
M/7-R;`(`7V1S<F5F*.MW:'D@;V@@=VAY/VMA;&EA<U]A<G)A>4/E86QI87,N
M(2XA9G)C;VYS="@@8F5A;&EA<RXA:V%L:6%S7V-R;W-S0JBI2BY+8F@R46)H
M,V9F=7)B;&5I<W)R<G)R969S*"@H*"D(8W)E,BQF4F5G97AP*#$F(71H:7,@
M:7,@82!C87-E(&EN<V5N<VET:79E(')E9V5X<&%I8V%R,BBK`V1T:&ES*&)I
M<RUR*#%F82!T97-T8&1N=6US1R%3("HCX7H4KD?AVC]D,#`P,6<Q+C`P,#$P
M92`@,2`@92`@,"`@8G(QJ*GK`6-A<C$II0%E8W)O<W-"*4LI2F)R,B[L`6-R
993$M<B@Q<'1H:7,@:7,@82!R96=E>'!A>```
EOF
'3.001' => <<'EOF',
M/?-R;`,`7V1S<F5F*.MW:'D@;V@@=VAY/VES<G)R<G)E9G,H*"@H*0AB<C&H
MJ2=E8W)O<W-"**BI,BDS8G(R+BAD;G5M<T<A4R`J(^%Z%*Y'X=H_9#`P,#%G
M,2XP,#`Q,&4@(#$@(&4@(#`@(&MA;&EA<U]A<G)A>4/E86QI87,N=2YU8V%R
M,2BK`V1T:&ES*&)I<RQF4F5G97AP*#%F82!T97-T8&9R8V]N<W0H(&)B:#)1
M8F@S9F9U<F)L96-R93$MD`$H,7!T:&ES(&ES(&$@<F5G97AP87AC87(R*80!
M:V%L:6%S7V-R;W-S0BXS+C)C<F4R+9`!*#$F(71H:7,@:7,@82!C87-E(&EN
:<V5N<VET:79E(')E9V5X<&%I96%L:6%S+G4`
EOF
'3.002' => <<'EOF',
M/?-R;`,`7V-A<C$HJP-D=&AI<RAB:7,L9E)E9V5X<"@Q9F$@=&5S=&!E8W)O
M<W-"**BI+"DM:V%L:6%S7V%R<F%Y0^5A;&EA<RX^+CYD;G5M<T<A4R`J(^%Z
M%*Y'X=H_9#`P,#%G,2XP,#`Q,&4@(#$@(&4@(#`@(&1S<F5F*.MW:'D@;V@@
M=VAY/V-R93(M$R@Q)B%T:&ES(&ES(&$@8V%S92!I;G-E;G-I=&EV92!R96=E
M>'!A:6)H,E%B:#-F9G5R8FQE8G(RJ*G$`6)R,2[%`6-A<C(I!V-R93$M$R@Q
M<'1H:7,@:7,@82!R96=E>'!A>&MA;&EA<U]C<F]S<T(N+2XL:7-R<G)R<F5F
9<R@H*"@I>F9R8V]N<W0H(&)E86QI87,N/@``
EOF
'3.003' => <<'EOF',
M/?-R;`,`7V)R,:BI!6-A<C$HJP-D=&AI<RAB:7,L9E)E9V5X<"@Q9F$@=&5S
M=&!B:#)18F@S9F9U<F)L96-R93$M&2@Q<'1H:7,@:7,@82!R96=E>'!A>&5A
M;&EA<^5A;&EA<V1S<F5F*.MW:'D@;V@@=VAY/VES<G)R<G)E9G,H*"@H*65K
M86QI87-?87)R87E#+EDN62Y9:V%L:6%S7V-R;W-S0JBIH0$NH@%C87(R*0UB
M<C(N!F1N=6US1R%3("HCX7H4KD?AVC]D,#`P,6<Q+C`P,#$P92`@,2`@92`@
M,"`@8W)E,BT9*#$F(71H:7,@:7,@82!C87-E(&EN<V5N<VET:79E(')E9V5X
:<&%I96-R;W-S0BFB`2FA`69R8V]N<W0H(&(`
EOF
'3.004' => <<'EOF',
M/?-R;`,`7V9R8V]N<W0H(&)C<F4R+&9296=E>'`H,28A=&AI<R!I<R!A(&-A
M<V4@:6YS96YS:71I=F4@<F5G97AP86EB<C*HJ4)B:#)18F@S9F9U<F)L96)R
M,2Y#9'-R968HZW=H>2!O:"!W:'D_96%L:6%SY6%L:6%S:V%L:6%S7V-R;W-S
M0JBI@P$NA`%E8W)O<W-"*80!*8,!9&YU;7-'(5,@*B/A>A2N1^':/V0P,#`Q
M9S$N,#`P,3!E("`Q("!E("`P("!C87(R**L#9'1H:7,H8FES+1$H,69A('1E
M<W1@8W)E,2T1*#%P=&AI<R!I<R!A(')E9V5X<&%X:V%L:6%S7V%R<F%Y0RYP
;+G`N<&-A<C$IR`%I<W)R<G)R969S*"@H*"E>
EOF
'3.005' => <<'EOF',
M/?-R;`,`7V-A<C$HJP-D=&AI<RAB:7,L9E)E9V5X<"@Q9F$@=&5S=&!K86QI
M87-?8W)O<W-"J*DQ+C)C<F4R+1,H,28A=&AI<R!I<R!A(&-A<V4@:6YS96YS
M:71I=F4@<F5G97AP86ED;G5M<T<A4R`J(^%Z%*Y'X=H_9#`P,#%G,2XP,#`Q
M,&4@(#$@(&4@(#`@(&)R,:BID@%K86QI87-?87)R87E#Y6%L:6%S+J,!+J,!
M96-R;W-S0BDR*3%B:#)18F@S9F9U<F)L96)R,BZ3`61S<F5F*.MW:'D@;V@@
M=VAY/V5A;&EA<RZC`69R8V]N<W0H(&)C87(R*0=C<F4Q+1,H,7!T:&ES(&ES
<(&$@<F5G97AP87AI<W)R<G)R969S*"@H*"G4`0``
EOF
'3.006' => <<'EOF',
M/?-R;`,`7V-A<C(HJP-D=&AI<RAB:7,L9E)E9V5X<"@Q9F$@=&5S=&!E86QI
M87/E86QI87-F<F-O;G-T*"!B96-R;W-S0BBHJ4(I0V-R93(M$R@Q)B%T:&ES
M(&ES(&$@8V%S92!I;G-E;G-I=&EV92!R96=E>'!A:6)R,:BI=V-A<C$I!VMA
M;&EA<U]A<G)A>4,N*BXJ+BIB<C(N>&1N=6US1R%3("HCX7H4KD?AVC]D,#`P
M,6<Q+C`P,#$P92`@,2`@92`@,"`@:7-R<G)R<F5F<R@H*"@HZW=H>2!O:"!W
M:'D_:V%L:6%S7V-R;W-S0BY#+D)D<W)E9BG3`6)H,E%B:#-F9G5R8FQE8W)E
8,2T3*#%P=&AI<R!I<R!A(')E9V5X<&%X
EOF
'3.007' => <<'EOF',
M/?-R;`,`7VMA;&EA<U]C<F]S<T*HJ0\N$&MA;&EA<U]A<G)A>4/E86QI87,N
M(2XA96%L:6%S+B%C87(Q**L#9'1H:7,H8FES+&9296=E>'`H,69A('1E<W1@
M8W)E,2U$*#%P=&AI<R!I<R!A(')E9V5X<&%X8W)E,BU$*#$F(71H:7,@:7,@
M82!C87-E(&EN<V5N<VET:79E(')E9V5X<&%I96-R;W-S0BD0*0]B:#)18F@S
M9F9U<F)L969R8V]N<W0H(&)B<C&HJ<,!9&YU;7-'(5,@*B/A>A2N1^':/V0P
M,#`Q9S$N,#`P,3!E("`Q("!E("`P("!D<W)E9BCK=VAY(&]H('=H>3]I<W)R
9<G)R969S*"@H*"GY`6)R,B[$`6-A<C(I.```
EOF
'3.008' => <<'EOF',
M/?-R;`,`7V)R,:BI!6MA;&EA<U]A<G)A>4/E86QI87,N%2X59G)C;VYS="@@
M8F1S<F5F*.MW:'D@;V@@=VAY/V)H,E%B:#-F9G5R8FQE8G(R+@9C<F4Q+&92
M96=E>'`H,7!T:&ES(&ES(&$@<F5G97AP87AC87(R**L#9'1H:7,H8FES+5,H
M,69A('1E<W1@96-R;W-S0BBHJ9,!*90!8W)E,BU3*#$F(71H:7,@:7,@82!C
M87-E(&EN<V5N<VET:79E(')E9V5X<&%I9&YU;7-'(5,@*B/A>A2N1^':/V0P
M,#`Q9S$N,#`P,3!E("`Q("!E("`P("!K86QI87-?8W)O<W-"+I0!+I,!8V%R
;,2ET:7-R<G)R<F5F<R@H*"@I+V5A;&EA<RX5
EOF
'3.009' => <<'EOF',
M/?-R;`,`7V5A;&EA<^5A;&EA<VMA;&EA<U]C<F]S<T*HJ1LN'&MA;&EA<U]A
M<G)A>4,N""X(+@AI<W)R<G)R969S*"@H*"CK=VAY(&]H('=H>3]F<F-O;G-T
M*"!B9&YU;7-'(5,@*B/A>A2N1^':/V0P,#`Q9S$N,#`P,3!E("`Q("!E("`P
M("!C87(Q**L#9'1H:7,H8FES+&9296=E>'`H,69A('1E<W1@8V%R,BF)`6)R
M,:BIL`%C<F4R+94!*#$F(71H:7,@:7,@82!C87-E(&EN<V5N<VET:79E(')E
M9V5X<&%I8G(R+K$!8F@R46)H,V9F=7)B;&5E8W)O<W-"*1PI&V1S<F5F*4)C
;<F4Q+94!*#%P=&AI<R!I<R!A(')E9V5X<&%X
EOF
'3.01' => <<'EOF',
M/?-R;`,`7VMA;&EA<U]C<F]S<T*HJ0\N$&-R93(L9E)E9V5X<"@Q)B%T:&ES
M(&ES(&$@8V%S92!I;G-E;G-I=&EV92!R96=E>'!A:6)R,:BI2F)H,E%B:#-F
M9G5R8FQE9G)C;VYS="@@8F1S<F5F*.MW:'D@;V@@=VAY/V5C<F]S<T(I$"D/
M9&YU;7-'(5,@*B/A>A2N1^':/V0P,#`Q9S$N,#`P,3!E("`Q("!E("`P("!I
M<W)R<G)R969S*"@H*"EK8V%R,BBK`V1T:&ES*&)I<RT9*#%F82!T97-T8&)R
M,BY+8V%R,2G#`6-R93$M&2@Q<'1H:7,@:7,@82!R96=E>'!A>&5A;&EA<^5A
:;&EA<VMA;&EA<U]A<G)A>4,NAP(NAP(NAP(`
EOF
'3.011' => <<'EOF',
M/?-R;`,`7VES<G)R<G)E9G,H*"@H*.MW:'D@;V@@=VAY/V9R8V]N<W0H(&)E
M8W)O<W-"**BI+RDP8G(RJ*DW8V%R,2BK`V1T:&ES*&)I<RQF4F5G97AP*#%F
M82!T97-T8&)R,2XX8W)E,2U+*#%P=&AI<R!I<R!A(')E9V5X<&%X96%L:6%S
MY6%L:6%S:V%L:6%S7V%R<F%Y0RZ"`2Z"`2Z"`61S<F5F*1%B:#)18F@S9F9U
M<F)L96MA;&EA<U]C<F]S<T(N,"XO8W)E,BU+*#$F(71H:7,@:7,@82!C87-E
M(&EN<V5N<VET:79E(')E9V5X<&%I8V%R,BD_9&YU;7-'(5,@*B/A>A2N1^':
:/V0P,#`Q9S$N,#`P,3!E("`Q("!E("`P("``
EOF
'3.012' => <<'EOF',
M/?-R;`,`7V-A<C$HJP-D=&AI<RAB:7,L9E)E9V5X<"@Q9F$@=&5S=&!I<W)R
M<G)R969S*"@H*"CK=VAY(&]H('=H>3]F<F-O;G-T*"!B96-R;W-S0BBHJ5$I
M4F5A;&EA<^5A;&EA<V-A<C(I!V)H,E%B:#-F9G5R8FQE:V%L:6%S7V-R;W-S
M0BY2+E%C<F4Q+1,H,7!T:&ES(&ES(&$@<F5G97AP87AB<C&HJ:4!9'-R968I
M,V)R,BZF`6-R93(M$R@Q)B%T:&ES(&ES(&$@8V%S92!I;G-E;G-I=&EV92!R
M96=E>'!A:61N=6US1R%3("HCX7H4KD?AVC]D,#`P,6<Q+C`P,#$P92`@,2`@
992`@,"`@:V%L:6%S7V%R<F%Y0RY<+EPN7```
EOF
'3.014' => <<'EOF',
M/?-R;`,`7VMA;&EA<U]C<F]S<T*HJ0\N$&5A;&EA<^5A;&EA<V)R,JBI(VMA
M;&EA<U]A<G)A>4,N&BX:+AIC87(Q**L#9'1H:7,H8FES+&9296=E>'`H,69A
M('1E<W1@9G)C;VYS="@@8F-R93(M2B@Q)B%T:&ES(&ES(&$@8V%S92!I;G-E
M;G-I=&EV92!R96=E>'!A:6-A<C(I/F)H,E%B:#-F9G5R8FQE8G(Q+B1I<W)R
M<G)R969S*"@H*"CK=VAY(&]H('=H>3]C<F4Q+4HH,7!T:&ES(&ES(&$@<F5G
M97AP87AD;G5M<T<A4R`J(^%Z%*Y'X=H_9#`P,#%G,2XP,#`Q,&4@(#$@(&4@
7(#`@(&1S<F5F*;H!96-R;W-S0BD0*0\`
EOF
'3.015' => <<'EOF',
M/?-R;`,`7VMA;&EA<U]C<F]S<T*HJ0\N$&)R,:BI%V1N=6US1R%3("HCX7H4
MKD?AVC]D,#`P,6<Q+C`P,#$P92`@,2`@92`@,"`@8W)E,BQF4F5G97AP*#$F
M(71H:7,@:7,@82!C87-E(&EN<V5N<VET:79E(')E9V5X<&%I8G(R+AAE8W)O
M<W-"*1`I#V5A;&EA<^5A;&EA<V9R8V]N<W0H(&)C87(Q**L#9'1H:7,H8FES
M+4LH,69A('1E<W1@8V%R,BFD`6-R93$M2R@Q<'1H:7,@:7,@82!R96=E>'!A
M>&)H,E%B:#-F9G5R8FQE9'-R968HZW=H>2!O:"!W:'D_:V%L:6%S7V%R<F%Y
;0RZ/`2Z/`2Z/`6ES<G)R<G)E9G,H*"@H*?$!
EOF
'4.002' => <<'EOF',
M/?-R;`0`7V-R93(L9E)E9V5X<"@Q)B%T:&ES(&ES(&$@8V%S92!I;G-E;G-I
M=&EV92!R96=E>'!A:6)R,JBI.&9R8V]N<W0H(&)B<C$N.6-R93$M!R@Q<'1H
M:7,@:7,@82!R96=E>'!A>&-A<C$HJP-D=&AI<RAB:7,M!R@Q9F$@=&5S=&!D
M<W)E9BCK=VAY(&]H('=H>3]E8W)O<W-"**BIFP$IG`%K86QI87-?87)R87E#
MY6%L:6%S+J\!+J\!8V%R,BEJ:7-R<G)R<F5F<R@H*"@IAP%D;G5M<T<A4R`J
M(^%Z%*Y'X=H_9#`P,#%G,2XP,#`Q,&4@(#$@(&4@(#`@(&)H,E%B:#-F9G5R
?8FQE96%L:6%S+J\!:V%L:6%S7V-R;W-S0BZ<`2Z;`0``
EOF
'4.003' => <<'EOF',
M/?-R;`0`7VMA;&EA<U]A<G)A>4/E86QI87,N#RX/8G(QJ*D<8V%R,2BK`V1T
M:&ES*&)I<RQF4F5G97AP*#%F82!T97-T8&ES<G)R<G)E9G,H*"@H*.MW:'D@
M;V@@=VAY/V)H,E%B:#-F9G5R8FQE8W)E,2TP*#%P=&AI<R!I<R!A(')E9V5X
M<&%X8G(R+AUE86QI87,N#V1S<F5F*5!E8W)O<W-"**BIH0$IH@%K86QI87-?
M8W)O<W-"+J(!+J$!8W)E,BTP*#$F(71H:7,@:7,@82!C87-E(&EN<V5N<VET
M:79E(')E9V5X<&%I9G)C;VYS="@@8F1N=6US1R%3("HCX7H4KD?AVC]D,#`P
;,6<Q+C`P,#$P92`@,2`@92`@,"`@8V%R,BDD
EOF
'4.004' => <<'EOF',
M/?-R;`0`7V-R93(L9E)E9V5X<"@Q)B%T:&ES(&ES(&$@8V%S92!I;G-E;G-I
M=&EV92!R96=E>'!A:6)R,JBI.&1N=6US1R%3("HCX7H4KD?AVC]D,#`P,6<Q
M+C`P,#$P92`@,2`@92`@,"`@:V%L:6%S7V-R;W-S0JBI="YU8V%R,2BK`V1T
M:&ES*&)I<RT'*#%F82!T97-T8&)R,2XY8W)E,2T'*#%P=&AI<R!I<R!A(')E
M9V5X<&%X8F@R46)H,V9F=7)B;&5K86QI87-?87)R87E#Y6%L:6%S+M`!+M`!
M96-R;W-S0BEU*71D<W)E9BCK=VAY(&]H('=H>3]C87(R*7YE86QI87,NT`%I
:<W)R<G)R969S*"@H*"GM`69R8V]N<W0H(&(`
EOF
'4.005' => <<'EOF',
M/?-R;`0`7V5A;&EA<^5A;&EA<V-A<C$HJP-D=&AI<RAB:7,L9E)E9V5X<"@Q
M9F$@=&5S=&!C<F4R+1\H,28A=&AI<R!I<R!A(&-A<V4@:6YS96YS:71I=F4@
M<F5G97AP86EB:#)18F@S9F9U<F)L965C<F]S<T(HJ*ES*71C<F4Q+1\H,7!T
M:&ES(&ES(&$@<F5G97AP87AK86QI87-?87)R87E#+@@N""X(:V%L:6%S7V-R
M;W-S0BYT+G-B<C*HJ;H!:7-R<G)R<F5F<R@H*"@HZW=H>2!O:"!W:'D_8V%R
M,BD39&YU;7-'(5,@*B/A>A2N1^':/V0P,#`Q9S$N,#`P,3!E("`Q("!E("`P
:("!F<F-O;G-T*"!B9'-R968IS0%B<C$NNP$`
EOF
'4.006' => <<'EOF',
M/?-R;`0`7V)R,JBI!6-A<C$HJP-D=&AI<RAB:7,L9E)E9V5X<"@Q9F$@=&5S
M=&!C<F4Q+1DH,7!T:&ES(&ES(&$@<F5G97AP87AI<W)R<G)R969S*"@H*"CK
M=VAY(&]H('=H>3]D;G5M<T<A4R`J(^%Z%*Y'X=H_9#`P,#%G,2XP,#`Q,&4@
M(#$@(&4@(#`@(&-A<C(I#6-R93(M&2@Q)B%T:&ES(&ES(&$@8V%S92!I;G-E
M;G-I=&EV92!R96=E>'!A:65A;&EA<^5A;&EA<VMA;&EA<U]C<F]S<T*HJ=@!
M+MD!9'-R968I5&MA;&EA<U]A<G)A>4,NQ0$NQ0$NQ0%B:#)18F@S9F9U<F)L
=96)R,2X&9G)C;VYS="@@8F5C<F]S<T(IV0$IV`$`
EOF
'4.007' => <<'EOF',
M/?-R;`0`7V5A;&EA<^5A;&EA<VMA;&EA<U]A<G)A>4,N""X(+@AB<C&HJ21D
M;G5M<T<A4R`J(^%Z%*Y'X=H_9#`P,#%G,2XP,#`Q,&4@(#$@(&4@(#`@(&-A
M<C$HJP-D=&AI<RAB:7,L9E)E9V5X<"@Q9F$@=&5S=&!C87(R*5AK86QI87-?
M8W)O<W-"J*F(`2Z)`61S<F5F*.MW:'D@;V@@=VAY/V)R,BXE9G)C;VYS="@@
M8F-R93$M9"@Q<'1H:7,@:7,@82!R96=E>'!A>&)H,E%B:#-F9G5R8FQE8W)E
M,BUD*#$F(71H:7,@:7,@82!C87-E(&EN<V5N<VET:79E(')E9V5X<&%I:7-R
;<G)R<F5F<R@H*"@IE0%E8W)O<W-"*8D!*8@!
EOF
'4.008' => <<'EOF',
M/?-R;`0`7V5C<F]S<T(HJ*D**0MC87(R**L#9'1H:7,H8FES+&9296=E>'`H
M,69A('1E<W1@8V%R,2D49&YU;7-'(5,@*B/A>A2N1^':/V0P,#`Q9S$N,#`P
M,3!E("`Q("!E("`P("!K86QI87-?87)R87E#Y6%L:6%S+G`N<&5A;&EA<RYP
M9G)C;VYS="@@8F1S<F5F*.MW:'D@;V@@=VAY/VES<G)R<G)E9G,H*"@H*9(!
M8W)E,BT@*#$F(71H:7,@:7,@82!C87-E(&EN<V5N<VET:79E(')E9V5X<&%I
M8G(RJ*G?`6MA;&EA<U]C<F]S<T(N"RX*8G(Q+N`!8W)E,2T@*#%P=&AI<R!I
:<R!A(')E9V5X<&%X8F@R46)H,V9F=7)B;&4`
EOF
'4.009' => <<'EOF',
M/?-R;`0`7V-A<C(HJP-D=&AI<RAB:7,L9E)E9V5X<"@Q9F$@=&5S=&!D<W)E
M9BCK=VAY(&]H('=H>3]F<F-O;G-T*"!B8W)E,2T3*#%P=&AI<R!I<R!A(')E
M9V5X<&%X96-R;W-S0BBHJ6,I9&)R,JBI:V-R93(M$R@Q)B%T:&ES(&ES(&$@
M8V%S92!I;G-E;G-I=&EV92!R96=E>'!A:65A;&EA<^5A;&EA<VES<G)R<G)E
M9G,H*"@H*2IB<C$N;&)H,E%B:#-F9G5R8FQE9&YU;7-'(5,@*B/A>A2N1^':
M/V0P,#`Q9S$N,#`P,3!E("`Q("!E("`P("!K86QI87-?8W)O<W-"+F0N8VMA
:;&EA<U]A<G)A>4,NH0$NH0$NH0%C87(Q*0<`
EOF
'4.01' => <<'EOF',
M/?-R;`0`7VES<G)R<G)E9G,H*"@H*.MW:'D@;V@@=VAY/V-A<C(HJP-D=&AI
M<RAB:7,L9E)E9V5X<"@Q9F$@=&5S=&!B<C*HJ4)F<F-O;G-T*"!B8W)E,2TN
M*#%P=&AI<R!I<R!A(')E9V5X<&%X9'-R968I$6-A<C$I(F)R,2Y#:V%L:6%S
M7V-R;W-S0JBIB0$NB@%D;G5M<T<A4R`J(^%Z%*Y'X=H_9#`P,#%G,2XP,#`Q
M,&4@(#$@(&4@(#`@(&5C<F]S<T(IB@$IB0%C<F4R+2XH,28A=&AI<R!I<R!A
M(&-A<V4@:6YS96YS:71I=F4@<F5G97AP86EK86QI87-?87)R87E#Y6%L:6%S
=+H,"+H,"8F@R46)H,V9F=7)B;&5E86QI87,N@P(`
EOF
'4.011' => <<'EOF',
M/?-R;`0`7V)R,:BI!6-A<C(HJP-D=&AI<RAB:7,L9E)E9V5X<"@Q9F$@=&5S
M=&!C<F4Q+1DH,7!T:&ES(&ES(&$@<F5G97AP87AB<C(N!FMA;&EA<U]A<G)A
M>4/E86QI87,N5RY79G)C;VYS="@@8F-R93(M&2@Q)B%T:&ES(&ES(&$@8V%S
M92!I;G-E;G-I=&EV92!R96=E>'!A:6-A<C$I#6)H,E%B:#-F9G5R8FQE9&YU
M;7-'(5,@*B/A>A2N1^':/V0P,#`Q9S$N,#`P,3!E("`Q("!E("`P("!K86QI
M87-?8W)O<W-"J*GE`2[F`6ES<G)R<G)E9G,H*"@H*.MW:'D@;V@@=VAY/V5A
;;&EA<RY79'-R968I^P%E8W)O<W-"*>8!*>4!
EOF
'4.012' => <<'EOF',
M/?-R;`0`7V)R,:BI!6-R93$L9E)E9V5X<"@Q<'1H:7,@:7,@82!R96=E>'!A
M>&MA;&EA<U]A<G)A>4/E86QI87,N-BXV:V%L:6%S7V-R;W-S0JBI32Y.8V%R
M,2BK`V1T:&ES*&)I<RT-*#%F82!T97-T8&-R93(M#2@Q)B%T:&ES(&ES(&$@
M8V%S92!I;G-E;G-I=&EV92!R96=E>'!A:6)R,BX&9&YU;7-'(5,@*B/A>A2N
M1^':/V0P,#`Q9S$N,#`P,3!E("`Q("!E("`P("!E8W)O<W-"*4XI36-A<C(I
M5V9R8V]N<W0H(&)B:#)18F@S9F9U<F)L961S<F5F*.MW:'D@;V@@=VAY/VES
7<G)R<G)E9G,H*"@H*?L!96%L:6%S+C8`
EOF
'4.014' => <<'EOF',
M/?-R;`0`7V1N=6US1R%3("HCX7H4KD?AVC]D,#`P,6<Q+C`P,#$P92`@,2`@
M92`@,"`@8V%R,2BK`V1T:&ES*&)I<RQF4F5G97AP*#%F82!T97-T8&-A<C(I
M,V1S<F5F*.MW:'D@;V@@=VAY/V5C<F]S<T(HJ*EP*7%B<C&HJ7AF<F-O;G-T
M*"!B:7-R<G)R<F5F<R@H*"@I7&-R93$M/R@Q<'1H:7,@:7,@82!R96=E>'!A
M>&-R93(M/R@Q)B%T:&ES(&ES(&$@8V%S92!I;G-E;G-I=&EV92!R96=E>'!A
M:6)R,BYY:V%L:6%S7V%R<F%Y0^5A;&EA<R[O`2[O`6)H,E%B:#-F9G5R8FQE
:96%L:6%S+N\!:V%L:6%S7V-R;W-S0BYQ+G``
EOF
'4.015' => <<'EOF',
M/?-R;`0`7V)H,E%B:#-F9G5R8FQE:7-R<G)R<F5F<R@H*"@HZW=H>2!O:"!W
M:'D_8G(RJ*DN96%L:6%SY6%L:6%S96-R;W-S0BBHJ44I1F-A<C$HJP-D=&AI
M<RAB:7,L9E)E9V5X<"@Q9F$@=&5S=&!D;G5M<T<A4R`J(^%Z%*Y'X=H_9#`P
M,#%G,2XP,#`Q,&4@(#$@(&4@(#`@(&-R93$M6R@Q<'1H:7,@:7,@82!R96=E
M>'!A>&9R8V]N<W0H(&)C87(R*4]C<F4R+5LH,28A=&AI<R!I<R!A(&-A<V4@
M:6YS96YS:71I=F4@<F5G97AP86EB<C$N+VMA;&EA<U]C<F]S<T(N1BY%9'-R
7968I'VMA;&EA<U]A<G)A>4,N-RXW+C<`
EOF
'4.016' => <<'EOF',
M/?-R;`0`7VMA;&EA<U]C<F]S<T*HJ0\N$&)R,:BI%V5C<F]S<T(I$"D/96%L
M:6%SY6%L:6%S8G(R+AAB:#)18F@S9F9U<F)L961S<F5F*.MW:'D@;V@@=VAY
M/V-R93$L9E)E9V5X<"@Q<'1H:7,@:7,@82!R96=E>'!A>&-A<C$HJP-D=&AI
M<RAB:7,M6R@Q9F$@=&5S=&!K86QI87-?87)R87E#+BLN*RXK8W)E,BU;*#$F
M(71H:7,@:7,@82!C87-E(&EN<V5N<VET:79E(')E9V5X<&%I9G)C;VYS="@@
M8FES<G)R<G)E9G,H*"@H*4ID;G5M<T<A4R`J(^%Z%*Y'X=H_9#`P,#%G,2XP
6,#`Q,&4@(#$@(&4@(#`@(&-A<C(I?```
EOF
'4.017' => <<'EOF',
M/?-R;`0`7VES<G)R<G)E9G,H*"@H*.MW:'D@;V@@=VAY/V-R93$L9E)E9V5X
M<"@Q<'1H:7,@:7,@82!R96=E>'!A>&1N=6US1R%3("HCX7H4KD?AVC]D,#`P
M,6<Q+C`P,#$P92`@,2`@92`@,"`@8G(QJ*EM:V%L:6%S7V-R;W-S0JBI?2Y^
M96%L:6%SY6%L:6%S8G(R+FYE8W)O<W-"*7XI?6-R93(M(B@Q)B%T:&ES(&ES
M(&$@8V%S92!I;G-E;G-I=&EV92!R96=E>'!A:6-A<C$HJP-D=&AI<RAB:7,M
M(B@Q9F$@=&5S=&!F<F-O;G-T*"!B9'-R968I$6-A<C(IT`%K86QI87-?87)R
:87E#+H@!+H@!+H@!8F@R46)H,V9F=7)B;&4`
EOF
'4.018' => <<'EOF',
M/?-R;`0`7V1S<F5F*.MW:'D@;V@@=VAY/V)H,E%B:#-F9G5R8FQE9&YU;7-'
M(5,@*B/A>A2N1^':/V0P,#`Q9S$N,#`P,3!E("`Q("!E("`P("!C<F4R+&92
M96=E>'`H,28A=&AI<R!I<R!A(&-A<V4@:6YS96YS:71I=F4@<F5G97AP86EC
M87(R**L#9'1H:7,H8FES+5,H,69A('1E<W1@:7-R<G)R<F5F<R@H*"@I"&)R
M,JBIL`%F<F-O;G-T*"!B8G(Q+K$!96-R;W-S0BBHJ<P!*<T!:V%L:6%S7V-R
M;W-S0B[-`2[,`6-R93$M4R@Q<'1H:7,@:7,@82!R96=E>'!A>&MA;&EA<U]A
A<G)A>4/E86QI87,NC@(NC@)C87(Q*88!96%L:6%S+HX"
EOF
'4.019' => <<'EOF',
M/?-R;`0`7VMA;&EA<U]C<F]S<T*HJ0\N$&ES<G)R<G)E9G,H*"@H*.MW:'D@
M;V@@=VAY/V)H,E%B:#-F9G5R8FQE9G)C;VYS="@@8F-A<C(HJP-D=&AI<RAB
M:7,L9E)E9V5X<"@Q9F$@=&5S=&!D;G5M<T<A4R`J(^%Z%*Y'X=H_9#`P,#%G
M,2XP,#`Q,&4@(#$@(&4@(#`@(&)R,JBIF`%E86QI87/E86QI87-C<F4R+5@H
M,28A=&AI<R!I<R!A(&-A<V4@:6YS96YS:71I=F4@<F5G97AP86EK86QI87-?
M87)R87E#+J(!+J(!+J(!8V%R,2E,8G(Q+ID!9'-R968I(V-R93$M6"@Q<'1H
;:7,@:7,@82!R96=E>'!A>&5C<F]S<T(I$"D/
EOF
'4.02' => <<'EOF',
M/?-R;`0`7V)H,E%B:#-F9G5R8FQE96%L:6%SY6%L:6%S8V%R,2BK`V1T:&ES
M*&)I<RQF4F5G97AP*#%F82!T97-T8&5C<F]S<T(HJ*E&*4=I<W)R<G)R969S
M*"@H*"CK=VAY(&]H('=H>3]K86QI87-?87)R87E#+A8N%BX68W)E,2TM*#%P
M=&AI<R!I<R!A(')E9V5X<&%X8G(QJ*F7`6MA;&EA<U]C<F]S<T(N1RY&8G(R
M+I@!8W)E,BTM*#$F(71H:7,@:7,@82!C87-E(&EN<V5N<VET:79E(')E9V5X
M<&%I9G)C;VYS="@@8F1S<F5F*5IC87(R*2%D;G5M<T<A4R`J(^%Z%*Y'X=H_
99#`P,#%G,2XP,#`Q,&4@(#$@(&4@(#`@(```
EOF
'4.021' => <<'EOF',
M/?-R;`0`7V5A;&EA<^5A;&EA<VMA;&EA<U]C<F]S<T*HJ1LN'&-A<C(HJP-D
M=&AI<RAB:7,L9E)E9V5X<"@Q9F$@=&5S=&!C87(Q*25B<C*HJ4MI<W)R<G)R
M969S*"@H*"CK=VAY(&]H('=H>3]D;G5M<T<A4R`J(^%Z%*Y'X=H_9#`P,#%G
M,2XP,#`Q,&4@(#$@(&4@(#`@(&9R8V]N<W0H(&)C<F4R+3$H,28A=&AI<R!I
M<R!A(&-A<V4@:6YS96YS:71I=F4@<F5G97AP86EC<F4Q+3$H,7!T:&ES(&ES
M(&$@<F5G97AP87AB<C$N3&5C<F]S<T(I'"D;8F@R46)H,V9F=7)B;&5D<W)E
69BE=:V%L:6%S7V%R<F%Y0RX(+@@N"```
EOF
'4.022' => <<'EOF',
M/?-R;`0`7V)H,E%B:#-F9G5R8FQE:V%L:6%S7V%R<F%Y0^5A;&EA<RX=+AUC
M87(R**L#9'1H:7,H8FES+&9296=E>'`H,69A('1E<W1@9'-R968HZW=H>2!O
M:"!W:'D_8V%R,2DL8G(QJ*ED96-R;W-S0BBHJ6\I<&1N=6US1R%3("HCX7H4
MKD?AVC]D,#`P,6<Q+C`P,#$P92`@,2`@92`@,"`@8G(R+F5E86QI87,N'6MA
M;&EA<U]C<F]S<T(N<"YO9G)C;VYS="@@8F-R93(M."@Q)B%T:&ES(&ES(&$@
M8V%S92!I;G-E;G-I=&EV92!R96=E>'!A:6-R93$M."@Q<'1H:7,@:7,@82!R
796=E>'!A>&ES<G)R<G)E9G,H*"@H*4\`
EOF
'4.023' => <<'EOF',
M/?-R;`0`7VMA;&EA<U]A<G)A>4/E86QI87,N#RX/9G)C;VYS="@@8F-R93(L
M9E)E9V5X<"@Q)B%T:&ES(&ES(&$@8V%S92!I;G-E;G-I=&EV92!R96=E>'!A
M:6ES<G)R<G)E9G,H*"@H*.MW:'D@;V@@=VAY/V5A;&EA<RX/:V%L:6%S7V-R
M;W-S0JBIA@$NAP%B<C*HJ9`!9&YU;7-'(5,@*B/A>A2N1^':/V0P,#`Q9S$N
M,#`P,3!E("`Q("!E("`P("!E8W)O<W-"*8<!*88!8F@R46)H,V9F=7)B;&5D
M<W)E9BEE8V%R,2BK`V1T:&ES*&)I<RTH*#%F82!T97-T8&)R,2Z1`6-R93$M
=*"@Q<'1H:7,@:7,@82!R96=E>'!A>&-A<C(IYP$`
EOF
'4.024' => <<'EOF',
M/?-R;`0`7V1N=6US1R%3("HCX7H4KD?AVC]D,#`P,6<Q+C`P,#$P92`@,2`@
M92`@,"`@8V%R,BBK`V1T:&ES*&)I<RQF4F5G97AP*#%F82!T97-T8&MA;&EA
M<U]A<G)A>4/E86QI87,N72Y=:7-R<G)R<F5F<R@H*"@HZW=H>2!O:"!W:'D_
M96%L:6%S+EUF<F-O;G-T*"!B:V%L:6%S7V-R;W-S0JBIH0$NH@%B<C*HJ:L!
M8V%R,2DS9'-R968I=F-R93(M/R@Q)B%T:&ES(&ES(&$@8V%S92!I;G-E;G-I
M=&EV92!R96=E>'!A:6)H,E%B:#-F9G5R8FQE8G(Q+JP!96-R;W-S0BFB`2FA
<`6-R93$M/R@Q<'1H:7,@:7,@82!R96=E>'!A>```
EOF
'4.025' => <<'EOF',
M/?-R;`0`7V-A<C(HJP-D=&AI<RAB:7,L9E)E9V5X<"@Q9F$@=&5S=&!C<F4Q
M+1,H,7!T:&ES(&ES(&$@<F5G97AP87AI<W)R<G)R969S*"@H*"CK=VAY(&]H
M('=H>3]F<F-O;G-T*"!B:V%L:6%S7V%R<F%Y0^5A;&EA<RYQ+G%B<C*HJ7YE
M86QI87,N<61N=6US1R%3("HCX7H4KD?AVC]D,#`P,6<Q+C`P,#$P92`@,2`@
M92`@,"`@8F@R46)H,V9F=7)B;&5C87(Q*0=D<W)E9BE.96-R;W-S0BBHJ=@!
M*=D!8W)E,BT3*#$F(71H:7,@:7,@82!C87-E(&EN<V5N<VET:79E(')E9V5X
;<&%I8G(Q+G]K86QI87-?8W)O<W-"+MD!+M@!
EOF
'5.001' => <<'EOF',
M/?-R;`4`7V)R,:BI!69R8V]N<W0H(&)C87(Q**L#9'1H:7,H8FES+&9296=E
M>'`H,69A('1E<W1@8G(R+@9B:#)18F@S9F9U<F)L96-R93$M(R@Q<'1H:7,@
M:7,@82!R96=E>'!A>&1S<F5F*.MW:'D@;V@@=VAY/V5C<F]S<T(HJ*E\*7UD
M;G5M<T<A4R`J(^%Z%*Y'X=H_9#`P,#%G,2XP,#`Q,&4@(#$@(&4@(#`@(&MA
M;&EA<U]A<G)A>4/E86QI87,NN@$NN@%E86QI87,NN@%K86QI87-?8W)O<W-"
M+GTN?&-A<C(I%V-R93(M(R@Q)B%T:&ES(&ES(&$@8V%S92!I;G-E;G-I=&EV
:92!R96=E>'!A:6ES<G)R<G)E9G,H*"@H*6@`
EOF
);
}

