package Color::Library::Dictionary::NBS_ISCC::B;

use strict;
use warnings;

use base qw/Color::Library::Dictionary/;

__PACKAGE__->_register_dictionary;

package Color::Library::Dictionary::NBS_ISCC::B;

=pod

=head1 NAME

Color::Library::Dictionary::NBS_ISCC::B - (NBS/ISCC B) Colour Terminology in Biology

=head1 DESCRIPTION

    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


    Color Name Dictionary: (B) Colour Terminology in Biology

    Colour Terminology in Biology,
    H.A. Dade 

This combinatorial dictionary seems to be derived from Latin (eg. viridi-caeruleus). Its 128 names map to 789 colors, using 214 of the 267 centroids. But averaging over 6 colors per name means it has little descriptive power.

L<http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#B>

=head1 COLORS

	aerugineus          aerugineus         #93c592

	aerugineus          aerugineus         #679267

	aerugineus          aerugineus         #6aab8e

	aerugineus          aerugineus         #3b7861

	aerugineus          aerugineus         #8da399

	aerugineus          aerugineus         #5e716a

	aerugineus          aerugineus         #66ada4

	aerugineus          aerugineus         #317873

	armeniacus          armeniacus         #cb6d51

	armeniacus          armeniacus         #d99058

	atroardesiacus      atroardesiacus     #3a4b47

	atroardesiacus      atroardesiacus     #4e5755

	atroardesiacus      atroardesiacus     #00304e

	atroardesiacus      atroardesiacus     #536878

	atroardesiacus      atroardesiacus     #36454f

	atroardesiacus      atroardesiacus     #202830

	atroardesiacus      atroardesiacus     #51585e

	atroardesiacus      atroardesiacus     #202428

	atroardesiacus      atroardesiacus     #5d555b

	atroardesiacus      atroardesiacus     #555555

	atroardesiacus      atroardesiacus     #222222

	atrocyaneus         atrocyaneus        #317873

	atrocyaneus         atrocyaneus        #004b49

	atrocyaneus         atrocyaneus        #367588

	atrocyaneus         atrocyaneus        #004958

	atrocyaneus         atrocyaneus        #436b95

	atrocyaneus         atrocyaneus        #00304e

	atrocyaneus         atrocyaneus        #536878

	atroherbaceus       atroherbaceus      #867e36

	atroherbaceus       atroherbaceus      #665d1e

	atroherbaceus       atroherbaceus      #8a9a5b

	atroherbaceus       atroherbaceus      #4a5d23

	atroherbaceus       atroherbaceus      #515744

	atroherbaceus       atroherbaceus      #355e3b

	atroherbaceus       atroherbaceus      #3b7861

	atroherbaceus       atroherbaceus      #5e716a

	atrolazulinus       atrolazulinus      #00a1c2

	atrolazulinus       atrolazulinus      #0067a5

	atrolazulinus       atrolazulinus      #00416a

	atrolazulinus       atrolazulinus      #436b95

	atrolazulinus       atrolazulinus      #00304e

	atrolazulinus       atrolazulinus      #536878

	atrolazulinus       atrolazulinus      #36454f

	atrolazulinus       atrolazulinus      #545aa7

	atrolazulinus       atrolazulinus      #4e5180

	atrolazulinus       atrolazulinus      #4c516d

	atrolividus         atrolividus        #86608e

	atrolividus         atrolividus        #563c5c

	atrolividus         atrolividus        #796878

	atrolividus         atrolividus        #50404d

	atrolividus         atrolividus        #5d3954

	atrolividus         atrolividus        #673147

	atromurinus         atromurinus        #5b504f

	atromurinus         atromurinus        #555555

	atropurpureus       atropurpureus      #602f6b

	atropurpureus       atropurpureus      #86608e

	atropurpureus       atropurpureus      #563c5c

	atropurpureus       atropurpureus      #50404d

	atropurpureus       atropurpureus      #9e4f88

	atropurpureus       atropurpureus      #702963

	atropurpureus       atropurpureus      #915c83

	atropurpureus       atropurpureus      #5d3954

	atropurpureus       atropurpureus      #341731

	atropurpureus       atropurpureus      #673147

	atrovenetus         atrovenetus        #3b7861

	atrovenetus         atrovenetus        #1b4d3e

	atrovenetus         atrovenetus        #5e716a

	atrovenetus         atrovenetus        #3a4b47

	atrovenetus         atrovenetus        #4e5755

	atrovenetus         atrovenetus        #007a74

	atrovenetus         atrovenetus        #317873

	atrovenetus         atrovenetus        #004b49

	atrovenetus         atrovenetus        #367588

	atrovenetus         atrovenetus        #536878

	atrovenetus         atrovenetus        #36454f

	atrovinosus         atrovinosus        #905d5d

	atrovinosus         atrovinosus        #543d3f

	atrovinosus         atrovinosus        #674c47

	atrovinosus         atrovinosus        #50404d

	atrovinosus         atrovinosus        #673147

	atrovinosus         atrovinosus        #915f6d

	atroviolaceus       atroviolaceus      #36454f

	atroviolaceus       atroviolaceus      #272458

	atroviolaceus       atroviolaceus      #4e5180

	atroviolaceus       atroviolaceus      #252440

	atroviolaceus       atroviolaceus      #4c516d

	atroviolaceus       atroviolaceus      #604e81

	atroviolaceus       atroviolaceus      #2f2140

	atroviolaceus       atroviolaceus      #554c69

	atroviolaceus       atroviolaceus      #602f6b

	atroviolaceus       atroviolaceus      #563c5c

	atroviolaceus       atroviolaceus      #796878

	atroviolaceus       atroviolaceus      #50404d

	atroviridis         atroviridis        #7e9f2e

	atroviridis         atroviridis        #8a9a5b

	atroviridis         atroviridis        #4a5d23

	atroviridis         atroviridis        #515744

	atroviridis         atroviridis        #679267

	atroviridis         atroviridis        #355e3b

	atroviridis         atroviridis        #3eb489

	atroviridis         atroviridis        #007959

	atroviridis         atroviridis        #3b7861

	atroviridis         atroviridis        #1b4d3e

	atroviridis         atroviridis        #5e716a

	atroviridis         atroviridis        #3a4b47

	atroviridis         atroviridis        #4e5755

	atroviridis         atroviridis        #317873

	atroviridis         atroviridis        #004b49

	aurantiacus         aurantiacus        #f38400

	aurantiacus         aurantiacus        #ed872d

	aurantiacus         aurantiacus        #eaa221

	avellaneus          avellaneus         #c19a6b

	avellaneus          avellaneus         #826644

	avellaneus          avellaneus         #ae9b82

	badius              badius             #ab4e52

	badius              badius             #905d5d

	badius              badius             #543d3f

	badius              badius             #882d17

	badius              badius             #79443b

	badius              badius             #674c47

	brunneo-vinosus     brunneovinosus     #905d5d

	brunneo-vinosus     brunneovinosus     #543d3f

	brunneo-vinosus     brunneovinosus     #5c504f

	brunneo-vinosus     brunneovinosus     #674c47

	brunneo-vinosus     brunneovinosus     #635147

	brunneo-vinosus     brunneovinosus     #5b504f

	brunneo-vinosus     brunneovinosus     #50404d

	brunneo-vinosus     brunneovinosus     #673147

	bubalinus           bubalinus          #f4c2c2

	bubalinus           bubalinus          #e3a857

	bubalinus           bubalinus          #fad6a5

	bubalinus           bubalinus          #f8de7e

	bubalinus           bubalinus          #c9ae5d

	bubalinus           bubalinus          #f3e5ab

	caeruleo-glaucus    caeruleoglaucus    #c7e6d7

	caeruleo-glaucus    caeruleoglaucus    #96ded1

	caeruleo-glaucus    caeruleoglaucus    #66aabc

	caeruleo-glaucus    caeruleoglaucus    #bcd4e6

	caeruleo-glaucus    caeruleoglaucus    #91a3b0

	caeruleo-griseus    caeruleogriseus    #bcd4e6

	caeruleo-griseus    caeruleogriseus    #91a3b0

	caeruleo-griseus    caeruleogriseus    #536878

	caeruleo-griseus    caeruleogriseus    #81878b

	caeruleo-griseus    caeruleogriseus    #8c92ac

	caeruleus           caeruleus          #c7e6d7

	caeruleus           caeruleus          #96ded1

	caeruleus           caeruleus          #66ada4

	caeruleus           caeruleus          #239eba

	caeruleus           caeruleus          #9cd1dc

	caeruleus           caeruleus          #66aabc

	caeruleus           caeruleus          #a1caf1

	caeruleus           caeruleus          #70a3cc

	caeruleus           caeruleus          #bcd4e6

	caeruleus           caeruleus          #91a3b0

	caesius             caesius            #66aabc

	caesius             caesius            #367588

	caesius             caesius            #70a3cc

	caesius             caesius            #436b95

	caesius             caesius            #91a3b0

	caesius             caesius            #536878

	caesius             caesius            #8791bf

	caesius             caesius            #4e5180

	caesius             caesius            #8c92ac

	caesius             caesius            #4c516d

	castaneus           castaneus          #79443b

	castaneus           castaneus          #3e1d1e

	castaneus           castaneus          #674c47

	castaneus           castaneus          #43302e

	chlorinus           chlorinus          #8a9a5b

	chlorinus           chlorinus          #44944a

	chlorinus           chlorinus          #679267

	cinnamomeus         cinnamomeus        #d9a6a9

	cinnamomeus         cinnamomeus        #d99058

	cinnamomeus         cinnamomeus        #a67b5b

	cinnamomeus         cinnamomeus        #e3a857

	cinnamomeus         cinnamomeus        #be8a3d

	cinnamomeus         cinnamomeus        #996515

	cinnamomeus         cinnamomeus        #c19a6b

	cinnamomeus         cinnamomeus        #826644

	citrino-viridis     citrinoviridis     #b9b459

	citrino-viridis     citrinoviridis     #98943e

	citrino-viridis     citrinoviridis     #867e36

	corallinus          corallinus         #e4717a

	corallinus          corallinus         #ab4e52

	corallinus          corallinus         #d9603b

	corallinus          corallinus         #cb6d51

	croceus             croceus            #f4c2c2

	croceus             croceus            #d99058

	croceus             croceus            #fbc97f

	croceus             croceus            #e3a857

	croceus             croceus            #fad6a5

	cyaneus             cyaneus            #007791

	cyaneus             cyaneus            #367588

	cyaneus             cyaneus            #0067a5

	cyaneus             cyaneus            #436b95

	electrinus          electrinus         #fada5e

	electrinus          electrinus         #d4af37

	electrinus          electrinus         #f8de7e

	electrinus          electrinus         #c9ae5d

	ferro-griseus       ferrogriseus       #57554c

	flavo-virens        flavovirens        #beb72e

	flavo-virens        flavovirens        #eae679

	flavo-virens        flavovirens        #b9b459

	flavo-virens        flavovirens        #ebe8a4

	flavo-virens        flavovirens        #bdda57

	flavo-virens        flavovirens        #7e9f2e

	flavo-virens        flavovirens        #c9dc89

	flavo-virens        flavovirens        #8a9a5b

	flavo-virens        flavovirens        #dadfb7

	flavus              flavus             #fada5e

	flavus              flavus             #d4af37

	flavus              flavus             #f8de7e

	flavus              flavus             #e9e450

	flavus              flavus             #beb72e

	fulvus              fulvus             #ae6938

	fulvus              fulvus             #be8a3d

	fulvus              fulvus             #996515

	fumosus             fumosus            #c1b6b3

	fumosus             fumosus            #c2ac99

	fumosus             fumosus            #ae9b82

	fumosus             fumosus            #c2b280

	fumosus             fumosus            #bfb8a5

	fumosus             fumosus            #f2f3f4

	fumosus             fumosus            #b9b8b5

	fusco-niger         fusconiger         #5c504f

	fusco-niger         fusconiger         #3e322c

	fusco-niger         fusconiger         #5b504f

	fusco-niger         fusconiger         #483c32

	fusco-niger         fusconiger         #50404d

	fusco-niger         fusconiger         #5d555b

	fusco-niger         fusconiger         #38152c

	fusco-niger         fusconiger         #555555

	fuscus              fuscus             #905d5d

	fuscus              fuscus             #5c504f

	fuscus              fuscus             #5b504f

	fuscus              fuscus             #50404d

	fuscus              fuscus             #5d555b

	glauco-griseus      glaucogriseus      #8f9779

	glauco-griseus      glaucogriseus      #c7e6d7

	glauco-griseus      glaucogriseus      #8da399

	glauco-griseus      glaucogriseus      #b2beb5

	glauco-griseus      glaucogriseus      #7d8984

	glauco-griseus      glaucogriseus      #91a3b0

	glauco-griseus      glaucogriseus      #536878

	glauco-griseus      glaucogriseus      #b4bcc0

	glauco-griseus      glaucogriseus      #81878b

	glauco-venetus      glaucovenetus      #8da399

	glauco-venetus      glaucovenetus      #5e716a

	glauco-venetus      glaucovenetus      #66ada4

	glauco-venetus      glaucovenetus      #317873

	glaucus             glaucus            #93c592

	glaucus             glaucus            #c7e6d7

	glaucus             glaucus            #8da399

	glaucus             glaucus            #66ada4

	glaucus             glaucus            #9cd1dc

	glaucus             glaucus            #bcd4e6

	griseo-caeruleus    griseocaeruleus    #a1caf1

	griseo-caeruleus    griseocaeruleus    #70a3cc

	griseo-caeruleus    griseocaeruleus    #bcd4e6

	griseo-caeruleus    griseocaeruleus    #91a3b0

	griseo-caeruleus    griseocaeruleus    #b3bce2

	griseo-caeruleus    griseocaeruleus    #8791bf

	griseo-caeruleus    griseocaeruleus    #c0c8e1

	griseo-caeruleus    griseocaeruleus    #8c92ac

	griseo-chlorinus    griseochlorinus    #eae679

	griseo-chlorinus    griseochlorinus    #b9b459

	griseo-chlorinus    griseochlorinus    #ebe8a4

	griseo-chlorinus    griseochlorinus    #b9b57d

	griseo-chlorinus    griseochlorinus    #c9dc89

	griseo-chlorinus    griseochlorinus    #8a9a5b

	griseo-chlorinus    griseochlorinus    #dadfb7

	griseo-lavendulus   griseolavendulus   #8791bf

	griseo-lavendulus   griseolavendulus   #c0c8e1

	griseo-lavendulus   griseolavendulus   #8c92ac

	griseo-lavendulus   griseolavendulus   #c4c3dd

	griseo-lavendulus   griseolavendulus   #9690ab

	griseo-lavendulus   griseolavendulus   #554c69

	griseo-lavendulus   griseolavendulus   #b695c0

	griseo-lavendulus   griseolavendulus   #aa98a9

	griseo-lavendulus   griseolavendulus   #796878

	griseo-lavendulus   griseolavendulus   #8b8589

	griseo-lavendulus   griseolavendulus   #c3a6b1

	griseo-lazulinus    griseolazulinus    #91a3b0

	griseo-lazulinus    griseolazulinus    #536878

	griseo-lazulinus    griseolazulinus    #8c92ac

	griseo-lazulinus    griseolazulinus    #4c516d

	griseo-lilacinus    griseolilacinus    #dea5a4

	griseo-lilacinus    griseolilacinus    #aa8a9e

	griseo-lilacinus    griseolilacinus    #c3a6b1

	griseo-lineus       griseolineus       #4997d0

	griseo-lineus       griseolineus       #a1caf1

	griseo-lineus       griseolineus       #70a3cc

	griseo-lineus       griseolineus       #8791bf

	griseo-lineus       griseolineus       #c0c8e1

	griseo-lineus       griseolineus       #8c82b6

	griseo-olivaceous   griseoolivaceous   #8c8767

	griseo-olivaceous   griseoolivaceous   #8a8776

	griseo-olivaceous   griseoolivaceous   #57554c

	griseo-roseus       griseoroseus       #ea9399

	griseo-roseus       griseoroseus       #e4717a

	griseo-roseus       griseoroseus       #f9ccca

	griseo-roseus       griseoroseus       #dea5a4

	griseo-roseus       griseoroseus       #c08081

	griseo-roseus       griseoroseus       #ead8d7

	griseo-roseus       griseoroseus       #c4aead

	griseo-roseus       griseoroseus       #de6fa1

	griseo-roseus       griseoroseus       #efbbcc

	griseo-roseus       griseoroseus       #d597ae

	griseo-roseus       griseoroseus       #c17e91

	griseo-sepiaceus    griseosepiaceus    #977f73

	griseo-sepiaceus    griseosepiaceus    #958070

	griseo-sepiaceus    griseosepiaceus    #635147

	griseo-sepiaceus    griseosepiaceus    #5b504f

	griseo-sepiaceus    griseosepiaceus    #c19a6b

	griseo-sepiaceus    griseosepiaceus    #826644

	griseo-sepiaceus    griseosepiaceus    #ae9b82

	griseo-sepiaceus    griseosepiaceus    #7e6d5a

	griseo-venetus      griseovenetus      #96ded1

	griseo-venetus      griseovenetus      #66ada4

	griseo-violaceus    griseoviolaceus    #8c92ac

	griseo-violaceus    griseoviolaceus    #4c516d

	griseo-violaceus    griseoviolaceus    #604e81

	griseo-violaceus    griseoviolaceus    #9690ab

	griseo-violaceus    griseoviolaceus    #554c69

	griseo-viridis      griseoviridis      #44944a

	griseo-viridis      griseoviridis      #93c592

	griseo-viridis      griseoviridis      #679267

	griseo-viridis      griseoviridis      #6aab8e

	herbaceus           herbaceus          #b9b459

	herbaceus           herbaceus          #98943e

	herbaceus           herbaceus          #867e36

	herbaceus           herbaceus          #665d1e

	herbaceus           herbaceus          #7e9f2e

	herbaceus           herbaceus          #8a9a5b

	herbaceus           herbaceus          #8f9779

	herbaceus           herbaceus          #4a5d23

	herbaceus           herbaceus          #515744

	herbaceus           herbaceus          #679267

	herbaceus           herbaceus          #355e3b

	herbaceus           herbaceus          #3b7861

	herbaceus           herbaceus          #5e716a

	herbaceus           herbaceus          #3a4b47

	hinnuleus           hinnuleus          #977f73

	hinnuleus           hinnuleus          #a67b5b

	hinnuleus           hinnuleus          #6f4e37

	hinnuleus           hinnuleus          #958070

	hinnuleus           hinnuleus          #635147

	incarnatus          incarnatus         #e4717a

	incarnatus          incarnatus         #dea5a4

	incarnatus          incarnatus         #f88379

	incarnatus          incarnatus         #e66761

	incarnatus          incarnatus         #f4c2c2

	incarnatus          incarnatus         #d9a6a9

	isabellinus         isabellinus        #826644

	isabellinus         isabellinus        #a18f60

	isabellinus         isabellinus        #967117

	latericius          latericius         #79443b

	latericius          latericius         #977f73

	latericius          latericius         #674c47

	latericius          latericius         #a67b5b

	latericius          latericius         #6f4e37

	latericius          latericius         #635147

	lavendulo-griseus   lavendulogriseus   #91a3b0

	lavendulo-griseus   lavendulogriseus   #b4bcc0

	lavendulo-griseus   lavendulogriseus   #81878b

	lavendulo-griseus   lavendulogriseus   #bfb9bd

	lavendulo-griseus   lavendulogriseus   #8b8589

	lavendulo-griseus   lavendulogriseus   #848482

	lavendulus          lavendulus         #6c79b8

	lavendulus          lavendulus         #b3bce2

	lavendulus          lavendulus         #8791bf

	lavendulus          lavendulus         #7e73b8

	lavendulus          lavendulus         #dcd0ff

	lavendulus          lavendulus         #8c82b6

	lavendulus          lavendulus         #c4c3dd

	lavendulus          lavendulus         #9690ab

	lavendulus          lavendulus         #d6cadd

	lazulino-ardesiacus lazulinoardesiacus #00304e

	lazulino-ardesiacus lazulinoardesiacus #536878

	lazulino-ardesiacus lazulinoardesiacus #36454f

	lazulino-ardesiacus lazulinoardesiacus #4c516d

	lazulinus           lazulinus          #00a1c2

	lazulinus           lazulinus          #4997d0

	lazulinus           lazulinus          #0067a5

	lazulinus           lazulinus          #436b95

	lazulinus           lazulinus          #30267a

	lazulinus           lazulinus          #6c79b8

	lazulinus           lazulinus          #545aa7

	lilacinus           lilacinus          #7e73b8

	lilacinus           lilacinus          #8c82b6

	lilacinus           lilacinus          #d399e6

	lilacinus           lilacinus          #b695c0

	lilacinus           lilacinus          #86608e

	lilacinus           lilacinus          #d6cadd

	lilacinus           lilacinus          #aa98a9

	lilacinus           lilacinus          #9e4f88

	lilacinus           lilacinus          #b784a7

	lilacinus           lilacinus          #915c83

	lilacinus           lilacinus          #aa8a9e

	lilacinus           lilacinus          #de6fa1

	lilacinus           lilacinus          #efbbcc

	lilacinus           lilacinus          #d597ae

	lilacinus           lilacinus          #e8ccd7

	lilacinus           lilacinus          #c3a6b1

	lineus              lineus             #4997d0

	lineus              lineus             #a1caf1

	lineus              lineus             #70a3cc

	lineus              lineus             #bcd4e6

	lineus              lineus             #91a3b0

	lineus              lineus             #6c79b8

	lineus              lineus             #b3bce2

	lineus              lineus             #8791bf

	lineus              lineus             #c0c8e1

	lineus              lineus             #8c92ac

	livido-purpureus    lividopurpureus    #86608e

	livido-purpureus    lividopurpureus    #b784a7

	livido-purpureus    lividopurpureus    #915c83

	livido-purpureus    lividopurpureus    #836479

	livido-purpureus    lividopurpureus    #a8516e

	livido-purpureus    lividopurpureus    #915f6d

	livido-ruber        lividoruber        #a8516e

	livido-ruber        lividoruber        #915f6d

	livido-violaceus    lividoviolaceus    #604e97

	livido-violaceus    lividoviolaceus    #8c82b6

	livido-violaceus    lividoviolaceus    #604e81

	livido-violaceus    lividoviolaceus    #875692

	livido-violaceus    lividoviolaceus    #86608e

	livido-violaceus    lividoviolaceus    #796878

	luteolus            luteolus           #fbc97f

	luteolus            luteolus           #e3a857

	luteolus            luteolus           #fada5e

	luteolus            luteolus           #d4af37

	luteolus            luteolus           #f8de7e

	luteolus            luteolus           #c9ae5d

	luteus              luteus             #ed872d

	luteus              luteus             #eaa221

	luteus              luteus             #f3c300

	luteus              luteus             #d4af37

	malachiteus         malachiteus        #93c592

	malachiteus         malachiteus        #679267

	malachiteus         malachiteus        #3b7861

	malvaceus           malvaceus          #d399e6

	malvaceus           malvaceus          #875692

	malvaceus           malvaceus          #b695c0

	malvaceus           malvaceus          #9e4f88

	malvaceus           malvaceus          #de6fa1

	melleus             melleus            #e3a857

	melleus             melleus            #996515

	melleus             melleus            #c19a6b

	melleus             melleus            #826644

	melleus             melleus            #c9ae5d

	melleus             melleus            #ab9144

	melleus             melleus            #967117

	miniatus            miniatus           #e25822

	miniatus            miniatus           #d9603b

	miniatus            miniatus           #cb6d51

	miniatus            miniatus           #9e4732

	murinus             murinus            #5b504f

	murinus             murinus            #4b3621

	murinus             murinus            #848482

	murinus             murinus            #555555

	niger               niger              #5b504f

	niger               niger              #57554c

	niger               niger              #36454f

	niger               niger              #50404d

	niger               niger              #5d555b

	niger               niger              #555555

	niger               niger              #222222

	obscuro-virens      obscurovirens      #665d1e

	obscuro-virens      obscurovirens      #5b5842

	obscuro-virens      obscurovirens      #4a5d23

	obscuro-virens      obscurovirens      #515744

	obscuro-virens      obscurovirens      #3b7861

	obscuro-virens      obscurovirens      #1b4d3e

	obscuro-virens      obscurovirens      #5e716a

	obscuro-virens      obscurovirens      #3a4b47

	ochraceus           ochraceus          #f4c2c2

	ochraceus           ochraceus          #d9a6a9

	ochraceus           ochraceus          #ed872d

	ochraceus           ochraceus          #d99058

	ochraceus           ochraceus          #eaa221

	ochraceus           ochraceus          #c98500

	ochraceus           ochraceus          #be8a3d

	olivaceo-bubalinus  olivaceobubalinus  #f3e5ab

	olivaceo-bubalinus  olivaceobubalinus  #c2b280

	olivaceo-citrinus   olivaceocitrinus   #af8d13

	olivaceo-citrinus   olivaceocitrinus   #ab9144

	olivaceo-citrinus   olivaceocitrinus   #967117

	olivaceo-citrinus   olivaceocitrinus   #6c541e

	olivaceo-citrinus   olivaceocitrinus   #98943e

	olivaceo-citrinus   olivaceocitrinus   #867e36

	olivaceo-citrinus   olivaceocitrinus   #665d1e

	olivaceo-griseus    olivaceogriseus    #5b5842

	olivaceo-griseus    olivaceogriseus    #8a8776

	olivaceo-griseus    olivaceogriseus    #57554c

	olivaceo-niger      olivaceoniger      #57554c

	olivaceo-niger      olivaceoniger      #515744

	olivaceo-niger      olivaceoniger      #31362b

	olivaceo-niger      olivaceoniger      #4e5755

	olivaceo-niger      olivaceoniger      #555555

	olivaceus           olivaceus          #7e6d5a

	olivaceus           olivaceus          #483c32

	olivaceus           olivaceus          #ab9144

	olivaceus           olivaceus          #a18f60

	olivaceus           olivaceus          #967117

	olivaceus           olivaceus          #6c541e

	olivaceus           olivaceus          #98943e

	olivaceus           olivaceus          #867e36

	olivaceus           olivaceus          #665d1e

	olivaceus           olivaceus          #5b5842

	persicinus          persicinus         #f88379

	persicinus          persicinus         #f4c2c2

	persicinus          persicinus         #d9a6a9

	persicinus          persicinus         #d9603b

	persicinus          persicinus         #cb6d51

	persicinus          persicinus         #fd943f

	persicinus          persicinus         #fab57f

	persicinus          persicinus         #d99058

	pistaceus           pistaceus          #8a9a5b

	pistaceus           pistaceus          #8f9779

	pistaceus           pistaceus          #93c592

	pistaceus           pistaceus          #679267

	pistaceus           pistaceus          #6aab8e

	pistaceus           pistaceus          #8da399

	pistaceus           pistaceus          #5e716a

	plumbeo-niger       plumbeoniger       #51585e

	plumbeus            plumbeus           #91a3b0

	plumbeus            plumbeus           #536878

	plumbeus            plumbeus           #81878b

	plumbeus            plumbeus           #51585e

	prasinus            prasinus           #c9dc89

	prasinus            prasinus           #b6e5af

	prasinus            prasinus           #93c592

	prasinus            prasinus           #8ed1b2

	prasinus            prasinus           #c7e6d7

	primulinus          primulinus         #f3e5ab

	primulinus          primulinus         #eae679

	primulinus          primulinus         #b9b459

	purpureo-ardesiacus purpureoardesiacus #796878

	purpureo-ardesiacus purpureoardesiacus #50404d

	purpureo-ardesiacus purpureoardesiacus #5d3954

	purpureo-ardesiacus purpureoardesiacus #836479

	purpureo-griseus    purpureogriseus    #796878

	purpureo-griseus    purpureogriseus    #8b8589

	purpureo-griseus    purpureogriseus    #5d555b

	purpureo-griseus    purpureogriseus    #848482

	purpureus           purpureus          #9a4eae

	purpureus           purpureus          #875692

	purpureus           purpureus          #602f6b

	purpureus           purpureus          #86608e

	purpureus           purpureus          #870074

	purpureus           purpureus          #9e4f88

	purpureus           purpureus          #702963

	purpureus           purpureus          #b3446c

	purpureus           purpureus          #78184a

	purpureus           purpureus          #a8516e

	purpureus           purpureus          #915f6d

	roseo-bubalinus     roseobubalinus     #f4c2c2

	roseo-bubalinus     roseobubalinus     #d9a6a9

	roseo-bubalinus     roseobubalinus     #ecd5c5

	roseo-bubalinus     roseobubalinus     #c7ada3

	roseo-bubalinus     roseobubalinus     #c2ac99

	roseo-bubalinus     roseobubalinus     #fad6a5

	roseo-vinosus       roseovinosus       #f4c2c2

	roseo-vinosus       roseovinosus       #d9a6a9

	roseo-vinosus       roseovinosus       #ecd5c5

	roseo-vinosus       roseovinosus       #c7ada3

	roseo-vinosus       roseovinosus       #b4745e

	roseo-vinosus       roseovinosus       #a87c6d

	roseus              roseus             #ffb5ba

	roseus              roseus             #ea9399

	roseus              roseus             #e4717a

	roseus              roseus             #ab4e52

	roseus              roseus             #f88379

	roseus              roseus             #e66761

	roseus              roseus             #9e4f88

	roseus              roseus             #ffc8d6

	roseus              roseus             #e68fac

	roseus              roseus             #de6fa1

	roseus              roseus             #efbbcc

	roseus              roseus             #ce4676

	roseus              roseus             #b3446c

	ruber               ruber              #be0032

	ruber               ruber              #bc3f4a

	ruber               ruber              #ce4676

	ruber               ruber              #78184a

	ruber               ruber              #a8516e

	ruber               ruber              #915f6d

	salmoneus           salmoneus          #d9a6a9

	salmoneus           salmoneus          #ecd5c5

	salmoneus           salmoneus          #c2ac99

	salmoneus           salmoneus          #fab57f

	salmoneus           salmoneus          #d99058

	salmoneus           salmoneus          #e3a857

	salmoneus           salmoneus          #fad6a5

	salmoneus           salmoneus          #f3e5ab

	senatus             senatus            #be6516

	senatus             senatus            #c98500

	senatus             senatus            #be8a3d

	senatus             senatus            #996515

	sepiaceus           sepiaceus          #6f4e37

	sepiaceus           sepiaceus          #635147

	sepiaceus           sepiaceus          #826644

	sepiaceus           sepiaceus          #7e6d5a

	sepiaceus           sepiaceus          #483c32

	sepiaceus           sepiaceus          #6c541e

	stramineus          stramineus         #f8de7e

	stramineus          stramineus         #f3e5ab

	stramineus          stramineus         #eae679

	stramineus          stramineus         #b9b459

	stramineus          stramineus         #ebe8a4

	sublazulinus        sublazulinus       #4997d0

	sublazulinus        sublazulinus       #0067a5

	sublazulinus        sublazulinus       #a1caf1

	sublazulinus        sublazulinus       #70a3cc

	sublazulinus        sublazulinus       #bcd4e6

	sublazulinus        sublazulinus       #91a3b0

	sublazulinus        sublazulinus       #30267a

	sublazulinus        sublazulinus       #6c79b8

	sublazulinus        sublazulinus       #545aa7

	sublazulinus        sublazulinus       #b3bce2

	sublazulinus        sublazulinus       #8791bf

	sublazulinus        sublazulinus       #c0c8e1

	sublazulinus        sublazulinus       #8c92ac

	submurinus          submurinus         #c1b6b3

	submurinus          submurinus         #8e8279

	submurinus          submurinus         #bfb8a5

	submurinus          submurinus         #b9b8b5

	subolivaceo-griseus subolivaceogriseus #f0ead6

	subolivaceo-griseus subolivaceogriseus #bfb8a5

	subolivaceo-griseus subolivaceogriseus #8c8767

	subolivaceo-griseus subolivaceogriseus #8a8776

	subpurpureo-griseus subpurpureogriseus #bfb9bd

	subpurpureo-griseus subpurpureogriseus #8b8589

	subvinoso-griseus   subvinosogriseus   #c4aead

	subvinoso-griseus   subvinosogriseus   #c1b6b3

	subvinoso-griseus   subvinosogriseus   #ad8884

	subvinoso-griseus   subvinosogriseus   #905d5d

	subvinoso-griseus   subvinosogriseus   #8f817f

	subvinoso-griseus   subvinosogriseus   #958070

	subvinoso-griseus   subvinosogriseus   #aa98a9

	subvinoso-griseus   subvinosogriseus   #796878

	subvinoso-griseus   subvinosogriseus   #bfb9bd

	subvinoso-griseus   subvinosogriseus   #8b8589

	subvinoso-griseus   subvinosogriseus   #e8ccd7

	subvinoso-griseus   subvinosogriseus   #c3a6b1

	subvinosus          subvinosus         #f9ccca

	subvinosus          subvinosus         #dea5a4

	subvinosus          subvinosus         #ead8d7

	subvinosus          subvinosus         #c4aead

	subvinosus          subvinosus         #c1b6b3

	subvinosus          subvinosus         #ecd5c5

	subvinosus          subvinosus         #c7ada3

	subviolaceous       subviolaceous      #30267a

	subviolaceous       subviolaceous      #6c79b8

	subviolaceous       subviolaceous      #545aa7

	subviolaceous       subviolaceous      #b3bce2

	subviolaceous       subviolaceous      #8791bf

	subviolaceous       subviolaceous      #c0c8e1

	subviolaceous       subviolaceous      #8c92ac

	subviolaceous       subviolaceous      #7e73b8

	subviolaceous       subviolaceous      #dcd0ff

	subviolaceous       subviolaceous      #8c82b6

	subviolaceous       subviolaceous      #c4c3dd

	subviolaceous       subviolaceous      #9690ab

	subviolaceous       subviolaceous      #d399e6

	subviolaceous       subviolaceous      #d5badb

	subviolaceous       subviolaceous      #b695c0

	subviridi-griseus   subviridigriseus   #b4bcc0

	subviridi-griseus   subviridigriseus   #b9b8b5

	subviridis          subviridis         #bdda57

	subviridis          subviridis         #c9dc89

	subviridis          subviridis         #dadfb7

	subviridis          subviridis         #83d37d

	subviridis          subviridis         #93c592

	sulphureus          sulphureus         #e9e450

	sulphureus          sulphureus         #eae679

	sulphureus          sulphureus         #b9b459

	sulphureus          sulphureus         #ebe8a4

	testaceus           testaceus          #f88379

	testaceus           testaceus          #d9a6a9

	testaceus           testaceus          #b4745e

	testaceus           testaceus          #a87c6d

	testaceus           testaceus          #79443b

	testaceus           testaceus          #d99058

	testaceus           testaceus          #a67b5b

	umbrinus            umbrinus           #79443b

	umbrinus            umbrinus           #674c47

	umbrinus            umbrinus           #80461b

	umbrinus            umbrinus           #a67b5b

	umbrinus            umbrinus           #6f4e37

	umbrinus            umbrinus           #996515

	umbrinus            umbrinus           #826644

	umbrinus            umbrinus           #4b3621

	venetus             venetus            #00a693

	venetus             venetus            #007a74

	venetus             venetus            #66ada4

	venetus             venetus            #317873

	vinoso-bubalinus    vinosobubalinus    #c1b6b3

	vinoso-bubalinus    vinosobubalinus    #f4c2c2

	vinoso-bubalinus    vinosobubalinus    #d9a6a9

	vinoso-bubalinus    vinosobubalinus    #ecd5c5

	vinoso-bubalinus    vinosobubalinus    #c7ada3

	vinoso-bubalinus    vinosobubalinus    #c2ac99

	vinoso-bubalinus    vinosobubalinus    #c19a6b

	vinoso-bubalinus    vinosobubalinus    #ae9b82

	vinoso-griseus      vinosogriseus      #905d5d

	vinoso-griseus      vinosogriseus      #5c504f

	vinoso-griseus      vinosogriseus      #977f73

	vinoso-griseus      vinosogriseus      #958070

	vinoso-griseus      vinosogriseus      #635147

	vinoso-griseus      vinosogriseus      #796878

	vinoso-lividus      vinosolividus      #ad8884

	vinoso-lividus      vinosolividus      #905d5d

	vinoso-lividus      vinosolividus      #a87c6d

	vinoso-lividus      vinosolividus      #674c47

	vinoso-lividus      vinosolividus      #c17e91

	vinoso-lividus      vinosolividus      #915f6d

	vinoso-purpureus    vinosopurpureus    #836479

	vinosus             vinosus            #e4717a

	vinosus             vinosus            #c08081

	vinosus             vinosus            #905d5d

	vinosus             vinosus            #f88379

	violaceo-ardesiacus violaceoardesiacus #796878

	violaceo-ardesiacus violaceoardesiacus #50404d

	violaceo-ardesiacus violaceoardesiacus #5d555b

	violaceo-griseus    violaceogriseus    #536878

	violaceo-griseus    violaceogriseus    #4c516d

	violaceo-griseus    violaceogriseus    #796878

	violaceo-griseus    violaceogriseus    #8b8589

	violaceo-griseus    violaceogriseus    #5d555b

	violaceo-griseus    violaceogriseus    #848482

	violaceo-griseus    violaceogriseus    #555555

	violaceo-niger      violaceoniger      #50404d

	violaceo-niger      violaceoniger      #5d555b

	violaceo-niger      violaceoniger      #555555

	violaceo-niger      violaceoniger      #222222

	violaceus           violaceus          #30267a

	violaceus           violaceus          #545aa7

	violaceus           violaceus          #4e5180

	violaceus           violaceus          #9065ca

	violaceus           violaceus          #7e73b8

	violaceus           violaceus          #604e97

	violaceus           violaceus          #8c82b6

	violaceus           violaceus          #604e81

	violaceus           violaceus          #9a4eae

	violaceus           violaceus          #602f6b

	violaceus           violaceus          #86608e

	viridi-caeruleus    viridicaeruleus    #3eb489

	viridi-caeruleus    viridicaeruleus    #6aab8e

	viridi-caeruleus    viridicaeruleus    #c7e6d7

	viridi-caeruleus    viridicaeruleus    #96ded1

	viridi-caeruleus    viridicaeruleus    #66ada4

	viridi-flavus       viridiflavus       #e9e450

	viridi-flavus       viridiflavus       #beb72e

	viridi-flavus       viridiflavus       #eae679

	viridi-flavus       viridiflavus       #b9b459

	viridi-flavus       viridiflavus       #ebe8a4

	viridi-flavus       viridiflavus       #8db600

	viridi-flavus       viridiflavus       #bdda57

	viridi-flavus       viridiflavus       #7e9f2e

	viridi-flavus       viridiflavus       #c9dc89

	viridi-flavus       viridiflavus       #8a9a5b

	viridi-glaucus      viridiglaucus      #c9dc89

	viridi-glaucus      viridiglaucus      #dadfb7

	viridi-glaucus      viridiglaucus      #8ed1b2

	viridi-glaucus      viridiglaucus      #c7e6d7

	viridi-griseus      viridigriseus      #8a9a5b

	viridi-griseus      viridigriseus      #8f9779

	viridi-griseus      viridigriseus      #8da399

	viridi-griseus      viridigriseus      #5e716a

	viridi-griseus      viridigriseus      #b2beb5

	viridi-griseus      viridigriseus      #7d8984

	viridi-griseus      viridigriseus      #81878b

	viridi-griseus      viridigriseus      #51585e

	viridi-griseus      viridigriseus      #b9b8b5

	viridi-griseus      viridigriseus      #848482

	viridi-niger        viridiniger        #555555

	viridi-olivaceus    viridiolivaceus    #a18f60

	viridi-olivaceus    viridiolivaceus    #867e36

	viridi-olivaceus    viridiolivaceus    #8a9a5b

	viridi-olivaceus    viridiolivaceus    #8f9779

	viridis             viridis            #8db600

	viridis             viridis            #bdda57

	viridis             viridis            #7e9f2e

	viridis             viridis            #83d37d

	viridis             viridis            #93c592

	viridis             viridis            #3eb489


=cut

sub _load_color_list() {
    return [
['nbs-iscc-b:aerugineus.135','aerugineus','aerugineus ',[147,197,146],'93c592',9684370],
['nbs-iscc-b:aerugineus.136','aerugineus','aerugineus ',[103,146,103],'679267',6787687],
['nbs-iscc-b:aerugineus.144','aerugineus','aerugineus ',[106,171,142],'6aab8e',6990734],
['nbs-iscc-b:aerugineus.145','aerugineus','aerugineus ',[59,120,97],'3b7861',3897441],
['nbs-iscc-b:aerugineus.149','aerugineus','aerugineus ',[141,163,153],'8da399',9282457],
['nbs-iscc-b:aerugineus.150','aerugineus','aerugineus ',[94,113,106],'5e716a',6189418],
['nbs-iscc-b:aerugineus.163','aerugineus','aerugineus ',[102,173,164],'66ada4',6729124],
['nbs-iscc-b:aerugineus.164','aerugineus','aerugineus ',[49,120,115],'317873',3242099],
['nbs-iscc-b:armeniacus.37','armeniacus','armeniacus',[203,109,81],'cb6d51',13331793],
['nbs-iscc-b:armeniacus.53','armeniacus','armeniacus',[217,144,88],'d99058',14258264],
['nbs-iscc-b:atroardesiacus.151','atroardesiacus','atroardesiacus',[58,75,71],'3a4b47',3820359],
['nbs-iscc-b:atroardesiacus.156','atroardesiacus','atroardesiacus',[78,87,85],'4e5755',5134165],
['nbs-iscc-b:atroardesiacus.183','atroardesiacus','atroardesiacus',[0,48,78],'00304e',12366],
['nbs-iscc-b:atroardesiacus.186','atroardesiacus','atroardesiacus',[83,104,120],'536878',5466232],
['nbs-iscc-b:atroardesiacus.187','atroardesiacus','atroardesiacus',[54,69,79],'36454f',3556687],
['nbs-iscc-b:atroardesiacus.188','atroardesiacus','atroardesiacus',[32,40,48],'202830',2107440],
['nbs-iscc-b:atroardesiacus.192','atroardesiacus','atroardesiacus',[81,88,94],'51585e',5331038],
['nbs-iscc-b:atroardesiacus.193','atroardesiacus','atroardesiacus',[32,36,40],'202428',2106408],
['nbs-iscc-b:atroardesiacus.234','atroardesiacus','atroardesiacus',[93,85,91],'5d555b',6116699],
['nbs-iscc-b:atroardesiacus.266','atroardesiacus','atroardesiacus',[85,85,85],'555555',5592405],
['nbs-iscc-b:atroardesiacus.267','atroardesiacus','atroardesiacus',[34,34,34],'222222',2236962],
['nbs-iscc-b:atrocyaneus.164','atrocyaneus','atrocyaneus ',[49,120,115],'317873',3242099],
['nbs-iscc-b:atrocyaneus.165','atrocyaneus','atrocyaneus ',[0,75,73],'004b49',19273],
['nbs-iscc-b:atrocyaneus.173','atrocyaneus','atrocyaneus ',[54,117,136],'367588',3569032],
['nbs-iscc-b:atrocyaneus.174','atrocyaneus','atrocyaneus ',[0,73,88],'004958',18776],
['nbs-iscc-b:atrocyaneus.182','atrocyaneus','atrocyaneus ',[67,107,149],'436b95',4418453],
['nbs-iscc-b:atrocyaneus.183','atrocyaneus','atrocyaneus ',[0,48,78],'00304e',12366],
['nbs-iscc-b:atrocyaneus.186','atrocyaneus','atrocyaneus ',[83,104,120],'536878',5466232],
['nbs-iscc-b:atroherbaceus.106','atroherbaceus','atroherbaceus',[134,126,54],'867e36',8814134],
['nbs-iscc-b:atroherbaceus.107','atroherbaceus','atroherbaceus',[102,93,30],'665d1e',6708510],
['nbs-iscc-b:atroherbaceus.120','atroherbaceus','atroherbaceus',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-b:atroherbaceus.125','atroherbaceus','atroherbaceus',[74,93,35],'4a5d23',4873507],
['nbs-iscc-b:atroherbaceus.127','atroherbaceus','atroherbaceus',[81,87,68],'515744',5330756],
['nbs-iscc-b:atroherbaceus.137','atroherbaceus','atroherbaceus',[53,94,59],'355e3b',3497531],
['nbs-iscc-b:atroherbaceus.145','atroherbaceus','atroherbaceus',[59,120,97],'3b7861',3897441],
['nbs-iscc-b:atroherbaceus.150','atroherbaceus','atroherbaceus',[94,113,106],'5e716a',6189418],
['nbs-iscc-b:atrolazulinus.176','atrolazulinus','atrolazulinus',[0,161,194],'00a1c2',41410],
['nbs-iscc-b:atrolazulinus.178','atrolazulinus','atrolazulinus',[0,103,165],'0067a5',26533],
['nbs-iscc-b:atrolazulinus.179','atrolazulinus','atrolazulinus',[0,65,106],'00416a',16746],
['nbs-iscc-b:atrolazulinus.182','atrolazulinus','atrolazulinus',[67,107,149],'436b95',4418453],
['nbs-iscc-b:atrolazulinus.183','atrolazulinus','atrolazulinus',[0,48,78],'00304e',12366],
['nbs-iscc-b:atrolazulinus.186','atrolazulinus','atrolazulinus',[83,104,120],'536878',5466232],
['nbs-iscc-b:atrolazulinus.187','atrolazulinus','atrolazulinus',[54,69,79],'36454f',3556687],
['nbs-iscc-b:atrolazulinus.196','atrolazulinus','atrolazulinus',[84,90,167],'545aa7',5528231],
['nbs-iscc-b:atrolazulinus.200','atrolazulinus','atrolazulinus',[78,81,128],'4e5180',5132672],
['nbs-iscc-b:atrolazulinus.204','atrolazulinus','atrolazulinus',[76,81,109],'4c516d',5001581],
['nbs-iscc-b:atrolividus.223','atrolividus','atrolividus',[134,96,142],'86608e',8806542],
['nbs-iscc-b:atrolividus.224','atrolividus','atrolividus',[86,60,92],'563c5c',5651548],
['nbs-iscc-b:atrolividus.228','atrolividus','atrolividus',[121,104,120],'796878',7956600],
['nbs-iscc-b:atrolividus.229','atrolividus','atrolividus',[80,64,77],'50404d',5259341],
['nbs-iscc-b:atrolividus.242','atrolividus','atrolividus',[93,57,84],'5d3954',6109524],
['nbs-iscc-b:atrolividus.259','atrolividus','atrolividus',[103,49,71],'673147',6762823],
['nbs-iscc-b:atromurinus.64','atromurinus','atromurinus',[91,80,79],'5b504f',5984335],
['nbs-iscc-b:atromurinus.266','atromurinus','atromurinus',[85,85,85],'555555',5592405],
['nbs-iscc-b:atropurpureus.219','atropurpureus','atropurpureus',[96,47,107],'602f6b',6303595],
['nbs-iscc-b:atropurpureus.223','atropurpureus','atropurpureus',[134,96,142],'86608e',8806542],
['nbs-iscc-b:atropurpureus.224','atropurpureus','atropurpureus',[86,60,92],'563c5c',5651548],
['nbs-iscc-b:atropurpureus.229','atropurpureus','atropurpureus',[80,64,77],'50404d',5259341],
['nbs-iscc-b:atropurpureus.237','atropurpureus','atropurpureus',[158,79,136],'9e4f88',10375048],
['nbs-iscc-b:atropurpureus.238','atropurpureus','atropurpureus',[112,41,99],'702963',7350627],
['nbs-iscc-b:atropurpureus.241','atropurpureus','atropurpureus',[145,92,131],'915c83',9526403],
['nbs-iscc-b:atropurpureus.242','atropurpureus','atropurpureus',[93,57,84],'5d3954',6109524],
['nbs-iscc-b:atropurpureus.243','atropurpureus','atropurpureus',[52,23,49],'341731',3413809],
['nbs-iscc-b:atropurpureus.259','atropurpureus','atropurpureus',[103,49,71],'673147',6762823],
['nbs-iscc-b:atrovenetus.145','atrovenetus','atrovenetus',[59,120,97],'3b7861',3897441],
['nbs-iscc-b:atrovenetus.146','atrovenetus','atrovenetus',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-b:atrovenetus.150','atrovenetus','atrovenetus',[94,113,106],'5e716a',6189418],
['nbs-iscc-b:atrovenetus.151','atrovenetus','atrovenetus',[58,75,71],'3a4b47',3820359],
['nbs-iscc-b:atrovenetus.156','atrovenetus','atrovenetus',[78,87,85],'4e5755',5134165],
['nbs-iscc-b:atrovenetus.160','atrovenetus','atrovenetus',[0,122,116],'007a74',31348],
['nbs-iscc-b:atrovenetus.164','atrovenetus','atrovenetus',[49,120,115],'317873',3242099],
['nbs-iscc-b:atrovenetus.165','atrovenetus','atrovenetus',[0,75,73],'004b49',19273],
['nbs-iscc-b:atrovenetus.173','atrovenetus','atrovenetus',[54,117,136],'367588',3569032],
['nbs-iscc-b:atrovenetus.186','atrovenetus','atrovenetus',[83,104,120],'536878',5466232],
['nbs-iscc-b:atrovenetus.187','atrovenetus','atrovenetus',[54,69,79],'36454f',3556687],
['nbs-iscc-b:atrovinosus.19','atrovinosus','atrovinosus',[144,93,93],'905d5d',9461085],
['nbs-iscc-b:atrovinosus.20','atrovinosus','atrovinosus',[84,61,63],'543d3f',5520703],
['nbs-iscc-b:atrovinosus.46','atrovinosus','atrovinosus',[103,76,71],'674c47',6769735],
['nbs-iscc-b:atrovinosus.229','atrovinosus','atrovinosus',[80,64,77],'50404d',5259341],
['nbs-iscc-b:atrovinosus.259','atrovinosus','atrovinosus',[103,49,71],'673147',6762823],
['nbs-iscc-b:atrovinosus.262','atrovinosus','atrovinosus',[145,95,109],'915f6d',9527149],
['nbs-iscc-b:atroviolaceus.187','atroviolaceus','atroviolaceus',[54,69,79],'36454f',3556687],
['nbs-iscc-b:atroviolaceus.197','atroviolaceus','atroviolaceus',[39,36,88],'272458',2565208],
['nbs-iscc-b:atroviolaceus.200','atroviolaceus','atroviolaceus',[78,81,128],'4e5180',5132672],
['nbs-iscc-b:atroviolaceus.201','atroviolaceus','atroviolaceus',[37,36,64],'252440',2434112],
['nbs-iscc-b:atroviolaceus.204','atroviolaceus','atroviolaceus',[76,81,109],'4c516d',5001581],
['nbs-iscc-b:atroviolaceus.211','atroviolaceus','atroviolaceus',[96,78,129],'604e81',6311553],
['nbs-iscc-b:atroviolaceus.212','atroviolaceus','atroviolaceus',[47,33,64],'2f2140',3088704],
['nbs-iscc-b:atroviolaceus.215','atroviolaceus','atroviolaceus',[85,76,105],'554c69',5590121],
['nbs-iscc-b:atroviolaceus.219','atroviolaceus','atroviolaceus',[96,47,107],'602f6b',6303595],
['nbs-iscc-b:atroviolaceus.224','atroviolaceus','atroviolaceus',[86,60,92],'563c5c',5651548],
['nbs-iscc-b:atroviolaceus.228','atroviolaceus','atroviolaceus',[121,104,120],'796878',7956600],
['nbs-iscc-b:atroviolaceus.229','atroviolaceus','atroviolaceus',[80,64,77],'50404d',5259341],
['nbs-iscc-b:atroviridis.117','atroviridis','atroviridis',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-b:atroviridis.120','atroviridis','atroviridis',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-b:atroviridis.125','atroviridis','atroviridis',[74,93,35],'4a5d23',4873507],
['nbs-iscc-b:atroviridis.127','atroviridis','atroviridis',[81,87,68],'515744',5330756],
['nbs-iscc-b:atroviridis.136','atroviridis','atroviridis',[103,146,103],'679267',6787687],
['nbs-iscc-b:atroviridis.137','atroviridis','atroviridis',[53,94,59],'355e3b',3497531],
['nbs-iscc-b:atroviridis.140','atroviridis','atroviridis',[62,180,137],'3eb489',4109449],
['nbs-iscc-b:atroviridis.141','atroviridis','atroviridis',[0,121,89],'007959',31065],
['nbs-iscc-b:atroviridis.145','atroviridis','atroviridis',[59,120,97],'3b7861',3897441],
['nbs-iscc-b:atroviridis.146','atroviridis','atroviridis',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-b:atroviridis.150','atroviridis','atroviridis',[94,113,106],'5e716a',6189418],
['nbs-iscc-b:atroviridis.151','atroviridis','atroviridis',[58,75,71],'3a4b47',3820359],
['nbs-iscc-b:atroviridis.156','atroviridis','atroviridis',[78,87,85],'4e5755',5134165],
['nbs-iscc-b:atroviridis.164','atroviridis','atroviridis',[49,120,115],'317873',3242099],
['nbs-iscc-b:atroviridis.165','atroviridis','atroviridis',[0,75,73],'004b49',19273],
['nbs-iscc-b:aurantiacus.48','aurantiacus','aurantiacus',[243,132,0],'f38400',15959040],
['nbs-iscc-b:aurantiacus.50','aurantiacus','aurantiacus',[237,135,45],'ed872d',15566637],
['nbs-iscc-b:aurantiacus.68','aurantiacus','aurantiacus',[234,162,33],'eaa221',15376929],
['nbs-iscc-b:avellaneus.76','avellaneus','avellaneus',[193,154,107],'c19a6b',12687979],
['nbs-iscc-b:avellaneus.77','avellaneus','avellaneus',[130,102,68],'826644',8545860],
['nbs-iscc-b:avellaneus.79','avellaneus','avellaneus',[174,155,130],'ae9b82',11443074],
['nbs-iscc-b:badius.15','badius','badius',[171,78,82],'ab4e52',11226706],
['nbs-iscc-b:badius.19','badius','badius',[144,93,93],'905d5d',9461085],
['nbs-iscc-b:badius.20','badius','badius',[84,61,63],'543d3f',5520703],
['nbs-iscc-b:badius.40','badius','badius',[136,45,23],'882d17',8924439],
['nbs-iscc-b:badius.43','badius','badius',[121,68,59],'79443b',7947323],
['nbs-iscc-b:badius.46','badius','badius',[103,76,71],'674c47',6769735],
['nbs-iscc-b:brunneovinosus.19','brunneovinosus','brunneo-vinosus',[144,93,93],'905d5d',9461085],
['nbs-iscc-b:brunneovinosus.20','brunneovinosus','brunneo-vinosus',[84,61,63],'543d3f',5520703],
['nbs-iscc-b:brunneovinosus.23','brunneovinosus','brunneo-vinosus',[92,80,79],'5c504f',6049871],
['nbs-iscc-b:brunneovinosus.46','brunneovinosus','brunneo-vinosus',[103,76,71],'674c47',6769735],
['nbs-iscc-b:brunneovinosus.61','brunneovinosus','brunneo-vinosus',[99,81,71],'635147',6508871],
['nbs-iscc-b:brunneovinosus.64','brunneovinosus','brunneo-vinosus',[91,80,79],'5b504f',5984335],
['nbs-iscc-b:brunneovinosus.229','brunneovinosus','brunneo-vinosus',[80,64,77],'50404d',5259341],
['nbs-iscc-b:brunneovinosus.259','brunneovinosus','brunneo-vinosus',[103,49,71],'673147',6762823],
['nbs-iscc-b:bubalinus.28','bubalinus','bubalinus',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-b:bubalinus.71','bubalinus','bubalinus',[227,168,87],'e3a857',14919767],
['nbs-iscc-b:bubalinus.73','bubalinus','bubalinus',[250,214,165],'fad6a5',16438949],
['nbs-iscc-b:bubalinus.86','bubalinus','bubalinus',[248,222,126],'f8de7e',16309886],
['nbs-iscc-b:bubalinus.87','bubalinus','bubalinus',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-b:bubalinus.89','bubalinus','bubalinus',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-b:caeruleoglaucus.148','caeruleoglaucus','caeruleo-glaucus',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-b:caeruleoglaucus.162','caeruleoglaucus','caeruleo-glaucus',[150,222,209],'96ded1',9887441],
['nbs-iscc-b:caeruleoglaucus.172','caeruleoglaucus','caeruleo-glaucus',[102,170,188],'66aabc',6728380],
['nbs-iscc-b:caeruleoglaucus.184','caeruleoglaucus','caeruleo-glaucus',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-b:caeruleoglaucus.185','caeruleoglaucus','caeruleo-glaucus',[145,163,176],'91a3b0',9544624],
['nbs-iscc-b:caeruleogriseus.184','caeruleogriseus','caeruleo-griseus',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-b:caeruleogriseus.185','caeruleogriseus','caeruleo-griseus',[145,163,176],'91a3b0',9544624],
['nbs-iscc-b:caeruleogriseus.186','caeruleogriseus','caeruleo-griseus',[83,104,120],'536878',5466232],
['nbs-iscc-b:caeruleogriseus.191','caeruleogriseus','caeruleo-griseus',[129,135,139],'81878b',8488843],
['nbs-iscc-b:caeruleogriseus.203','caeruleogriseus','caeruleo-griseus',[140,146,172],'8c92ac',9212588],
['nbs-iscc-b:caeruleus.148','caeruleus','caeruleus',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-b:caeruleus.162','caeruleus','caeruleus',[150,222,209],'96ded1',9887441],
['nbs-iscc-b:caeruleus.163','caeruleus','caeruleus',[102,173,164],'66ada4',6729124],
['nbs-iscc-b:caeruleus.168','caeruleus','caeruleus',[35,158,186],'239eba',2334394],
['nbs-iscc-b:caeruleus.171','caeruleus','caeruleus',[156,209,220],'9cd1dc',10277340],
['nbs-iscc-b:caeruleus.172','caeruleus','caeruleus',[102,170,188],'66aabc',6728380],
['nbs-iscc-b:caeruleus.180','caeruleus','caeruleus',[161,202,241],'a1caf1',10603249],
['nbs-iscc-b:caeruleus.181','caeruleus','caeruleus',[112,163,204],'70a3cc',7381964],
['nbs-iscc-b:caeruleus.184','caeruleus','caeruleus',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-b:caeruleus.185','caeruleus','caeruleus',[145,163,176],'91a3b0',9544624],
['nbs-iscc-b:caesius.172','caesius','caesius',[102,170,188],'66aabc',6728380],
['nbs-iscc-b:caesius.173','caesius','caesius',[54,117,136],'367588',3569032],
['nbs-iscc-b:caesius.181','caesius','caesius',[112,163,204],'70a3cc',7381964],
['nbs-iscc-b:caesius.182','caesius','caesius',[67,107,149],'436b95',4418453],
['nbs-iscc-b:caesius.185','caesius','caesius',[145,163,176],'91a3b0',9544624],
['nbs-iscc-b:caesius.186','caesius','caesius',[83,104,120],'536878',5466232],
['nbs-iscc-b:caesius.199','caesius','caesius',[135,145,191],'8791bf',8884671],
['nbs-iscc-b:caesius.200','caesius','caesius',[78,81,128],'4e5180',5132672],
['nbs-iscc-b:caesius.203','caesius','caesius',[140,146,172],'8c92ac',9212588],
['nbs-iscc-b:caesius.204','caesius','caesius',[76,81,109],'4c516d',5001581],
['nbs-iscc-b:castaneus.43','castaneus','castaneus',[121,68,59],'79443b',7947323],
['nbs-iscc-b:castaneus.44','castaneus','castaneus',[62,29,30],'3e1d1e',4070686],
['nbs-iscc-b:castaneus.46','castaneus','castaneus',[103,76,71],'674c47',6769735],
['nbs-iscc-b:castaneus.47','castaneus','castaneus',[67,48,46],'43302e',4403246],
['nbs-iscc-b:chlorinus.120','chlorinus','chlorinus',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-b:chlorinus.131','chlorinus','chlorinus',[68,148,74],'44944a',4494410],
['nbs-iscc-b:chlorinus.136','chlorinus','chlorinus',[103,146,103],'679267',6787687],
['nbs-iscc-b:cinnamomeus.29','cinnamomeus','cinnamomeus',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-b:cinnamomeus.53','cinnamomeus','cinnamomeus',[217,144,88],'d99058',14258264],
['nbs-iscc-b:cinnamomeus.57','cinnamomeus','cinnamomeus',[166,123,91],'a67b5b',10910555],
['nbs-iscc-b:cinnamomeus.71','cinnamomeus','cinnamomeus',[227,168,87],'e3a857',14919767],
['nbs-iscc-b:cinnamomeus.72','cinnamomeus','cinnamomeus',[190,138,61],'be8a3d',12487229],
['nbs-iscc-b:cinnamomeus.74','cinnamomeus','cinnamomeus',[153,101,21],'996515',10052885],
['nbs-iscc-b:cinnamomeus.76','cinnamomeus','cinnamomeus',[193,154,107],'c19a6b',12687979],
['nbs-iscc-b:cinnamomeus.77','cinnamomeus','cinnamomeus',[130,102,68],'826644',8545860],
['nbs-iscc-b:citrinoviridis.102','citrinoviridis','citrino-viridis ',[185,180,89],'b9b459',12170329],
['nbs-iscc-b:citrinoviridis.103','citrinoviridis','citrino-viridis ',[152,148,62],'98943e',9999422],
['nbs-iscc-b:citrinoviridis.106','citrinoviridis','citrino-viridis ',[134,126,54],'867e36',8814134],
['nbs-iscc-b:corallinus.3','corallinus','corallinus ',[228,113,122],'e4717a',14971258],
['nbs-iscc-b:corallinus.15','corallinus','corallinus ',[171,78,82],'ab4e52',11226706],
['nbs-iscc-b:corallinus.35','corallinus','corallinus ',[217,96,59],'d9603b',14245947],
['nbs-iscc-b:corallinus.37','corallinus','corallinus ',[203,109,81],'cb6d51',13331793],
['nbs-iscc-b:croceus.28','croceus','croceus',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-b:croceus.53','croceus','croceus',[217,144,88],'d99058',14258264],
['nbs-iscc-b:croceus.70','croceus','croceus',[251,201,127],'fbc97f',16501119],
['nbs-iscc-b:croceus.71','croceus','croceus',[227,168,87],'e3a857',14919767],
['nbs-iscc-b:croceus.73','croceus','croceus',[250,214,165],'fad6a5',16438949],
['nbs-iscc-b:cyaneus.169','cyaneus','cyaneus',[0,119,145],'007791',30609],
['nbs-iscc-b:cyaneus.173','cyaneus','cyaneus',[54,117,136],'367588',3569032],
['nbs-iscc-b:cyaneus.178','cyaneus','cyaneus',[0,103,165],'0067a5',26533],
['nbs-iscc-b:cyaneus.182','cyaneus','cyaneus',[67,107,149],'436b95',4418453],
['nbs-iscc-b:electrinus.83','electrinus','electrinus',[250,218,94],'fada5e',16439902],
['nbs-iscc-b:electrinus.84','electrinus','electrinus',[212,175,55],'d4af37',13938487],
['nbs-iscc-b:electrinus.86','electrinus','electrinus',[248,222,126],'f8de7e',16309886],
['nbs-iscc-b:electrinus.87','electrinus','electrinus',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-b:ferrogriseus.113','ferrogriseus','ferro-griseus',[87,85,76],'57554c',5723468],
['nbs-iscc-b:flavovirens.99','flavovirens','flavo-virens',[190,183,46],'beb72e',12498734],
['nbs-iscc-b:flavovirens.101','flavovirens','flavo-virens',[234,230,121],'eae679',15394425],
['nbs-iscc-b:flavovirens.102','flavovirens','flavo-virens',[185,180,89],'b9b459',12170329],
['nbs-iscc-b:flavovirens.104','flavovirens','flavo-virens',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-b:flavovirens.116','flavovirens','flavo-virens',[189,218,87],'bdda57',12442199],
['nbs-iscc-b:flavovirens.117','flavovirens','flavo-virens',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-b:flavovirens.119','flavovirens','flavo-virens',[201,220,137],'c9dc89',13229193],
['nbs-iscc-b:flavovirens.120','flavovirens','flavo-virens',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-b:flavovirens.121','flavovirens','flavo-virens',[218,223,183],'dadfb7',14344119],
['nbs-iscc-b:flavus.83','flavus','flavus',[250,218,94],'fada5e',16439902],
['nbs-iscc-b:flavus.84','flavus','flavus',[212,175,55],'d4af37',13938487],
['nbs-iscc-b:flavus.86','flavus','flavus',[248,222,126],'f8de7e',16309886],
['nbs-iscc-b:flavus.98','flavus','flavus',[233,228,80],'e9e450',15328336],
['nbs-iscc-b:flavus.99','flavus','flavus',[190,183,46],'beb72e',12498734],
['nbs-iscc-b:fulvus.54','fulvus','fulvus',[174,105,56],'ae6938',11430200],
['nbs-iscc-b:fulvus.72','fulvus','fulvus',[190,138,61],'be8a3d',12487229],
['nbs-iscc-b:fulvus.74','fulvus','fulvus',[153,101,21],'996515',10052885],
['nbs-iscc-b:fumosus.10','fumosus','fumosus',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-b:fumosus.33','fumosus','fumosus',[194,172,153],'c2ac99',12758169],
['nbs-iscc-b:fumosus.79','fumosus','fumosus',[174,155,130],'ae9b82',11443074],
['nbs-iscc-b:fumosus.90','fumosus','fumosus',[194,178,128],'c2b280',12759680],
['nbs-iscc-b:fumosus.93','fumosus','fumosus',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-b:fumosus.263','fumosus','fumosus',[242,243,244],'f2f3f4',15922164],
['nbs-iscc-b:fumosus.264','fumosus','fumosus',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-b:fusconiger.23','fusconiger','fusco-niger',[92,80,79],'5c504f',6049871],
['nbs-iscc-b:fusconiger.62','fusconiger','fusco-niger',[62,50,44],'3e322c',4076076],
['nbs-iscc-b:fusconiger.64','fusconiger','fusco-niger',[91,80,79],'5b504f',5984335],
['nbs-iscc-b:fusconiger.81','fusconiger','fusco-niger',[72,60,50],'483c32',4734002],
['nbs-iscc-b:fusconiger.229','fusconiger','fusco-niger',[80,64,77],'50404d',5259341],
['nbs-iscc-b:fusconiger.234','fusconiger','fusco-niger',[93,85,91],'5d555b',6116699],
['nbs-iscc-b:fusconiger.260','fusconiger','fusco-niger',[56,21,44],'38152c',3675436],
['nbs-iscc-b:fusconiger.266','fusconiger','fusco-niger',[85,85,85],'555555',5592405],
['nbs-iscc-b:fuscus.19','fuscus','fuscus',[144,93,93],'905d5d',9461085],
['nbs-iscc-b:fuscus.23','fuscus','fuscus',[92,80,79],'5c504f',6049871],
['nbs-iscc-b:fuscus.64','fuscus','fuscus',[91,80,79],'5b504f',5984335],
['nbs-iscc-b:fuscus.229','fuscus','fuscus',[80,64,77],'50404d',5259341],
['nbs-iscc-b:fuscus.234','fuscus','fuscus',[93,85,91],'5d555b',6116699],
['nbs-iscc-b:glaucogriseus.122','glaucogriseus','glauco-griseus',[143,151,121],'8f9779',9410425],
['nbs-iscc-b:glaucogriseus.148','glaucogriseus','glauco-griseus',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-b:glaucogriseus.149','glaucogriseus','glauco-griseus',[141,163,153],'8da399',9282457],
['nbs-iscc-b:glaucogriseus.154','glaucogriseus','glauco-griseus',[178,190,181],'b2beb5',11714229],
['nbs-iscc-b:glaucogriseus.155','glaucogriseus','glauco-griseus',[125,137,132],'7d8984',8227204],
['nbs-iscc-b:glaucogriseus.185','glaucogriseus','glauco-griseus',[145,163,176],'91a3b0',9544624],
['nbs-iscc-b:glaucogriseus.186','glaucogriseus','glauco-griseus',[83,104,120],'536878',5466232],
['nbs-iscc-b:glaucogriseus.190','glaucogriseus','glauco-griseus',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-b:glaucogriseus.191','glaucogriseus','glauco-griseus',[129,135,139],'81878b',8488843],
['nbs-iscc-b:glaucovenetus.149','glaucovenetus','glauco-venetus ',[141,163,153],'8da399',9282457],
['nbs-iscc-b:glaucovenetus.150','glaucovenetus','glauco-venetus ',[94,113,106],'5e716a',6189418],
['nbs-iscc-b:glaucovenetus.163','glaucovenetus','glauco-venetus ',[102,173,164],'66ada4',6729124],
['nbs-iscc-b:glaucovenetus.164','glaucovenetus','glauco-venetus ',[49,120,115],'317873',3242099],
['nbs-iscc-b:glaucus.135','glaucus','glaucus',[147,197,146],'93c592',9684370],
['nbs-iscc-b:glaucus.148','glaucus','glaucus',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-b:glaucus.149','glaucus','glaucus',[141,163,153],'8da399',9282457],
['nbs-iscc-b:glaucus.163','glaucus','glaucus',[102,173,164],'66ada4',6729124],
['nbs-iscc-b:glaucus.171','glaucus','glaucus',[156,209,220],'9cd1dc',10277340],
['nbs-iscc-b:glaucus.184','glaucus','glaucus',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-b:griseocaeruleus.180','griseocaeruleus','griseo-caeruleus',[161,202,241],'a1caf1',10603249],
['nbs-iscc-b:griseocaeruleus.181','griseocaeruleus','griseo-caeruleus',[112,163,204],'70a3cc',7381964],
['nbs-iscc-b:griseocaeruleus.184','griseocaeruleus','griseo-caeruleus',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-b:griseocaeruleus.185','griseocaeruleus','griseo-caeruleus',[145,163,176],'91a3b0',9544624],
['nbs-iscc-b:griseocaeruleus.198','griseocaeruleus','griseo-caeruleus',[179,188,226],'b3bce2',11779298],
['nbs-iscc-b:griseocaeruleus.199','griseocaeruleus','griseo-caeruleus',[135,145,191],'8791bf',8884671],
['nbs-iscc-b:griseocaeruleus.202','griseocaeruleus','griseo-caeruleus',[192,200,225],'c0c8e1',12634337],
['nbs-iscc-b:griseocaeruleus.203','griseocaeruleus','griseo-caeruleus',[140,146,172],'8c92ac',9212588],
['nbs-iscc-b:griseochlorinus.101','griseochlorinus','griseo-chlorinus',[234,230,121],'eae679',15394425],
['nbs-iscc-b:griseochlorinus.102','griseochlorinus','griseo-chlorinus',[185,180,89],'b9b459',12170329],
['nbs-iscc-b:griseochlorinus.104','griseochlorinus','griseo-chlorinus',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-b:griseochlorinus.105','griseochlorinus','griseo-chlorinus',[185,181,125],'b9b57d',12170621],
['nbs-iscc-b:griseochlorinus.119','griseochlorinus','griseo-chlorinus',[201,220,137],'c9dc89',13229193],
['nbs-iscc-b:griseochlorinus.120','griseochlorinus','griseo-chlorinus',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-b:griseochlorinus.121','griseochlorinus','griseo-chlorinus',[218,223,183],'dadfb7',14344119],
['nbs-iscc-b:griseolavendulus.199','griseolavendulus','griseo-lavendulus',[135,145,191],'8791bf',8884671],
['nbs-iscc-b:griseolavendulus.202','griseolavendulus','griseo-lavendulus',[192,200,225],'c0c8e1',12634337],
['nbs-iscc-b:griseolavendulus.203','griseolavendulus','griseo-lavendulus',[140,146,172],'8c92ac',9212588],
['nbs-iscc-b:griseolavendulus.213','griseolavendulus','griseo-lavendulus',[196,195,221],'c4c3dd',12895197],
['nbs-iscc-b:griseolavendulus.214','griseolavendulus','griseo-lavendulus',[150,144,171],'9690ab',9867435],
['nbs-iscc-b:griseolavendulus.215','griseolavendulus','griseo-lavendulus',[85,76,105],'554c69',5590121],
['nbs-iscc-b:griseolavendulus.222','griseolavendulus','griseo-lavendulus',[182,149,192],'b695c0',11965888],
['nbs-iscc-b:griseolavendulus.227','griseolavendulus','griseo-lavendulus',[170,152,169],'aa98a9',11180201],
['nbs-iscc-b:griseolavendulus.228','griseolavendulus','griseo-lavendulus',[121,104,120],'796878',7956600],
['nbs-iscc-b:griseolavendulus.233','griseolavendulus','griseo-lavendulus',[139,133,137],'8b8589',9143689],
['nbs-iscc-b:griseolavendulus.253','griseolavendulus','griseo-lavendulus',[195,166,177],'c3a6b1',12822193],
['nbs-iscc-b:griseolazulinus.185','griseolazulinus','griseo-lazulinus',[145,163,176],'91a3b0',9544624],
['nbs-iscc-b:griseolazulinus.186','griseolazulinus','griseo-lazulinus',[83,104,120],'536878',5466232],
['nbs-iscc-b:griseolazulinus.203','griseolazulinus','griseo-lazulinus',[140,146,172],'8c92ac',9212588],
['nbs-iscc-b:griseolazulinus.204','griseolazulinus','griseo-lazulinus',[76,81,109],'4c516d',5001581],
['nbs-iscc-b:griseolilacinus.5','griseolilacinus','griseo-lilacinus',[222,165,164],'dea5a4',14591396],
['nbs-iscc-b:griseolilacinus.244','griseolilacinus','griseo-lilacinus',[170,138,158],'aa8a9e',11176606],
['nbs-iscc-b:griseolilacinus.253','griseolilacinus','griseo-lilacinus',[195,166,177],'c3a6b1',12822193],
['nbs-iscc-b:griseolineus.177','griseolineus','griseo-lineus',[73,151,208],'4997d0',4822992],
['nbs-iscc-b:griseolineus.180','griseolineus','griseo-lineus',[161,202,241],'a1caf1',10603249],
['nbs-iscc-b:griseolineus.181','griseolineus','griseo-lineus',[112,163,204],'70a3cc',7381964],
['nbs-iscc-b:griseolineus.199','griseolineus','griseo-lineus',[135,145,191],'8791bf',8884671],
['nbs-iscc-b:griseolineus.202','griseolineus','griseo-lineus',[192,200,225],'c0c8e1',12634337],
['nbs-iscc-b:griseolineus.210','griseolineus','griseo-lineus',[140,130,182],'8c82b6',9208502],
['nbs-iscc-b:griseoolivaceous.109','griseoolivaceous','griseo-olivaceous',[140,135,103],'8c8767',9209703],
['nbs-iscc-b:griseoolivaceous.112','griseoolivaceous','griseo-olivaceous',[138,135,118],'8a8776',9078646],
['nbs-iscc-b:griseoolivaceous.113','griseoolivaceous','griseo-olivaceous',[87,85,76],'57554c',5723468],
['nbs-iscc-b:griseoroseus.2','griseoroseus','griseo-roseus',[234,147,153],'ea9399',15373209],
['nbs-iscc-b:griseoroseus.3','griseoroseus','griseo-roseus',[228,113,122],'e4717a',14971258],
['nbs-iscc-b:griseoroseus.4','griseoroseus','griseo-roseus',[249,204,202],'f9ccca',16370890],
['nbs-iscc-b:griseoroseus.5','griseoroseus','griseo-roseus',[222,165,164],'dea5a4',14591396],
['nbs-iscc-b:griseoroseus.6','griseoroseus','griseo-roseus',[192,128,129],'c08081',12615809],
['nbs-iscc-b:griseoroseus.7','griseoroseus','griseo-roseus',[234,216,215],'ead8d7',15390935],
['nbs-iscc-b:griseoroseus.8','griseoroseus','griseo-roseus',[196,174,173],'c4aead',12889773],
['nbs-iscc-b:griseoroseus.248','griseoroseus','griseo-roseus',[222,111,161],'de6fa1',14577569],
['nbs-iscc-b:griseoroseus.249','griseoroseus','griseo-roseus',[239,187,204],'efbbcc',15711180],
['nbs-iscc-b:griseoroseus.250','griseoroseus','griseo-roseus',[213,151,174],'d597ae',13997998],
['nbs-iscc-b:griseoroseus.251','griseoroseus','griseo-roseus',[193,126,145],'c17e91',12680849],
['nbs-iscc-b:griseosepiaceus.45','griseosepiaceus','griseo-sepiaceus',[151,127,115],'977f73',9928563],
['nbs-iscc-b:griseosepiaceus.60','griseosepiaceus','griseo-sepiaceus',[149,128,112],'958070',9797744],
['nbs-iscc-b:griseosepiaceus.61','griseosepiaceus','griseo-sepiaceus',[99,81,71],'635147',6508871],
['nbs-iscc-b:griseosepiaceus.64','griseosepiaceus','griseo-sepiaceus',[91,80,79],'5b504f',5984335],
['nbs-iscc-b:griseosepiaceus.76','griseosepiaceus','griseo-sepiaceus',[193,154,107],'c19a6b',12687979],
['nbs-iscc-b:griseosepiaceus.77','griseosepiaceus','griseo-sepiaceus',[130,102,68],'826644',8545860],
['nbs-iscc-b:griseosepiaceus.79','griseosepiaceus','griseo-sepiaceus',[174,155,130],'ae9b82',11443074],
['nbs-iscc-b:griseosepiaceus.80','griseosepiaceus','griseo-sepiaceus',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-b:griseovenetus.162','griseovenetus','griseo-venetus',[150,222,209],'96ded1',9887441],
['nbs-iscc-b:griseovenetus.163','griseovenetus','griseo-venetus',[102,173,164],'66ada4',6729124],
['nbs-iscc-b:griseoviolaceus.203','griseoviolaceus','griseo-violaceus',[140,146,172],'8c92ac',9212588],
['nbs-iscc-b:griseoviolaceus.204','griseoviolaceus','griseo-violaceus',[76,81,109],'4c516d',5001581],
['nbs-iscc-b:griseoviolaceus.211','griseoviolaceus','griseo-violaceus',[96,78,129],'604e81',6311553],
['nbs-iscc-b:griseoviolaceus.214','griseoviolaceus','griseo-violaceus',[150,144,171],'9690ab',9867435],
['nbs-iscc-b:griseoviolaceus.215','griseoviolaceus','griseo-violaceus',[85,76,105],'554c69',5590121],
['nbs-iscc-b:griseoviridis.131','griseoviridis','griseo-viridis',[68,148,74],'44944a',4494410],
['nbs-iscc-b:griseoviridis.135','griseoviridis','griseo-viridis',[147,197,146],'93c592',9684370],
['nbs-iscc-b:griseoviridis.136','griseoviridis','griseo-viridis',[103,146,103],'679267',6787687],
['nbs-iscc-b:griseoviridis.144','griseoviridis','griseo-viridis',[106,171,142],'6aab8e',6990734],
['nbs-iscc-b:herbaceus.102','herbaceus','herbaceus',[185,180,89],'b9b459',12170329],
['nbs-iscc-b:herbaceus.103','herbaceus','herbaceus',[152,148,62],'98943e',9999422],
['nbs-iscc-b:herbaceus.106','herbaceus','herbaceus',[134,126,54],'867e36',8814134],
['nbs-iscc-b:herbaceus.107','herbaceus','herbaceus',[102,93,30],'665d1e',6708510],
['nbs-iscc-b:herbaceus.117','herbaceus','herbaceus',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-b:herbaceus.120','herbaceus','herbaceus',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-b:herbaceus.122','herbaceus','herbaceus',[143,151,121],'8f9779',9410425],
['nbs-iscc-b:herbaceus.125','herbaceus','herbaceus',[74,93,35],'4a5d23',4873507],
['nbs-iscc-b:herbaceus.127','herbaceus','herbaceus',[81,87,68],'515744',5330756],
['nbs-iscc-b:herbaceus.136','herbaceus','herbaceus',[103,146,103],'679267',6787687],
['nbs-iscc-b:herbaceus.137','herbaceus','herbaceus',[53,94,59],'355e3b',3497531],
['nbs-iscc-b:herbaceus.145','herbaceus','herbaceus',[59,120,97],'3b7861',3897441],
['nbs-iscc-b:herbaceus.150','herbaceus','herbaceus',[94,113,106],'5e716a',6189418],
['nbs-iscc-b:herbaceus.151','herbaceus','herbaceus',[58,75,71],'3a4b47',3820359],
['nbs-iscc-b:hinnuleus.45','hinnuleus','hinnuleus',[151,127,115],'977f73',9928563],
['nbs-iscc-b:hinnuleus.57','hinnuleus','hinnuleus',[166,123,91],'a67b5b',10910555],
['nbs-iscc-b:hinnuleus.58','hinnuleus','hinnuleus',[111,78,55],'6f4e37',7294519],
['nbs-iscc-b:hinnuleus.60','hinnuleus','hinnuleus',[149,128,112],'958070',9797744],
['nbs-iscc-b:hinnuleus.61','hinnuleus','hinnuleus',[99,81,71],'635147',6508871],
['nbs-iscc-b:incarnatus.3','incarnatus','incarnatus',[228,113,122],'e4717a',14971258],
['nbs-iscc-b:incarnatus.5','incarnatus','incarnatus',[222,165,164],'dea5a4',14591396],
['nbs-iscc-b:incarnatus.26','incarnatus','incarnatus',[248,131,121],'f88379',16286585],
['nbs-iscc-b:incarnatus.27','incarnatus','incarnatus',[230,103,97],'e66761',15099745],
['nbs-iscc-b:incarnatus.28','incarnatus','incarnatus',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-b:incarnatus.29','incarnatus','incarnatus',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-b:isabellinus.77','isabellinus','isabellinus',[130,102,68],'826644',8545860],
['nbs-iscc-b:isabellinus.91','isabellinus','isabellinus',[161,143,96],'a18f60',10588000],
['nbs-iscc-b:isabellinus.94','isabellinus','isabellinus',[150,113,23],'967117',9859351],
['nbs-iscc-b:latericius.43','latericius','latericius',[121,68,59],'79443b',7947323],
['nbs-iscc-b:latericius.45','latericius','latericius',[151,127,115],'977f73',9928563],
['nbs-iscc-b:latericius.46','latericius','latericius',[103,76,71],'674c47',6769735],
['nbs-iscc-b:latericius.57','latericius','latericius',[166,123,91],'a67b5b',10910555],
['nbs-iscc-b:latericius.58','latericius','latericius',[111,78,55],'6f4e37',7294519],
['nbs-iscc-b:latericius.61','latericius','latericius',[99,81,71],'635147',6508871],
['nbs-iscc-b:lavendulogriseus.185','lavendulogriseus','lavendulo-griseus ',[145,163,176],'91a3b0',9544624],
['nbs-iscc-b:lavendulogriseus.190','lavendulogriseus','lavendulo-griseus ',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-b:lavendulogriseus.191','lavendulogriseus','lavendulo-griseus ',[129,135,139],'81878b',8488843],
['nbs-iscc-b:lavendulogriseus.232','lavendulogriseus','lavendulo-griseus ',[191,185,189],'bfb9bd',12564925],
['nbs-iscc-b:lavendulogriseus.233','lavendulogriseus','lavendulo-griseus ',[139,133,137],'8b8589',9143689],
['nbs-iscc-b:lavendulogriseus.265','lavendulogriseus','lavendulo-griseus ',[132,132,130],'848482',8684674],
['nbs-iscc-b:lavendulus.195','lavendulus','lavendulus',[108,121,184],'6c79b8',7109048],
['nbs-iscc-b:lavendulus.198','lavendulus','lavendulus',[179,188,226],'b3bce2',11779298],
['nbs-iscc-b:lavendulus.199','lavendulus','lavendulus',[135,145,191],'8791bf',8884671],
['nbs-iscc-b:lavendulus.206','lavendulus','lavendulus',[126,115,184],'7e73b8',8287160],
['nbs-iscc-b:lavendulus.209','lavendulus','lavendulus',[220,208,255],'dcd0ff',14471423],
['nbs-iscc-b:lavendulus.210','lavendulus','lavendulus',[140,130,182],'8c82b6',9208502],
['nbs-iscc-b:lavendulus.213','lavendulus','lavendulus',[196,195,221],'c4c3dd',12895197],
['nbs-iscc-b:lavendulus.214','lavendulus','lavendulus',[150,144,171],'9690ab',9867435],
['nbs-iscc-b:lavendulus.226','lavendulus','lavendulus',[214,202,221],'d6cadd',14076637],
['nbs-iscc-b:lazulinoardesiacus.183','lazulinoardesiacus','lazulino-ardesiacus',[0,48,78],'00304e',12366],
['nbs-iscc-b:lazulinoardesiacus.186','lazulinoardesiacus','lazulino-ardesiacus',[83,104,120],'536878',5466232],
['nbs-iscc-b:lazulinoardesiacus.187','lazulinoardesiacus','lazulino-ardesiacus',[54,69,79],'36454f',3556687],
['nbs-iscc-b:lazulinoardesiacus.204','lazulinoardesiacus','lazulino-ardesiacus',[76,81,109],'4c516d',5001581],
['nbs-iscc-b:lazulinus.176','lazulinus','lazulinus',[0,161,194],'00a1c2',41410],
['nbs-iscc-b:lazulinus.177','lazulinus','lazulinus',[73,151,208],'4997d0',4822992],
['nbs-iscc-b:lazulinus.178','lazulinus','lazulinus',[0,103,165],'0067a5',26533],
['nbs-iscc-b:lazulinus.182','lazulinus','lazulinus',[67,107,149],'436b95',4418453],
['nbs-iscc-b:lazulinus.194','lazulinus','lazulinus',[48,38,122],'30267a',3155578],
['nbs-iscc-b:lazulinus.195','lazulinus','lazulinus',[108,121,184],'6c79b8',7109048],
['nbs-iscc-b:lazulinus.196','lazulinus','lazulinus',[84,90,167],'545aa7',5528231],
['nbs-iscc-b:lilacinus.206','lilacinus','lilacinus',[126,115,184],'7e73b8',8287160],
['nbs-iscc-b:lilacinus.210','lilacinus','lilacinus',[140,130,182],'8c82b6',9208502],
['nbs-iscc-b:lilacinus.217','lilacinus','lilacinus',[211,153,230],'d399e6',13867494],
['nbs-iscc-b:lilacinus.222','lilacinus','lilacinus',[182,149,192],'b695c0',11965888],
['nbs-iscc-b:lilacinus.223','lilacinus','lilacinus',[134,96,142],'86608e',8806542],
['nbs-iscc-b:lilacinus.226','lilacinus','lilacinus',[214,202,221],'d6cadd',14076637],
['nbs-iscc-b:lilacinus.227','lilacinus','lilacinus',[170,152,169],'aa98a9',11180201],
['nbs-iscc-b:lilacinus.237','lilacinus','lilacinus',[158,79,136],'9e4f88',10375048],
['nbs-iscc-b:lilacinus.240','lilacinus','lilacinus',[183,132,167],'b784a7',12027047],
['nbs-iscc-b:lilacinus.241','lilacinus','lilacinus',[145,92,131],'915c83',9526403],
['nbs-iscc-b:lilacinus.244','lilacinus','lilacinus',[170,138,158],'aa8a9e',11176606],
['nbs-iscc-b:lilacinus.248','lilacinus','lilacinus',[222,111,161],'de6fa1',14577569],
['nbs-iscc-b:lilacinus.249','lilacinus','lilacinus',[239,187,204],'efbbcc',15711180],
['nbs-iscc-b:lilacinus.250','lilacinus','lilacinus',[213,151,174],'d597ae',13997998],
['nbs-iscc-b:lilacinus.252','lilacinus','lilacinus',[232,204,215],'e8ccd7',15256791],
['nbs-iscc-b:lilacinus.253','lilacinus','lilacinus',[195,166,177],'c3a6b1',12822193],
['nbs-iscc-b:lineus.177','lineus','lineus',[73,151,208],'4997d0',4822992],
['nbs-iscc-b:lineus.180','lineus','lineus',[161,202,241],'a1caf1',10603249],
['nbs-iscc-b:lineus.181','lineus','lineus',[112,163,204],'70a3cc',7381964],
['nbs-iscc-b:lineus.184','lineus','lineus',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-b:lineus.185','lineus','lineus',[145,163,176],'91a3b0',9544624],
['nbs-iscc-b:lineus.195','lineus','lineus',[108,121,184],'6c79b8',7109048],
['nbs-iscc-b:lineus.198','lineus','lineus',[179,188,226],'b3bce2',11779298],
['nbs-iscc-b:lineus.199','lineus','lineus',[135,145,191],'8791bf',8884671],
['nbs-iscc-b:lineus.202','lineus','lineus',[192,200,225],'c0c8e1',12634337],
['nbs-iscc-b:lineus.203','lineus','lineus',[140,146,172],'8c92ac',9212588],
['nbs-iscc-b:lividopurpureus.223','lividopurpureus','livido-purpureus',[134,96,142],'86608e',8806542],
['nbs-iscc-b:lividopurpureus.240','lividopurpureus','livido-purpureus',[183,132,167],'b784a7',12027047],
['nbs-iscc-b:lividopurpureus.241','lividopurpureus','livido-purpureus',[145,92,131],'915c83',9526403],
['nbs-iscc-b:lividopurpureus.245','lividopurpureus','livido-purpureus',[131,100,121],'836479',8610937],
['nbs-iscc-b:lividopurpureus.258','lividopurpureus','livido-purpureus',[168,81,110],'a8516e',11030894],
['nbs-iscc-b:lividopurpureus.262','lividopurpureus','livido-purpureus',[145,95,109],'915f6d',9527149],
['nbs-iscc-b:lividoruber.258','lividoruber','livido-ruber',[168,81,110],'a8516e',11030894],
['nbs-iscc-b:lividoruber.262','lividoruber','livido-ruber',[145,95,109],'915f6d',9527149],
['nbs-iscc-b:lividoviolaceus.207','lividoviolaceus','livido-violaceus',[96,78,151],'604e97',6311575],
['nbs-iscc-b:lividoviolaceus.210','lividoviolaceus','livido-violaceus',[140,130,182],'8c82b6',9208502],
['nbs-iscc-b:lividoviolaceus.211','lividoviolaceus','livido-violaceus',[96,78,129],'604e81',6311553],
['nbs-iscc-b:lividoviolaceus.218','lividoviolaceus','livido-violaceus',[135,86,146],'875692',8869522],
['nbs-iscc-b:lividoviolaceus.223','lividoviolaceus','livido-violaceus',[134,96,142],'86608e',8806542],
['nbs-iscc-b:lividoviolaceus.228','lividoviolaceus','livido-violaceus',[121,104,120],'796878',7956600],
['nbs-iscc-b:luteolus.70','luteolus','luteolus',[251,201,127],'fbc97f',16501119],
['nbs-iscc-b:luteolus.71','luteolus','luteolus',[227,168,87],'e3a857',14919767],
['nbs-iscc-b:luteolus.83','luteolus','luteolus',[250,218,94],'fada5e',16439902],
['nbs-iscc-b:luteolus.84','luteolus','luteolus',[212,175,55],'d4af37',13938487],
['nbs-iscc-b:luteolus.86','luteolus','luteolus',[248,222,126],'f8de7e',16309886],
['nbs-iscc-b:luteolus.87','luteolus','luteolus',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-b:luteus.50','luteus','luteus',[237,135,45],'ed872d',15566637],
['nbs-iscc-b:luteus.68','luteus','luteus',[234,162,33],'eaa221',15376929],
['nbs-iscc-b:luteus.82','luteus','luteus',[243,195,0],'f3c300',15975168],
['nbs-iscc-b:luteus.84','luteus','luteus',[212,175,55],'d4af37',13938487],
['nbs-iscc-b:malachiteus.135','malachiteus','malachiteus',[147,197,146],'93c592',9684370],
['nbs-iscc-b:malachiteus.136','malachiteus','malachiteus',[103,146,103],'679267',6787687],
['nbs-iscc-b:malachiteus.145','malachiteus','malachiteus',[59,120,97],'3b7861',3897441],
['nbs-iscc-b:malvaceus.217','malvaceus','malvaceus',[211,153,230],'d399e6',13867494],
['nbs-iscc-b:malvaceus.218','malvaceus','malvaceus',[135,86,146],'875692',8869522],
['nbs-iscc-b:malvaceus.222','malvaceus','malvaceus',[182,149,192],'b695c0',11965888],
['nbs-iscc-b:malvaceus.237','malvaceus','malvaceus',[158,79,136],'9e4f88',10375048],
['nbs-iscc-b:malvaceus.248','malvaceus','malvaceus',[222,111,161],'de6fa1',14577569],
['nbs-iscc-b:melleus.71','melleus','melleus',[227,168,87],'e3a857',14919767],
['nbs-iscc-b:melleus.74','melleus','melleus',[153,101,21],'996515',10052885],
['nbs-iscc-b:melleus.76','melleus','melleus',[193,154,107],'c19a6b',12687979],
['nbs-iscc-b:melleus.77','melleus','melleus',[130,102,68],'826644',8545860],
['nbs-iscc-b:melleus.87','melleus','melleus',[201,174,93],'c9ae5d',13217373],
['nbs-iscc-b:melleus.88','melleus','melleus',[171,145,68],'ab9144',11243844],
['nbs-iscc-b:melleus.94','melleus','melleus',[150,113,23],'967117',9859351],
['nbs-iscc-b:miniatus.34','miniatus','miniatus ',[226,88,34],'e25822',14833698],
['nbs-iscc-b:miniatus.35','miniatus','miniatus ',[217,96,59],'d9603b',14245947],
['nbs-iscc-b:miniatus.37','miniatus','miniatus ',[203,109,81],'cb6d51',13331793],
['nbs-iscc-b:miniatus.38','miniatus','miniatus ',[158,71,50],'9e4732',10372914],
['nbs-iscc-b:murinus.64','murinus','murinus ',[91,80,79],'5b504f',5984335],
['nbs-iscc-b:murinus.78','murinus','murinus ',[75,54,33],'4b3621',4929057],
['nbs-iscc-b:murinus.265','murinus','murinus ',[132,132,130],'848482',8684674],
['nbs-iscc-b:murinus.266','murinus','murinus ',[85,85,85],'555555',5592405],
['nbs-iscc-b:niger.64','niger','niger',[91,80,79],'5b504f',5984335],
['nbs-iscc-b:niger.113','niger','niger',[87,85,76],'57554c',5723468],
['nbs-iscc-b:niger.187','niger','niger',[54,69,79],'36454f',3556687],
['nbs-iscc-b:niger.229','niger','niger',[80,64,77],'50404d',5259341],
['nbs-iscc-b:niger.234','niger','niger',[93,85,91],'5d555b',6116699],
['nbs-iscc-b:niger.266','niger','niger',[85,85,85],'555555',5592405],
['nbs-iscc-b:niger.267','niger','niger',[34,34,34],'222222',2236962],
['nbs-iscc-b:obscurovirens.107','obscurovirens','obscuro-virens',[102,93,30],'665d1e',6708510],
['nbs-iscc-b:obscurovirens.110','obscurovirens','obscuro-virens',[91,88,66],'5b5842',5986370],
['nbs-iscc-b:obscurovirens.125','obscurovirens','obscuro-virens',[74,93,35],'4a5d23',4873507],
['nbs-iscc-b:obscurovirens.127','obscurovirens','obscuro-virens',[81,87,68],'515744',5330756],
['nbs-iscc-b:obscurovirens.145','obscurovirens','obscuro-virens',[59,120,97],'3b7861',3897441],
['nbs-iscc-b:obscurovirens.146','obscurovirens','obscuro-virens',[27,77,62],'1b4d3e',1789246],
['nbs-iscc-b:obscurovirens.150','obscurovirens','obscuro-virens',[94,113,106],'5e716a',6189418],
['nbs-iscc-b:obscurovirens.151','obscurovirens','obscuro-virens',[58,75,71],'3a4b47',3820359],
['nbs-iscc-b:ochraceus.28','ochraceus','ochraceus',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-b:ochraceus.29','ochraceus','ochraceus',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-b:ochraceus.50','ochraceus','ochraceus',[237,135,45],'ed872d',15566637],
['nbs-iscc-b:ochraceus.53','ochraceus','ochraceus',[217,144,88],'d99058',14258264],
['nbs-iscc-b:ochraceus.68','ochraceus','ochraceus',[234,162,33],'eaa221',15376929],
['nbs-iscc-b:ochraceus.69','ochraceus','ochraceus',[201,133,0],'c98500',13206784],
['nbs-iscc-b:ochraceus.72','ochraceus','ochraceus',[190,138,61],'be8a3d',12487229],
['nbs-iscc-b:olivaceobubalinus.89','olivaceobubalinus','olivaceo-bubalinus',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-b:olivaceobubalinus.90','olivaceobubalinus','olivaceo-bubalinus',[194,178,128],'c2b280',12759680],
['nbs-iscc-b:olivaceocitrinus.85','olivaceocitrinus','olivaceo-citrinus',[175,141,19],'af8d13',11504915],
['nbs-iscc-b:olivaceocitrinus.88','olivaceocitrinus','olivaceo-citrinus',[171,145,68],'ab9144',11243844],
['nbs-iscc-b:olivaceocitrinus.94','olivaceocitrinus','olivaceo-citrinus',[150,113,23],'967117',9859351],
['nbs-iscc-b:olivaceocitrinus.95','olivaceocitrinus','olivaceo-citrinus',[108,84,30],'6c541e',7099422],
['nbs-iscc-b:olivaceocitrinus.103','olivaceocitrinus','olivaceo-citrinus',[152,148,62],'98943e',9999422],
['nbs-iscc-b:olivaceocitrinus.106','olivaceocitrinus','olivaceo-citrinus',[134,126,54],'867e36',8814134],
['nbs-iscc-b:olivaceocitrinus.107','olivaceocitrinus','olivaceo-citrinus',[102,93,30],'665d1e',6708510],
['nbs-iscc-b:olivaceogriseus.110','olivaceogriseus','olivaceo-griseus',[91,88,66],'5b5842',5986370],
['nbs-iscc-b:olivaceogriseus.112','olivaceogriseus','olivaceo-griseus',[138,135,118],'8a8776',9078646],
['nbs-iscc-b:olivaceogriseus.113','olivaceogriseus','olivaceo-griseus',[87,85,76],'57554c',5723468],
['nbs-iscc-b:olivaceoniger.113','olivaceoniger','olivaceo-niger',[87,85,76],'57554c',5723468],
['nbs-iscc-b:olivaceoniger.127','olivaceoniger','olivaceo-niger',[81,87,68],'515744',5330756],
['nbs-iscc-b:olivaceoniger.128','olivaceoniger','olivaceo-niger',[49,54,43],'31362b',3225131],
['nbs-iscc-b:olivaceoniger.156','olivaceoniger','olivaceo-niger',[78,87,85],'4e5755',5134165],
['nbs-iscc-b:olivaceoniger.266','olivaceoniger','olivaceo-niger',[85,85,85],'555555',5592405],
['nbs-iscc-b:olivaceus.80','olivaceus','olivaceus ',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-b:olivaceus.81','olivaceus','olivaceus ',[72,60,50],'483c32',4734002],
['nbs-iscc-b:olivaceus.88','olivaceus','olivaceus ',[171,145,68],'ab9144',11243844],
['nbs-iscc-b:olivaceus.91','olivaceus','olivaceus ',[161,143,96],'a18f60',10588000],
['nbs-iscc-b:olivaceus.94','olivaceus','olivaceus ',[150,113,23],'967117',9859351],
['nbs-iscc-b:olivaceus.95','olivaceus','olivaceus ',[108,84,30],'6c541e',7099422],
['nbs-iscc-b:olivaceus.103','olivaceus','olivaceus ',[152,148,62],'98943e',9999422],
['nbs-iscc-b:olivaceus.106','olivaceus','olivaceus ',[134,126,54],'867e36',8814134],
['nbs-iscc-b:olivaceus.107','olivaceus','olivaceus ',[102,93,30],'665d1e',6708510],
['nbs-iscc-b:olivaceus.110','olivaceus','olivaceus ',[91,88,66],'5b5842',5986370],
['nbs-iscc-b:persicinus.26','persicinus','persicinus',[248,131,121],'f88379',16286585],
['nbs-iscc-b:persicinus.28','persicinus','persicinus',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-b:persicinus.29','persicinus','persicinus',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-b:persicinus.35','persicinus','persicinus',[217,96,59],'d9603b',14245947],
['nbs-iscc-b:persicinus.37','persicinus','persicinus',[203,109,81],'cb6d51',13331793],
['nbs-iscc-b:persicinus.49','persicinus','persicinus',[253,148,63],'fd943f',16618559],
['nbs-iscc-b:persicinus.52','persicinus','persicinus',[250,181,127],'fab57f',16430463],
['nbs-iscc-b:persicinus.53','persicinus','persicinus',[217,144,88],'d99058',14258264],
['nbs-iscc-b:pistaceus.120','pistaceus','pistaceus',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-b:pistaceus.122','pistaceus','pistaceus',[143,151,121],'8f9779',9410425],
['nbs-iscc-b:pistaceus.135','pistaceus','pistaceus',[147,197,146],'93c592',9684370],
['nbs-iscc-b:pistaceus.136','pistaceus','pistaceus',[103,146,103],'679267',6787687],
['nbs-iscc-b:pistaceus.144','pistaceus','pistaceus',[106,171,142],'6aab8e',6990734],
['nbs-iscc-b:pistaceus.149','pistaceus','pistaceus',[141,163,153],'8da399',9282457],
['nbs-iscc-b:pistaceus.150','pistaceus','pistaceus',[94,113,106],'5e716a',6189418],
['nbs-iscc-b:plumbeoniger.192','plumbeoniger','plumbeo-niger',[81,88,94],'51585e',5331038],
['nbs-iscc-b:plumbeus.185','plumbeus','plumbeus',[145,163,176],'91a3b0',9544624],
['nbs-iscc-b:plumbeus.186','plumbeus','plumbeus',[83,104,120],'536878',5466232],
['nbs-iscc-b:plumbeus.191','plumbeus','plumbeus',[129,135,139],'81878b',8488843],
['nbs-iscc-b:plumbeus.192','plumbeus','plumbeus',[81,88,94],'51585e',5331038],
['nbs-iscc-b:prasinus.119','prasinus','prasinus',[201,220,137],'c9dc89',13229193],
['nbs-iscc-b:prasinus.134','prasinus','prasinus',[182,229,175],'b6e5af',11986351],
['nbs-iscc-b:prasinus.135','prasinus','prasinus',[147,197,146],'93c592',9684370],
['nbs-iscc-b:prasinus.143','prasinus','prasinus',[142,209,178],'8ed1b2',9359794],
['nbs-iscc-b:prasinus.148','prasinus','prasinus',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-b:primulinus.89','primulinus','primulinus',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-b:primulinus.101','primulinus','primulinus',[234,230,121],'eae679',15394425],
['nbs-iscc-b:primulinus.102','primulinus','primulinus',[185,180,89],'b9b459',12170329],
['nbs-iscc-b:purpureoardesiacus.228','purpureoardesiacus','purpureo-ardesiacus',[121,104,120],'796878',7956600],
['nbs-iscc-b:purpureoardesiacus.229','purpureoardesiacus','purpureo-ardesiacus',[80,64,77],'50404d',5259341],
['nbs-iscc-b:purpureoardesiacus.242','purpureoardesiacus','purpureo-ardesiacus',[93,57,84],'5d3954',6109524],
['nbs-iscc-b:purpureoardesiacus.245','purpureoardesiacus','purpureo-ardesiacus',[131,100,121],'836479',8610937],
['nbs-iscc-b:purpureogriseus.228','purpureogriseus','purpureo-griseus',[121,104,120],'796878',7956600],
['nbs-iscc-b:purpureogriseus.233','purpureogriseus','purpureo-griseus',[139,133,137],'8b8589',9143689],
['nbs-iscc-b:purpureogriseus.234','purpureogriseus','purpureo-griseus',[93,85,91],'5d555b',6116699],
['nbs-iscc-b:purpureogriseus.265','purpureogriseus','purpureo-griseus',[132,132,130],'848482',8684674],
['nbs-iscc-b:purpureus.216','purpureus','purpureus',[154,78,174],'9a4eae',10112686],
['nbs-iscc-b:purpureus.218','purpureus','purpureus',[135,86,146],'875692',8869522],
['nbs-iscc-b:purpureus.219','purpureus','purpureus',[96,47,107],'602f6b',6303595],
['nbs-iscc-b:purpureus.223','purpureus','purpureus',[134,96,142],'86608e',8806542],
['nbs-iscc-b:purpureus.236','purpureus','purpureus',[135,0,116],'870074',8847476],
['nbs-iscc-b:purpureus.237','purpureus','purpureus',[158,79,136],'9e4f88',10375048],
['nbs-iscc-b:purpureus.238','purpureus','purpureus',[112,41,99],'702963',7350627],
['nbs-iscc-b:purpureus.255','purpureus','purpureus',[179,68,108],'b3446c',11748460],
['nbs-iscc-b:purpureus.256','purpureus','purpureus',[120,24,74],'78184a',7870538],
['nbs-iscc-b:purpureus.258','purpureus','purpureus',[168,81,110],'a8516e',11030894],
['nbs-iscc-b:purpureus.262','purpureus','purpureus',[145,95,109],'915f6d',9527149],
['nbs-iscc-b:roseobubalinus.28','roseobubalinus','roseo-bubalinus',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-b:roseobubalinus.29','roseobubalinus','roseo-bubalinus',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-b:roseobubalinus.31','roseobubalinus','roseo-bubalinus',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-b:roseobubalinus.32','roseobubalinus','roseo-bubalinus',[199,173,163],'c7ada3',13086115],
['nbs-iscc-b:roseobubalinus.33','roseobubalinus','roseo-bubalinus',[194,172,153],'c2ac99',12758169],
['nbs-iscc-b:roseobubalinus.73','roseobubalinus','roseo-bubalinus',[250,214,165],'fad6a5',16438949],
['nbs-iscc-b:roseovinosus.28','roseovinosus','roseo-vinosus',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-b:roseovinosus.29','roseovinosus','roseo-vinosus',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-b:roseovinosus.31','roseovinosus','roseo-vinosus',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-b:roseovinosus.32','roseovinosus','roseo-vinosus',[199,173,163],'c7ada3',13086115],
['nbs-iscc-b:roseovinosus.39','roseovinosus','roseo-vinosus',[180,116,94],'b4745e',11826270],
['nbs-iscc-b:roseovinosus.42','roseovinosus','roseo-vinosus',[168,124,109],'a87c6d',11041901],
['nbs-iscc-b:roseus.1','roseus','roseus',[255,181,186],'ffb5ba',16758202],
['nbs-iscc-b:roseus.2','roseus','roseus',[234,147,153],'ea9399',15373209],
['nbs-iscc-b:roseus.3','roseus','roseus',[228,113,122],'e4717a',14971258],
['nbs-iscc-b:roseus.15','roseus','roseus',[171,78,82],'ab4e52',11226706],
['nbs-iscc-b:roseus.26','roseus','roseus',[248,131,121],'f88379',16286585],
['nbs-iscc-b:roseus.27','roseus','roseus',[230,103,97],'e66761',15099745],
['nbs-iscc-b:roseus.237','roseus','roseus',[158,79,136],'9e4f88',10375048],
['nbs-iscc-b:roseus.246','roseus','roseus',[255,200,214],'ffc8d6',16763094],
['nbs-iscc-b:roseus.247','roseus','roseus',[230,143,172],'e68fac',15110060],
['nbs-iscc-b:roseus.248','roseus','roseus',[222,111,161],'de6fa1',14577569],
['nbs-iscc-b:roseus.249','roseus','roseus',[239,187,204],'efbbcc',15711180],
['nbs-iscc-b:roseus.254','roseus','roseus',[206,70,118],'ce4676',13518454],
['nbs-iscc-b:roseus.255','roseus','roseus',[179,68,108],'b3446c',11748460],
['nbs-iscc-b:ruber.11','ruber','ruber',[190,0,50],'be0032',12451890],
['nbs-iscc-b:ruber.12','ruber','ruber',[188,63,74],'bc3f4a',12336970],
['nbs-iscc-b:ruber.254','ruber','ruber',[206,70,118],'ce4676',13518454],
['nbs-iscc-b:ruber.256','ruber','ruber',[120,24,74],'78184a',7870538],
['nbs-iscc-b:ruber.258','ruber','ruber',[168,81,110],'a8516e',11030894],
['nbs-iscc-b:ruber.262','ruber','ruber',[145,95,109],'915f6d',9527149],
['nbs-iscc-b:salmoneus.29','salmoneus','salmoneus',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-b:salmoneus.31','salmoneus','salmoneus',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-b:salmoneus.33','salmoneus','salmoneus',[194,172,153],'c2ac99',12758169],
['nbs-iscc-b:salmoneus.52','salmoneus','salmoneus',[250,181,127],'fab57f',16430463],
['nbs-iscc-b:salmoneus.53','salmoneus','salmoneus',[217,144,88],'d99058',14258264],
['nbs-iscc-b:salmoneus.71','salmoneus','salmoneus',[227,168,87],'e3a857',14919767],
['nbs-iscc-b:salmoneus.73','salmoneus','salmoneus',[250,214,165],'fad6a5',16438949],
['nbs-iscc-b:salmoneus.89','salmoneus','salmoneus',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-b:senatus.51','senatus','senatus',[190,101,22],'be6516',12477718],
['nbs-iscc-b:senatus.69','senatus','senatus',[201,133,0],'c98500',13206784],
['nbs-iscc-b:senatus.72','senatus','senatus',[190,138,61],'be8a3d',12487229],
['nbs-iscc-b:senatus.74','senatus','senatus',[153,101,21],'996515',10052885],
['nbs-iscc-b:sepiaceus.58','sepiaceus','sepiaceus',[111,78,55],'6f4e37',7294519],
['nbs-iscc-b:sepiaceus.61','sepiaceus','sepiaceus',[99,81,71],'635147',6508871],
['nbs-iscc-b:sepiaceus.77','sepiaceus','sepiaceus',[130,102,68],'826644',8545860],
['nbs-iscc-b:sepiaceus.80','sepiaceus','sepiaceus',[126,109,90],'7e6d5a',8285530],
['nbs-iscc-b:sepiaceus.81','sepiaceus','sepiaceus',[72,60,50],'483c32',4734002],
['nbs-iscc-b:sepiaceus.95','sepiaceus','sepiaceus',[108,84,30],'6c541e',7099422],
['nbs-iscc-b:stramineus.86','stramineus','stramineus',[248,222,126],'f8de7e',16309886],
['nbs-iscc-b:stramineus.89','stramineus','stramineus',[243,229,171],'f3e5ab',15984043],
['nbs-iscc-b:stramineus.101','stramineus','stramineus',[234,230,121],'eae679',15394425],
['nbs-iscc-b:stramineus.102','stramineus','stramineus',[185,180,89],'b9b459',12170329],
['nbs-iscc-b:stramineus.104','stramineus','stramineus',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-b:sublazulinus.177','sublazulinus','sublazulinus',[73,151,208],'4997d0',4822992],
['nbs-iscc-b:sublazulinus.178','sublazulinus','sublazulinus',[0,103,165],'0067a5',26533],
['nbs-iscc-b:sublazulinus.180','sublazulinus','sublazulinus',[161,202,241],'a1caf1',10603249],
['nbs-iscc-b:sublazulinus.181','sublazulinus','sublazulinus',[112,163,204],'70a3cc',7381964],
['nbs-iscc-b:sublazulinus.184','sublazulinus','sublazulinus',[188,212,230],'bcd4e6',12375270],
['nbs-iscc-b:sublazulinus.185','sublazulinus','sublazulinus',[145,163,176],'91a3b0',9544624],
['nbs-iscc-b:sublazulinus.194','sublazulinus','sublazulinus',[48,38,122],'30267a',3155578],
['nbs-iscc-b:sublazulinus.195','sublazulinus','sublazulinus',[108,121,184],'6c79b8',7109048],
['nbs-iscc-b:sublazulinus.196','sublazulinus','sublazulinus',[84,90,167],'545aa7',5528231],
['nbs-iscc-b:sublazulinus.198','sublazulinus','sublazulinus',[179,188,226],'b3bce2',11779298],
['nbs-iscc-b:sublazulinus.199','sublazulinus','sublazulinus',[135,145,191],'8791bf',8884671],
['nbs-iscc-b:sublazulinus.202','sublazulinus','sublazulinus',[192,200,225],'c0c8e1',12634337],
['nbs-iscc-b:sublazulinus.203','sublazulinus','sublazulinus',[140,146,172],'8c92ac',9212588],
['nbs-iscc-b:submurinus.10','submurinus','submurinus',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-b:submurinus.63','submurinus','submurinus',[142,130,121],'8e8279',9339513],
['nbs-iscc-b:submurinus.93','submurinus','submurinus',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-b:submurinus.264','submurinus','submurinus',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-b:subolivaceogriseus.92','subolivaceogriseus','subolivaceo-griseus',[240,234,214],'f0ead6',15788758],
['nbs-iscc-b:subolivaceogriseus.93','subolivaceogriseus','subolivaceo-griseus',[191,184,165],'bfb8a5',12564645],
['nbs-iscc-b:subolivaceogriseus.109','subolivaceogriseus','subolivaceo-griseus',[140,135,103],'8c8767',9209703],
['nbs-iscc-b:subolivaceogriseus.112','subolivaceogriseus','subolivaceo-griseus',[138,135,118],'8a8776',9078646],
['nbs-iscc-b:subpurpureogriseus.232','subpurpureogriseus','subpurpureo-griseus',[191,185,189],'bfb9bd',12564925],
['nbs-iscc-b:subpurpureogriseus.233','subpurpureogriseus','subpurpureo-griseus',[139,133,137],'8b8589',9143689],
['nbs-iscc-b:subvinosogriseus.8','subvinosogriseus','subvinoso-griseus',[196,174,173],'c4aead',12889773],
['nbs-iscc-b:subvinosogriseus.10','subvinosogriseus','subvinoso-griseus',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-b:subvinosogriseus.18','subvinosogriseus','subvinoso-griseus',[173,136,132],'ad8884',11372676],
['nbs-iscc-b:subvinosogriseus.19','subvinosogriseus','subvinoso-griseus',[144,93,93],'905d5d',9461085],
['nbs-iscc-b:subvinosogriseus.22','subvinosogriseus','subvinoso-griseus',[143,129,127],'8f817f',9404799],
['nbs-iscc-b:subvinosogriseus.60','subvinosogriseus','subvinoso-griseus',[149,128,112],'958070',9797744],
['nbs-iscc-b:subvinosogriseus.227','subvinosogriseus','subvinoso-griseus',[170,152,169],'aa98a9',11180201],
['nbs-iscc-b:subvinosogriseus.228','subvinosogriseus','subvinoso-griseus',[121,104,120],'796878',7956600],
['nbs-iscc-b:subvinosogriseus.232','subvinosogriseus','subvinoso-griseus',[191,185,189],'bfb9bd',12564925],
['nbs-iscc-b:subvinosogriseus.233','subvinosogriseus','subvinoso-griseus',[139,133,137],'8b8589',9143689],
['nbs-iscc-b:subvinosogriseus.252','subvinosogriseus','subvinoso-griseus',[232,204,215],'e8ccd7',15256791],
['nbs-iscc-b:subvinosogriseus.253','subvinosogriseus','subvinoso-griseus',[195,166,177],'c3a6b1',12822193],
['nbs-iscc-b:subvinosus.4','subvinosus','subvinosus',[249,204,202],'f9ccca',16370890],
['nbs-iscc-b:subvinosus.5','subvinosus','subvinosus',[222,165,164],'dea5a4',14591396],
['nbs-iscc-b:subvinosus.7','subvinosus','subvinosus',[234,216,215],'ead8d7',15390935],
['nbs-iscc-b:subvinosus.8','subvinosus','subvinosus',[196,174,173],'c4aead',12889773],
['nbs-iscc-b:subvinosus.10','subvinosus','subvinosus',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-b:subvinosus.31','subvinosus','subvinosus',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-b:subvinosus.32','subvinosus','subvinosus',[199,173,163],'c7ada3',13086115],
['nbs-iscc-b:subviolaceous.194','subviolaceous','subviolaceous',[48,38,122],'30267a',3155578],
['nbs-iscc-b:subviolaceous.195','subviolaceous','subviolaceous',[108,121,184],'6c79b8',7109048],
['nbs-iscc-b:subviolaceous.196','subviolaceous','subviolaceous',[84,90,167],'545aa7',5528231],
['nbs-iscc-b:subviolaceous.198','subviolaceous','subviolaceous',[179,188,226],'b3bce2',11779298],
['nbs-iscc-b:subviolaceous.199','subviolaceous','subviolaceous',[135,145,191],'8791bf',8884671],
['nbs-iscc-b:subviolaceous.202','subviolaceous','subviolaceous',[192,200,225],'c0c8e1',12634337],
['nbs-iscc-b:subviolaceous.203','subviolaceous','subviolaceous',[140,146,172],'8c92ac',9212588],
['nbs-iscc-b:subviolaceous.206','subviolaceous','subviolaceous',[126,115,184],'7e73b8',8287160],
['nbs-iscc-b:subviolaceous.209','subviolaceous','subviolaceous',[220,208,255],'dcd0ff',14471423],
['nbs-iscc-b:subviolaceous.210','subviolaceous','subviolaceous',[140,130,182],'8c82b6',9208502],
['nbs-iscc-b:subviolaceous.213','subviolaceous','subviolaceous',[196,195,221],'c4c3dd',12895197],
['nbs-iscc-b:subviolaceous.214','subviolaceous','subviolaceous',[150,144,171],'9690ab',9867435],
['nbs-iscc-b:subviolaceous.217','subviolaceous','subviolaceous',[211,153,230],'d399e6',13867494],
['nbs-iscc-b:subviolaceous.221','subviolaceous','subviolaceous',[213,186,219],'d5badb',14007003],
['nbs-iscc-b:subviolaceous.222','subviolaceous','subviolaceous',[182,149,192],'b695c0',11965888],
['nbs-iscc-b:subviridigriseus.190','subviridigriseus','subviridi-griseus',[180,188,192],'b4bcc0',11844800],
['nbs-iscc-b:subviridigriseus.264','subviridigriseus','subviridi-griseus',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-b:subviridis.116','subviridis','subviridis',[189,218,87],'bdda57',12442199],
['nbs-iscc-b:subviridis.119','subviridis','subviridis',[201,220,137],'c9dc89',13229193],
['nbs-iscc-b:subviridis.121','subviridis','subviridis',[218,223,183],'dadfb7',14344119],
['nbs-iscc-b:subviridis.130','subviridis','subviridis',[131,211,125],'83d37d',8639357],
['nbs-iscc-b:subviridis.135','subviridis','subviridis',[147,197,146],'93c592',9684370],
['nbs-iscc-b:sulphureus.98','sulphureus','sulphureus',[233,228,80],'e9e450',15328336],
['nbs-iscc-b:sulphureus.101','sulphureus','sulphureus',[234,230,121],'eae679',15394425],
['nbs-iscc-b:sulphureus.102','sulphureus','sulphureus',[185,180,89],'b9b459',12170329],
['nbs-iscc-b:sulphureus.104','sulphureus','sulphureus',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-b:testaceus.26','testaceus','testaceus',[248,131,121],'f88379',16286585],
['nbs-iscc-b:testaceus.29','testaceus','testaceus',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-b:testaceus.39','testaceus','testaceus',[180,116,94],'b4745e',11826270],
['nbs-iscc-b:testaceus.42','testaceus','testaceus',[168,124,109],'a87c6d',11041901],
['nbs-iscc-b:testaceus.43','testaceus','testaceus',[121,68,59],'79443b',7947323],
['nbs-iscc-b:testaceus.53','testaceus','testaceus',[217,144,88],'d99058',14258264],
['nbs-iscc-b:testaceus.57','testaceus','testaceus',[166,123,91],'a67b5b',10910555],
['nbs-iscc-b:umbrinus.43','umbrinus','umbrinus',[121,68,59],'79443b',7947323],
['nbs-iscc-b:umbrinus.46','umbrinus','umbrinus',[103,76,71],'674c47',6769735],
['nbs-iscc-b:umbrinus.55','umbrinus','umbrinus',[128,70,27],'80461b',8406555],
['nbs-iscc-b:umbrinus.57','umbrinus','umbrinus',[166,123,91],'a67b5b',10910555],
['nbs-iscc-b:umbrinus.58','umbrinus','umbrinus',[111,78,55],'6f4e37',7294519],
['nbs-iscc-b:umbrinus.74','umbrinus','umbrinus',[153,101,21],'996515',10052885],
['nbs-iscc-b:umbrinus.77','umbrinus','umbrinus',[130,102,68],'826644',8545860],
['nbs-iscc-b:umbrinus.78','umbrinus','umbrinus',[75,54,33],'4b3621',4929057],
['nbs-iscc-b:venetus.159','venetus','venetus',[0,166,147],'00a693',42643],
['nbs-iscc-b:venetus.160','venetus','venetus',[0,122,116],'007a74',31348],
['nbs-iscc-b:venetus.163','venetus','venetus',[102,173,164],'66ada4',6729124],
['nbs-iscc-b:venetus.164','venetus','venetus',[49,120,115],'317873',3242099],
['nbs-iscc-b:vinosobubalinus.10','vinosobubalinus','vinoso-bubalinus',[193,182,179],'c1b6b3',12695219],
['nbs-iscc-b:vinosobubalinus.28','vinosobubalinus','vinoso-bubalinus',[244,194,194],'f4c2c2',16040642],
['nbs-iscc-b:vinosobubalinus.29','vinosobubalinus','vinoso-bubalinus',[217,166,169],'d9a6a9',14263977],
['nbs-iscc-b:vinosobubalinus.31','vinosobubalinus','vinoso-bubalinus',[236,213,197],'ecd5c5',15521221],
['nbs-iscc-b:vinosobubalinus.32','vinosobubalinus','vinoso-bubalinus',[199,173,163],'c7ada3',13086115],
['nbs-iscc-b:vinosobubalinus.33','vinosobubalinus','vinoso-bubalinus',[194,172,153],'c2ac99',12758169],
['nbs-iscc-b:vinosobubalinus.76','vinosobubalinus','vinoso-bubalinus',[193,154,107],'c19a6b',12687979],
['nbs-iscc-b:vinosobubalinus.79','vinosobubalinus','vinoso-bubalinus',[174,155,130],'ae9b82',11443074],
['nbs-iscc-b:vinosogriseus.19','vinosogriseus','vinoso-griseus',[144,93,93],'905d5d',9461085],
['nbs-iscc-b:vinosogriseus.23','vinosogriseus','vinoso-griseus',[92,80,79],'5c504f',6049871],
['nbs-iscc-b:vinosogriseus.45','vinosogriseus','vinoso-griseus',[151,127,115],'977f73',9928563],
['nbs-iscc-b:vinosogriseus.60','vinosogriseus','vinoso-griseus',[149,128,112],'958070',9797744],
['nbs-iscc-b:vinosogriseus.61','vinosogriseus','vinoso-griseus',[99,81,71],'635147',6508871],
['nbs-iscc-b:vinosogriseus.228','vinosogriseus','vinoso-griseus',[121,104,120],'796878',7956600],
['nbs-iscc-b:vinosolividus.18','vinosolividus','vinoso-lividus',[173,136,132],'ad8884',11372676],
['nbs-iscc-b:vinosolividus.19','vinosolividus','vinoso-lividus',[144,93,93],'905d5d',9461085],
['nbs-iscc-b:vinosolividus.42','vinosolividus','vinoso-lividus',[168,124,109],'a87c6d',11041901],
['nbs-iscc-b:vinosolividus.46','vinosolividus','vinoso-lividus',[103,76,71],'674c47',6769735],
['nbs-iscc-b:vinosolividus.251','vinosolividus','vinoso-lividus',[193,126,145],'c17e91',12680849],
['nbs-iscc-b:vinosolividus.262','vinosolividus','vinoso-lividus',[145,95,109],'915f6d',9527149],
['nbs-iscc-b:vinosopurpureus.245','vinosopurpureus','vinoso-purpureus',[131,100,121],'836479',8610937],
['nbs-iscc-b:vinosus.3','vinosus','vinosus',[228,113,122],'e4717a',14971258],
['nbs-iscc-b:vinosus.6','vinosus','vinosus',[192,128,129],'c08081',12615809],
['nbs-iscc-b:vinosus.19','vinosus','vinosus',[144,93,93],'905d5d',9461085],
['nbs-iscc-b:vinosus.26','vinosus','vinosus',[248,131,121],'f88379',16286585],
['nbs-iscc-b:violaceoardesiacus.228','violaceoardesiacus','violaceo-ardesiacus',[121,104,120],'796878',7956600],
['nbs-iscc-b:violaceoardesiacus.229','violaceoardesiacus','violaceo-ardesiacus',[80,64,77],'50404d',5259341],
['nbs-iscc-b:violaceoardesiacus.234','violaceoardesiacus','violaceo-ardesiacus',[93,85,91],'5d555b',6116699],
['nbs-iscc-b:violaceogriseus.186','violaceogriseus','violaceo-griseus',[83,104,120],'536878',5466232],
['nbs-iscc-b:violaceogriseus.204','violaceogriseus','violaceo-griseus',[76,81,109],'4c516d',5001581],
['nbs-iscc-b:violaceogriseus.228','violaceogriseus','violaceo-griseus',[121,104,120],'796878',7956600],
['nbs-iscc-b:violaceogriseus.233','violaceogriseus','violaceo-griseus',[139,133,137],'8b8589',9143689],
['nbs-iscc-b:violaceogriseus.234','violaceogriseus','violaceo-griseus',[93,85,91],'5d555b',6116699],
['nbs-iscc-b:violaceogriseus.265','violaceogriseus','violaceo-griseus',[132,132,130],'848482',8684674],
['nbs-iscc-b:violaceogriseus.266','violaceogriseus','violaceo-griseus',[85,85,85],'555555',5592405],
['nbs-iscc-b:violaceoniger.229','violaceoniger','violaceo-niger',[80,64,77],'50404d',5259341],
['nbs-iscc-b:violaceoniger.234','violaceoniger','violaceo-niger',[93,85,91],'5d555b',6116699],
['nbs-iscc-b:violaceoniger.266','violaceoniger','violaceo-niger',[85,85,85],'555555',5592405],
['nbs-iscc-b:violaceoniger.267','violaceoniger','violaceo-niger',[34,34,34],'222222',2236962],
['nbs-iscc-b:violaceus.194','violaceus','violaceus',[48,38,122],'30267a',3155578],
['nbs-iscc-b:violaceus.196','violaceus','violaceus',[84,90,167],'545aa7',5528231],
['nbs-iscc-b:violaceus.200','violaceus','violaceus',[78,81,128],'4e5180',5132672],
['nbs-iscc-b:violaceus.205','violaceus','violaceus',[144,101,202],'9065ca',9463242],
['nbs-iscc-b:violaceus.206','violaceus','violaceus',[126,115,184],'7e73b8',8287160],
['nbs-iscc-b:violaceus.207','violaceus','violaceus',[96,78,151],'604e97',6311575],
['nbs-iscc-b:violaceus.210','violaceus','violaceus',[140,130,182],'8c82b6',9208502],
['nbs-iscc-b:violaceus.211','violaceus','violaceus',[96,78,129],'604e81',6311553],
['nbs-iscc-b:violaceus.216','violaceus','violaceus',[154,78,174],'9a4eae',10112686],
['nbs-iscc-b:violaceus.219','violaceus','violaceus',[96,47,107],'602f6b',6303595],
['nbs-iscc-b:violaceus.223','violaceus','violaceus',[134,96,142],'86608e',8806542],
['nbs-iscc-b:viridicaeruleus.140','viridicaeruleus','viridi-caeruleus',[62,180,137],'3eb489',4109449],
['nbs-iscc-b:viridicaeruleus.144','viridicaeruleus','viridi-caeruleus',[106,171,142],'6aab8e',6990734],
['nbs-iscc-b:viridicaeruleus.148','viridicaeruleus','viridi-caeruleus',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-b:viridicaeruleus.162','viridicaeruleus','viridi-caeruleus',[150,222,209],'96ded1',9887441],
['nbs-iscc-b:viridicaeruleus.163','viridicaeruleus','viridi-caeruleus',[102,173,164],'66ada4',6729124],
['nbs-iscc-b:viridiflavus.98','viridiflavus','viridi-flavus',[233,228,80],'e9e450',15328336],
['nbs-iscc-b:viridiflavus.99','viridiflavus','viridi-flavus',[190,183,46],'beb72e',12498734],
['nbs-iscc-b:viridiflavus.101','viridiflavus','viridi-flavus',[234,230,121],'eae679',15394425],
['nbs-iscc-b:viridiflavus.102','viridiflavus','viridi-flavus',[185,180,89],'b9b459',12170329],
['nbs-iscc-b:viridiflavus.104','viridiflavus','viridi-flavus',[235,232,164],'ebe8a4',15460516],
['nbs-iscc-b:viridiflavus.115','viridiflavus','viridi-flavus',[141,182,0],'8db600',9287168],
['nbs-iscc-b:viridiflavus.116','viridiflavus','viridi-flavus',[189,218,87],'bdda57',12442199],
['nbs-iscc-b:viridiflavus.117','viridiflavus','viridi-flavus',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-b:viridiflavus.119','viridiflavus','viridi-flavus',[201,220,137],'c9dc89',13229193],
['nbs-iscc-b:viridiflavus.120','viridiflavus','viridi-flavus',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-b:viridiglaucus.119','viridiglaucus','viridi-glaucus',[201,220,137],'c9dc89',13229193],
['nbs-iscc-b:viridiglaucus.121','viridiglaucus','viridi-glaucus',[218,223,183],'dadfb7',14344119],
['nbs-iscc-b:viridiglaucus.143','viridiglaucus','viridi-glaucus',[142,209,178],'8ed1b2',9359794],
['nbs-iscc-b:viridiglaucus.148','viridiglaucus','viridi-glaucus',[199,230,215],'c7e6d7',13100759],
['nbs-iscc-b:viridigriseus.120','viridigriseus','viridi-griseus',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-b:viridigriseus.122','viridigriseus','viridi-griseus',[143,151,121],'8f9779',9410425],
['nbs-iscc-b:viridigriseus.149','viridigriseus','viridi-griseus',[141,163,153],'8da399',9282457],
['nbs-iscc-b:viridigriseus.150','viridigriseus','viridi-griseus',[94,113,106],'5e716a',6189418],
['nbs-iscc-b:viridigriseus.154','viridigriseus','viridi-griseus',[178,190,181],'b2beb5',11714229],
['nbs-iscc-b:viridigriseus.155','viridigriseus','viridi-griseus',[125,137,132],'7d8984',8227204],
['nbs-iscc-b:viridigriseus.191','viridigriseus','viridi-griseus',[129,135,139],'81878b',8488843],
['nbs-iscc-b:viridigriseus.192','viridigriseus','viridi-griseus',[81,88,94],'51585e',5331038],
['nbs-iscc-b:viridigriseus.264','viridigriseus','viridi-griseus',[185,184,181],'b9b8b5',12171445],
['nbs-iscc-b:viridigriseus.265','viridigriseus','viridi-griseus',[132,132,130],'848482',8684674],
['nbs-iscc-b:viridiniger.266','viridiniger','viridi-niger',[85,85,85],'555555',5592405],
['nbs-iscc-b:viridiolivaceus.91','viridiolivaceus','viridi-olivaceus ',[161,143,96],'a18f60',10588000],
['nbs-iscc-b:viridiolivaceus.106','viridiolivaceus','viridi-olivaceus ',[134,126,54],'867e36',8814134],
['nbs-iscc-b:viridiolivaceus.120','viridiolivaceus','viridi-olivaceus ',[138,154,91],'8a9a5b',9083483],
['nbs-iscc-b:viridiolivaceus.122','viridiolivaceus','viridi-olivaceus ',[143,151,121],'8f9779',9410425],
['nbs-iscc-b:viridis.115','viridis','viridis',[141,182,0],'8db600',9287168],
['nbs-iscc-b:viridis.116','viridis','viridis',[189,218,87],'bdda57',12442199],
['nbs-iscc-b:viridis.117','viridis','viridis',[126,159,46],'7e9f2e',8298286],
['nbs-iscc-b:viridis.130','viridis','viridis',[131,211,125],'83d37d',8639357],
['nbs-iscc-b:viridis.135','viridis','viridis',[147,197,146],'93c592',9684370],
['nbs-iscc-b:viridis.140','viridis','viridis',[62,180,137],'3eb489',4109449]
    ];
}

sub _description {
    return {
          'subtitle' => 'Colour Terminology in Biology',
          'title' => 'NBS/ISCC B',
          'description' => '    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


    Color Name Dictionary: (B) Colour Terminology in Biology

    Colour Terminology in Biology,
    H.A. Dade 

This combinatorial dictionary seems to be derived from Latin (eg. viridi-caeruleus). Its 128 names map to 789 colors, using 214 of the 267 centroids. But averaging over 6 colors per name means it has little descriptive power.

[http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#B]
'
        }

}

1;
