#' Vector transmission model
#' 
#' @description A basic model with several compartments to model vector-borne transmission
#' 
#' @details The model tracks the dynamics of susceptible, infected, and recovered hosts, and susceptible and infected vectors. Infection, recovery, and waning immunity processes are implemented for hosts. Births and deaths and infection processes are implemented for vectors.
#' 
#' This code was generated by the modelbuilder R package.  
#' The model is implemented as a set of ordinary differential equations using the deSolve package. 
#' The following R packages need to be loaded for the function to work: deSolve. 
#' 
#' @param Sh : starting value for Susceptible hosts : numeric
#' @param Ih : starting value for Infected hosts : numeric
#' @param Rh : starting value for Recovered hosts : numeric
#' @param Sv : starting value for Susceptible Vectors : numeric
#' @param Iv : starting value for Infected Vectors : numeric
#' @param b1 : rate at which susceptible hosts are infected by vectors : numeric
#' @param b2 : rate at which susceptible vectors are infected by hosts : numeric
#' @param g : recovery rate of hosts : numeric
#' @param w : wanning immunity rate : numeric
#' @param n : vector birth rate : numeric
#' @param m : vector death rate : numeric
#' @param tstart : Start time of simulation : numeric
#' @param tfinal : Final time of simulation : numeric
#' @param dt : Time step : numeric
#' @return The function returns the output as a list. 
#' The time-series from the simulation is returned as a dataframe saved as list element \code{ts}. 
#' The \code{ts} dataframe has one column per compartment/variable. The first column is time.   
#' @examples  
#' # To run the simulation with default parameters:  
#' result <- simulate_Vector_transmission_model_ode() 
#' # To choose values other than the standard one, specify them like this:  
#' result <- simulate_Vector_transmission_model_ode(Sh = 2000,Ih = 2,Rh = 0,Sv = 2000,Iv = 2) 
#' # You can display or further process the result, like this:  
#' plot(result$ts[,'time'],result$ts[,'Sh'],xlab='Time',ylab='Numbers',type='l') 
#' print(paste('Max number of Sh: ',max(result$ts[,'Sh']))) 
#' @section Warning: This function does not perform any error checking. So if you try to do something nonsensical (e.g. have negative values for parameters), the code will likely abort with an error message.
#' @section Model Author: Andreas Handel
#' @section Model creation date: 2020-09-01
#' @section Code Author: generated by the \code{modelbuilder} R package 
#' @section Code creation date: 2021-07-19
#' @export 
 
simulate_Vector_transmission_model_ode <- function(Sh = 1000, Ih = 1, Rh = 0, Sv = 1000, Iv = 1, b1 = 0.003, b2 = 0.003, g = 2, w = 0, n = 200, m = 0.1, tstart = 0, tfinal = 100, dt = 0.1) 
{ 
  ############################## 
  #Block of ODE equations for deSolve 
  ############################## 
  Vector_transmission_model_ode_fct <- function(t, y, parms) 
  {
    with( as.list(c(y,parms)), { #lets us access variables and parameters stored in y and parms by name 
    #StartODES
    #Susceptible hosts : infection of susceptible hosts : waning immunity :
    dSh_mb = -b1*Sh*Iv +w*Rh
    #Infected hosts : infection of susceptible hosts : recovery of infected :
    dIh_mb = +b1*Sh*Iv -g*Ih
    #Recovered hosts : recovery of infected hosts : waning immunity :
    dRh_mb = +g*Ih -w*Rh
    #Susceptible Vectors : vector births : infection of susceptible vectors : death of susceptible vectors :
    dSv_mb = +n -b2*Sv*Ih -m*Sv
    #Infected Vectors : infection of susceptible vectors : death of infected vectors :
    dIv_mb = +b2*Sv*Ih -m*Iv
    #EndODES
    list(c(dSh_mb,dIh_mb,dRh_mb,dSv_mb,dIv_mb)) 
  } ) } #close with statement, end ODE code block 
 
  ############################## 
  #Main function code block 
  ############################## 
  #Creating named vectors 
  varvec_mb = c(Sh = Sh, Ih = Ih, Rh = Rh, Sv = Sv, Iv = Iv) 
  parvec_mb = c(b1 = b1, b2 = b2, g = g, w = w, n = n, m = m) 
  timevec_mb = seq(tstart, tfinal,by = dt) 
  #Running the model 
  simout = deSolve::ode(y = varvec_mb, parms = parvec_mb, times = timevec_mb,  func = Vector_transmission_model_ode_fct, rtol = 1e-12, atol = 1e-12) 
  #Setting up empty list and returning result as data frame called ts 
  result <- list() 
  result$ts <- as.data.frame(simout) 
  return(result) 
} 
