---
title: "Introduction to Athlytics"
output: rmarkdown::html_vignette
vignette: >
  %\VignetteIndexEntry{Introduction to Athlytics}
  %\VignetteEngine{knitr::rmarkdown}
  %\VignetteEncoding{UTF-8}
---




``` r
library(Athlytics)
library(rStrava) # Required for authentication
```

## Introduction

The `Athlytics` R package simplifies advanced analysis of athletic performance and training load using data sourced directly from the Strava API. It provides functions to fetch Strava data (via the `rStrava` package), calculate key metrics based on sports science principles, and generate insightful visualizations for training monitoring.

This vignette covers:

*   Installation
*   Authentication with the Strava API
*   Overview and examples of core analysis functions

## Installation

You can install the released version of Athlytics from CRAN (once available) with:

```r
# install.packages("Athlytics") 
```

Alternatively, install the development version from GitHub:

```r
# install.packages('remotes') # If needed
# remotes::install_github('HzaCode/Athlytics') 
```

## Authentication with Strava

`Athlytics` requires a valid Strava API token to fetch data. This is handled using the `rStrava` package.

1.  **Create a Strava API Application:**
    *   Go to <https://developers.strava.com/> and navigate to "My API Application" under your settings, or follow their "Getting Started" guide to create an app.
    *   Create an application (e.g., "My Athlytics App").
    *   Set "Authorization Callback Domain" to `localhost`.
    *   Note your **Client ID** and **Client Secret**.

2.  **Authenticate in R:**
    *   Use `rStrava::strava_oauth()`.
    *   Using `cache = TRUE` is highly recommended to securely store the token and avoid re-authenticating every session.

```r
# --- Example Authentication Code ---
# (Do not run this block directly in the vignette build)

# Replace with your actual credentials or use environment variables
# app_name <- 'MyAthlyticsApp' 
# client_id <- "YOUR_CLIENT_ID" 
# client_secret <- "YOUR_SECRET"

# This function performs the OAuth handshake (may open browser)
# stoken <- rStrava::strava_oauth(
#   app_name,
#   client_id = client_id,
#   client_secret = client_secret,
#   app_scope = "activity:read_all", # Ensure necessary scopes
#   cache = TRUE # IMPORTANT: caches the token
# )

# For the examples below, we assume you have obtained a valid token
# object named 'stoken' in your interactive R session.
# Replace this placeholder line if running interactively:
stoken_placeholder <- "<Replace this string with your actual stoken object>"

# You would use the real 'stoken' object in function calls, e.g.:
# some_data <- calculate_acwr(stoken = stoken, ...)
```

**Important:** The following examples use `stoken = stoken_placeholder` and have `eval=FALSE`. You need to replace `stoken_placeholder` with your actual token object (`stoken`) and potentially set `eval=TRUE` if running interactively with a valid, cached token.

## Example Analysis Visualizations

These functions generate plots to analyze trends and performance.

### 1. Load Exposure (Acute vs. Chronic)

Visualizes the relationship between short-term (acute) and long-term (chronic) training load to assess readiness and potential injury risk. Note that TSS/HRSS calculations based on activity summaries are approximations.


``` r
# Calculate using approximate TSS for Rides (Requires FTP)
exposure_data_tss <- calculate_exposure(
    stoken = stoken_placeholder, 
    activity_type = "Ride",
    load_metric = "tss",
    user_ftp = 280, # Example FTP, replace with yours
    acute_period = 7,
    chronic_period = 28
)

# Plot the result
plot_exposure(exposure_data = exposure_data_tss, risk_zones = TRUE)

# Calculate using approximate HRSS for Runs (Requires Max & Resting HR)
hrss_data <- calculate_exposure(
    stoken = stoken_placeholder,
    activity_type = "Run",
    load_metric = "hrss",
    user_max_hr = 190,     # Example Max HR
    user_resting_hr = 50, # Example Resting HR
    acute_period = 7,
    chronic_period = 42
)

plot_exposure(exposure_data = hrrss_data, risk_zones = TRUE)
```

### 2. ACWR Trend

Tracks the Acute:Chronic Workload Ratio over time, highlighting periods of potentially risky load increases.


``` r
# Calculate ACWR using duration for Runs
acwr_data_run <- calculate_acwr(
    stoken = stoken_placeholder,
    activity_type = "Run",
    load_metric = "duration_mins",
    acute_period = 7,
    chronic_period = 28
)

# Plot the trend
plot_acwr(acwr_data = acwr_data_run, highlight_zones = TRUE)
```

### 3. Efficiency Factor (EF) Trend

Monitors aerobic efficiency, typically calculated as output (Pace or Power) divided by input (Heart Rate). Requires activities with both relevant metrics.


``` r
# Calculate EF (Pace/HR) for Runs and Rides
ef_data_pacehr <- calculate_ef(
    stoken = stoken_placeholder,
    activity_type = c("Run", "Ride"),
    ef_metric = "Pace_HR"
)

# Plot the trend
plot_ef(ef_data = ef_data_pacehr, add_trend_line = TRUE)
```

### 4. Personal Bests (PBs)

Visualizes the progression of estimated best efforts over various distances (currently primarily for running).

**Note:** Fetching PB data can be slow and API-intensive.


``` r
# Calculate PBs for 1k, 5k, 10k Runs
# Limit activities checked for speed
pb_data_run <- calculate_pbs(
    stoken = stoken_placeholder,
    distance_meters = c(1000, 5000, 10000),
    activity_type = "Run",
    max_activities = 50 # Limit for example
)

# Plot the progression, highlighting new PBs
plot_pbs(pb_data = pb_data_run)
```

### 5. Decoupling Trend

Assesses cardiovascular drift during activities by comparing performance (Pace or Power) in the first half vs. the second half relative to Heart Rate.

**Note:** Fetching decoupling data now uses direct Strava API calls via `httr` (instead of `rStrava::get_activity_streams`) to obtain detailed activity streams. This can be very slow and may hit API rate limits more easily, especially for many activities. Consider using the `max_activities` parameter.


``` r
# Calculate Pace/HR decoupling for Runs
# Limit activities checked for speed
decoupling_data_run <- calculate_decoupling(
    stoken = stoken_placeholder,
    activity_type = "Run",
    decouple_metric = "Pace_HR",
    max_activities = 20 # Limit for example
)

# Plot the trend
plot_decoupling(decoupling_data = decoupling_data_run, add_trend_line = TRUE)
```

### 2. Strava API Setup

*   Sign up for a Strava account if you don't have one.
*   Go to <https://developers.strava.com/> and navigate to "My API Application" under your settings or follow their "Getting Started" guide to create an app.
*   When creating your app, you'll be asked for:
    *   Application Name (e.g., `MyAthlyticsApp`)

## Further Information

This vignette provides a basic overview. For details on specific functions, their parameters, and underlying calculations, please consult the function help pages (e.g., `?calculate_acwr`, `?plot_acwr`) in R.

For bug reports or feature requests, please use the GitHub repository issues tracker: <https://github.com/HzaCode/Athlytics/issues> 
