% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run-pgs-statistics.R
\name{analyze.pgs.binary.predictiveness}
\alias{analyze.pgs.binary.predictiveness}
\title{Analyze PGS Predictiveness for Binary Phenotypes}
\usage{
analyze.pgs.binary.predictiveness(
  pgs.data,
  pgs.columns,
  phenotype.columns,
  covariate.columns = NULL,
  phenotype.type = "binary",
  cutoff.threshold = NULL,
  output.dir = NULL,
  filename.prefix = NULL,
  file.extension = "png",
  width = 8,
  height = 8,
  xaxes.cex = 1.5,
  yaxes.cex = 1.5,
  titles.cex = 1.5
)
}
\arguments{
\item{pgs.data}{A data frame containing the PGS, phenotype, and covariate columns.}

\item{pgs.columns}{A character vector specifying the names of the PGS columns
in \code{pgs.data} to be analyzed. All specified columns must be numeric.}

\item{phenotype.columns}{A character vector specifying the names of the phenotype columns in \code{data} to be analyzed. If binary phenotypes are specified, they must be factors with two levels (0 and 1).}

\item{covariate.columns}{A character vector specifying the names of covariate columns in \code{data} to be included in the regression model. Default is NULL.}

\item{phenotype.type}{A character string specifying the type of phenotype. Must be either 'continuous' or 'binary'. All provided phenotype columns must match this type.}

\item{cutoff.threshold}{A numeric value or a named list specifying the cutoff threshold for converting continuous phenotypes to binary. If a named list, it must contain entries for each continuous phenotype.}

\item{output.dir}{A character string specifying the directory where the ROC plots will be saved. If NULL, no plots are saved.}

\item{filename.prefix}{A character string specifying the prefix for the output filenames. If NULL, defaults to 'ApplyPolygenicScore-Plot'.}

\item{file.extension}{A character string specifying the file extension for the output plots. Default is 'png'.}

\item{width}{Numeric value specifying the width of the output plot in inches.}

\item{height}{Numeric value specifying the height of the output plot in inches.}

\item{xaxes.cex}{Numeric size for all x-axis labels.}

\item{yaxes.cex}{Numeric size for all y-axis labels.}

\item{titles.cex}{Numeric size for all plot titles.}
}
\value{
A list containing a data frame of logistic regression results and a plot object of corresponding ROC curves.

\strong{Output Formatting}
  \item{results.df}{A data frame with columns:
    \itemize{
      \item \code{phenotype}: Name of the phenotype column.
      \item \code{PGS}: Name of the PGS column.
      \item \code{AUC}: Area Under the Reciever Operator Curve.
      \item \code{OR}: Odds Ratio for the PGS from logistic regression.
      \item \code{OR.Lower.CI}: Lower 95% Confidence Interval for the Odds Ratio.
      \item \code{OR.Upper.CI}: Upper 95% Confidence Interval for the Odds Ratio.
      \item \code{p.value}: P-value for the PGS coefficient.
    }
    Values for \code{AUC}, \code{OR}, \code{OR.Lower.CI}, \code{OR.Upper.CI}, and \code{p.value} may be NA
    if model fitting or ROC calculation fails (e.g., due to no complete cases,
    no variance in PGS, or ROC calculation errors).
  }
  \item{roc.plot}{A \code{multipanelplot} object (from \code{BoutrosLab.plotting.general})
    if \code{output.dir} is \code{NULL}, otherwise \code{NULL} if plots are saved to file.}

Each phenotype is plotted in a separate panel, with ROC curves for each PGS specified in \code{pgs.columns}.
}
\description{
This function performs logistic regression to evaluate the predictiveness of polygenic scores
for binary or continuous phenotypes. For continuous phenotypes, it converts them
to binary based on a specified cutoff threshold. It calculates and returns AUC,
Odds Ratios (OR), and p-values for each PGS. Corresponding ROC curves are plotted automatically.
}
\examples{
set.seed(100);

pgs.data <- data.frame(
    PGS = rnorm(100, 0, 1),
    continuous.phenotype = rnorm(100, 2, 1),
    binary.phenotype = sample(c(0, 1), 100, replace = TRUE),
    covariate1 = rnorm(100, 0, 1)
    );
 temp.dir <- tempdir();

# Basic analysis with binary phenotype
analyze.pgs.binary.predictiveness(
    pgs.data,
    output.dir = temp.dir,
    filename.prefix = 'basic-plot',
    pgs.columns = 'PGS',
    phenotype.columns = 'binary.phenotype',
    phenotype.type = 'binary',
    covariate.columns = 'covariate1',
    width = 6,
    height = 6
    );

# Analysis with continuous phenotype and cutoff threshold
analyze.pgs.binary.predictiveness(
   pgs.data,
   output.dir = temp.dir,
   filename.prefix = 'continuous-plot',
   pgs.columns = 'PGS',
   phenotype.columns = 'continuous.phenotype',
   phenotype.type = 'continuous',
   cutoff.threshold = 1.5, # Convert to binary using this threshold
   covariate.columns = NULL,
   width = 6,
   height = 6
   );

}
