% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utilities.R
\name{custom_filter}
\alias{custom_filter}
\title{Special custom filtering for compositional data}
\usage{
custom_filter(data, ..., prop = NULL, special = NULL)
}
\arguments{
\item{data}{A data frame containing the compositional variables which should
be used to perform the filtering.}

\item{...}{Any additional arguments specified to the dplyr \code{\link[dplyr:filter]{filter()}} function.
Filtering conditions for any additional variables can also be specified here.}

\item{prop}{A character/numeric vector indicating the columns containing the
compositional variables in `data`.}

\item{special}{A character string specifying the filtering condition.
Four special keywords can be specified here for filtering
  1. richness: A positive integer value to filter communities with
               a specific number of compositional variables (variables with non-zero values).
  2. evenness: A numeric value between 0 and 1, to filter rows based on
               the relative abundances of the compositional variables where a higher
               value signifies a more even community with equal proportions
               of all variables.
  3. equi: A boolean variable indicating whether to filter rows containing
           equi-proportional communities, i.e., communities where all variables
           have the same non-zero proportion.
  4. monos: A boolean value indicating whether to filter communities
            containing a single compositional variable, i.e., richness == 1.
These keywords can be combined using any logical operators and can even
be combined with any other variables in the data. Please use the exact
keywords (case-sensitive) in the query to get appropriate results. See
examples for more details.}
}
\value{
A subset of the original data which matches the specified filtering conditions.
}
\description{
A handy wrapper around the dplyr \code{\link[dplyr:filter]{filter()}} function
enabling the user to filter rows which satisfy specific conditions
for compositional data like all equi-proportional communities, or communities
with a given value of richness without having to make any changes to the data
or adding any additional columns. All other functionalities are same as the
dplyr \code{\link[dplyr:filter]{filter()}} function.
}
\examples{
library(DImodels)
library(dplyr)

## Load data
data(sim3)

# The special filter keywords should be specified as a string
# Filter communities containing 3 species
head(custom_filter(data = sim3, prop = 4:12,
                   special = "richness == 3"))

# Filter communities at richness 6 OR evenness 0
head(custom_filter(data = sim3, prop = 4:12,
                   special = "richness == 6 | evenness == 0"), 12)

# Filter all monoculture AND treatment "A" (treatment is column present in data)
head(custom_filter(data = sim3, prop = 4:12,
                   special = "monos == TRUE & treatment == 'A'"), 10)

# Filter all equi proportional communities but NOT monocultures
head(custom_filter(data = sim3, prop = 4:12,
                   special = "equi == TRUE & monos == FALSE"))

# Can also use normal filter
sim3 \%>\% custom_filter(p1 == 1)

# Both special filtering and normal filtering can be combined as well
sim3 \%>\% custom_filter(prop = paste0("p", 1:9),
                       special = "richness == 1",
                       community \%in\% c(7, 9))
}
