#' Truth table
#'
#' Internal function to evaluate importance of predictor combinations
#' @importFrom LogicReg eval.logreg logreg logreg.anneal.control
#' @importFrom methods is
#'
#' @param data Input data.
#' @param tree Logic tree constructed for a sample.
#' @param Xs Input predictors.
#' @param mtype Model type: classification or linear or survival regression.

#' @details
#' Creates a matrix of all interactions contained in one sample's logic tree.
#'
#' @return \code{mat.truth}: Matrix of all interactions contained in one sample's logic tree
#'
#' @references
#' Wolf BJ, Hill EG, Slate EH. Logic Forest: an ensemble classifier for discovering logical combinations of binary markers. Bioinformatics. 2010;26(17):2183-2189. doi:10.1093/bioinformatics/btq354
#'
#' @author Bethany J. Wolf \email{wolfb@@musc.edu}
#'
#' @seealso \code{\link{prime.imp}}
#'
#' @keywords internal

TTab<-function(data, tree, Xs, mtype)
{
  if(!is(tree,"logregtree"))
    stop("tree must be an object of class logregtree")
  mod.var<-tree$trees[,3]
  mod.var<-sort(mod.var[mod.var!=0])
  mod.var<-mod.var[!duplicated(mod.var)]
  model.var<-c()
  for (i in 1:length(mod.var))
  {
    modvar<-Xs[mod.var[i]]
    model.var<-append(model.var, modvar)
  }
  mat.perms<-Perms(length(mod.var))
  nms<-colnames(data)
  if (is.null(colnames(data))) {colnames(mat.perms)<-paste("X", model.var, sep="")}
  if (length(nms)>0) {colnames(mat.perms)<-nms[mod.var]}
  mat.bin<-matrix(0, nrow(mat.perms), max(mod.var))
  mat.bin[,mod.var]<-mat.perms
  pred.out<-eval.logreg(tree, mat.bin)
  mat.truth<-cbind(mat.perms, outcome=pred.out)
  if(mtype==1) {truth<-ifelse(tree$coef>0 | is.na(tree$coef),1,0)}
  if(mtype==2) {truth<-1}
  if(mtype==3) {truth<-1}
  if(mtype==4) {truth<-1}
  ids.truth<-mat.truth[,"outcome"]==truth
  mat.truth<-mat.truth[ids.truth,-ncol(mat.truth),drop=FALSE]
  mat.truth
}
