% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MCPtest.R
\name{MCPtest}
\alias{MCPtest}
\title{Multiple comparison procedures to the means.}
\usage{
MCPtest(
  y,
  trt = NULL,
  dferror = NULL,
  mserror = NULL,
  replication = NULL,
  alpha = 0.05,
  main = NULL,
  MCP = "all",
  ismean = FALSE,
  parallel = FALSE
)
}
\arguments{
\item{y}{Model (aov or lm), numeric vector containing the
response variable or the mean of the treatments.}

\item{trt}{Constant (y = model) or a vector containing
the treatments.}

\item{dferror}{Degrees of freedom of the Mean Square Error.}

\item{mserror}{Mean Square Error.}

\item{replication}{Number de repetitions of the treatments in the experiment.
For unbalanced data should be informed the harmonic mean of repetitions.
This argument should be informed only if \code{ismean = TRUE}.}

\item{alpha}{Significant level. The default is \code{alpha = 0.05}.}

\item{main}{Title of the analysis.}

\item{MCP}{Allows choosing the multiple comparison test;
the \emph{defaut} is "all". This option will go perform all tests.
However, the options are: the Skott-Knott midrange test ("MGM"),
the Skott-Knott Range test ("MGR"), the Tukey midrange test ("TM"),
the Scott-Knott's test ("SK").}

\item{ismean}{Logic. If \code{FALSE} (default), the \code{y} argument represents
a model (aov or lm) or a numeric vector containing the response variable.
If \code{TRUE} the \code{y} argument represents the mean of treatments.}

\item{parallel}{Logic. If \code{FALSE} (default), the \code{MCPtest} function
runs without parallelization. Otherwise, the code is executed with
parallelization. Note that the parallelization is not always more efficient.}
}
\value{
\code{MCPtest} returns the print of a list of results. First,
    the summary of \code{y}. Second, the statistics
    of the test chosen. And finally, the  mean group results for each test.
    If \code{MCPtest} function is stored
    in an object, the results will be printed and
    also stored in the object.
}
\description{
\code{MCPtest} applies several tests of multiple comparisons present
    in the literature. The tests chosen are based on the evaluation
    of the researcher to decide the choice of test for analysis
    in the experiment.
}
\details{
The \code{MCP} argument allows you to choose various tests
    of multiple comparisons at once. For example,
    \code{MCP = c("MGM", "MGR")}, and so on.
}
\examples{
# Simulated data (completely randomized design)

# Response variable
rv <- c(100.08, 105.66, 97.64, 100.11, 102.60, 121.29, 100.80,
        99.11, 104.43, 122.18, 119.49, 124.37, 123.19, 134.16,
        125.67, 128.88, 148.07, 134.27, 151.53, 127.31)

# Treatments
treat <- factor(rep(LETTERS[1:5], each = 4))

# Anova
res     <- anova(aov(rv~treat))
DFerror <- res$Df[2]
MSerror <- res$`Mean Sq`[2]

# Loading the MCPtests package
library(MCPtests)

# applying the tests
results <- MCPtest(y = rv,
                  trt = treat,
                  dferror = DFerror,
                  mserror = MSerror,
                  alpha = 0.05,
                  main = "Multiple Comparison Procedure: MGM test",
                  MCP = c("MGM"))

# Other option for the MCP argument is "all". All tests are used.

results$Groups     # Results of the tests
results$Statistics # Main arguments of the tests
results$Summary    # Summary of the response variable

# Using the y argument as aov or lm model
res  <- aov(rv~treat)

MCPtest(y = res, trt = "treat", alpha = 0.05,
       main = "Multiple Comparison Procedure: MGM test",
       MCP = c("MGM"))

# For unbalanced data: It will be used the harmonic mean of
#                       the number of experiment replicates

# Using the previous example
rv <- rv[-1]
treat <- treat[-1]

res  <- lm(rv~treat) # Linear model

# Multiple comparison procedure: MGR test
MCPtest(y = res, trt = "treat", alpha = 0.05,
       main = "Multiple Comparison Procedure: MGR test",
       MCP = c("MGR"))

# Assuming that the available data are the averages
#  of the treatments and the analysis of variance

# Analysis of Variance Table

# Response: rv
#            Df Sum Sq Mean Sq F value    Pr(>F)
# treat      4 4135.2 1033.80  14.669 4.562e-05 ***
# Residuals 15 1057.1   70.47

mean.treat <- c(100.87, 105.95, 117.62, 127.97, 140.30)
treat <- factor(LETTERS[1:5])
DFerror <- 15
MSerror <- 70.47488
replic <- 4

MCPtest(y = mean.treat,
       trt = treat,
       dferror = DFerror,
       mserror = MSerror,
       replication = replic,
       alpha = 0.05,
       main = "Multiple Comparison Procedure: MGM test",
       MCP = c("MGM"),
       ismean = TRUE)

}
\references{
BATISTA, Ben Deivide de Oliveira. Testes de comparacoes multiplas
    baseados na distribuicao da \emph{midrange} \emph{estudentizada}
    externamente. 2016. 194f. Tese (Doutorado em Estatistica e
    Experimentacao Agropecuaria) - Universidade Federal de Lavras, 2016.
    (Portuguese, Brazil)

SCOTT, A. J.; KNOTT, M. A cluster analysis method for grouping means in
    the analysis of variance. Biometrics, International Biometric Society,
    v. 30, n. 3, p. 507-512, 1974.

DUNCAN, D. B. Multiple range and multiple F Tests. Biometrics, International
    Biometric Society, v. 11, n. 1,p. 1-42, 1955.
}
