% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model_adjpin.R
\name{adjpin}
\alias{adjpin}
\title{Estimation of adjusted PIN model}
\usage{
adjpin(data, method = "ECM", initialsets = "GE", num_init = 20,
              restricted = list(), ..., verbose = TRUE)
}
\arguments{
\item{data}{A dataframe with 2 variables: the first
corresponds to buyer-initiated trades (buys), and the second corresponds
to seller-initiated trades (sells).}

\item{method}{A character string referring to the method
used to estimate the model of \insertCite{Duarte09;textual}{PINstimation}.
It takes one of two values: \code{"ML"} refers to the standard maximum likelihood
estimation, and \code{"ECM"} refers to the expectation-conditional maximization
algorithm. The default value is \code{"ECM"}. Details of the ECM method,
and comparative results can be found in
\insertCite{Ghachem2022;textual}{PINstimation}, and in
\insertCite{Ghachem2022b;textual}{PINstimation}.}

\item{initialsets}{It can either be a character string referring to
prebuilt algorithms generating initial parameter sets or a dataframe
containing custom initial parameter sets.
If \code{initialsets} is a character string, it refers to the method of generation
of the initial parameter sets, and takes one of three values: \code{"GE"}, \code{"CL"},
or \code{"RANDOM"}. \code{"GE"} refers to initial parameter sets generated by the
algorithm of \insertCite{Ersan2022b;textual}{PINstimation}, and implemented
in \code{initials_adjpin()}, \code{"CL"} refers to initial parameter sets generated by
the algorithm of \insertCite{ChengLai2021;textual}{PINstimation}, and
implemented in \code{initials_adjpin_cl()}, while \code{"RANDOM"} generates random
initial parameter sets as implemented in \code{initials_adjpin_rnd()}.
The default value is \code{"GE"}. If \code{initialsets} is a dataframe, the function
\code{adjpin()} will estimate the AdjPIN model using the provided initial
parameter sets.}

\item{num_init}{An integer specifying the maximum number of
initial parameter sets to be used in the estimation.
If \code{initialsets="GE"}, the generation of initial parameter sets will stop
when the number of initial parameter sets reaches \code{num_init}. It can stop
earlier if the number of all possible generated initial parameter sets is
lower than \code{num_init}. If \code{initialsets="RANDOM"}, exactly \code{num_init}
initial parameter sets are returned. If \code{initialsets="CL"}: then \code{num_init}
is ignored, and all \code{256} initial parameter sets are used. The default
value is \code{20}. \verb{[i]} The argument \code{num_init} is ignored when the argument
\code{initialsets} is a dataframe.}

\item{restricted}{A binary list that allows estimating restricted
AdjPIN models by specifying which model parameters are assumed to be equal.
It contains one or multiple of the following four elements
\verb{\{theta, mu, eps, d\}}. For instance, If \code{theta} is set to \code{TRUE},
then the probability of liquidity shock in no-information days, and in
information days is assumed to be the same (\thetaB\code{=}\thetaS). If any of
the remaining rate elements \verb{\{mu, eps, d\}} is set to \code{TRUE},
(say  \code{mu=TRUE}), then the rate is assumed to be the same on the buy side,
and on the sell side (\mub\code{=}\mus). If more than one element is set to
\code{TRUE}, then the restrictions are combined. For instance, if the argument
\code{restricted} is set to \code{list(theta=TRUE, eps=TRUE, d=TRUE)}, then the
restricted AdjPIN model is estimated, where \thetaB\code{=}\thetaS, \eb\code{=}\es,
and \Db\code{=}\Ds. If the value of the argument \code{restricted} is the empty list
(\code{list()}), then all parameters of the model are assumed to be independent,
and the unrestricted model is estimated. The default value is the empty
list \code{list()}.}

\item{...}{Additional arguments passed on to the function \code{adjpin()}. The
recognized arguments are \code{hyperparams}, and \code{fact}. The argument
\code{hyperparams} consists of a list containing the hyperparameters of the \code{ECM}
algorithm. When not empty, it contains one or more  of the following
elements: \code{maxeval}, and \code{tolerance}. It is used only when the \code{method}
argument is set to \code{"ECM"}. The argument \code{fact} is a binary value that
determines which likelihood functional form is used: A factorization of
the likelihood function by \insertCite{Ersan2022b;textual}{PINstimation}
when it is set to \code{TRUE}, otherwise, the original likelihood function of
\insertCite{Duarte09;textual}{PINstimation}. The default value is \code{TRUE}.
More about these arguments are in the Details section.}

\item{verbose}{A binary variable that determines whether
detailed information about the steps of the estimation of the AdjPIN model
is displayed. No output is produced when \code{verbose} is set to
\code{FALSE}. The default value is \code{TRUE}.}
}
\value{
Returns an object of class \code{estimate.adjpin}.
}
\description{
Estimates the Adjusted Probability of Informed Trading
(\code{adjPIN}) as well as the Probability of Symmetric Order-flow Shock
(\code{PSOS}) from the \code{AdjPIN} model of Duarte and Young(2009).
}
\details{
The argument 'data' should be a numeric dataframe, and contain
at least two variables. Only the first two variables will be considered:
The first variable is assumed to correspond to the total number of
buyer-initiated trades, while the second variable is assumed to
correspond to the total number of seller-initiated trades. Each row or
observation correspond to a trading day. \code{NA} values will be ignored.

If \code{initialsets} is neither a dataframe, nor a character string from the
set \verb{\{"GE",} \verb{"CL",} \verb{"RANDOM"\}}, the estimation of the \code{AdjPIN} model is
aborted. The default initial parameters (\code{"GE"}) for the estimation
method are generated using a modified hierarchical agglomerative
clustering. For more information, see \code{initials_adjpin()}.

The argument \code{hyperparams}  contains the hyperparameters of the \code{ECM}
algorithm. It is either empty or contains one or two  of the following
elements:

\itemize{
\item \code{maxeval}: (\code{integer}) It stands for maximum number of iterations of
the \code{ECM}  algorithm for each initial parameter set. When missing, \code{maxeval}
takes the default value of \code{100}.

\item \code{tolerance} (\code{numeric}) The \code{ECM} algorithm is stopped when the
(relative) change of log-likelihood is  smaller than tolerance. When
missing, \code{tolerance} takes the default value of \code{0.001}.
}
}
\examples{
# We use 'generatedata_adjpin()' to generate a S4 object of type 'dataset'
# with 60 observations.

sim_data <- generatedata_adjpin(days = 60)

# The actual dataset of 60 observations is stored in the slot 'data' of the
# S4 object 'sim_data'. Each observation corresponds to a day and contains
# the total number of buyer-initiated transactions ('B') and seller-
# initiated transactions ('S') on that day.

xdata <- sim_data@data

# ------------------------------------------------------------------------ #
# Compare the unrestricted AdjPIN model with various restricted models     #
# ------------------------------------------------------------------------ #

# Estimate the unrestricted AdjPIN model using the ECM algorithm (default),
# and show the estimation output

estimate.adjpin.0 <- adjpin(xdata, verbose = FALSE)

show(estimate.adjpin.0)

# Estimate the restricted AdjPIN model where mub=mus
\donttest{
estimate.adjpin.1 <- adjpin(xdata, restricted = list(mu = TRUE),
                                  verbose = FALSE)

# Estimate the restricted AdjPIN model where eps.b=eps.s

estimate.adjpin.2 <- adjpin(xdata, restricted = list(eps = TRUE),
                                  verbose = FALSE)

# Estimate the restricted AdjPIN model where d.b=d.s

estimate.adjpin.3 <- adjpin(xdata, restricted = list(d = TRUE),
                                  verbose = FALSE)

# Compare the different values of adjusted PIN

estimates <- list(estimate.adjpin.0, estimate.adjpin.1,
                  estimate.adjpin.2, estimate.adjpin.3)

adjpins <- sapply(estimates, function(x) x@adjpin)

psos <- sapply(estimates, function(x) x@psos)

summary <- cbind(adjpins, psos)
rownames(summary) <- c("unrestricted", "same.mu", "same.eps", "same.d")

show(round(summary, 5))
}
}
\references{
\insertAllCited
}
