% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/process-plate.R
\name{process_plate}
\alias{process_plate}
\title{Process a plate and save output values to a CSV}
\usage{
process_plate(
  plate,
  filename = NULL,
  output_dir = "normalised_data",
  write_output = TRUE,
  normalisation_type = "RAU",
  data_type = "Median",
  include_raw_mfi = TRUE,
  adjust_blanks = FALSE,
  verbose = TRUE,
  reference_dilution = 1/400,
  ...
)
}
\arguments{
\item{plate}{(\code{Plate()}) a plate object}

\item{filename}{(\code{character(1)}) The name of the output CSV file with normalised MFI values.
If not provided or equals to \code{NULL}, the output filename will be based on the normalisation type
and the plate name, precisely: \verb{\{plate_name\}_\{normalisation_type\}.csv}.
By default the \code{plate_name} is the filename of the input file that contains the plate data.
For more details please refer to \link[PvSTATEM]{Plate}.

If the passed filename does not contain \code{.csv} extension, the default extension \code{.csv} will be added.
Filename can also be a path to a file, e.g. \code{path/to/file.csv}. In this case, the \code{output_dir} and \code{filename} will be joined together.
However, if the passed filepath is an absolute path and the \code{output_dir} parameter is also provided, the \code{output_dir} parameter will be ignored.
If a file already exists under a specified filepath, the function will overwrite it.}

\item{output_dir}{(\code{character(1)}) The directory where the output CSV file should be saved.
Please note that any directory path provided will create all necessary directories (including parent directories) if they do not exist.
If it equals to \code{NULL} the current working directory will be used. Default is 'normalised_data'.}

\item{write_output}{(\code{logical(1)}) whether or not to write the output to a file
specified by \code{filename} parameter. The default is \code{TRUE}.}

\item{normalisation_type}{(\code{character(1)}) type of normalisation to use. Available options are:
\cr \code{c(RAU, nMFI)}.}

\item{data_type}{(\code{character(1)}) type of data to use for the computation. Median is the default}

\item{include_raw_mfi}{(\code{logical(1)}) include raw MFI values in the output. The default is \code{TRUE}.
In case this option is \code{TRUE}, the output dataframe contains two columns for each analyte: one for the normalised values and one for the raw MFI values.
The normalised columns are named as \code{AnalyteName} and \code{AnalyteName_raw}, respectively.}

\item{adjust_blanks}{(\code{logical(1)}) adjust blanks before computing RAU values. The default is \code{FALSE}}

\item{verbose}{(\code{logical(1)}) print additional information. The default is \code{TRUE}}

\item{reference_dilution}{(\code{numeric(1)}) target dilution to use as reference for the nMFI normalisation. Ignored in case of RAU normalisation.
Default is \code{1/400}.
It should refer to a dilution of a standard curve sample in the given plate object.
This parameter could be either a numeric value or a string.
In case it is a character string, it should have the format \verb{1/d+}, where \verb{d+} is any positive integer.}

\item{...}{Additional arguments to be passed to the fit model function (\code{create_standard_curve_model_analyte})}
}
\value{
a data frame with normalised values
}
\description{
Depending on the \code{normalisation_type} argument, the function will compute the RAU or nMFI values for each analyte in the plate.
\strong{RAU} is the default normalisation type.

The behaviour of the function, in the case of RAU normalisation type, can be summarised as follows:
\enumerate{
\item Adjust blanks if not already done.
\item Fit a model to each analyte using standard curve samples.
\item Compute RAU values for each analyte using the corresponding model.
\item Aggregate computed RAU values into a single data frame.
\item Save the computed RAU values to a CSV file.
}

More info about the RAU normalisation can be found in
\code{create_standard_curve_model_analyte} function documentation \link[PvSTATEM]{create_standard_curve_model_analyte} or in the Model reference \link[PvSTATEM]{Model}.

In case the normalisation type is \strong{nMFI}, the function will:
\enumerate{
\item Adjust blanks if not already done.
\item Compute nMFI values for each analyte using the target dilution.
\item Aggregate computed nMFI values into a single data frame.
\item Save the computed nMFI values to a CSV file.
}

More info about the nMFI normalisation can be found in \code{get_nmfi} function documentation \link[PvSTATEM]{get_nmfi}.
}
\examples{

plate_file <- system.file("extdata", "CovidOISExPONTENT_CO_reduced.csv", package = "PvSTATEM")
# a plate file with reduced number of analytes to speed up the computation
layout_file <- system.file("extdata", "CovidOISExPONTENT_CO_layout.xlsx", package = "PvSTATEM")

plate <- read_luminex_data(plate_file, layout_file, verbose = FALSE)

example_dir <- tempdir(check = TRUE) # a temporary directory
# create and save dataframe with computed dilutions
process_plate(plate, output_dir = example_dir)

# process plate without adjusting blanks and save the output to a file with a custom name
process_plate(plate,
  filename = "plate_without_blanks_adjusted.csv",
  output_dir = example_dir, adjust_blanks = FALSE
)


# nMFI normalisation
process_plate(plate,
  output_dir = example_dir,
  normalisation_type = "nMFI", reference_dilution = 1 / 400
)

}
