% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pkbd_functions.R
\name{dpkb}
\alias{dpkb}
\alias{rpkb}
\title{The Poisson kernel-based Distribution (PKBD)}
\usage{
dpkb(x, mu, rho, logdens = FALSE)

rpkb(
  n,
  mu,
  rho,
  method = "rejacg",
  tol.eps = .Machine$double.eps^0.25,
  max.iter = 1000
)
}
\arguments{
\item{x}{\eqn{n \times d}-matrix (or data.frame) of \eqn{n} data point on the
sphere \eqn{\mathcal{S}^{d-1}}, with \eqn{d \ge 2}.}

\item{mu}{location vector parameter with length indicating the dimension of
generated points.}

\item{rho}{Concentration parameter, with \eqn{0 \le} \code{rho} \eqn{< 1}.}

\item{logdens}{Logical; if 'TRUE', densities are returned in logarithmic
scale.}

\item{n}{number of observations.}

\item{method}{string that indicates the method used for sampling
observations. The available methods are
\itemize{
\item \code{'rejvmf'} acceptance-rejection algorithm using
von Mises-Fisher envelopes (Algorithm in Table 2 of
Golzy and Markatou 2020);
\item \code{'rejacg'} using angular central Gaussian envelopes
(Algorithm in Table 1 of Sablica et al. 2023);
\item \code{'rejpsaw'} using projected Saw distributions
(Algorithm in Table 2 of Sablica et al. 2023).
}}

\item{tol.eps}{the desired accuracy of convergence tolerance
(for 'rejacg' method).}

\item{max.iter}{the maximum number of iterations (for 'rejacg' method).}
}
\value{
\code{dpkb} gives the density value;
\code{rpkb} generates random observations from the PKBD.
}
\description{
The Poisson kernel-based densities are based on the normalized Poisson kernel
and are defined on the \eqn{(d-1)}-dimensional unit sphere. Given a vector
\eqn{\mathbf{\mu} \in \mathcal{S}^{d-1}}, where \eqn{\mathcal{S}^{d-1}= 
\{x \in \mathbb{R}^d : ||x|| = 1\}}, and a parameter \eqn{\rho} such that
\eqn{0 < \rho < 1}, the probability density function of a \eqn{d}-variate
Poisson kernel-based density is defined by:
\deqn{f(\mathbf{x}|\rho, \mathbf{\mu}) = \frac{1-\rho^2}{\omega_d 
||\mathbf{x} - \rho \mathbf{\mu}||^d},}
where \eqn{\mu} is a vector orienting the center of the distribution,
\eqn{\rho} is a parameter to control the concentration of the distribution
around the vector \eqn{\mu} and it is related to the variance of the
distribution. Recall that, for \eqn{x = (x_1, \ldots, x_d) \in \mathbb{R}^d},
\eqn{||x|| = \sqrt{x_1^2 + \ldots + x_d^2}}. Furthermore, \eqn{\omega_d =
2\pi^{d/2} [\Gamma(d/2)]^{-1}} is the surface area of the unit sphere in
\eqn{\mathbb{R}^d} (see Golzy and Markatou, 2020). When \eqn{\rho \to 0},
the Poisson kernel-based density tends to the uniform density on the sphere.
Connections of the PKBDs to other distributions are discussed in detail in
Golzy and Markatou (2020). Here we note that when \eqn{d=2}, PKBDs reduce to
the wrapped Cauchy distribution. Additionally, with precise choice of the
parameters \eqn{\rho} and \eqn{\mu} the two-dimensional PKBD becomes a
two-dimensional projected normal distribution. However, the connection with
the \eqn{d}-dimensional projected normal distributions does not carry beyond
\eqn{d=2}.
Golzy and Markatou (2020) proposed an acceptance-rejection method for
simulating data from a PKBD using von Mises-Fisher envelopes (\code{rejvmf}
method). Furthermore Sablica, Hornik and Leydold (2023) proposed new ways for
simulating from the PKBD, using angular central Gaussian envelopes
(\code{rejacg}) or using the projected Saw distributions (\code{rejpsaw}).
}
\details{
This function \code{dpkb()} computes the density value for a given point
\code{x} from the Poisson kernel-based distribution with mean direction
vector \code{mu} and concentration parameter \code{rho}.

The number of observations generated is determined by \code{n} for
\code{rpkb()}. This function returns the \eqn{(n \times d)}-matrix of
generated \eqn{n} observations on \eqn{\mathcal{S}^{(d-1)}}.

A limitation of the \code{rejvmf} is that the method does not ensure the
computational feasibility of the sampler for \eqn{\rho} approaching 1.

If the chosen method is 'rejacg', the function \code{uniroot}, from the
\code{stat} package, is used to estimate the beta parameter. In this case,
the complete results are provided as output.
}
\note{
If the required packages (\code{movMF} for \code{rejvmf} method, and
\code{Tinflex} for \code{rejpsaw}) are not installed, the function will display a
message asking the user to install the missing package(s).
}
\examples{
# Generate some data from pkbd density
pkbd_dat <- rpkb(10, c(0.5, 0), 0.5)

# Calculate the PKBD density values
dens_val <- dpkb(pkbd_dat, c(0.5, 0.5), 0.5)

}
\references{
Golzy, M. and Markatou, M. (2020) Poisson Kernel-Based Clustering on the
Sphere: Convergence Properties, Identifiability, and a Method of Sampling,
Journal of Computational and Graphical Statistics, 29:4, 758-770,
DOI: 10.1080/10618600.2020.1740713.

Sablica L., Hornik K. and Leydold J. (2023) "Efficient sampling from the PKBD
distribution", Electronic Journal of Statistics, 17(2), 2180-2209.
}
