
#' Distance between a point and a group
#'
#'
#' This function calculates the distance between the point i of the \code{data} matrix and all the components in the group \code{num}.
#'
#' @param data Matrix with \code{dim(data)[1]} points of \code{dim(data)[2]} dimensions.
#' @param i Number. Number of the row of \code{data} where the point is.
#' @param distance Function. This function designs how the distance is going to be calculated. It must have as input two vectors and as output the distance of these vectors.
#' @param FHW_output List. List with:
#'
#' \itemize{
#'   \item centers: the information of the centers updated.
#'   \item grouping: the information of the groups updated. List. Each component of the list contains a vector with the points that belong to that group. More specifically, the list component i has a vector with the numbers of the row of the matrix \code{data} where the points belonging to group i are.
#'    }
#'
#' @param num Number. Number of the group from \code{FHW_output$grouping}.
#'
#' @return Returns a vector. The component j contains the distance between the point in the row i of the \code{data} matrix and the point j of the group \code{num}.
#'
#'
#'
#'
#' @examples
#'
#'set.seed(451)
#'data=rbind(matrix(runif(20,1,5), nrow = 2, ncol = 10),
#'          matrix(runif(20,20,30), nrow = 2, ncol = 10),
#'          matrix(runif(20,50,70), nrow = 2, ncol = 10))
#'k=3
#'seed=5
#'
#'
#'FHW_output=Hartigan_and_Wong(data,
#'                             Euclideandistance,
#'                             k,
#'                             centers_function_mean,
#'                             init_centers_random,
#'                             seed=seed,
#'                             10)
#'
#'d_i_other_group(data, 1, Euclideandistance, FHW_output,2)
#'
#'
#'
#' @export
#'
d_i_other_group<-function(data, i, distance, FHW_output,num){

  Other_group=FHW_output$grouping[[num]]
  dGNoGi=rep(0,length(Other_group) )
  for(j in 1:length(Other_group)){
    dGNoGi[j]=distance(data[i,],data[Other_group[j],])
  }
  return(dGNoGi)
}

