#' slotlim_once
#'
#' @description
#' Run a single SlotLim pass: compute \code{rb}, \code{P}, \code{TBA}, \code{SAM},
#' and catch advice \code{Ay_percent}; optionally show a composite plot (\code{P}, \code{TBA}, \code{SAM}, \code{Ay_percent}).
#'
#' @param Cy Numeric. Historical catch.
#' @param b_index Numeric vector of a biomass or abundance index in descending time order
#'   (most recent first).
#' @param method Character. Method for calculating \code{rb} ("annual", "1over2", or "2over3").
#' @param minLS,maxLS,Lc Numeric. Slot limits and length at first capture.
#' @param growth_model One of \code{"vb"}, \code{"gompertz"}, \code{"schnute"}.
#' @param Linf,K,l0 von Bertalanffy (vB) parameters; \code{l0} is the start length (default 0).
#' @param Gom_Linf,Gom_K,Gom_l0 Gompertz parameters; requires \code{0 < Gom_l0 < Gom_Linf}.
#' @param g1,g2,l2 Schnute parameters; \code{l2} is length at \code{tmax}; requires \code{g1 > 0}, \code{l2 > 0},
#'   and this parameterization assumes \code{g2 != 0}.
#' @param tmax Numeric. Maximum observed age; used for integration bounds and (if \code{M} is \code{NULL})
#'   to compute default \code{M}.
#' @param M Numeric or `NULL`. Natural mortality. If `NULL`, defaults to \eqn{M = 4.899 \times tmax^{-0.916}}.
#' @param lower,upper Optional values at specified percentiles. If provided, used directly by \code{SAM()}.
#' @param LF Optional numeric vector of length-frequency data. If \code{lower}/\code{upper} are \code{NULL} and
#'   \code{LF} is supplied, the function computes percentiles via \code{percentile(LF, probs)} and uses them.
#' @param probs Numeric vector of probabilities in \eqn{[0,1]} passed to \code{percentile()} when \code{LF} is used.
#'   Default \code{c(0.025, 0.975)}.
#' @param constraint Numeric (default 1). Passed to \code{SAM()}.
#' @param T1,T2 Optional numerics passed to \code{catch_advice()}.
#' @param plots Logical; if \code{TRUE}, a \eqn{2 \times 2} composite plot is printed (if \pkg{patchwork} is available).
#' @param length_units Optional character; x-axis units for the \code{prop_target} and \code{SAM} plots (e.g., "mm").
#'
#' @details
#' Precedence for size inputs: if both \code{lower} and \code{upper} are provided, they are used.
#' Otherwise, if \code{LF} is provided, they are derived via \code{percentile(LF, probs)}. Else error.
#'
#' @return A list with \code{Ay}, \code{Ay_percent}, \code{TBA}, \code{SAM}, \code{rb}, \code{P},
#'   and (if \code{plots=TRUE}) a composite plot. Also returns the resolved \code{M} and the
#'   \code{lower}/\code{upper} bounds actually used; \code{tmax} is echoed back.
#'
#' @examples
#' # Minimal, fast example (no plotting), passing lower/upper directly:
#' slotlim_once(
#'   Cy = 1000,
#'   b_index = c(0.5, 0.6, 0.7, 0.6, 0.5), method = "2over3",
#'   minLS = 120, maxLS = 240, Lc = 80,
#'   growth_model = "vb", Linf = 405, K = 0.118, l0 = 0,
#'   tmax = 34,
#'   lower = 100, upper = 220
#' )
#'
#' \donttest{
#' # Derive lower/upper from length-frequency percentiles:
#' set.seed(1)
#' LF <- rnorm(200, mean = 180, sd = 40)  # toy example LF
#'
#' # Compute M from tmax:
#' slotlim_once(
#'   Cy = 1000,
#'   b_index = c(0.5, 0.6, 0.7, 0.6, 0.5),
#'   minLS = 120, maxLS = 240, Lc = 80,
#'   growth_model = "vb", Linf = 405, K = 0.118, l0 = 0,
#'   tmax = 34,
#'   LF = LF, probs = c(0.05, 0.95),
#'   method = "1over2"  # rb method chosen
#' )
#'
#' # Use explicit M (still provide tmax for bounds):
#' slotlim_once(
#'   Cy = 1000,
#'   b_index = c(0.5, 0.6, 0.7, 0.6, 0.5),
#'   minLS = 120, maxLS = 240, Lc = 80,
#'   growth_model = "vb", Linf = 405, K = 0.118, l0 = 0,
#'   tmax = 34,
#'   M = 0.19,
#'   LF = LF, probs = c(0.025, 0.975),
#'   method = "1over2"  # rb method chosen
#' )
#'
#' # Plotting example (needs ggplot2 and patchwork):
#' slotlim_once(
#'   Cy = 1000,
#'   b_index = c(0.5, 0.6, 0.7, 0.6, 0.5),
#'   minLS = 120, maxLS = 240, Lc = 80,
#'   growth_model = "vb", Linf = 405, K = 0.118, l0 = 0,
#'   tmax = 34,
#'   LF = LF, probs = c(0.025, 0.975),
#'   method = "1over2",  # rb method chosen
#'   plots = TRUE, length_units = "mm"
#' )
#' }
#'
#' @export
slotlim_once <- function(
    Cy = NULL,
    b_index = NULL,
    method = c("annual", "1over2", "2over3"),
    minLS = NULL, maxLS = NULL, Lc = NULL,
    growth_model = c("vb", "gompertz", "schnute"),
    # vB
    Linf = NULL, K = NULL, l0 = 0, tmax = NULL,
    # Gompertz
    Gom_Linf = NULL, Gom_K = NULL, Gom_l0 = NULL,
    # Schnute (l2 is length at tmax)
    g1 = NULL, g2 = NULL, l2 = NULL,
    # Mortality (optional; defaults via tmax if NULL)
    M = NULL,
    # SAM & advice
    lower = NULL, upper = NULL,
    LF = NULL, probs = c(0.025, 0.975),
    constraint = 1, T1 = NULL, T2 = NULL,
    # plot control
    plots = FALSE, length_units = NULL
) {
  growth_model <- match.arg(growth_model)

  # ---- validate core inputs ----
  if (!is.numeric(b_index) || length(b_index) < 3L)
    stop("`b_index` must be numeric with at least 3 values (for rb '1over2').", call. = FALSE)
  if (!is.numeric(Cy) || length(Cy) != 1L || !is.finite(Cy))
    stop("`Cy` must be a finite numeric scalar.", call. = FALSE)
  if (!is.numeric(tmax) || length(tmax) != 1L || !is.finite(tmax) || tmax <= 0)
    stop("`tmax` must be a positive finite numeric scalar.", call. = FALSE)

  # ---- resolve/validate M ----
  if (is.null(M)) M <- 4.899 * (tmax ^ -0.916)
  if (!is.numeric(M) || length(M) != 1L || !is.finite(M) || M <= 0)
    stop("Resolved `M` must be a positive finite numeric scalar.", call. = FALSE)

  # ---- rb using user-defined method ----
  rb_val <- rb(b_index = b_index, method = method, na.rm = FALSE, digits = NULL)

  # ---- targeted proportion (P) ----
  if (isTRUE(plots)) {
    P_out <- switch(
      growth_model,
      "vb" = prop_target(minLS = minLS, maxLS = maxLS, Lc = Lc,
                         growth_model = "vb", Linf = Linf, K = K, l0 = l0,
                         tmax = tmax, M = M, plot = TRUE, length_units = length_units),
      "gompertz" = prop_target(minLS = minLS, maxLS = maxLS, Lc = Lc,
                               growth_model = "gompertz", Gom_Linf = Gom_Linf, Gom_K = Gom_K,
                               Gom_l0 = Gom_l0, tmax = tmax, M = M,
                               plot = TRUE, length_units = length_units),
      "schnute" = prop_target(minLS = minLS, maxLS = maxLS, Lc = Lc,
                              growth_model = "schnute", g1 = g1, g2 = g2, l2 = l2,
                              tmax = tmax, M = M, plot = TRUE, length_units = length_units)
    )
    P_val  <- P_out$proportion
    P_plot <- P_out$plot
  } else {
    P_val <- switch(
      growth_model,
      "vb" = prop_target(minLS = minLS, maxLS = maxLS, Lc = Lc,
                         growth_model = "vb", Linf = Linf, K = K, l0 = l0,
                         tmax = tmax, M = M, plot = FALSE),
      "gompertz" = prop_target(minLS = minLS, maxLS = maxLS, Lc = Lc,
                               growth_model = "gompertz", Gom_Linf = Gom_Linf, Gom_K = Gom_K,
                               Gom_l0 = Gom_l0, tmax = tmax, M = M, plot = FALSE),
      "schnute" = prop_target(minLS = minLS, maxLS = maxLS, Lc = Lc,
                              growth_model = "schnute", g1 = g1, g2 = g2, l2 = l2,
                              tmax = tmax, M = M, plot = FALSE)
    )
    P_plot <- NULL
  }

  # ---- lower/upper derivation if needed ----
  lower_used <- lower; upper_used <- upper
  if (is.null(lower_used) || is.null(upper_used)) {
    if (is.null(LF))
      stop("Provide either `lower` and `upper`, or `LF` with `probs` to compute them.", call. = FALSE)
    pc <- percentile(LF, probs = probs)
    vals <- unlist(pc, use.names = FALSE)
    if (length(vals) < 2L)
      stop("`probs` must yield at least two percentiles (e.g., c(0.025, 0.975)).", call. = FALSE)
    lower_used <- vals[1]; upper_used <- vals[length(vals)]
    if (!is.finite(lower_used) || !is.finite(upper_used))
      stop("Percentile computation produced non-finite values; check `LF`.", call. = FALSE)
  }

  # ---- SAM ----
  if (isTRUE(plots)) {
    sam <- SAM(lower = lower_used, upper = upper_used,
               minLS = minLS, maxLS = maxLS,
               constraint = constraint, plot = TRUE,
               length_units = length_units)
    sam_plot <- sam$plot
  } else {
    sam <- SAM(lower = lower_used, upper = upper_used,
               minLS = minLS, maxLS = maxLS,
               constraint = constraint, plot = FALSE)
    sam_plot <- NULL
  }
  SAM_val <- sam$SAM

  # ---- TBA ----
  if (isTRUE(plots)) {
    tba <- TBA(P_targeted = P_val, rb = rb_val, plot = TRUE)
    tba_plot <- tba$plot
  } else {
    tba <- TBA(P_targeted = P_val, rb = rb_val, plot = FALSE)
    tba_plot <- NULL
  }
  TBA_val <- tba$TBA

  # ---- Advice ----
  if (isTRUE(plots)) {
    adv <- catch_advice(Cy = Cy, TBA = TBA_val, SAM = SAM_val,
                        T1 = T1, T2 = T2, plot = TRUE)
    adv_plot <- adv$plot
  } else {
    adv <- catch_advice(Cy = Cy, TBA = TBA_val, SAM = SAM_val,
                        T1 = T1, T2 = T2, plot = FALSE)
    adv_plot <- NULL
  }

  # ---- composite using wrap_plots ----
  composite <- NULL
  if (isTRUE(plots) &&
      requireNamespace("patchwork", quietly = TRUE) &&
      requireNamespace("ggplot2", quietly = TRUE)) {
    composite <- patchwork::wrap_plots(
      P_plot, tba_plot, sam_plot, adv_plot, ncol = 2
    ) + patchwork::plot_annotation(title = "")
    print(composite)
  } else if (isTRUE(plots)) {
    warning("Install 'patchwork' and 'ggplot2' to display the composite.", call. = FALSE)
  }

  # ---- return (visible) ----
  out <- list(
    Ay = adv$Ay,
    Ay_percent = adv$Ay_percent,
    TBA = TBA_val,
    SAM = SAM_val,
    rb = rb_val,
    P = P_val,
    M = M,
    tmax = tmax,
    lower = lower_used,
    upper = upper_used
  )
  if (isTRUE(plots)) out$plots <- list(composite = composite)
  out
}
