#' @importFrom geometry delaunayn

#' @title Core-center
#' @description Given a game, this function computes its core center.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param tol A tolerance parameter, as a non-negative number. By default, \code{tol=1e-12}.
#' @return The core-center, as a vector.
#' @details The core of a game \eqn{v\in G^N} is the set of all its stable imputations:
#' \deqn{C(v)=\{x\in\mathbb{R}^n : x(N)=v(N), x(S)\ge v(S)\ \forall S \in 2^N\},}
#' where \eqn{x(S)=\sum_{i\in S} x_i}. A game is said to be balanced if its core is not empty.
#'
#' The core-center of a balanced game \eqn{v}, \eqn{CC(v)}, is defined as the expectation
#' of the uniform distribution over \eqn{C(v)}, and thus can be interpreted
#' as the centroid or center of gravity of \eqn{C(v)}.
#' Let \eqn{\mu} be the \eqn{(n-1)}-dimensional Lebesgue measure and let \eqn{V(C)=\mu(C(v))}
#' be the volume (measure) of the core. If \eqn{V(C)>0}, then, for each \eqn{i\in N},
#' \deqn{CC_i(v)=\frac{1}{V(C)}\int_{C(v)}x_i d\mu}.
#' @examples
#' v1 <- claimsgame(E=8,d=c(3,5,6))
#' corecentervalue(v1)
#' plotcoreset(v1,solutions="corecenter")
#'
#' v2 <- c(0,0,0,0,0,0,0,0,1,4,1,3,6,8,10)
#' corecentervalue(v2)
#' plotcoreset(v2,solutions="corecenter")
#'
#' # What if the game is not full-dimensional because of a dummy player?
#' v3 <- c(440,0,0,0,440,440,440,15,14,7,455,454,447,60,500)
#' dummynull(v3) # player 1 is a dummy in v3, so the core is degenerate
#' # For coredimension to detect that, tolerance has to be appropriate:
#' coredimension(v=v3,tol=100*.Machine$double.eps) # tolerance too small
#' coredimension(v=v3) # default tolerance, 1e-12, big enough
#'
#' # Now how to compute the corecenter?
#' # When given a degenerate game, corecentervalue computes an approximation:
#' ( cc.approx <- corecentervalue(v=v3) ) # approximate core-center
#' # However, knowing that player 1 is a dummy and that the core-center assigns
#' # dummies their individual worth...
#' v3.without1 <- subgame(v=v3,S=14) # subgame without player 1
#' ( cc.exact <- c(v3[1],corecentervalue(v3.without1)) ) # "exact" core-center
#'
#' # Plotting both results:
#' plotcoreset(v3,allocations=rbind(cc.approx,cc.exact),projected=TRUE)
#' @references Gonzalez-Díaz, J. & Sánchez-Rodríguez, E. (2007). A natural selection from the core of a TU game: the core-center. International Journal of Game Theory, 36(1), 27-46.
#' @seealso \link{balancedcheck}, \link{corecenterhitrun}, \link{coredimension}, \link{corevertices}, \link{corevertices234}
#' @export

corecentervalue <- function(v, binary = FALSE, tol = 1e-12) {
  nC=length(v)#Número de coaliciones.
  n=log(nC+1)/log(2)#Número de jugadores.
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }
  if (binary == FALSE) {
    v <- lex2bin(v)
  }
  if (!balancedcheck(v,binary=TRUE)) {
    message("The core of 'v' is empty.")
    return(NULL)
  }

  ################################
  #####Creación de variables######
  ################################

  #Caso directo: n=2.
  if(n==2)
  {
    m <- c(max(0,v[3]-v[2]),max(0,v[3]-v[1]))
    V <- matrix(c(m[1],v[3]-m[1],v[3]-m[2],m[2]),ncol=2,byrow=T)
    CC <- colSums(V)/2
    return(CC)
  }
  #Si n>2, seguimos con la función.

  CC <- rep(0,n) #Inicializamos core-center

  ################################
  ######Cuerpo de la función######
  ################################

  #Calculamos los vértices del core y estudiamos los casos básicos.

  V <- corevertices(v,binary=TRUE)

  if(dim(V)[1]==1)
  {#Si el core es un punto, el core-center coincidirá con dicho punto.
    CC=as.vector(V)
  }else if(dim(V)[1]==2)
  {#Si es un segmento, calculamos el punto medio.
    CC=colSums(V)/2
  }else
  {#En cualquier otro caso:
    #Primero estudiamos si hay columnas iguales a 0 (colapsa un hiperplano).
    #Si las hay, las quitamos y trabajamos con una dimensión menos.
    #Si aplicamos la triangulación con un hiperplano colapsado, da error.
    nocolapso=which(abs(colSums(V))>tol)
    V=V[,nocolapso]
    #Por eficiencia, quitamos la última coordenada.
    V=V[,1:(dim(V)[2]-1)]
    #Realizamos la triangulación de Delaunay.
    #Options="Qt": Triangulated output.

    # Ojo ahora. Si core full-dimensional, usamos Qt. Si degenera, QJ.
    if (coredimension(v,binary=TRUE,tol=tol)<n-1.5) { # si degenera...
      message(paste(
        "The core of 'v' is not full-dimensional. A perturbation has been introduced",
        "to avoid numerical issues, so the result might not be fully accurate.",
        sep = "\n"
      ))
      P=delaunayn(V,options="QJ",output.options=T)
    } else { # si no degenera...
      P=delaunayn(V,options="Qt",output.options=T)
    }

    #Los vértices de cada triángulo vienen dados por P$tri.
    #Calculamos el centro de cada triángulo y lo multiplicamos por su volumen.
    #Trabajamos el los hiperplanos no colapsados.
    #Sumamos solo en las coordenadas no colapsadas, las otras son cero.
    for(i in 1:length(P$areas))
    {
      CC[nocolapso[-length(nocolapso)]]=CC[nocolapso[-length(nocolapso)]]+
        colSums(V[P$tri[i,],])*P$areas[i]/(length(nocolapso))
    }
    #Dividimos la ponderación por la suma del volumen total.
    CC=CC/sum(P$areas)
    #Falta aplicar la eficiencia:
    CC[n]=v[nC]-sum(CC)
  }

  return(CC)

}
